#include "gimme_config_h.h"

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#endif

#include "args.h"
#include "basic_info.h"
#include "cdstatus_output.h"

int basicInfo(const cdstatus_args * cdsargs)
{
    int status, type, drive;

    drive = open(cdsargs->drivename, O_RDONLY | O_NONBLOCK);
    if (drive == -1)
    {
		snprintf(output_buffer, OUTPUT_BUFFSIZE, "Error opening drive %s.", cdsargs->drivename);
        conditional_perror(CRITICAL, output_buffer);
        exit(EXIT_FAILURE);
    }

    status = ioctl(drive, CDROM_DRIVE_STATUS);
    if (status == -1)
    {
        conditional_perror(CRITICAL, "Error getting drive status (while performing ioctl)");
        exit(EXIT_FAILURE);
    }

    switch (status)
    {
    case CDS_NO_INFO:
        conditional_puts(WARNING, "No information is available (i.e. this function is not implemented on the drive.");
        exit(EXIT_SUCCESS);
    case CDS_NO_DISC:
        conditional_puts(WARNING, "No disc is in the drive.");
        exit(EXIT_SUCCESS);
    case CDS_TRAY_OPEN:
        conditional_puts(CRITICAL, "The drive tray is open (or no disc is inserted).  Please close it then retry.");
        exit(EXIT_SUCCESS);
    case CDS_DRIVE_NOT_READY:
        conditional_puts(CRITICAL, "The drive is not ready.");
        exit(EXIT_SUCCESS);
    case CDS_DISC_OK:
        conditional_puts(DEBUG, "The drive reads disc ok. Attempting to retrieve disc info.");
        break;
    default:
		conditional_printf(CRITICAL, "Error: Invalid status code returned: %d\n", status);
        exit(EXIT_FAILURE);
    }

    type = ioctl(drive, CDROM_DISC_STATUS);
    if (type == -1)
    {
        conditional_perror(CRITICAL, "Unexpected error reading disc info from drive, exiting");
        exit(EXIT_FAILURE);
    }

    switch (type)
    {
    case CDS_NO_INFO:
        conditional_puts(CRITICAL, "Unable to read disc type info.  Function not implemented.");
        exit(EXIT_FAILURE);
    case CDS_NO_DISC:
        conditional_puts(CRITICAL, "\"No disc in drive\" status returned.  Questionable since disc presence and drive readiness was just verified.");
        exit(EXIT_FAILURE);
    case CDS_AUDIO:
        conditional_puts(DEBUG, "Disc appears to be an audio cd.");
        break;
    case CDS_DATA_1:
        conditional_puts(NORMAL, "Disc appears to be a DATA MODE 1 cd.");
        break;
    case CDS_DATA_2:
        conditional_puts(NORMAL, "Disc appears to be a DATA MODE 2 cd.");
        break;
    case CDS_XA_2_1:
        conditional_puts(NORMAL, "Disc appears to be of type XA 2 1");
        break;
    case CDS_XA_2_2:
        conditional_puts(NORMAL, "Disc appears to be of type XA 2 2");
        break;
    case CDS_MIXED:
        conditional_puts(NORMAL, "Disc appears to be MIXED MODE. Will try to gather info on contained audio tracks.");
        break;
    default:
		conditional_printf(WARNING, "Unknown type returned: %d.  Possible error occured.\n", type);
    }
    (void)close(drive);
    return type;
}
