/* TODO: add config file handling for --max-read-sets and 
 * --identical-reads */

#include "gimme_config_h.h"

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "cdstatus.h"
#include "handle_args.h"
#include "print_help.h"
#include "cdstatus_output.h"

#define MAX_CONFIG_LINE_LENGTH 1024

static void processConfigFile(cdstatus_args * cdsargs);
static void processCommandLine(int argc, char *argv[], cdstatus_args * cdsargs);
static void initializeDefaults(cdstatus_args * cdsargs);

void handleArgs(int argc, char *argv[], cdstatus_args * cdsargs)
{

	initializeDefaults(cdsargs);
	/* process the config file first, so that the command line overrides the config
	 * file options */
	processConfigFile(cdsargs);
	processCommandLine(argc, argv, cdsargs);

	return;
}

static void processConfigFile(cdstatus_args * cdsargs)
{
	char *home_path;
	char config_name[CDSTATUS_PATH_MAX];
	char config_line[MAX_CONFIG_LINE_LENGTH];
	struct stat config_stat;
	FILE *config_file;
	char *optname;
	char *optvalue;
	struct passwd *userinfo;

	config_name[0] = '\0';
	config_line[0] = '\0';

	userinfo = getpwuid(getuid());

	/* .cdstatus config values */
	home_path=userinfo->pw_dir;
	if (home_path==NULL )
	{
		/* can't determine home directory, so not much point in looking for config... */
		return;
	}
	else if(strlen(home_path) > (sizeof(config_name)-strlen(".cdstatus")))
	{
		/* return again... we won't be able to build the file path if it exceeds the
		 * MAX_PATH plus '.cdstatus' */
		return;
	}

	(void)snprintf(config_name, sizeof(config_name)-1, "/%s/.cdstatus",home_path);
	config_name[sizeof(config_name)-1]='\0';
	conditional_printf(DEBUG, "Using config file %s\n", config_name);

	if (stat(config_name, &config_stat) == -1)
	{
		if( errno != ENOENT)
		{
 			conditional_perror(WARNING, "Error checking for config file, not using");
		}
		else
		{
			conditional_puts(DEBUG, "Config file not found, not using.");
		}
		return;
	}
	
	if(!S_ISREG(config_stat.st_mode))
	{
		conditional_puts(WARNING, "Location pointed to by $HOME/.cdstatus is not a regular file. Ignoring.");
		return;
	}

	config_file = fopen(config_name, "r");
	if (config_file == NULL)
	{
		conditional_perror(WARNING, "Error opening config file, will not be used");
		return;
	}

	/* past the point where we don't have a config file */
	while (fgets(config_line, sizeof(config_line), config_file))
	{
		if ((config_line[0] == ';') || (config_line[0] == '\n') || (config_line[0] == ' ') || (config_line[0] == '#'))
		{
			continue;
		}
		if (strchr(config_line, '\n'))
		{
			*(strchr(config_line, '\n')) = '\0';
		}
		optvalue = strchr(config_line, '=');
		if (optvalue == NULL)
		{
			continue;
		}
		*optvalue = '\0';
		++optvalue;
		optname = config_line;

		if (strlen(optvalue) == 0)
		{
			conditional_printf(WARNING, "Empty configuration value in user config file .cdstatus (%s) skipped/ignored.\n", optname);
			continue;
		}

		if (strncmp("drive", optname, strlen("drive"))==0)
		{
			strncpy(cdsargs->drivename, optvalue, sizeof(cdsargs->drivename)-1);
			cdsargs->drivename[sizeof(cdsargs->drivename)-1]='\0';
		}
		else if (strncmp("outputdir", optname, strlen("outputdir"))==0)
		{
			strncpy(cdsargs->outputdir, optvalue, sizeof(cdsargs->outputdir)-1);
			cdsargs->outputdir[sizeof(cdsargs->outputdir)-1]='\0';
		}
		else if (strncmp("retries", optname, strlen("retries"))==0)
		{
			cdsargs->max_retries = atoi(optvalue);
		}
		else if (strncmp("basename", optname, strlen("basename"))==0)
		{
			strncpy(cdsargs->basename, optvalue, sizeof(cdsargs->basename)-1);
			cdsargs->basename[sizeof(cdsargs->basename)-1]='\0';
		}
		else if (strncmp("readchunk", optname, strlen("readchunk"))==0)
		{
			cdsargs->read_chunk_size = atoi(optvalue);
		}
		else if (strncmp("cddb_site", optname, strlen("cddb_site"))==0)
		{
			strncpy(cdsargs->cddb_site, optvalue, sizeof(cdsargs->cddb_site)-1);
			cdsargs->cddb_site[sizeof(cdsargs->cddb_site)-1]='\0';
		}
		else if (strncmp("no-cddb", optname, strlen("no-cddb"))==0)
		{
			cdsargs->cddb = 0;
		}
		else if (strncmp("cddb", optname, strlen("cddb"))==0)
		{
			if (strncmp(optvalue, "yes", strlen("yes"))==0)
			{
				cdsargs->cddb = 1;
			}
			else if (strncmp(optvalue, "no", strlen("no"))==0)
			{
				cdsargs->cddb = 0;
			}
			else
			{
				conditional_printf(WARNING, "Invalid cddb option value in .cdstatus config file: %s\n", optvalue);
			}
		}
		else if (strncmp("format", optname, strlen("format"))==0)
		{
            if(optvalue[0]=='"')
            {
                strncpy(cdsargs->format, optvalue+1, sizeof(cdsargs->format)-1);
                cdsargs->format[sizeof(cdsargs->format)-1]='\0';
            }
            else
            {
			    strncpy(cdsargs->format, optvalue, sizeof(cdsargs->format)-1);
				cdsargs->format[sizeof(cdsargs->format)-1] = '\0';
            }
			cdsargs->format_on = 1;
		}
		else if (strncmp("default_first_match", optname, strlen("default_first_match"))==0)
		{
			if (strncmp(optvalue, "yes", strlen("yes"))==0)
			{
				cdsargs->default_first_match = 1;
			}
			else if (strncmp(optvalue, "no", strlen("no"))==0)
			{
				cdsargs->default_first_match = 0;
			}
			else
			{
				conditional_printf(WARNING, "Invalid value for default_first_match (%s) in .cdstatus config file.\n", optvalue);
			}
		}
		else if (strncmp("rip", optname, strlen("rip"))==0)
		{
			if (strncmp(optvalue, "yes", strlen("yes"))==0)
			{
				cdsargs->readtest = 1;
			}
			else if (strncmp(optvalue, "no", strlen("no"))==0)
			{
				cdsargs->readtest = 0;
			}
			else
			{
				conditional_printf(WARNING, "Invalid rip option value in .cdstatus config file: %s\n", optvalue);
			}
		}
		else if (strncmp("lame", optname, strlen("lame"))==0)
		{
			if (strncmp(optvalue, "yes", strlen("yes"))==0)
			{
				cdsargs->encoder = ENC_LAME;
                cdsargs->readtest = 1;
				conditional_puts(WARNING, "Please notice that the lame and lameopts options are deprecated. They will disappear in a future version." \
					" Please make use of the new, equivalent encoder=lame and encopts options.");
			}
			else if (strncmp(optvalue, "no", strlen("no"))==0)
			{
				cdsargs->encoder = ENC_WAVE;
				conditional_puts(WARNING, "Please note that the lame and lameopts options are deprecated. They will disappear in a future version." \
					" The equivalent is encoder=encoder-name and encopts=whatever. For now, when setting lame=no, the WAVE 'encoder' will be used.");
			}
			else
			{
				conditional_printf(WARNING, "Invalid lame option value in .cdstatus config file: %s\n", optvalue);
			}
		}
		else if (strncmp("lameopts", optname, strlen("lameopts"))==0)
		{
			strncpy(cdsargs->encopts, optvalue, sizeof(cdsargs->encopts)-1);
			cdsargs->encopts[sizeof(cdsargs->encopts)-1] = '\0';
			conditional_puts(WARNING, "Please note that the lame and lameopts options are deprecated and will be removed.  Use encoder=lame and encopts" \
				" instead.");
		}
		else if (strncmp("encoder", optname, strlen("encoder"))==0)
		{
			/* check for different valid encoders */
			if(strncmp(optvalue, "null", strlen("null"))==0)
			{
				cdsargs->encoder=ENC_NULL;
			}
			else if(strncmp(optvalue, "raw", strlen("raw"))==0)
			{
				cdsargs->encoder=ENC_RAW;
			}
			else if(strncmp(optvalue, "wave", strlen("wave"))==0)
			{
				cdsargs->encoder=ENC_WAVE;
			}
			else if(strncmp(optvalue, "lame", strlen("lame"))==0)
			{
				cdsargs->encoder=ENC_LAME;
			}
			else if(strncmp(optvalue, "toolame", strlen("toolame"))==0)
			{
				cdsargs->encoder=ENC_TOOLAME;
			}
			else if(strncmp(optvalue, "oggenc", strlen("oggenc"))==0)
			{
				cdsargs->encoder=ENC_OGGENC;
			}
			else
			{
				conditional_printf(WARNING, "Invalid encoder output format chosen in config file: %s, "
					"reverting to default.\n", optvalue);
			}
		}
	}
	return;
}

static void processCommandLine(int argc, char *argv[], cdstatus_args * cdsargs)
{
	int current_option;

#ifdef HAVE_GETOPT_H
	struct option long_options[] =
	{
		{"basename",		required_argument,	0, 'b' },
		{"cddb",			no_argument,		0, 'c' },
		{"debug",			no_argument,		0, 'D' },
		{"drive",			required_argument,	0, 'd' },
		{"encoder",			required_argument,	0, 'e' },
		{"encopts",			required_argument, 	0, 'E' },
		{"format",			required_argument,	0, 'f' },
		{"help",			no_argument, 		0, 'h' },
		{"identical-reads", required_argument,	0, 'i' },
		{"lame",			no_argument,		0, 'l' },
		{"lameopts",		required_argument,	0, 'L' },
		{"max-read-sets",	required_argument,	0, 'm' },
		{"no-cddb",			no_argument,		0, 'C' },
		{"no-mangle",		no_argument,		0, 'M' },
		{"nomangle",		no_argument,		0, 'M' },
		{"outputdir",		required_argument,	0, 'o' },
		{"readchunk",		required_argument,	0, 'k' },
		{"readtest",		no_argument,		0, 'r' },
		{"rip",				no_argument,		0, 'r' },
		{"reset",			no_argument, 		0, 'R' },
		{"retries",			required_argument,	0, 't' },
		{"start",			required_argument,	0, 's' },
		{"stop",			required_argument,	0, 'S' },
		{"silent",			no_argument,		0, 'q' },
		{"version",			no_argument,		0, 'V' }
	};

	while((current_option = getopt_long(argc, argv, "b:cDd:e:E:f:hi:lL:m:CMMo:k:rRt:s:S:qV", long_options, NULL)) != -1)
#else
	while((current_option = getopt(argc, argv, "b:cDd:e:E:f:hi:lL:m:CMMo:k:rRt:s:S:qV")) != -1)
#endif
	{
		switch(current_option)
		{
			case 'R':
				if(argc==2)
				{
					cdsargs->reset = 1;
				}
				else
				{
					conditional_puts(CRITICAL, "Cannot use device reset in conjunction with other options.");
					exit(EXIT_FAILURE);
				}
				break;
			case 'r':
				cdsargs->readtest = 1;
				break;
			case 's':
				cdsargs->start = atoi(optarg);
				break;
			case 'S':
				cdsargs->stop = atoi(optarg);
				break;
			case 't':
				cdsargs->max_retries = atoi(optarg);
				break;
			case 'i':
				cdsargs->identical_reads = atoi(optarg);
				break;
			case 'm':
				cdsargs->max_read_sets = atoi(optarg);
				break;
			case 'd':
				strncpy(cdsargs->drivename, optarg, sizeof(cdsargs->drivename) - 1);
				cdsargs->drivename[sizeof(cdsargs->drivename) - 1] = '\0';
				break;
			case 'o':
				strncpy(cdsargs->outputdir, optarg, sizeof(cdsargs->outputdir) - 1);
				cdsargs->outputdir[sizeof(cdsargs->outputdir) - 1] = '\0';
				break;
			case 'b':
				strncpy(cdsargs->basename, optarg, sizeof(cdsargs->basename)-1);
				cdsargs->basename[sizeof(cdsargs->basename)-1] = '\0';
				break;
			case 'k':
				cdsargs->read_chunk_size = atoi(optarg);
				break;
			case 'c':
				cdsargs->cddb = 1;
				break;
			case 'C':
				cdsargs->cddb = 0;
				break;
			case 'f':
				cdsargs->format_on = 1;
				strncpy(cdsargs->format, optarg, sizeof(cdsargs->format)-1);
				cdsargs->format[sizeof(cdsargs->format)-1] = '\0';
				break;
			case 'M':
				cdsargs->noMangle = 1;
				break;
			case 'q':
				current_priority = WARNING;
				break;
			case 'D':
				current_priority = DEBUG;
				break;
			case 'l':
				cdsargs->encoder = ENC_LAME;
				cdsargs->readtest = 1;
				conditional_puts(WARNING, "Please use the 'lame' argument to -e or --encoder instead "
					"of -l to specify lame. -l will be going away.");
				break;
			case 'L':
				strncpy(cdsargs->encopts, optarg, sizeof(cdsargs->encopts)-1);
				cdsargs->encopts[sizeof(cdsargs->encopts)-1] = '\0';
				conditional_puts(WARNING, "Please use -E to specify encopts instead of the deprecated "
					"-L for lameopts.");
				break;
			case 'e':
				if(strcmp(optarg, "null")==0)
				{
					cdsargs->encoder=ENC_NULL;
				}
				else if(strcmp(optarg, "raw")==0)
				{
					cdsargs->encoder=ENC_RAW;
				}
				else if(strcmp(optarg, "wave")==0)
				{
					cdsargs->encoder=ENC_WAVE;
				}
				else if(strcmp(optarg, "lame")==0)
				{
					cdsargs->encoder=ENC_LAME;
				}
				else if(strcmp(optarg, "toolame")==0)
				{
					cdsargs->encoder=ENC_TOOLAME;
				}
				else if(strcmp(optarg, "oggenc")==0)
				{
					cdsargs->encoder=ENC_OGGENC;
				}
				else
				{
					conditional_printf(WARNING, "Invalid argument to encoder option: %s.", optarg);
				}
				break;
			case 'E':
				strncpy(cdsargs->encopts, optarg, sizeof(cdsargs->encopts)-1);
				cdsargs->encopts[sizeof(cdsargs->encopts)-1] = '\0';
				break;
			case 'h':
				printHelp();
				exit(EXIT_SUCCESS);
				break;
			case 'V':
				/* main.c already emits version. we just have to exit without
				 * doing anything further */
				exit(EXIT_SUCCESS);
			case '?':
				conditional_puts(CRITICAL, "Invalid option selected.  Please verify options, seeing -h for details.");
				exit(EXIT_FAILURE);
				break;
			default:
				conditional_puts(CRITICAL, "Somehow fell through all option handling code. Possible bug in getopt(). Aborting!");
				exit(EXIT_FAILURE);
		}
	}

	if (cdsargs->reset!=0)
	{
		if (cdsargs->readtest!=0)
		{
			conditional_puts(CRITICAL, "Error, cannot use data extraction and --reset.");
			exit(EXIT_FAILURE);
		}
	}

	if (cdsargs->max_retries < 0)
	{
		conditional_puts(WARNING, "Warning: negative value specified for retry count, setting to 0.");
		cdsargs->max_retries = 0;
	}
	/* The reason that 75 is special is that it is the number of 'frames' in
	 * a second of cd audio, for reading in Minute-Second-Frame mode. Most
	 * drives don't allow reading more than a full second of audio data at
	 * once. */
	if (cdsargs->read_chunk_size > 75)
	{
		conditional_puts(WARNING, "Warning: readchunk value over 75 will not work properly on most hardware.  Dropping to 75.");
		cdsargs->read_chunk_size = 75;
	}
	else if (cdsargs->read_chunk_size <= 0)
	{
		conditional_puts(WARNING, "Warning: readchunk value of 0 or below will cause all kinds of problems.  Fixing to 1.");
		cdsargs->read_chunk_size = 1;
	}

	if (cdsargs->stop < cdsargs->start)
	{
		conditional_puts(WARNING, "--stop value must be greater than or equal to --start value (which defaults to 1)");
		exit(EXIT_FAILURE);
	}
	if (cdsargs->start < 1)
	{
		conditional_puts(WARNING, "--start value must be greater than or equal to 1");
		exit(EXIT_FAILURE);
	}

	return;
}

static void initializeDefaults(cdstatus_args * cdsargs)
{
	strncpy(cdsargs->outputdir, ".", sizeof(cdsargs->outputdir)-1);
	cdsargs->outputdir[sizeof(cdsargs->outputdir)-1]='\0';
	strncpy(cdsargs->basename, "track", sizeof(cdsargs->basename)-1);
	cdsargs->basename[sizeof(cdsargs->basename)-1] = '\0';
	strncpy(cdsargs->drivename, "/dev/cdrom", sizeof(cdsargs->drivename)-1);
	cdsargs->drivename[sizeof(cdsargs->drivename)-1] = '\0';
	strncpy(cdsargs->cddb_site, "freedb.freedb.org", sizeof(cdsargs->cddb_site)-1);
	cdsargs->cddb_site[sizeof(cdsargs->cddb_site)-1] = '\0';
	strncpy(cdsargs->encopts, "", sizeof(cdsargs->encopts)-1);
	cdsargs->encopts[sizeof(cdsargs->encopts)-1] = '\0';
	cdsargs->read_chunk_size = 1;
	cdsargs->max_retries = 3;
	cdsargs->cddb = 0;
	cdsargs->default_first_match = 0;
	cdsargs->start = 1;
	cdsargs->stop = 99;
	cdsargs->encoder = ENC_WAVE;
	cdsargs->readtest = 0;
	cdsargs->format_on = 0;
	strncpy(cdsargs->format, "#Artist#/#Album#/#Artist#_-_#Album#_-_#TrkNum#_-_#TrkName#", sizeof(cdsargs->format)-1);
	cdsargs->format[sizeof(cdsargs->format)-1] = '\0';
	cdsargs->noMangle = 0;
	cdsargs->identical_reads=1;
	cdsargs->max_read_sets=1;
	current_priority = NORMAL;
}
