#include "gimme_config_h.h"

/* splint barfs on trying to parse these files, so we pretend we don't need
 * them if we're running under splint so that it doesn't error out */
#ifndef S_SPLINT_S

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#endif

/* S_SPLINT_S */
#endif

#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif

#include "cd_toc_info.h"
#include "cdstatus_output.h"
#include "read_toc.h"

/* These are all defined in <linux/cdrom.h>.  If for some reason we don't
 * have that file, we can attempt to use their normal values from it. This
 * probably won't work, because the absence of the file usually will mean
 * we are working on a different (non-linux) platform where these values
 * won't hold true.  Worth trying though... */
#ifndef HAVE_LINUX_CDROM_H
#ifndef CDROM_MSF
#define CDROM_MSF 0x02
#endif

#ifndef CDROMREADTOCHDR
#define CDROMREADTOCHDR 0x5305
#endif

#ifndef CDROMREADTOCENTRY
#define CDROMREADTOCENTRY 0x5306
#endif

#ifndef CD_FRAMES
#define CD_FRAMES 75
#endif

#ifndef CDROM_DATA_TRACK
#define CDROM_DATA_TRACK 0x04
#endif

#ifndef CDROM_LEADOUT
#define CDROM_LEADOUT 0xAA
#endif

#ifndef CDROMVOLREAD
#define CDROMVOLREAD 0x5313
#endif
/* HAVE_LINUX_CDROM_H */
#endif

/* Yes this is silly. The value will never change. The intent is to disambiguate
 * that places where it is used, it is referring to the number of seconds in a
 * minute, and not some other arbitrary (unclear) usage of the number 60. */
#define SECONDS_PER_MINUTE 60

int readTOC(int drive, cd_toc_info * cdtocinfo, struct cdrom_tochdr *cdtochdr)
{
	struct cdrom_tocentry cdtocentry;
	struct cdrom_volctrl cdvolume;
	unsigned int counter;
	int status;

	memset(&cdtocentry, 0, sizeof(cdtocentry));
	memset(&cdvolume, 0, sizeof(cdvolume));

	conditional_puts(DEBUG, "Reading table of contents.");
	status = ioctl(drive, CDROMREADTOCHDR, cdtochdr);
	if (status == -1)
	{
		conditional_perror(CRITICAL, "Unable to read TOC");
		return errno;
	}

	conditional_printf(DEBUG, "Disc appears to have %d tracks. Querying for track info.", cdtochdr->cdth_trk1);

	for (counter = cdtochdr->cdth_trk0; counter <= cdtochdr->cdth_trk1; ++counter)
	{
		cdtocentry.cdte_format = CDROM_MSF;
		cdtocentry.cdte_track = counter;
		status = ioctl(drive, CDROMREADTOCENTRY, &cdtocentry);
		if (status == -1)
		{
			snprintf(output_buffer, OUTPUT_BUFFSIZE, "Error querying track info for track %u\n", counter);
			conditional_perror(WARNING, output_buffer);
			return errno;
		}
		cdtocinfo[counter].min = cdtocentry.cdte_addr.msf.minute;
		cdtocinfo[counter].sec = cdtocentry.cdte_addr.msf.second;
		cdtocinfo[counter].frame_local = cdtocentry.cdte_addr.msf.frame;
		cdtocinfo[counter].frame_global = cdtocinfo[counter].frame_local;
		(cdtocinfo[counter].frame_global) += ((cdtocinfo[counter].min) * CD_FRAMES * SECONDS_PER_MINUTE);
		(cdtocinfo[counter].frame_global) += ((cdtocinfo[counter].sec) * CD_FRAMES);
		cdtocinfo[counter].frame_global-=CD_MSF_OFFSET; 
		cdtocinfo[counter].data = (cdtocentry.cdte_ctrl & CDROM_DATA_TRACK);

		conditional_printf(DEBUG, "Track %u starts at: minute: %u, second: %u, frame: %u; global frame: %u\n", counter, cdtocinfo[counter].min, cdtocinfo[counter].sec, cdtocinfo[counter].frame_local, cdtocinfo[counter].frame_global);
	}

	conditional_puts(DEBUG, "Gathering info on leadout track.");
	cdtocentry.cdte_format = CDROM_MSF;
	cdtocentry.cdte_track = CDROM_LEADOUT;

	status = ioctl(drive, CDROMREADTOCENTRY, &cdtocentry);
	if (status == -1)
	{
		conditional_perror(WARNING, "Error querying track info for leadout track");
		return errno;
	}
	else
	{
		cdtocinfo[0].min = cdtocentry.cdte_addr.msf.minute;
		cdtocinfo[0].sec = cdtocentry.cdte_addr.msf.second;
		cdtocinfo[0].frame_local = cdtocentry.cdte_addr.msf.frame;
		cdtocinfo[0].frame_global = cdtocinfo[0].frame_local;
		(cdtocinfo[0].frame_global) += ((cdtocinfo[0].min) * CD_FRAMES * SECONDS_PER_MINUTE);
		(cdtocinfo[0].frame_global) += ((cdtocinfo[0].sec) * CD_FRAMES);
		cdtocinfo[0].frame_global-=CD_MSF_OFFSET; 

		conditional_printf(DEBUG, "Leadout track starts at: minute: %u, second: %u, frame: %u; global frame: %u\n", cdtocinfo[0].min, cdtocinfo[0].sec, cdtocinfo[0].frame_local, cdtocinfo[0].frame_global);
	}

	conditional_puts(DEBUG, "Querying drive volume settings:");
	status = ioctl(drive, CDROMVOLREAD, &cdvolume);
	if (status == -1)
	{
		conditional_perror(WARNING, "Unable to read volume settings");
		return errno;
	}
	conditional_printf(NORMAL, "Volume settings for drive: Channel 1: %d, Channel 2: %d, Channel 3: %d, Channel 4: %d\n", cdvolume.channel0, cdvolume.channel1, cdvolume.channel2, cdvolume.channel3);
	
	return 0;
}
