// Copyright 2023-2024 Gentoo Authors
// Distributed under the terms of the GNU General Public License v2

namespace Gdmt.Ebuild

module Private =
    open System
    open System.IO

    open SimpleLog.SimpleLog

    open Gdmt.Shared
    open Gdmt.Shared.Types

    open Gdmt.Ebuild.Entities

    let FindEbuildFiles (path: string) : StringArray =
        match path with
        | p when Directory.Exists p -> Directory.GetFiles(path, "*.ebuild")
        | p when File.Exists p -> [| p |]
        | _ -> $"The ebuild file does not exist, given {path}" |> Exception |> raise


    let expandEbuildCommandAbbreviation (ebuildCommand: String) =
        let abbreviations =
            Map(
                [ ("cg", "config")
                  ("cl", "clean")
                  ("co", "compile")
                  ("cu", "configure")
                  ("d", "digest")
                  ("f", "fetch")
                  ("h", "help")
                  ("i", "install")
                  ("ma", "manifest")
                  ("me", "merge")
                  ("pa", "package")
                  ("pe", "prerm")
                  ("po", "postinst")
                  ("pp", "prepare")
                  ("pr", "preinst")
                  ("ps", "postrm")
                  ("q", "qmerge")
                  ("r", "rpm")
                  ("s", "setup")
                  ("t", "test")
                  ("um", "unmerge")
                  ("up", "unpack") ]
            )

        match ebuildCommand with
        | k when abbreviations.ContainsKey k -> abbreviations[k]
        | unabbreviated -> unabbreviated

    let RunEbuildCommands
        (ebuild: string)
        (ebuildCommands: StringArray)
        (envMap: ProcessEnvironmentMap)
        (niceness: int)
        : unit =
        let args =
            [ "nice"; "-n"; $"{niceness}"; "ebuild"; ebuild ]
            @ (ebuildCommands |> Array.map expandEbuildCommandAbbreviation |> Array.toList)

        LogMessage Debug $"Running ebuild command for {ebuild}"

        try
            ExecProcess(args, envMap).Run().Check()

            LogMessage Success $"Ebuild command for {ebuild} finished successfully"
        with _ex ->
            LogMessage Error $"Ebuild command for {ebuild} failed"

            reraise ()

    let MakeCleanedEnvTuple (envVar: string) : EnvVarPair = (envVar, "")

    let EbuildCleanedEnvMap: EnvVarMap =
        [ "ANT_HOME"
          "ANT_RESPECT_JAVA_HOME"
          "CARGO_HOME"
          "CONAN_USER_HOME"
          "DBUS_SESSION_BUS_ADDRES"
          "DISPLAY"
          "GOPATH"
          "GPG_TTY"
          "GRADLE_USER_HOME"
          "HEX_HOME"
          "HOME"
          "ICEAUTHORITY"
          "IPYTHONDIR"
          "JULIA_DEPOT_PATH"
          "JUPYTER_CONFIG_DIR"
          "MIX_HOME"
          "NIMBLE_DIR"
          "NPM_CONFIG_USERCONFIG"
          "OPAMROOT"
          "PINENTRY_USER_DATA"
          "PIP_CACHE_DIR"
          "PYLINTHOME"
          "XDG_CACHE_HOME"
          "XDG_CONFIG_HOME"
          "XDG_DATA_DIRS"
          "XDG_DATA_HOME"
          "XDG_LOG_HOME"
          "XDG_STATE_HOME"
          "_JAVA_OPTIONS" ]
        |> List.map MakeCleanedEnvTuple

    let RunEbuildCommandsForEach
        (ebuildFiles: StringArray)
        (ebuildCommands: StringArray)
        (portageSettings: PortageSettings)
        (processSettings: ProcessSettings)
        : unit =
        let envMap =
            portageSettings.GetPortageEnv()
            @ EbuildCleanedEnvMap
            @ [ ("MAKEOPTS", $"-j {processSettings.Jobs} -l {processSettings.LoadAverage}") ]

        LogMessage Debug $"Enabled Portage FEATURES:    {portageSettings.GetFeatures}"
        LogMessage Debug $"Enabled Portage USE flags:   {portageSettings.GetUse}"
        LogMessage Debug $"Build process niceness:      {processSettings.Niceness}"
        LogMessage Debug $"Build process jobs:          {processSettings.Jobs}"
        LogMessage Debug $"Build process load average:  {processSettings.LoadAverage}"

        for ebuild in ebuildFiles do
            RunEbuildCommands ebuild ebuildCommands envMap processSettings.Niceness
