/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2014  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

class CustomPattern : Pattern {
    public Piece[] board = new Piece[BOARD_SIZE];

    public CustomPattern () {
        base.empty ("");
        for (int i=0; i < board.length; i++) {
            board[i] = Piece.BLACK;
        }
        update ();
    }

    public CustomPattern.from_text (string dots) {
        base.empty ("");
        Game.compile (dots, ref board);
        update ();
    }
        
    /**
     * Make a copy that can be used by the pattern selector.
     * In particular, dots must be compilable.
     */
    public CustomPattern.copy (CustomPattern original) {
        base.empty ("");
        assert (original.count_whites () == 60);
        board = original.board;
        update ();
    }

    protected override void background (int n, int x, int y) {
        pixbuf_data[n + 0] = 0;
        pixbuf_data[n + 1] = 
        pixbuf_data[n + 2] = 150 - 30 / (3 - (x + y) % 3);
        pixbuf_data[n + 3] = 255;
    }

    /**
     * Construct a textual representation consistent with the contents of board[].
     */
    private void board_to_dots () {
        dots = "";
        for (int i=0; i < board.length / 2; i++) {
            var x = i % BOARD_WIDTH;
            var y = i / BOARD_WIDTH;
            if (x >= 1 && x <= 11 && y >= 1 && y <= 11)
                dots += board[i] == Piece.WHITE ? "P" : "H";
        }
    }

    public int count_whites () {
        var num_whites = 0;
        foreach (var p in board)
            if (p == Piece.WHITE) num_whites++;
        return num_whites;
    }

    /**
     * Make this pattern internally consistent (called after changing board[]).
     * (1) Update dots.
     * (2) Update pixbuf_data with the image of the pattern and a meter reading
     *     to indicate the number of whites and blacks. The meter reading is removed
     *     when there are exactly 60 whites and 60 blacks.
     */
    public void update () {
        board_to_dots ();
        create_pixbuf ();
    }

    protected override void create_pixbuf () {
        create_pixbuf_from_board (ref board);

        var num_whites = count_whites ();
        if (this is BlankPattern || num_whites == 60) return;

        // Meter reading
        int image_width = BOARD_WIDTH * dot_width;  // pixels
        int sizeof_pixel = 4; // bytes
        for (var x = dot_width; x < image_width - dot_width; x++) {
            var col = num_whites - 60 + image_width / 2 <= x ? 128 : 255;
            for (var y = 0; y < dot_width - 1; y++) {
                var n = (y * image_width + x) * sizeof_pixel;
                pixbuf_data[n] = col;
                pixbuf_data[n + 1] = pixbuf_data[n + 2] = 0;
                pixbuf_data[n + 3] = 255;
            }
        }

        button.image = new Gtk.Image.from_pixbuf (pixbuf);
    }
}// class CustomPattern


class BlankPattern : CustomPattern {
}


}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
