(* This file is part of the Kind 2 model checker.

   Copyright (c) 2015 by the Board of Trustees of the University of Iowa

   Licensed under the Apache License, Version 2.0 (the "License"); you
   may not use this file except in compliance with the License.  You
   may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0 

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
   implied. See the License for the specific language governing
   permissions and limitations under the License. 

*)

open Lib

module N = LustreNode
module Id = LustreIdent
module I = LustreIndex
module E = LustreExpr
module C = LustreContract
module SVar = StateVar
module SVM = StateVar.StateVarMap
module SVS = StateVar.StateVarSet

(* Name of the parsing functions in the rust [parse] module. *)
let parse_bool_fun, parse_int_fun, parse_real_fun = "bool", "int", "real"

(* File name at the end of a path. *)
let file_name_of_path file_path =
  try (
    let last_slash = String.rindex file_path '/' in
    if last_slash = (String.length file_path) - 1 then (
      Format.sprintf "[fmt_cp_target] illegal argument \"%s\"" file_path
      |> failwith
    ) ;
    String.sub file_path (last_slash + 1) (
      (String.length file_path) - (last_slash + 1)
    )
  ) with Not_found -> file_path

(* Formats a position as a link to the rust doc of the lustre file. *)
let fmt_pos_as_link fmt pos =
  let name, line, _ = file_row_col_of_pos pos in
  let name =
    match name with
    | "" -> Flags.input_file ()
    | name -> file_name_of_path name
  in
  Format.fprintf fmt "[%s line %d](../src/lus/%s.html#%d)"
    name line name line

(* Formatter for types as rust expressions. *)
let fmt_type fmt t = match Type.node_of_type t with
| Type.Bool -> Format.fprintf fmt "Bool"
| Type.Int
| Type.IntRange _ -> Format.fprintf fmt "Int"
| Type.Real -> Format.fprintf fmt "Real"
| _ ->
  Format.asprintf "type %a is not supported" Type.pp_print_type t
  |> failwith


(* Unsafe string representation of an ident, used for rust identifiers. *)
let mk_id_legal = Id.string_of_ident false

(* Same as [mk_id_legal] but capitalizes the first letter to fit rust
conventions for type naming. *)
let mk_id_type id = mk_id_legal id |> String.capitalize_ascii

(* Crate documentation for implementation, lint attributes. *)
let fmt_prefix blah name fmt typ = Format.fprintf fmt "\
//! %s for lustre node `%s` (see [%s](struct.%s.html)).
//!
//! Code generated by the [Kind 2 model checker][kind 2].
//!
//! [kind 2]: http://kind2-mc.github.io/kind2/ (The Kind 2 model checker)

// Deactiving lint warnings the transformation does not respect.
#![allow(
  non_upper_case_globals, non_snake_case, non_camel_case_types,
  unused_variables, unused_parens
)]

use helpers::* ;
" blah name typ typ

(* Specialization of [fmt_prefix] for implementation. *)
let fmt_prefix_implem = fmt_prefix "Implementation"

(* Specialization of [fmt_prefix] for oracles. *)
let fmt_prefix_oracle = fmt_prefix "Oracle"

(* Crate entry point. *)
let fmt_main fmt () = Format.fprintf fmt "\
/// Entry point.
fn main() {
  clap_and_run()
}
"

(* Helpers modules: cla parsing, types, traits for systems and stdin
parsing. *)
let fmt_helpers fmt systems = Format.fprintf fmt "\
/// Types and structures for systems.
pub mod helpers {
  use std::io::{ Stdin, stdin } ;
  use std::process::exit ;

  /// Prints usage.
  pub fn help() {
    println!(\"\") ;
    println!(\"\\
Options:
  -h, --help
    prints this message@.  \
  @[<v>%a@]
Usage:
  Inputs (outputs) are read (printed) as comma-separated values on a single
  line.
  The read-eval-print loop runs forever, write \\\"exit\\\" or \\\"quit\\\"
  to exit it cleanly.
Default system: \\\"%s\\\".\\
    \") ;
    println!(\"\")
  }

  /// Prints usage, an error, and exits with status `2`.
  pub fn error<T: ::std::fmt::Display>(e: T) {
    help() ;
    println!(\"Error: {}\", e) ;
    println!(\"\") ;
    exit(2)
  }

  /// Handles CLA.
  pub fn clap_and_run() {
    use std::env::args ;
    let mut args = args() ;
    // Skipping first argument (name of binary).
    match args.next() {
      Some(_) => (),
      None => unreachable!(),
    } ;
    if let Some(arg) = args.next() {
      match & arg as & str {
        \"-h\" | \"--help\" => {
          help() ;
          exit(0)
        },\
@.        @[<v>%a@]
        arg => error(
          format!(\"unexpected argument \\\"{}\\\".\", arg)
        ),
      }
    } ;
    // If no argument given, run top system.
    super::%s::run()
  }

  /// Alias for `i64`.
  pub type Int = i64 ;
  /// Alias for `f64`.
  pub type Real = f64 ;
  /// Alias for `bool`.
  pub type Bool = bool ;

  /// Stores an `Stdin` and a buffer to read lines.
  pub struct InputReader {
    /// Standard input.
    stdin: Stdin,
    /// String buffer.
    buff: String,
  }
  impl InputReader {
    /// Creates an input reader.
    pub fn mk() -> Self {
      InputReader {
        stdin: stdin(),
        buff: String::with_capacity(100),
      }
    }
    /// Reads comma separated inputs from standard input.
    pub fn read_inputs(& mut self) -> Result<Vec<String>, String> {
      self.buff.clear() ;
      match self.stdin.read_line(& mut self.buff) {
        Ok(_) => (),
        Err(e) => return Err(
          format!(\"could not read line from stdin: {}\", e)
        ),
      } ;
      let chars = self.buff.trim_left().chars() ;
      let mut buff = String::new() ;
      let mut vec = vec![] ;
      for c in chars {
        match c {
          ' ' | '\\t' => (),
          ',' | '\\n' => {
            vec.push(buff.clone()) ;
            buff.clear()
          },
          _ => buff.push(c),
        }
      } ;
      if vec.len() > 1 {
        match vec[0].trim() {
          \"exit\" | \"quit\" => exit(0),
          _ => ()
        }
      } ;
      Ok(vec)
    }
  }

  /// Trait all systems must implement.
  pub trait Sys: Sized {
    /// Type of inputs.
    type Input ;
    /// Type of outputs.
    type Output ;
    /// Number of inputs expected.
    fn arity() -> usize ;
    /// Parses a vector of inputs.
    fn input_of(Vec<String>) -> Result<Self::Input, String> ;
    /// Initial state of the system.
    fn init(Self::Input) -> Result<Self, String> ;
    /// Computes the next step.
    fn next(self, Self::Input) -> Result<Self, String> ;
    /// Reads inputs from standard input, computes initial state, prints output.
    fn read_init(reader: & mut InputReader) -> Result<Self, String> {
      match Self::input_of( try!(reader.read_inputs()) ) {
        Ok(inputs) => {
          let init = try!( Self::init(inputs) ) ;
          println!(\"{}\", init.output_str()) ;
          Ok(init)
        },
        Err(s) => Err(s),
      }
    }
    /// Reads inputs from standard input, computes next step, prints output.
    fn read_next(self, reader: & mut InputReader) -> Result<Self, String> {
      match Self::input_of( try!(reader.read_inputs()) ) {
        Ok(inputs) => {
          let next = try!( self.next(inputs) ) ;
          println!(\"{}\", next.output_str()) ;
          Ok(next)
        },
        Err(s) => Err(s),
      }
    }
    /// Output of the system.
    fn output(& self) -> Self::Output ;
    /// String representation of the output.
    fn output_str(& self) -> String ;
    /// Runs a never-ending, read-eval-print loop on the system.
    fn run() -> ! {
      let mut reader = InputReader::mk() ;
      let mut state = match Self::read_init(& mut reader) {
        Ok(init) => init,
        Err(e) => {
          println!(\"(Error: {})\", e) ;
          exit(2)
        }
      } ;
      loop {
        match state.read_next(& mut reader) {
          Ok(next) => state = next,
          Err(e) => {
            println!(\"(Error: {})\", e) ;
            exit(2)
          }
        }
      }
    }
  }
}

/// Parsing functions.
pub mod parse {
  use helpers::{ Int, Real, Bool } ;
  use std::fmt::Display ;
  use std::str::FromStr ;
  /// Generic parser to factor error handling out.
  fn generic<
    Out, Error: Display, F: Fn(& str) -> Result<Out, Error>
  >(s: & str, f: F, typ3: & 'static str) -> Result<Out, String> {
    match f(s) {
      Ok(res) => Ok(res),
      Err(e) => Err(
        format!(\"could not parse \\\"{}\\\" as {}: {}\", s, typ3, e)
      ),
    }
  }
  /// Parses a [`Bool`](../type.Bool.html).
  pub fn %s(s: & str) -> Result<Bool, String> {
    generic(
      s,
      |s| match s {
        \"true\" | \"on\" => Ok(true),
        \"false\" | \"off\" => Ok(false),
        _ => Err(
          format!(\"legal values: true, on, false, off\")
        ),
      },
      \"a bool\"
    )
  }
  /// Parses an [`Int`](../type.Int.html).
  pub fn %s(s: & str) -> Result<Int, String> {
    generic(s, |s| Int::from_str(s), \"an int\")
  }
  /// Parses a [`Real`](../type.Real.html).
  pub fn %s(s: & str) -> Result<Real, String> {
    generic(s, |s| Real::from_str(s), \"a real\")
  }
}
"
(pp_print_list
  ( fun fmt { N.name ; N.inputs ; N.outputs } ->
      Format.fprintf fmt "\
        --%s@   @[<v>\
          inputs:  @[<v>%a@]@ \
          outputs: @[<v>%a@]\
        @]\
      "
      (mk_id_legal name |> String.lowercase_ascii)
      ( pp_print_list
        ( fun fmt (_, svar) ->
            Format.fprintf fmt "%a (%s)"
              fmt_type (SVar.type_of_state_var svar)
              (SVar.name_of_state_var svar)
        )
        "@ "
      ) (I.bindings inputs)
      ( pp_print_list
        ( fun fmt (_, svar) ->
            Format.fprintf fmt "%a (%s)"
              fmt_type (SVar.type_of_state_var svar)
              (SVar.name_of_state_var svar)
        )
        "@ "
      ) (I.bindings outputs)
  ) "@ "
) systems
( match List.rev systems with
  | { N.name } :: _ -> mk_id_legal name |> String.lowercase_ascii
  | _ -> failwith "Can't generate helpers, no top system to print." )
(pp_print_list
  ( fun fmt { N.name } ->
      Format.fprintf fmt "\"--%s\" => super::%s::run(),"
        (mk_id_legal name |> String.lowercase_ascii)
        (mk_id_type name)
  ) "@ "
) systems
( match List.rev systems with
  | { N.name } :: _ -> mk_id_type name
  | _ -> failwith "Can't generate helpers, no top system to print." )
parse_bool_fun parse_int_fun parse_real_fun

(* Continuation type for the term-to-Rust printer.
Used to specify what should happen after the next step. *)
type continue =
| T of Term.t (* Next step is to print a term. *)
| S of string (* Next step is to print a string. *)


(* [wrap_with_sep e s [t1 ; ... ; tn]] creates the list
[[Ss ; T t1 ; S s ; ... ; S s ; tn ; e]]. *)
let wrap_with_sep ending sep kids =
  let ending = [ S ending ] in
  let rec loop kids lst = match kids with
    | [] -> List.rev_append lst ending
    | [kid] -> (S sep) :: (T kid) :: ending |> List.rev_append lst
    | kid :: kids -> (T kid) :: (S sep) :: lst |> loop kids
  in
  loop kids []

(* Prints a variable. Prefixes with ["self."] variables unrolled at 0 and
constant variables. *)
let fmt_var pref fmt var =
  if Var.is_state_var_instance var then (
    let off = Var.offset_of_state_var_instance var |> Numeral.to_int in
    let from = match off with
      | 0 -> "self."
      | 1 -> ""
      | _ ->
        Format.asprintf "unexpected var %a" Var.pp_print_var var
        |> failwith
    in
    Var.state_var_of_state_var_instance var
    |> SVar.name_of_state_var
    |> Format.fprintf fmt "%s%s%s" from pref
  ) else if Var.is_const_state_var var then (
    (* Constant input. Can't be just a constant, otherwise it would have been
    propagated. *)
    Var.state_var_of_state_var_instance var
    |> SVar.name_of_state_var
    |> Format.fprintf fmt "%s%s" pref
  ) else
    Format.asprintf "unexpected var %a" Var.pp_print_var var
    |> failwith

(* Goes down a term, printing what it can until it reaches a leaf. Then, calls
[fmt_term_up] on the continuation. *)
let rec fmt_term_down svar_pref next fmt term =
match Term.destruct term with
| Term.T.App (sym, kid :: kids) -> (
  let node = Symbol.node_of_symbol sym in
  match node with
  (* Unary. *)
  | `NOT ->
    Format.fprintf fmt "(! " ;
    assert (kids = []) ;
    fmt_term_down svar_pref ([ S ")" ] :: next) fmt kid
  (* Binary. *)
  | `EQ
  | `MOD
  | `LEQ
  | `LT
  | `GEQ
  | `GT -> (
    match kids with
    | [rhs] ->
      let op =
        match node with
        | `EQ -> " == "
        | `MOD -> " % "
        | `LEQ -> " <= "
        | `LT -> " < "
        | `GEQ -> " >= "
        | `GT -> " > "
        | _ -> failwith "unreachable"
      in
      Format.fprintf fmt "(" ;
      fmt_term_down svar_pref (
        [ S op ; T rhs ; S ")" ] :: next
      ) fmt kid
    | [] -> failwith "implication of one kid"
    | _ ->
      Format.sprintf "implication of %d kids" ((List.length kids) + 1)
      |> failwith
  )
  (* Binary but rewritten. *)
  | `IMPLIES ->
    Term.mk_not kid :: kids
    |> Term.mk_or
    |> fmt_term_down svar_pref next fmt 
  (* Ternary. *)
  | `ITE -> (
    let _, t, e = match kids with
      | [ t ; e ] -> kid, t, e
      | _ -> failwith "illegal ite"
    in
    Format.fprintf fmt "( if " ;
    fmt_term_down svar_pref (
      [ S " { " ; T t ; S " } else {" ; T e ; S " } )" ] :: next
    ) fmt kid
  )
  (* N-ary. *)
  | `MINUS when kids = [] ->
    Format.fprintf fmt "- " ;
    fmt_term_down svar_pref next fmt kid
  | `MINUS
  | `PLUS
  | `TIMES
  | `DIV
  | `OR
  | `XOR
  | `AND ->
    let op =
      match node with
      | `MINUS -> " - "
      | `PLUS -> " + "
      | `TIMES -> " * "
      | `DIV -> " / "
      | `OR -> " | "
      | `XOR -> " ^ "
      | `AND -> " & "
      | _ -> failwith "unreachable"
    in
    Format.fprintf fmt "(" ;
    fmt_term_down svar_pref (
      (wrap_with_sep ")" op kids) :: next
    ) fmt kid
  | `DISTINCT
  | `INTDIV
  | `ABS
  | _ ->
    Format.asprintf "unsupported symbol %a" Symbol.pp_print_symbol sym
    |> failwith
  (*
  | `TO_REAL
  | `TO_INT
  | `TO_UINT8
  | `TO_UINT16
  | `TO_UINT32
  | `TO_UINT64
  | `IS_INT
  (* Illegal. *)
  | `NUMERAL of Numeral.t
  | `DECIMAL of Decimal.t
  | `TRUE
  | `FALSE -> Format.fprintf fmt "illegal" *)
)
| Term.T.App (_, []) -> failwith "application with no kids"
| Term.T.Var var ->
  fmt_var svar_pref fmt var ;
  fmt_term_up svar_pref fmt next
| Term.T.Const sym ->
  ( match Symbol.node_of_symbol sym with
    | `NUMERAL n -> Format.fprintf fmt "%a" Numeral.pp_print_numeral n
    | `DECIMAL d -> Format.fprintf fmt "%a" Decimal.pp_print_decimal_as_float d
    | `TRUE -> Format.fprintf fmt "true"
    | `FALSE -> Format.fprintf fmt "false"
    | _ -> Format.asprintf "Const %a" Symbol.pp_print_symbol sym |> failwith
  ) ;
  fmt_term_up svar_pref fmt next
(* | Term.T.Attr (kid,_) -> fmt_term_down svar_pref [] fmt kid *)

(* Goes up a continuation. Prints the strings it finds and calls
[fmt_term_down] on terms. *)
and fmt_term_up svar_pref fmt = function
| (next :: nexts) :: tail -> (
  let tail = nexts :: tail in
  match next with
  | S str ->
    Format.fprintf fmt "%s" str ;
    fmt_term_up svar_pref fmt tail
  | T term ->
    fmt_term_down svar_pref tail fmt term
)
| [] :: tail -> fmt_term_up svar_pref fmt tail
| [] -> ()

(* Formatter for terms as rust expressions. *)
let fmt_term svar_pref = fmt_term_down svar_pref []


(* Rust-level parsing function for a type. *)
let parser_for t = match Type.node_of_type t with
| Type.Bool -> parse_bool_fun
| Type.Int
| Type.IntRange _ -> parse_int_fun
| Type.Real -> parse_real_fun
| _ ->
  Format.asprintf "type %a is not supported" Type.pp_print_type t
  |> failwith

(* Prefix for all state variables. *)
let svar_pref = "svar_"


(* Gathers [LustreNode.equation] with [LustreNode.node_call] for ordering. *)
type equation =
| Eq of N.equation (* An equation. *)
| Call of (int * N.node_call) (* A call with a uid local to the node. *)

(* Identifier refering to the current state of the system called. *)
let id_of_call cnt { N.call_node_name } =
  Format.sprintf "%s_%d" (mk_id_legal call_node_name) cnt

(* Pretty prints an equation or a call. *)
let pp_print_equation fmt = function
| Eq eq -> N.pp_print_node_equation false fmt eq
| Call (cnt, call) ->
  Format.fprintf fmt "%a (%d)" (N.pp_print_call false) call cnt


(* Orders equations topologicaly based on the [expr_init] or [expr_next]
expression of the right-hand side of the equation. *)
let order_equations init_or_expr inputs equations =
  (* Checks if [svar] is defined in the equations or is an input. *)
  let is_defined sorted svar =
    List.exists (fun (_, svar') -> svar == svar') inputs
    || List.exists (function
      | Eq ((svar', _), _) -> svar == svar'
      | Call (_, { N.call_outputs }) ->
        I.bindings call_outputs |> List.exists (
          fun (_, svar') -> svar == svar'
        )
    ) sorted
  in
  (* Sorts equations. *)
  let rec loop count later to_do sorted = match to_do with
    (* Equation. *)
    | (Eq ((_, _), rhs)) as eq :: to_do ->
      let later, sorted =
        if
          init_or_expr rhs
          |> E.cur_term_of_expr E.base_offset
          (* Extract svars. *)
          |> Term.state_vars_at_offset_of_term E.base_offset
          (* All svars must be defined. *)
          |> SVS.for_all (is_defined sorted)
        then later, eq :: sorted else eq :: later, sorted
      in
      loop count later to_do sorted
    (* Node call. *)
    | (
      Call (
        _, { N.call_inputs ; N.call_defaults }
      ) as eq
    ) :: to_do ->
      if call_defaults != None then (
        Format.printf "Compilating of condacts is not supported.@.@." ;
        failwith "could not compile system"
      ) ;
      let later, sorted =
        if
          I.bindings call_inputs
          (* All input svar must be defined. *)
          |> List.for_all (fun (_, svar) -> is_defined sorted svar)
        then later, eq :: sorted else eq :: later, sorted
      in
      loop count later to_do sorted
    (* Done. *)
    | [] -> (
      let count = count + 1 in
      if count <= (List.length equations) + 1 then
        match later with
        | [] -> List.rev sorted
        | _ -> loop count [] later sorted
      else (
        Format.printf
          "Some equations use undefined variables:@.  @[<v 2>%a@]@.@." (
            pp_print_list pp_print_equation "@ "
          ) later ;
        failwith "could not compile system"
      )
    )
  in
  loop 0 [] equations []



(* Pretty prints calls for struct documentation. *)
let fmt_calls_doc fmt = function
| [] -> Format.fprintf fmt "No subsystems for this system.@."
| calls -> Format.fprintf fmt "\
  | Lustre identifier | Struct | Inputs | Outputs | Position |@.\
  /// |:---:|:---:|:---:|:---:|:---:|@.\
  /// %a@.\
" ( pp_print_list
    ( fun fmt {
        N.call_pos ; N.call_node_name ; N.call_inputs ; N.call_outputs
      } ->
        Format.fprintf fmt
          "\
            | `%s` @?\
            | [%s](struct.%s.html) @?\
            | %a @?\
            | %a @?\
            | %a |\
          "
          (mk_id_legal call_node_name)
          (mk_id_type call_node_name)
          (mk_id_type call_node_name)
          (pp_print_list (fun fmt (_, svar) ->
              SVar.name_of_state_var svar
              |> Format.fprintf fmt "`%s`"
            ) ", "
          ) (I.bindings call_inputs)
          (pp_print_list (fun fmt (_, svar) ->
              SVar.name_of_state_var svar
              |> Format.fprintf fmt "`%s`"
            ) ", "
          ) (I.bindings call_outputs)
          fmt_pos_as_link call_pos
    ) "@./// "
  ) calls


(* Pretty prints assertions for struct documentation. *)
let fmt_asserts_doc fmt = function
| [] -> Format.fprintf fmt "/// No assertions for this system.@."
| asserts -> Format.fprintf fmt "%a@." (
  pp_print_list (fun fmt (pos,_) ->
    Format.fprintf fmt
      "- `%a`"
      fmt_pos_as_link pos
  ) "@ /// "
) asserts

(* Pretty prints assumptions for a struct documentation. *)
let fmt_assumes_doc fmt = function
| [] -> Format.fprintf fmt "/// No assumptions for this system.@."
| assumes -> Format.fprintf fmt "\
  /// | State variable | Position | Number |@.\
  /// |:------:|:-----:|:-----:|@.\
  /// %a@.\
" ( pp_print_list (fun fmt {
      LustreContract.pos ; LustreContract.num ; LustreContract.svar
    } ->
      Format.fprintf fmt
        "| `%s` @?| %a @?| %d |"
        (SVar.name_of_state_var svar)
        fmt_pos_as_link pos
        num
    ) "@ /// "
  ) assumes


(* Writes the documentation for a struct for the implementation of a node. *)
let implem_doc_of_struct is_top fmt (
  name, inputs, outputs, calls, asserts, contract
) =
  Format.fprintf fmt "\
      /// Stores the state for %s `%s`.@.\
      ///@.\
      /// # Inputs@.\
      ///@.\
      /// | Lustre identifier | Type |@.\
      /// |:---:|:---|@.\
      /// %a@.\
      ///@.\
      /// # Outputs@.\
      ///@.\
      /// | Lustre identifier | Type |@.\
      /// |:---:|:---|@.\
      /// %a@.\
      ///@.\
      /// # Sub systems@.\
      ///@.\
      /// %a\
      ///@.\
      /// # Assertions@.\
      ///@.\
      /// %a\
      ///@.\
      /// # Assumptions@.\
      ///@.\
      %a\
      ///@.\
    "
    (if is_top then "**top node**" else "sub-node") name
    ( pp_print_list (fun fmt (_, svar) ->
        Format.fprintf fmt "| `%s` | %a |"
          (SVar.name_of_state_var svar)
          fmt_type (SVar.type_of_state_var svar)
      ) "@./// "
    ) inputs
    ( pp_print_list (fun fmt (_, svar) ->
        Format.fprintf fmt "| `%s` | %a |"
          (SVar.name_of_state_var svar)
          fmt_type (SVar.type_of_state_var svar)
      ) "@./// "
    ) outputs
    fmt_calls_doc calls
    fmt_asserts_doc asserts
    ( fun fmt -> function
      | None -> fmt_assumes_doc fmt []
      | Some { C.assumes } -> fmt_assumes_doc fmt assumes
    ) contract


(* Writes the documentation for a struct for the test oracle of a node. *)
let oracle_doc_of_struct is_top fmt (
  name, inputs, outputs, assumes, guarantees, modes, svar_source_map
) =
  Format.fprintf fmt "\
      /// Stores the state for the oracle for %s `%s`.@.\
      ///@.\
      /// # Inputs@.\
      ///@.\
      /// | Lustre identifier | Type | Source |@.\
      /// |:---:|:---:|:---|@.\
      /// %a@.\
      ///@.\
      /// # Outputs@.\
      ///@.\
      /// The outputs of the oracle are the guarantees of the original@.\
      /// system and the implications for each require of each mode.
      ///@.\
      /// That is, if a mode has requires `req_1`, ..., `req_n` and ensures@.\
      /// `ens_1`, ..., `ens_m` this will generate `m` outputs:@.\
      ///@.\
      /// - `(req_1 && ... && req_n) => ens_1`
      /// - ...
      /// - `(req_1 && ... && req_n) => ens_m`
      ///@.\
      /// Hence, an ensure output is false iff the mode is active and the
      /// ensure is false.
      ///@.\
      /// | Lustre identifier | Type |@.\
      /// |:---:|:---|@.\
      /// %a@.\
      ///@.\
      /// ## Guarantees@.\
      ///@.\
      /// %a\
      ///@.\
      /// %a\
      ///@.\
      /// # Assumptions@.\
      ///@.\
      %a\
      ///@.\
    "
    (if is_top then "**top node**" else "sub-node") name
    ( pp_print_list (fun fmt (_, svar) ->
        Format.fprintf fmt "| `%s` | %a | %a |"
          (SVar.name_of_state_var svar)
          fmt_type (SVar.type_of_state_var svar)
          N.pp_print_state_var_source (
            try
              SVM.find svar svar_source_map
            with Not_found ->
              Format.asprintf
                "can't find source of svar %a"
                SVar.pp_print_state_var svar
              |> failwith
          )
      ) "@./// "
    ) inputs
    ( pp_print_list (fun fmt (_, svar) ->
        Format.fprintf fmt "| `%s` | %a |"
          (SVar.name_of_state_var svar)
          fmt_type (SVar.type_of_state_var svar)
      ) "@./// "
    ) outputs
    ( fun fmt -> function
      | [] -> Format.fprintf fmt "No guarantees for this system.@."
      | guarantees ->
        Format.fprintf fmt "\
            | Lustre identifier | Guarantee number | Position |@.\
            /// |:---:|:---:|:---|@.\
            /// %a@.
          "
          ( pp_print_list
            ( fun fmt ({ C.pos ; C.num ; C.svar }, _) ->
                Format.fprintf fmt "| `%s` | %d | %a |"
                  (SVar.name_of_state_var svar)
                  num
                  fmt_pos_as_link pos
            ) "@./// "
          ) guarantees
    ) guarantees
    ( fun fmt -> function
      | [] -> Format.fprintf fmt "No modes for this system."
      | modes ->
        Format.fprintf fmt "%a@." (pp_print_list
          ( fun fmt { C.name ; C.pos ; C.ensures } ->
            Format.fprintf fmt "\
                ## Mode **%s**@.\
                ///@.\
                /// Position: *%a*.@.\
                ///@.\
                /// | Lustre identifier | Mode require number | Position |@.\
                /// |:---:|:---:|:---|@.\
                /// %a@.
              "
              (Id.string_of_ident false name)
              fmt_pos_as_link pos
              ( pp_print_list
                (fun fmt { C.pos ; C.num ; C.svar } ->
                  Format.fprintf fmt "| `%s` | %d | %a |"
                    (SVar.name_of_state_var svar)
                    num
                    fmt_pos_as_link pos
                ) "@./// "
              ) ensures
          ) "@.///@./// "
        ) modes
    ) modes
    fmt_assumes_doc assumes


(* Compiles a node to rust, writes it to a formatter. *)
let node_to_rust oracle_info is_top fmt (
  { N.locals ; N.contract ; N.state_var_source_map } as node
) =

  (* Format.printf "node: %a@.@." (Id.pp_print_ident false) name ; *)

  let is_input svar =
    try (
      match SVM.find svar state_var_source_map with
      | N.Input ->
        (* Format.printf
          "input: %a@.@." SVar.pp_print_state_var svar ; *)
        true
      | _ ->
        (* Format.printf
          "not input: %a@.@." SVar.pp_print_state_var svar ; *)
        false
    ) with Not_found -> (
      (* Format.printf "dunno what dat is: %a@.@." SVar.pp_print_state_var svar ; *)
      false
    )
  in

  (* Remove inputs from locals, they're in the state anyways. *)
  let locals =
    locals |> List.filter (
      fun local ->
        match I.bindings local with
        | (_, svar) :: tail ->
          let local_is_input = is_input svar in
          if List.exists (
            fun (_, svar) -> (is_input svar) <> local_is_input
          ) tail then failwith "\
            unreachable: indexed state variable is partially an input\
          " else (
            if not local_is_input then (
              true
            ) else (
              Format.printf "filtering local %a out@.@." SVar.pp_print_state_var svar ;
              false
            )
          )
        | [] -> failwith "unreachable: empty indexed state variable"
    )
  in

  let {
    N.inputs ; N.outputs ; N.locals ;
    N.equations ; N.state_var_source_map ; N.calls = real_calls ;
    N.asserts ; N.contract
  } as node =
    (* If there's a contract, add all assume and requires to locals.

    We need to do this because assumptions may mention pre of the mode
    requirements. *)
    let locals = match contract with
      | None -> locals
      | Some { C.assumes ; C.modes } ->
        let known =
          locals |> List.fold_left (
            fun set local ->
              I.bindings local |> List.fold_left (
                fun set (_, svar) -> SVS.add svar set
              ) set
          ) SVS.empty
        in
        (* Format.printf "known:@." ;
        SVS.iter (
          fun svar -> Format.printf "  %a@." SVar.pp_print_state_var svar
        ) known ;
        Format.printf "@." ; *)
        let locals, known =
          assumes |> List.fold_left (
            fun (locals, known) { C.svar } ->
              if SVS.mem svar known || is_input svar
              then locals, known else (
                (I.singleton I.empty_index svar) :: locals, SVS.add svar known
              )
          ) (locals, known)
        in
        (* Format.printf "known:@." ;
        SVS.iter (
          fun svar -> Format.printf "  %a@." SVar.pp_print_state_var svar
        ) known ;
        Format.printf "@." ; *)
        modes |> List.fold_left (
          fun (locals, known) { C.requires } ->
            requires |> List.fold_left (
              fun (locals, known) { C.svar } ->
                if SVS.mem svar known || is_input svar
                then locals, known else (
                  (I.singleton I.empty_index svar) :: locals, SVS.add svar known
                )
            ) (locals, known)
        ) (locals, known)
        |> fst
    in
    { node with N.locals = locals }
  in

  let calls, _ =
    real_calls |> List.fold_left (
      fun (l,cpt) c -> Call (cpt, c) :: l, cpt + 1
    ) ([], 0)
  in
  let equations =
    equations |> List.fold_left (fun eqs ( ((_ (* svar *), _), _) as eq ) ->
      (* if SVM.mem svar state_var_source_map
      then (Eq eq) :: eqs else eqs *)
      Eq eq :: eqs
    ) calls
  in
  let name = mk_id_legal node.N.name in
  let typ = mk_id_type node.N.name in

  let inputs, outputs, locals =
    I.bindings inputs, I.bindings outputs,
    locals |> List.map I.bindings |> List.flatten
    (* |> List.fold_left (fun locs index ->
      ( I.bindings index |> List.filter (fun (_, svar) ->
        SVM.mem svar state_var_source_map
        ) |> List.rev_append
      ) locs
    ) [] *)
  in

  (* Struct documentation for this system. *)
  ( match oracle_info with
    | None ->
      implem_doc_of_struct is_top fmt (
        name, inputs, outputs, real_calls, asserts, contract
      )
    | Some (assumes, guarantees, modes) ->
      oracle_doc_of_struct is_top fmt (
        name, inputs, outputs, assumes, guarantees, modes, state_var_source_map
      )
  ) ;

  (* Struct header. *)
  Format.fprintf fmt "pub struct %s {" typ ;

  (* Fields. *)
  inputs |> List.iter (fun (_, svar) ->
    Format.fprintf fmt "@.  /// Input: `%a`@.  pub %s%s: %a,"
      SVar.pp_print_state_var svar
      svar_pref
      (SVar.name_of_state_var svar)
      fmt_type (SVar.type_of_state_var svar)
  ) ;

  Format.fprintf fmt "@." ;

  outputs |> List.iter (fun (_, svar) ->
    Format.fprintf fmt "@.  /// Output: `%a`@.  pub %s%s: %a,"
      SVar.pp_print_state_var svar
      svar_pref
      (SVar.name_of_state_var svar)
      fmt_type (SVar.type_of_state_var svar)
  ) ;

  Format.fprintf fmt "@." ;

  locals |> List.iter (fun (_, svar) ->
    let source =
      try
        Format.asprintf ", %a"
          N.pp_print_state_var_source (
            SVM.find svar state_var_source_map
          )
      with Not_found -> ""
    in
    Format.fprintf
      fmt "@.  /// Local%s: `%a`@.  pub %s%s: %a,"
      source
      SVar.pp_print_state_var svar
      svar_pref
      (SVar.name_of_state_var svar)
      fmt_type (SVar.type_of_state_var svar)
  ) ;

  Format.fprintf fmt "@." ;

  calls |> List.iter (function
    | Call (cnt, ({ N.call_pos ; N.call_node_name } as call)) ->
      Format.fprintf
        fmt "@.  /// Call to `%a` (%a).@.  pub %s: %s,"
        (Id.pp_print_ident false) call_node_name
        fmt_pos_as_link call_pos
        (id_of_call cnt call)
        (mk_id_type call_node_name)
    | _ -> failwith "unreachable"
  ) ;

  Format.fprintf fmt "@.}@.@.impl Sys for %s {@." typ ;

  (* Input type. *)
  inputs
  |> Format.fprintf fmt "  type Input = (@.    @[<v>%a@]@.  ) ;@." (
    pp_print_list (fun fmt (_, svar) ->
      Format.fprintf fmt "%a, // %s%s (%a)"
        fmt_type (SVar.type_of_state_var svar)
        svar_pref
        (SVar.name_of_state_var svar)
        SVar.pp_print_state_var svar
    ) "@ "
  ) ;

  (* Output type. *)
  outputs |> Format.fprintf fmt "  type Output = (@.    @[<v>%a@]@.  ) ;@." (
    pp_print_list (fun fmt (_, svar) ->
      Format.fprintf fmt "%a, // %s%s (%a)"
        fmt_type (SVar.type_of_state_var svar)
        svar_pref
        (SVar.name_of_state_var svar)
        SVar.pp_print_state_var svar
    ) "@ "
  ) ;

  (* Arity. *)
  List.length inputs
  |> Format.fprintf fmt "  fn arity() -> usize { %d }@." ;

  (* Input parsing. *)
  let input_cpt = ref 0 in
  Format.fprintf fmt "  \
      @[<v>\
        fn input_of(vec: Vec<String>) -> Result<Self::Input, String> {@   \
          @[<v>\
            match vec.len() {@   \
              @[<v>\
                n if n == Self::arity() => {@   \
                  @[<v>\
                    Ok( (@   @[<v>%a@],@ ) )\
                  @]@ \
                },@ \
                n => Err(@   \
                  @[<v>\
                    format!(@   \
                      \"arity mismatch, expected {} but got {}: {:?}\",@   \
                      Self::arity(), n, vec@ \
                    )@ \
                  @]@ \
                ),\
              @]@ \
            }\
          @]@ \
        }\
      @]@.@.\
    "
    ( pp_print_list (fun fmt (_, svar) ->
        Format.fprintf fmt "try!( parse::%s(& vec[%d]) )" (
          SVar.type_of_state_var svar
          |> parser_for
        ) ! input_cpt ;
        input_cpt := 1 + !input_cpt
      ) ", @ "
    ) inputs ;

  (* Init. *)
  let input_cpt = ref 0 in
  let eqs_init =
    order_equations (fun expr -> expr.E.expr_init) inputs equations
  in
  assert (
    (List.length eqs_init) == (List.length equations)
  ) ;

  Format.fprintf fmt "  \
      fn init(input: Self::Input) -> Result<Self, String> {@.    \
        @[<v>\
          // |===| Retrieving inputs.@ \
          %a@ @ \
          // |===| Computing initial state.@ \
          %a@ @ \
          // |===| Checking assertions.@ \
          %a@ @ \
          %a\
          // |===| Returning initial state.@ \
          Ok( %s {@   \
            @[<v>\
              // |===| Inputs.@ %a@ @ \
              // |===| Outputs.@ %a@ @ \
              // |===| Locals.@ %a@ @ \
              // |===| Calls.@ %a\
            @]@ \
          } )\
        @]@.  \
      }@.@.\
    "

    ( pp_print_list (fun fmt (_, svar) ->
        Format.fprintf fmt "let %s%s = input.%d ;"
          svar_pref
          (SVar.name_of_state_var svar) !input_cpt ;
        input_cpt := 1 + !input_cpt
      ) "@ "
    ) inputs

    ( pp_print_list (fun fmt -> function
        | Eq ((svar, _), expr) ->
          expr.E.expr_init
          |> E.base_term_of_expr (Numeral.succ E.base_offset)
          |> Format.fprintf fmt "let %s%s = %a ;"
            svar_pref
            (SVar.name_of_state_var svar)
            (fmt_term svar_pref)
        | Call (
          cnt, ({ N.call_node_name ; N.call_inputs ; N.call_outputs } as call)
        ) ->
          Format.fprintf fmt
            "\
              let %s = try!( %s::init( (@   @[<v>%a,@]@ ) ) ) ;@ \
              let (@   @[<v>%a,@]@ ) = %s.output() ;@ \
            "
            (id_of_call cnt call)
            (mk_id_type call_node_name)
            ( pp_print_list (fun fmt (_, svar) ->
                Format.fprintf fmt "%s%s"
                  svar_pref (SVar.name_of_state_var svar)
              ) ",@ "
            ) (I.bindings call_inputs)
            ( pp_print_list (fun fmt (_, svar) ->
                Format.fprintf fmt "%s%s"
                  svar_pref (SVar.name_of_state_var svar)
              ) ",@ "
            ) (I.bindings call_outputs)
            (id_of_call cnt call)
      ) "@ "
    ) eqs_init

    ( fun fmt asserts ->
      if oracle_info = None
      then
        Format.fprintf fmt
          "%a@ @ "
          ( pp_print_list (fun fmt (pos, svar) ->
              Format.fprintf fmt
                "// Assertion at %a@ if ! %s%s {@   \
                  @[<v>\
                    return Err(@   \
                      \"assertion failure in system `%s`: %a\".to_string()@ \
                    )\
                  @]@ \
                } ;"
                fmt_pos_as_link pos
                svar_pref (SVar.name_of_state_var svar)
                name
                fmt_pos_as_link pos
            ) "@ "
          ) asserts
    ) asserts

    ( fun fmt -> function
      | _ when oracle_info = None -> ()
      | None -> ()
      | Some { LustreContract.assumes } ->
        ( pp_print_list (fun fmt {
            LustreContract.pos ; LustreContract.num ; LustreContract.svar
          } ->
            Format.fprintf fmt
              "// Assumption number %d at %a@ if ! %s%s {@   \
                @[<v>\
                  return Err(@   \
                    \"assumption failure: \
                      %a (assumption number %d)\".to_string()@ \
                  )\
                @]@ \
              } ;"
              num
              fmt_pos_as_link pos
              svar_pref (SVar.name_of_state_var svar)
              fmt_pos_as_link pos
              num
          ) "@ "
        ) fmt assumes
    ) contract

    typ

    ( pp_print_list (fun fmt (_, svar) ->
        let name = svar_pref ^ SVar.name_of_state_var svar in
        Format.fprintf fmt "%s: %s," name name
      ) "@ "
    ) inputs

    ( pp_print_list (fun fmt (_, svar) ->
        let name = svar_pref ^ SVar.name_of_state_var svar in
        Format.fprintf fmt "%s: %s," name name
      ) "@ "
    ) outputs

    ( pp_print_list (fun fmt (_, svar) ->
        let name = svar_pref ^ SVar.name_of_state_var svar in
        Format.fprintf fmt "%s: %s," name name
      ) "@ "
    ) locals

    ( pp_print_list (fun fmt -> function
        | Call (cpt, call) ->
          let name = id_of_call cpt call in
          Format.fprintf fmt "%s: %s," name name
        | _ -> failwith "unreachable"
      ) "@ "
    ) calls ;

  (* Next. *)
  let input_cpt = ref 0 in
  let eqs_next =
    order_equations (fun expr -> expr.E.expr_step) inputs equations
  in
  assert (
    (List.length eqs_next) == (List.length equations)
  ) ;

  Format.fprintf fmt "  \
      fn next(mut self, input: Self::Input) -> Result<Self, String> {@.    \
        @[<v>\
          // |===| Retrieving inputs.@ \
          %a@ @ \
          // |===| Computing next state.@ \
          %a@ @ \
          // |===| Checking assertions.@ \
          %a@ @ \
          // |===| Checking assumptions.@ \
          %a@ @ \
          // |===| Updating next state.@ \
          // |===| Inputs.@ %a@ @ \
          // |===| Outputs.@ %a@ @ \
          // |===| Locals.@ %a@ @ \
          // |===| Calls.@ %a@ @ \
          // |===| Return new state.@ Ok( self )\
        @]@.  \
      }@.@.\
    "

    ( pp_print_list (fun fmt (_, svar) ->
        Format.fprintf fmt "let %s%s = input.%d ;"
          svar_pref
          (SVar.name_of_state_var svar) !input_cpt ;
        input_cpt := 1 + !input_cpt
      ) "@ "
    ) inputs

    ( pp_print_list (fun fmt -> function
        | Eq ((svar, _), expr) ->
          (* Format.printf "eq: %a@.@." pp_print_equation eq ; *)
          expr.E.expr_step
          |> E.cur_term_of_expr (Numeral.succ E.base_offset)
          |> Format.fprintf fmt "let %s%s = %a ;"
            svar_pref
            (SVar.name_of_state_var svar)
            (fmt_term svar_pref)
        | Call (
          cnt, ({ N.call_inputs ; N.call_outputs } as call)
        ) ->
          Format.fprintf fmt
            "\
              let %s = try!( self.%s.next( (@   @[<v>%a,@]@ ) ) ) ;@ \
              let (@   @[<v>%a,@]@ ) = %s.output() ;\
            "
            (id_of_call cnt call)
            (id_of_call cnt call)
            ( pp_print_list (fun fmt (_, svar) ->
                Format.fprintf fmt "%s%s"
                  svar_pref (SVar.name_of_state_var svar)
              ) ",@ "
            ) (I.bindings call_inputs)
            ( pp_print_list (fun fmt (_, svar) ->
                Format.fprintf fmt "%s%s"
                  svar_pref (SVar.name_of_state_var svar)
              ) ",@ "
            ) (I.bindings call_outputs)
            (id_of_call cnt call)
      ) "@ "
    ) eqs_next

    ( pp_print_list (fun fmt (pos, svar) ->
        Format.fprintf fmt
          "// Assertion at %a@ if ! %s%s {@   \
            @[<v>\
              return Err(@   \
                \"assertion failure in system `%s`: %a\".to_string()@ \
              )\
            @]@ \
          } ;"
          fmt_pos_as_link pos
          svar_pref (SVar.name_of_state_var svar)
          name
          fmt_pos_as_link pos
      ) "@ "
    ) asserts

    ( fun fmt -> function
      | None -> ()
      | Some { LustreContract.assumes } ->
        ( pp_print_list (fun fmt {
            LustreContract.pos ; LustreContract.num ; LustreContract.svar
          } ->
            Format.fprintf fmt
              "// Assumption number %d at %a@ if ! %s%s {@   \
                @[<v>\
                  return Err(@   \
                    \"assumption failure: \
                      %a (assumption number %d)\".to_string()@ \
                  )\
                @]@ \
              } ;"
              num
              fmt_pos_as_link pos
              svar_pref (SVar.name_of_state_var svar)
              fmt_pos_as_link pos
              num
          ) "@ "
        ) fmt assumes
    ) contract

    ( pp_print_list (fun fmt (_, svar) ->
        let name = svar_pref ^ SVar.name_of_state_var svar in
        Format.fprintf fmt "self.%s = %s ;" name name
      ) "@ "
    ) inputs

    ( pp_print_list (fun fmt (_, svar) ->
        let name = svar_pref ^ SVar.name_of_state_var svar in
        Format.fprintf fmt "self.%s = %s ;" name name
      ) "@ "
    ) outputs

    ( pp_print_list (fun fmt (_, svar) ->
        let name = svar_pref ^ SVar.name_of_state_var svar in
        Format.fprintf fmt "self.%s = %s ;" name name
      ) "@ "
    ) locals

    ( pp_print_list (fun fmt -> function
        | Call (cpt, call) ->
          let name = id_of_call cpt call in
          Format.fprintf fmt "self.%s = %s ;" name name
        | _ -> failwith "unreachable"
      ) "@ "
    ) calls ;

  (* Output. *)
  outputs
  |> Format.fprintf fmt "  \
    fn output(& self) -> Self::Output {(@.    \
      @[<v>%a@],@.  \
    )}@.\
  " (
    pp_print_list (fun fmt (_, svar) ->
      Format.fprintf fmt "self.%s%s" svar_pref (SVar.name_of_state_var svar)
    ) ",@ "
  ) ;

  (* Output to string. *)
  Format.fprintf fmt "  \
    @[<v>\
      fn output_str(& self) -> String {@   \
        @[<v>\
          format!(@   \
            @[<v>\"%a\",@ %a@]@ \
          )\
        @]@ \
      }\
    @]@.\
  " (
    pp_print_list (fun fmt _ -> Format.fprintf fmt "{}") ", \\@ "
  ) outputs (
    pp_print_list (fun fmt (_, svar) ->
      Format.fprintf fmt "self.%s%s" svar_pref (SVar.name_of_state_var svar)
    ) ",@ "
  ) outputs ;

  Format.fprintf fmt "}@.@." ;

  calls |> List.map (
    function
    | Call (_, { N.call_node_name } ) -> call_node_name
    | _ -> failwith "unreachable"
  )


(* Appends the file name at the end of the path given as first argument to the
path given as second argument. *)
let fmt_cp_target file_path cp_path =
  file_name_of_path file_path |> Format.sprintf "%s/%s" cp_path


(* Copies a file. *)
let cp_file src tgt =
  let ic = open_in src in
  let oc = open_out tgt in
  let rec loop () =
    try (
      let line = input_line ic in
      output_string oc line ;
      output_char oc '\n' ;
      loop ()
    ) with End_of_file ->
      close_in ic ;
      close_out oc
  in
  loop ()


(* Dumps the default [Cargo.toml] file in a directory. Also creates the
[build.rs] file that includes the lustre files in the documentation. *)
let dump_toml is_oracle name dir =
  let rsc_dir = "rsc" in
  let build_file = Format.sprintf "%s/build.rs" rsc_dir in

  (* Generate cargo configuration file. *)
  let out_channel = Format.sprintf "%s/Cargo.toml" dir |> open_out in
  let fmt = Format.formatter_of_out_channel out_channel in
  Format.fprintf fmt
    "\
      [package]@.\
      name = \"%s_%s\"@.\
      version = \"1.0.0\"@.\
      authors = [\"Kind 2 <cesare-tinelli@uiowa.edu>\"]@.\
      build = \"%s\"@.@?\
    "
    name (if is_oracle then "oracle" else "implem") build_file ;

  close_out out_channel ;

  let rsc_path = Format.sprintf "%s/%s" dir rsc_dir in
  let lus_path = Format.sprintf "%s/lus" rsc_path in
  let build_file_path = Format.sprintf "%s/%s" dir build_file in

  (* Create resource / lus directory if needed. *)
  mk_dir rsc_path ;
  mk_dir lus_path ;

  (* Generate build file. *)
  let out_channel = open_out build_file_path in
  let fmt = Format.formatter_of_out_channel out_channel in
  Format.fprintf fmt "\
#![allow(non_upper_case_globals)]

use std::fs::{ OpenOptions, create_dir_all } ;
use std::path::Path ;
use std::io::{ Error, Read, Write, BufRead } ;
use std::io::Result as IoRes ;

static lus_path: & 'static str = \"rsc/lus\" ;
static tgt_path: & 'static str = \"target/doc/src/lus\" ;

fn no_access(e: Error) {
  panic!(
    \"could not access content of folder \\\"{}\\\": {}\", lus_path, e
  )
}

macro_rules! try_io {
  ($e:expr, failwith $( $fail:expr ),+) => (
    match $e {
      Ok(something) => something,
      Err(e) => panic!( $( $fail ),+ , e ),
    }
  )
}

fn copy_lus(from: & Path, to: & Path) {
  let src = try_io!(
    OpenOptions::new().read(true).open(from),
    failwith \"could not open source file {}: {}\", from.to_str().unwrap()
  ) ;
  let tgt = & mut try_io!(
    OpenOptions::new().create(true).write(true).truncate(true).open(to),
    failwith \"could not open target file {}: {}\", to.to_str().unwrap()
  ) ;

  let format = match Format::read(src) {
    Ok(format) => format,
    Err(e) => panic!(
      \"could not read source file {}: {}\", from.to_str().unwrap(), e
    ),
  } ;

  try_io!(
    write_header(tgt, from),
    failwith \"could not write target file {}: {}\", to.to_str().unwrap()
  ) ;

  try_io!(
    format.write(tgt),
    failwith \"could not write target file {}: {}\", to.to_str().unwrap()
  ) ;

  try_io!(
    write_footer(tgt),
    failwith \"could not write target file {}: {}\", to.to_str().unwrap()
  )
}

fn main() {
  let path = Path::new(lus_path) ;

  if ! path.is_dir() {
    panic!(
      \"expected to find source lustre files in \\\"{}\\\" but {} is a file\",
      lus_path, lus_path
    )
  } ;

  if ! path.exists() {
    panic!(
      \"expected to find source lustre files in \\\"{}\\\" but {} does not exist\",
      lus_path, lus_path
    )
  } ;

  // Create target dir.
  match create_dir_all(tgt_path) {
    Ok(()) => (),
    Err(e) => panic!(
      \"could not create target directory \\\"{}\\\": {}\", tgt_path, e
    ),
  } ;

  match path.read_dir() {

    Ok(entries) => for entry in entries.into_iter() {
      use std::path::PathBuf ;
      match entry {

        Ok(entry) => {
          let path_buf = entry.path() ;
          let src = path_buf.as_path() ;
          match src.file_name() {
            Some(name) => {
              let mut tgt = PathBuf::new() ;
              tgt.push(tgt_path) ;
              tgt.push(name) ;
              tgt.set_extension(\"lus.html\") ;
              copy_lus(src, & tgt)
            },
            None => (),
          }
        },

        Err(e) => no_access(e),
      }
    },

    Err(e) => no_access(e),
  }
}



struct Format {
  lines: Vec<String>,
}
impl Format {
  fn mark_lines<W: Write>(& self, w: & mut W) -> IoRes<()> {
    for n in 1..(self.lines.len() + 1) {
      try!( write!(w, \"<span id=\\\"{}\\\">{: >4}</span>\\n\", n, n) )
    } ;

    write!(w, \"</pre><pre class='rust '>\\n\")
  }
  pub fn write<W: Write>(self, w: & mut W) -> IoRes<()> {
    try!( self.mark_lines(w) ) ;
    for line in self.lines.into_iter() {
      try!( write!(w, \"{}\\n\", line) )
    } ;
    Ok(())
  }
  pub fn read<Reader: Read>(reader: Reader) -> Result<Self,String> {
    use std::io::BufReader ;
    let reader = BufReader::new(reader) ;

    let mut lines = Vec::with_capacity(1000) ;
    for line in reader.lines() {
      match line {
        Ok(line) => lines.push(line.to_string()),
        Err(e) => return Err( format!(\"could not read file: {}\", e) ),
      }
    } ;

    Ok( Format { lines: lines } )
  }
}








fn write_header<W: Write>(w: & mut W, src: & Path) -> IoRes<()> {
  let src = src.file_name().unwrap().to_str().unwrap() ;
  write!(
    w,
    \"
<!DOCTYPE html>
<html lang=\\\"en\\\">
<head>
    <meta charset=\\\"utf-8\\\">
    <meta name=\\\"viewport\\\" content=\\\"width=device-width, initial-scale=1.0\\\">
    <meta name=\\\"generator\\\" content=\\\"Kind 2\\\">
    <meta name=\\\"description\\\" content=\\\"Source to the Lustre file `{}`.\\\">
    <meta name=\\\"keywords\\\" content=\\\"rust, rustlang, rust-lang\\\">

    <title>{}.html -- source</title>

    <link rel=\\\"stylesheet\\\" type=\\\"text/css\\\" href=\\\"../../rustdoc.css\\\">
    <link rel=\\\"stylesheet\\\" type=\\\"text/css\\\" href=\\\"../../main.css\\\">

    
    
</head>
<body class=\\\"rustdoc\\\">
    <!--[if lte IE 8]>
    <div class=\\\"warning\\\">
        This old browser is unsupported and will most likely display funky
        things.
    </div>
    <![endif]-->

    

    <nav class=\\\"sidebar\\\">
        
        
    </nav>

    <nav class=\\\"sub\\\">
        <form class=\\\"search-form js-only\\\">
            <div class=\\\"search-container\\\">
                <input class=\\\"search-input\\\" name=\\\"search\\\"
                       autocomplete=\\\"off\\\"
                       placeholder=\\\"Click or press ‘S’ to search, ‘?’ for more options…\\\"
                       type=\\\"search\\\">
            </div>
        </form>
    </nav>

    <section id='main' class=\\\"content source\\\"><pre class=\\\"line-numbers\\\">\
    \",
    src,
    src
  )
}

fn write_footer<W: Write>(w: & mut W) -> IoRes<()> {
  write!(
    w,
    \"\
</pre>
</section>
    <section id='search' class=\\\"content hidden\\\"></section>

    <section class=\\\"footer\\\"></section>

    <aside id=\\\"help\\\" class=\\\"hidden\\\">
        <div>
            <h1 class=\\\"hidden\\\">Help</h1>

            <div class=\\\"shortcuts\\\">
                <h2>Keyboard Shortcuts</h2>

                <dl>
                    <dt>?</dt>
                    <dd>Show this help dialog</dd>
                    <dt>S</dt>
                    <dd>Focus the search field</dd>
                    <dt>&larrb;</dt>
                    <dd>Move up in search results</dd>
                    <dt>&rarrb;</dt>
                    <dd>Move down in search results</dd>
                    <dt>&#9166;</dt>
                    <dd>Go to active search result</dd>
                </dl>
            </div>

            <div class=\\\"infos\\\">
                <h2>Search Tricks</h2>

                <p>
                    Prefix searches with a type followed by a colon (e.g.
                    <code>fn:</code>) to restrict the search to a given type.
                </p>

                <p>
                    Accepted types are: <code>fn</code>, <code>mod</code>,
                    <code>struct</code>, <code>enum</code>,
                    <code>trait</code>, <code>type</code>, <code>macro</code>,
                    and <code>const</code>.
                </p>

                <p>
                    Search functions by type signature (e.g.
                    <code>vec -> usize</code> or <code>* -> vec</code>)
                </p>
            </div>
        </div>
    </aside>

    

    <script>
        window.rootPath = \\\"../../\\\";
        window.currentCrate = \\\"system_oracle\\\";
        window.playgroundUrl = \\\"\\\";
    </script>
    <script src=\\\"../../jquery.js\\\"></script>
    <script src=\\\"../../main.js\\\"></script>
    
    <script defer src=\\\"../../search-index.js\\\"></script>
</body>
</html>
    \"
  )
}
  " ;
  close_out out_channel ;

  (* Copy all input files to the lus directory. *)
  let cp src = fmt_cp_target src lus_path |> cp_file src in
  Flags.all_input_files () |> List.iter (
    fun file -> cp file
  )




let to_rust oracle_info target find_sub top =


  (* Format.printf "node: @[<v>%a@]@.@." (N.pp_print_node false) top ; *)
  let top_name, top_type = mk_id_legal top.N.name, mk_id_type top.N.name in
  (* Creating project directory if necessary. *)
  mk_dir target ;
  (* Creating source dir. *)
  let src_dir = Format.sprintf "%s/src" target in
  mk_dir src_dir ;
  (* Dump toml configuration file. *)
  dump_toml (oracle_info <> None) top_name target ;
  (* Opening writer to file. *)
  let file = Format.sprintf "%s/main.rs" src_dir in
  let out_channel = open_out file in
  let fmt = Format.formatter_of_out_channel out_channel in
  Format.pp_set_margin fmt max_int ;

  (* Write prefix and static stuff. *)
  Format.fprintf
    fmt "%a@.%a@.@."
    ( match oracle_info with
      | None -> fmt_prefix_implem top_name
      | _    -> fmt_prefix_oracle top_name
    ) top_type
    fmt_main ()
    (* (consts "unimplemented" "unimplemented" "unimplemented") *) ;

  let rec compile is_top systems compiled = function
    | node :: nodes ->
      let systems, compiled, nodes =
        if Id.Set.mem node.N.name compiled |> not then (
          (* Oracle info only makes sense for the top node. *)
          let oracle_info = if not is_top then None else oracle_info in
          (* Remembering we compiled this node. *)
          let compiled = Id.Set.add node.N.name compiled in
          
          node :: systems,
          compiled,
          nodes @ (
            (* Compiling nodes, getting subnodes back. *)
            node_to_rust oracle_info is_top fmt node
            (* Discarding subnodes we already compiled. *)
            |> List.fold_left (fun l call_id ->
              if Id.Set.mem call_id compiled |> not
              then (Id.to_scope call_id |> find_sub) :: l else l
            ) []
          )
        ) else systems, compiled, nodes
      in
      compile false systems compiled nodes
    | [] -> systems
  in

  let systems = compile true [] Id.Set.empty [ top ] in


  Format.fprintf fmt "@.@." ;


  Format.fprintf fmt "%a@.@." fmt_helpers systems ;

  (* Flush and close file writer. *)
  close_out out_channel

(*
let print_trie desc =
  Format.printf "%s: @[<v>%a@]@.@."
    desc
    ( I.pp_print_trie
      ( fun fmt (lst, svar) ->
        Format.fprintf fmt "%a -> %a"
          (I.pp_print_one_index true) (List.hd lst)
          SVar.pp_print_state_var svar
      ) "@ "
    )
*)

let implem_to_rust = to_rust None

let oracle_to_rust target find_sub top =

  (* Successor of the max index of some trie. *)
  let next_index_of trie = I.top_max_index trie |> succ in

  (* let is_ghost svar =
    try (
      match SVM.find svar top.N.state_var_source_map with
      | N.Ghost -> true
      | _ -> false
    ) with Not_found -> false
  in *)
  (* let print_svar_source svar =
    try
      Format.printf "%a: %a@."
        SVar.pp_print_state_var svar
        N.pp_print_state_var_source (SVM.find svar top.N.state_var_source_map)
    with
      Not_found -> Format.printf "%a: not found@." SVar.pp_print_state_var svar
  in

  Format.printf "@.@.Inputs:@." ;
  top.N.inputs |> I.bindings
  |> List.iter (fun (_, svar) -> print_svar_source svar) ;
  Format.printf "@.@.Outputs:@." ;
  top.N.outputs |> I.bindings
  |> List.iter (fun (_, svar) -> print_svar_source svar) ;
  Format.printf "@.@.Locals:@." ;
  top.N.locals |> List.iter (
    fun loc -> I.bindings loc |> List.iter (
      fun (_, svar) -> print_svar_source svar
    )
  ) ;
  Format.printf "@.@.Equations:@." ;
  top.N.equations |> List.iter (
    fun eq -> Format.printf "  %a@." (N.pp_print_node_equation false) eq
  ) ;
  Format.printf "@.@.Calls:@." ;
  top.N.calls |> List.iter (
    fun call -> Format.printf "  %a@." (N.pp_print_call false) call
  ) ;
  Format.printf "@.@." ; *)

  (* Appends two tries. *)
  let append lhs rhs =
    (I.empty, 0)
    |> I.fold (
      fun _ svar (trie, cnt) -> I.add [I.TupleIndex cnt] svar trie, cnt + 1
    ) lhs
    |> I.fold (
      fun _ svar (trie, cnt) -> I.add [I.TupleIndex cnt] svar trie, cnt + 1
    ) rhs
    |> fst
  in
  
  (* Adding to inputs. *)
  let inputs = try append top.N.inputs top.N.outputs with e -> Format.asprintf "exc: %s" (Printexc.to_string e) |> failwith in

  (* Outputs are guarantees and mode ensures. *)
  let (outputs, output_svars, output_eqs), oracle_info =
    match top.N.contract with
    | None -> failwith "cannot generate oracle for contract-free node"
    | Some contract ->
      let outputs, output_svars =
        contract.C.guarantees
        |> List.fold_left (fun (trie, outs) (svar, _) ->
          I.add (
            [ I.ListIndex (next_index_of trie) ]
          ) svar.C.svar trie,
          SVS.add svar.C.svar outs
        ) (I.empty, SVS.empty)
      in
      contract.C.modes |> List.fold_left (
        fun (trie, outs, eqs) {
          C.name ; C.requires ; C.ensures
        } ->
          ensures |> List.fold_left (
            fun (trie, outs, eqs) svar ->
              let output =
                SVar.mk_state_var
                  ( Format.sprintf "mode_%s_%d"
                      (Id.string_of_ident false name) svar.C.num
                  ) [] Type.t_bool
              in
              let expr =
                requires
                |> List.map (fun req -> E.mk_var req.C.svar)
                |> E.mk_and_n
                |> fun lhs -> E.mk_impl lhs (E.mk_var svar.C.svar)
              in
              I.add (
                [ I.ListIndex (next_index_of trie) ]
              ) output trie,
              SVS.add output outs,
              ((output, []), expr) :: eqs
          ) (trie, outs, eqs)
      ) (outputs, output_svars, []),
      Some (
        contract.C.assumes, contract.C.guarantees, contract.C.modes
      )
  in

  let locals =
    top.N.locals |> List.filter (
      fun loc ->
        I.bindings loc
        |> List.exists ( fun (_, svar) -> SVS.mem svar output_svars |> not )
    )
  in

  (* Format.printf "@.@.Inputs:@." ;
  inputs |> I.bindings
  |> List.iter (fun (_, svar) -> print_svar_source svar) ;
  Format.printf "@.@.Outputs:@." ;
  outputs |> I.bindings
  |> List.iter (fun (_, svar) -> print_svar_source svar) ;
  Format.printf "@.@.Locals:@." ;
  locals |> List.iter (
    fun loc -> I.bindings loc |> List.iter (
      fun (_, svar) -> print_svar_source svar
    )
  ) ;
  Format.printf "@.@.Equations:@." ;
  top.N.equations |> List.iter (
    fun eq -> Format.printf "  %a@." (N.pp_print_node_equation false) eq
  ) ;
  Format.printf "@.@." ; *)

  (* Only keep equations about ghost variables. *)
  let equations =
    (* let outputs = I.bindings top.N.outputs |> List.map snd in
    top.N.equations |> List.filter (
      fun (svar, _, _) ->
        is_ghost svar && (List.mem svar outputs |> not)
    ) *)
    output_eqs @ top.N.equations
  in

  (* Only keep calls about ghost variables. *)
  let calls =
    (* top.N.calls |> List.filter (
      fun ({ N.call_inputs }: N.node_call) ->
        I.bindings call_inputs |> List.for_all (
          fun (_, svar) -> is_ghost svar
        )
    ) *)
    top.N.calls
  in

  (* Creating node and compiling. *)
  let oracle =
    { top with
      N.inputs = inputs ;
      N.outputs = outputs ;
      N.locals = locals ;
      N.equations = equations ;
      N.calls = calls ;
    }
  in

  (* Compiling oracle. *)
  oracle |> to_rust oracle_info target find_sub ;

  match oracle_info with
  | None ->
    Format.asprintf
      "no contract for node %a" (Id.pp_print_ident false) top.N.name
    |> failwith
  | Some (_, guarantees, modes) -> (
    Format.asprintf "%a" (Id.pp_print_ident false) top.N.name,
    guarantees |> List.map (
      fun ({ C.pos ; C.num }, _) -> pos, num
    ),
    modes |> List.fold_left (
      fun l { C.name ; C.ensures } ->
        ensures
        |> List.map (
          fun { C.pos ; C.num } ->
            Id.string_of_ident false name, pos, num
        )
        |> fun res -> List.rev_append res l
    ) []
  ) ;




(* 
   Local Variables:
   compile-command: "make -C .. -k"
   indent-tabs-mode: nil
   End: 
*)




