/*
 * intf.c
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>

#include "intf.h"
#include "util.h"
#include "curs_util.h"

enum {
    overview,
    detailed,
    graphic
} m_bandwidth = overview;

enum {
    minute,
    hour,
    day
} m_b_graphic = minute;


/*
 * must be implemented by the calling application
 * is called by this module whenever a fatal error occurs
 */
extern void   quit (const char *fmt, ...);


/*
 * globals
 */
static int     bdev                    = 0;
static int     initialized             = 0;
static intf_t  intf[MAX_IF];


/*
 * private function prototypes
 */
static void intf_calc_graphic(struct intf_g_s *table);
static void intf_print_graphic(void);
static void intf_calc_rate(void);



void
intf_readfile(const char *path)
{

    FILE *       fd;
    char         buf[512];
    char *       p;
    char *       s;
    int          inum         = 0;
    int          i;
    int          n;
    int          w;
    time_t       now = time(0);

    if (!initialized) {
        memset(&intf, 0, sizeof(intf));
        initialized = 1;
    }

    for (i=0; i < MAX_IF; i++) {
        if (intf[i].name[0])
            intf[i].to_remove = 1;
    }

    errno = 0;
    if ( !(fd = fopen(path, "r")) )
        quit("Unable to open file %s: %s", path, strerror(errno));

    /*
     * ignore header lines
     */
    fgets(buf, sizeof(buf), fd);
    fgets(buf, sizeof(buf), fd);

    /*
     * /proc/net/dev example:
     *
     * Inter-|   Receive                                                |  Transmit
     * face |bytes    packets errs drop fifo frame compressed multicast|bytes    packets errs drop fifo colls carrier compressed
     *     lo:  450444    5419    0    0    0     0          0         0   450444    5419    0    0    0     0       0          0
     *   eth0:15019201   48550    0    0    0     0          0         0  7288864   62412    0    0    0     0       0          0
     */

    memset(buf, 0, sizeof(buf));

    for (; fgets(buf, sizeof(buf), fd) && inum < (MAX_IF - 1); inum++) {

        /*
         * ignore empty lines
         */
        if (buf[0] == '\r' || buf[0] == '\n')
            continue;

        /*
         *        ^  eth0: ...
         * look for -----^
         */
        if (!(p = strchr(buf, ':')))
            continue;

        /*
         * replace ':' with \0, p points now to '   eth0'
         * point s to '   15019201   48550 ....'
         */
        *p = '\0';
        s = p+1;

        /*
         * ignore all precending spaces, p points now to 'eth0'
         */
        for (p = &buf[0]; *p == ' '; p++);

        /*
         * check if this interface is already known
         */
        for (i=0; i < MAX_IF; i++)
            if (!strcmp(p, intf[i].name))
                goto j0;

        /*
         * nada, look for a free slot to save this interface
         */
        for (i=0; i < MAX_IF; i++) {

            if (!intf[i].name[0]) {

                /*
                 * hai! found one, store interface name into it and
                 * update time
                 */
                strncpy(intf[i].name, p, sizeof(intf[i].name));
                intf[i].time = now;
                intf[i].g_time = now;

                goto j0;
            }
        }

        /*
         * no more free slots, ignore this line
         */
        continue;

j0:

        #define I intf[i]

        /*
         * this interface is still up, don't remove it
         */
        I.to_remove = 0;

        w = sscanf(s, "%lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu\n",
            &I.rx_bytes, &I.rx_packets, &I.rx_errors, &I.rx_drop, &I.rx_fifo,
            &I.rx_frame, &I.rx_compressed, &I.rx_multicast,
            &I.tx_bytes, &I.tx_packets, &I.tx_errors, &I.tx_drop, &I.tx_fifo,
            &I.tx_frame, &I.tx_compressed, &I.tx_multicast);

        if (w != 16)
            continue;

        /*
         * do this thing in here at least every second
         */
        if ( (now - intf[i].g_time) >= 1 ) {

            if (!intf[i].s.rx_bytes_old)
                intf[i].s.rx_bytes_old = intf[i].rx_bytes;

            if (!intf[i].s.tx_bytes_old)
                intf[i].s.tx_bytes_old = intf[i].tx_bytes;

            intf[i].s.rx_bytes[intf[i].s.i] = (intf[i].rx_bytes - intf[i].s.rx_bytes_old);
            intf[i].s.tx_bytes[intf[i].s.i] = (intf[i].tx_bytes - intf[i].s.tx_bytes_old);

            intf[i].s.rx_bytes_old = intf[i].rx_bytes;
            intf[i].s.tx_bytes_old = intf[i].tx_bytes;

            intf[i].s.rx_bytes[intf[i].s.i] /= (now - intf[i].g_time);
            intf[i].s.tx_bytes[intf[i].s.i] /= (now - intf[i].g_time);

            intf_calc_graphic(&intf[i].s);

            /*
             * do this thing in here after 60 second iterations (= 1 minute)
             */
            if (intf[i].s.i == 60) {
                intf[i].s.i = 0;

                /*
                 * summarize all values in seconds table and assign this
                 * value to the current slot in the minutes table
                 * dividide by 60 to get the average for the last minute.
                 */
                for (n=0; n < 60; n++) {
                    intf[i].m.rx_bytes[intf[i].m.i] += intf[i].s.rx_bytes[n];
                    intf[i].m.tx_bytes[intf[i].m.i] += intf[i].s.tx_bytes[n];
                }

                intf[i].m.rx_bytes[intf[i].m.i] /= 60;
                intf[i].m.tx_bytes[intf[i].m.i] /= 60;

                intf_calc_graphic(&intf[i].m);

                /*
                 * do this thing in here after 60 minute iterations (= 1 hour)
                 */
                if (intf[i].m.i == 60) {
                    intf[i].m.i = 0;

                    for (n=0; n < 60; n++) {
                        intf[i].h.rx_bytes[intf[i].h.i] += intf[i].m.rx_bytes[n];
                        intf[i].h.tx_bytes[intf[i].h.i] += intf[i].m.tx_bytes[n];
                    }

                    intf[i].h.rx_bytes[intf[i].h.i] /= 60;
                    intf[i].h.tx_bytes[intf[i].h.i] /= 60;

                    intf_calc_graphic(&intf[i].h);

                    if (intf[i].h.i == 60)
                        intf[i].h.i = 0;
                }
            }
            
            intf[i].s.rx_bytes[intf[i].s.i] = 0;
            intf[i].s.tx_bytes[intf[i].s.i] = 0;

            intf[i].g_time = now;
        }
        
    }

    fclose(fd);

    for (i=0; i < MAX_IF; i++) {
        if (intf[i].to_remove)
            memset(&intf[i], 0, sizeof(struct intf_s));
    }

    intf_calc_rate();
    
}


void
intf_calc_rate(void)
{
    int         i;
    time_t      now   = time(0);
    time_t      t;

    for (i=0; i < MAX_IF; i++) {

        if (intf[i].name[0]) {

            if (!intf[i].rx_bytes_old)
                intf[i].rx_bytes_old = intf[i].rx_bytes;
            if (!intf[i].tx_bytes_old)
                intf[i].tx_bytes_old = intf[i].tx_bytes;
            if (!intf[i].rx_packets_old)
                intf[i].rx_packets_old = intf[i].rx_packets;
            if (!intf[i].tx_packets_old)
                intf[i].tx_packets_old = intf[i].tx_packets;

            t = (now - intf[i].time);

            if ( t >= 3) {

                t++;

                intf[i].rx_bytes_rate += (intf[i].rx_bytes - intf[i].rx_bytes_old);
                intf[i].tx_bytes_rate += (intf[i].tx_bytes - intf[i].tx_bytes_old);
                intf[i].rx_packets_rate += (intf[i].rx_packets - intf[i].rx_packets_old);
                intf[i].tx_packets_rate += (intf[i].tx_packets - intf[i].tx_packets_old);

                intf[i].rx_bytes_rate /= t;
                intf[i].tx_bytes_rate /= t;
                intf[i].rx_packets_rate /= t;
                intf[i].tx_packets_rate /= t;

                if (intf[i].rx_bytes_old > intf[i].rx_bytes)
                    intf[i].rx_bytes_rate = 0;

                if (intf[i].tx_bytes_old > intf[i].tx_bytes)
                    intf[i].tx_bytes_rate = 0;

                if (intf[i].rx_packets_old > intf[i].rx_packets)
                    intf[i].rx_packets_rate = 0;

                if (intf[i].tx_packets_old > intf[i].tx_packets)
                    intf[i].tx_packets_rate = 0;

                intf[i].rx_bytes_old = intf[i].rx_bytes;
                intf[i].tx_bytes_old = intf[i].tx_bytes;
                intf[i].rx_packets_old = intf[i].rx_packets;
                intf[i].tx_packets_old = intf[i].tx_packets;

                intf[i].time = now;

            }
        }
    }
}


void
intf_calc_graphic(struct intf_g_s *table)
{
    int n, w, q;
    unsigned int rx, rx_min=0, rx_max=0, rx_item=0;
    unsigned int tx, tx_min=0, tx_max=0, tx_item=0;

    memset(table->rx_g, '.', sizeof(table->rx_g));
    for (n=0; n < 6; n++) table->rx_g[n][60] = '\0';
    memset(table->tx_g, '.', sizeof(table->tx_g));
    for (n=0; n < 6; n++) table->tx_g[n][60] = '\0';

    rx_min = rx_max = table->rx_bytes[0];
    tx_min = tx_max = table->tx_bytes[0];

    for (n=0; n < 60; n++) {
        if (rx_min > table->rx_bytes[n]) rx_min = table->rx_bytes[n];
        if (rx_max < table->rx_bytes[n]) rx_max = table->rx_bytes[n];

        if (tx_min > table->tx_bytes[n]) tx_min = table->tx_bytes[n];
        if (tx_max < table->tx_bytes[n]) tx_max = table->tx_bytes[n];
    }

    rx_item = (rx_max - rx_min) / 6;
    tx_item = (tx_max - tx_min) / 6;

    for (n=0; n < 6; n++) {
        table->rx_scale[n] = (float) (rx_min + ((n+1) * rx_item) );
        table->tx_scale[n] = (float) (tx_min + ((n+1) * tx_item) );
    }

    table->rx_scale[5] = (float) rx_max;
    table->tx_scale[5] = (float) tx_max;

    /* fill up the 2 dimensional arrays */
    for (n=0; n < 60; n++) {

        rx = table->rx_bytes[n];
        tx = table->tx_bytes[n];

        for (w=0; w < 6; w++) {
            if (!tx) continue;
            if ( (float) rx <= table->rx_scale[w]) {
                for (q=0; q < w+1; q++) table->rx_g[q][n] = '*';
                break;
            }
        }

        for (w=0; w < 6; w++) {
            if (!tx) continue;
            if ( (float) tx <= table->tx_scale[w]) {
                for (q=0; q < w+1; q++) table->tx_g[q][n] = '*';
                break;
            }
        }
    }

    table->i++;
}


void
intf_print_bandwidth(void)
{
    int i;
    char *unit;

    NEXT_ROW;

    if (COLS <= 63) {
        printw("Screen not wide enough");
        return;
    }

    addstr("#   Interface           RX Rate      RX #     TX Rate      TX #");
    NEXT_ROW;
    hline(ACS_HLINE, COLS);

    for (i=0; i < MAX_IF; i++) {
        if (intf[i].name[0]) {
            NEXT_ROW;

            if (row >= LINES-1)
                return;

            if (i == bdev)
                attrset(A_REVERSE);
            printw("%-3d ", i);
            if (i == bdev)
                attroff(A_REVERSE);
            printw("%-14s ", intf[i].name);
            printw("%10.2f", sumup(intf[i].rx_bytes_rate, &unit));
            addstr(unit);
            printw("%10.1f", (float) intf[i].rx_packets_rate);
            printw("%10.2f", sumup(intf[i].tx_bytes_rate, &unit));
            addstr(unit);
            printw("%10.1f", (float) intf[i].tx_packets_rate);
        }
    }

    if ( (LINES - row) >= 23 ) {
        row++;
        intf_print_graphic();
    }
}

void
intf_print_graphic(void)
{
    int i, w, gi;
    char *u2=NULL, *unit=NULL, *rx_g, *tx_g;

    float rx_scale[6], tx_scale[6];

    if (COLS < 80) {
        NEXT_ROW;
        print_full_line("Need 80 characters wide screen");
        return;
    }

    if (LINES < 22) {
        NEXT_ROW;
        print_full_line("Need 22 lines high screen");
        return;
    }
    

    i = bdev;

    if (intf[i].name[0]) {

        switch (m_b_graphic) {
            case minute:
                gi = intf[i].s.i;
                u2 = "s";
                memcpy(&rx_scale, &intf[i].s.rx_scale, sizeof(rx_scale));
                memcpy(&tx_scale, &intf[i].s.tx_scale, sizeof(tx_scale));
                rx_g = &intf[i].s.rx_g[0][0];
                tx_g = &intf[i].s.tx_g[0][0];
                break;

            case hour:
                gi = intf[i].m.i;
                u2 = "m";
                memcpy(&rx_scale, &intf[i].m.rx_scale, sizeof(rx_scale));
                memcpy(&tx_scale, &intf[i].m.tx_scale, sizeof(tx_scale));
                rx_g = &intf[i].m.rx_g[0][0];
                tx_g = &intf[i].m.tx_g[0][0];
                break;

            case day:
                gi = intf[i].h.i;
                u2 = "h";
                memcpy(&rx_scale, &intf[i].h.rx_scale, sizeof(rx_scale));
                memcpy(&tx_scale, &intf[i].h.tx_scale, sizeof(tx_scale));
                rx_g = &intf[i].h.rx_g[0][0];
                tx_g = &intf[i].h.tx_g[0][0];
                break;

            default:
                return;
        }

        /* display graph for incoming traffic */
        if (rx_scale[4] >= 1073741824) {
            for (w=0; w < 6; w++) rx_scale[w] /= 1073741824;
            unit = "GB";
        } else if (rx_scale[4] >= 1048576) {
            for (w=0; w < 6; w++) rx_scale[w] /= 1048576;
            unit = "MB";
        } else if (rx_scale[4] >= 1024) {
            for (w=0; w < 6; w++) rx_scale[w] /= 1024;
            unit = "KB";
        } else {
            unit = "B ";
        }

        move(row+3,0);

        printw("RX");

        move(row+2,7);
        addstr(unit);
        for (w=0; w < 6; w++) {
            move(row+8-w,2);
            printw("%8.2f ", rx_scale[w]);
            addstr(rx_g + (w * 61) );
        }
        move(row+9,11);
        printw("1   5   10   15   20   25   30   35   40   45   50   55   60 %s", u2);
        move(row+10,11);
        hline(' ', COLS-12);
        move(row+10,11+gi);
        addch('^');

        /* display graph for outgoing traffic */
        if (tx_scale[4] >= 1073741824) {
            for (w=0; w < 6; w++) tx_scale[w] /= 1073741824;
            unit = "GB";
        } else if (tx_scale[4] >= 1048576) {
            for (w=0; w < 6; w++) tx_scale[w] /= 1048576;
            unit = "MB";
        } else if (tx_scale[4] >= 1024) {
            for (w=0; w < 6; w++) tx_scale[w] /= 1024;
            unit = "KB";
        } else {
            unit = "B ";
        }

        move(row+12,0);

        printw("TX");

        move(row+11,7);
        addstr(unit);
        for (w=0; w < 6; w++) {
            move(row+17-w,2);
            printw("%8.2f ", tx_scale[w]);
            addstr(tx_g + (w * 61) );
        }
        move(row+18,11);
        printw("1   5   10   15   20   25   30   35   40   45   50   55   60 %s", u2);
        move(row+19,11);
        hline(' ', COLS-12);
        move(row+19,11+gi);
        addch('^');

        move(row+20, 1);
        print_full_line("Time scale: s:Seconds m:Minutes h:Hours");

        row += 20;
    }
}

void
intf_print_detailed(void)
{
    int i;
    float u;
    char *unit;

    if ( COLS < 40 ) {
        print_full_line("Need 40 characters wide screen");
        return;
    }

    if ( LINES < 12 ) {
        print_full_line("Need 12 lines high screen");
        return;
    }

    i = bdev;

    if (intf[i].name[0]) {

        NEXT_ROW;
        NEXT_ROW;
        printw("                          RX               TX");
        NEXT_ROW;
        NEXT_ROW;
        u = sumup(intf[i].rx_bytes, &unit);
        printw(" Bytes:         %12.2f %s", u, unit);
        u = sumup(intf[i].tx_bytes, &unit);
        printw("  %12.2f %s", u, unit);

        NEXT_ROW;
        printw(" Packets:       %12lu     %12lu", intf[i].rx_packets, intf[i].tx_packets);
        NEXT_ROW;
        printw(" Errors:        %12lu     %12lu", intf[i].rx_errors, intf[i].tx_errors);
        NEXT_ROW;
        printw(" Drop:          %12lu     %12lu", intf[i].rx_drop, intf[i].tx_drop);
        NEXT_ROW;
        printw(" Fifo:          %12lu     %12lu", intf[i].rx_fifo, intf[i].tx_fifo);
        NEXT_ROW;
        printw(" Frame:         %12lu     %12lu", intf[i].rx_frame, intf[i].tx_frame);
        NEXT_ROW;
        printw(" Compressed:    %12lu     %12lu", intf[i].rx_compressed, intf[i].tx_compressed);
        NEXT_ROW;
        printw(" Multicast:     %12lu     %12lu", intf[i].rx_multicast, intf[i].tx_multicast);
    }
}

int
intf_handle_input(int ch)
{
    int w;

    switch (ch) {
        case 'o':
        case 0x1b: /* ESC */
            clear();
            m_bandwidth = overview;
            return 1;

        case 'd':
            clear();
            m_bandwidth = detailed;
            return 1;

        case 's':
            clear();
            m_b_graphic = minute;
            return 1;

        case 'm':
            clear();
            m_b_graphic = hour;
            return 1;

        case 'h':
            clear();
            m_b_graphic = day;
            return 1;

        case 'g':
        case 0x0d: /* CR */
            clear();
            m_bandwidth = graphic;
            return 1;

        case KEY_DOWN:
            if (bdev+1 == MAX_IF-1)
                break;

            for (w=bdev+1; !intf[w].name[0] && w < MAX_IF; w++);

            if (w < MAX_IF && intf[w].name[0]) {
                bdev = w;
                return 1;
            }
            break;

        case KEY_UP:
            if (bdev-1 < 0)
                break;

            for (w=bdev-1; !intf[w].name[0] && w >= 0; w--);

            if (w >= 0 && intf[w].name[0]) {
                bdev = w;
                return 1;
            }
            break;

        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
            if ( (ch - 48) >= 0 && (ch - 48) <= 9 ) {
                if (intf[ch - 48].name[0]) {
                    bdev = (ch - 48);
                    return 1;
                }
            }
            break;
    }

    return 0;
}

void
intf_draw(void)
{
    switch (m_bandwidth) {
        case overview:
            print_top("Bandwidth Usage  o:Overview d:Detailed g:Graphical");
            intf_print_bandwidth();
            print_to_end();
            break;

        case detailed:
            if (intf[bdev].name[0]) {
                print_top("Detailed statistics for %s", intf[bdev].name);
                intf_print_detailed();
                print_to_end();
            }
            break;

        case graphic:
            if (intf[bdev].name[0]) {
                print_top("Graphical statistics for %s  Unit [S:Seconds M:Minutes H:Hours]",
                    intf[bdev].name);
                intf_print_graphic();
            }
            break;
    }

}
