/* 
 * nstats
 *
 * prints some statistics about the given network traffic.
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <pcap.h>
#include <netdb.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <netinet/ether.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <netinet/icmp6.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <net/if.h>

#include "pcap.h"
#include "packet.h"
#include "conn.h"
#include "util.h"
#include "stats.h"

#define MAX_IF 255

#define OUT_ALL              (1<<0)
#define OUT_ETHERNET         (1<<1)
#define OUT_IP               (1<<2)
#define OUT_TOS              (1<<3)
#define OUT_PROT_BREAK       (1<<4)
#define OUT_PORT             (1<<5)
#define OUT_TCP              (1<<6)
#define OUT_ICMP             (1<<7)
#define OUT_DESTU            (1<<8)
#define OUT_REDIR            (1<<9)
#define OUT_TEXC             (1<<10)
#define OUT_IP6              (1<<11)
#define OUT_CONNS            (1<<12)
#define OUT_ECN              (1<<13)
#define OUT_ICMP6            (1<<14)
#define OUT_DESTU6           (1<<15)
#define OUT_TEXC6            (1<<16)
#define OUT_PARA6            (1<<17)
#define OUT_FRAG             (1<<18)

char if_addr[256][MAX_IF];
char dname[256];

int output_mode = 0;
int promisc = 1;
int runtime = 0;
int list_length = 10;
int do_resolve = 1;

struct ifconf ifc;

void print_output(void);


void
quit(const char *msg)
{
    fprintf(stderr, "%s\n", msg);
    exit(-1);
}


void
sig_abort(int unusued)
{
    print_output();
    exit(0);
}

void
sig_alrm(int unusued)
{
    print_output();
    exit(0);
}

int main(int argc, char *argv[])
{
    pcap_t *p;
    char err[PCAP_ERRBUF_SIZE], cf[FILENAME_MAX], *s, *t;
    char filter[4096], device[256], *dev;
    struct bpf_program bpf;
    char *buf, *ptr;
    struct ifreq *ifr;
    u_char *args = NULL;
    int i;

    memset(err, 0, sizeof(err));
    memset(cf, 0, sizeof(cf));
    memset(filter, 0, sizeof(filter));
    memset(device, 0, sizeof(device));
    memset(&stats, 0, sizeof(stats));
    memset(&bpf, 0, sizeof(bpf));

    dev = &device[0];

    while ( 1 ) {
        int c;

        if ( (c = getopt(argc, argv, "r:vVo:pf:i:t:l:n")) == -1)
            break;

        switch (c) {
            case 'r':
                strncpy(cf, optarg, sizeof(cf));
                break;
            case 'v':
            case 'V':
                printf("nstats 0.3.4\n");
                printf("Copyright (C) 2002 by Thomas Graf <tgr@reeler.org>\n");
                exit(0);
                break;
            case 'o':
                s = optarg;
                while ( s ) {
                    if ((t = strchr(s, ','))) {
                        *t = '\0';
                        t++;
                    }
                    if (!strcasecmp(s, "all"))
                        output_mode |= OUT_ALL;
                    else if (!strcasecmp(s, "ether"))
                        output_mode |= OUT_ETHERNET;
                    else if (!strcasecmp(s, "ip"))
                        output_mode |= OUT_IP;
                    else if (!strcasecmp(s, "tos"))
                        output_mode |= OUT_TOS;
                    else if (!strcasecmp(s, "proto"))
                        output_mode |= OUT_PROT_BREAK;
                    else if (!strcasecmp(s, "port"))
                        output_mode |= OUT_PORT;
                    else if (!strcasecmp(s, "tcpopt"))
                        output_mode |= OUT_TCP;
                    else if (!strcasecmp(s, "icmp"))
                        output_mode |= OUT_ICMP;
                    else if (!strcasecmp(s, "destu"))
                        output_mode |= OUT_DESTU;
                    else if (!strcasecmp(s, "redir"))
                        output_mode |= OUT_REDIR;
                    else if (!strcasecmp(s, "texc"))
                        output_mode |= OUT_TEXC;
                    else if (!strcasecmp(s, "ip6"))
                        output_mode |= OUT_IP6;
                    else if (!strcasecmp(s, "conn"))
                        output_mode |= OUT_CONNS;
                    else if (!strcasecmp(s, "ecn"))
                        output_mode |= OUT_ECN;
                    else if (!strcasecmp(s, "icmp6"))
                        output_mode |= OUT_ICMP6;
                    else if (!strcasecmp(s, "destu6"))
                        output_mode |= OUT_DESTU6;
                    else if (!strcasecmp(s, "texc6"))
                        output_mode |= OUT_TEXC6;
                    else if (!strcasecmp(s, "para6"))
                        output_mode |= OUT_PARA6;
                    else if (!strcasecmp(s, "frag"))
                        output_mode |= OUT_FRAG;
                    else if (!strcasecmp(s, "none"))
                        output_mode = 0;
                    else {
                        fprintf(stderr, "Unknown output mode %s\n", s);
                        exit(-1);
                    }

                    s = t;
                }
                break;
            case 'p':
                promisc = 0;
                break;
            case 'f':
                strncpy(filter, optarg, sizeof(filter));
                break;
            case 'i':
                strncpy(device, optarg, sizeof(device));
                break;
            case 't':
                runtime = atoi(optarg);
                break;
            case 'l':
                list_length = atoi(optarg);
                break;
            case 'n':
                do_resolve = 0;
                break;
            default:
                exit(-1);
        }
    }

    errno = 0;
    dname[0] = '.';
    if (getdomainname(&dname[1], sizeof(dname)-1) < 0) {
        perror("getdomainname");
        exit(-1);
    }

    if (!(buf = (char *) malloc(sizeof(struct ifreq) * MAX_IF))) {
        fprintf(stderr, "Out of Memory!\n");
        exit(-1);
    }

    ifc.ifc_len = sizeof(struct ifreq) * MAX_IF;
    ifc.ifc_buf = buf;

    if (ioctl(socket(AF_INET,SOCK_DGRAM,0), SIOCGIFCONF, &ifc, sizeof(ifc)) < 0) {
        perror("ioctl");
        exit(-1);
    }


#if HAVE_SA_LEN
    for(i=0,ptr=buf;ptr<buf+ifc.ifc_len;ptr+=(sizeof(ifr->ifr_name)+ifr->ifr_addr.sa_len)) {
#else
    for(i=0,ptr=buf;ptr<buf+ifc.ifc_len;ptr+=(sizeof(ifr->ifr_name)+sizeof(struct sockaddr))) {
#endif
        ifr = (struct ifreq *) ptr;

        if(ifr->ifr_addr.sa_family==AF_INET){
            struct sockaddr_in *inaddr = (struct sockaddr_in *) &ifr->ifr_netmask;
            if (i < MAX_IF)
                strncpy(if_addr[i], inet_ntoa(inaddr->sin_addr), sizeof(if_addr[i]));
        }
    }

    free(buf);

    if (!cf[0]) {

        if ( geteuid() != 0 ) {
            fprintf(stderr, "You need to be root.\n");
            exit(-1);
        }

        errno = 0;
        if ( signal(SIGINT, &sig_abort) < 0 ) {
            perror("signal");
            exit(-1);
        }

        if ( signal(SIGALRM, &sig_alrm) < 0 ) {
            perror("signal");
            exit(-1);
        }

        if (!runtime)
            printf("Press Ctrl-C to stop\n");
        else {
            struct itimerval itv = {
                {runtime, 0},
                {runtime, 0}
            };
            if ( setitimer(ITIMER_REAL, &itv, NULL) < 0 ) {
                perror("setitimer");
                exit(-1);
            }
        }
    }

    p = start_pcap( (cf[0]) ? cf : NULL, device, filter, promisc);

    pcap_loop(p, -1, look_at_packet, args);

    print_output();

    return 0;
}


void
print_sep(void)
{
    printf("-------------------------------------------------------------\n");
}

void
print_header(const char *str)
{
    printf("%-24s #      %%        bytes      %%     avg\n", str);
    print_sep();
}

void
print_line(const char *str, cnt_t p, cnt_t pt, cnt_t b, cnt_t bt)
{
    printf("%-16s%10llu %6.2f %12llu %6.2f %7.1f\n",
        str,
        p, (p*100) / (float) pt,
        b, (b*100) / (float) bt,
        (p != 0) ? b / (float) p : 0);
}

void
print_output(void)
{
    int i;
    struct servent *s;
    struct ip_conn_s *c = stats.ip.conns;
    struct ip_port_s *ps;
    char addrbuf[INET6_ADDRSTRLEN], addrbuf2[INET6_ADDRSTRLEN];
    char serv1[256], serv2[256];

#define IPV4 stats.ip.ipv4
#define IPV6 stats.ip.ipv6
#define ICMP6 IPV6.icmp
#define DESTU6 ICMP6.dst_unreach
#define TEXC6 ICMP6.time_excd
#define PARA6 ICMP6.param_prob
#define ICMP4 IPV4.icmp
#define ICMP4DU ICMP4.types.dest_unreach
#define TEX ICMP4.types.time_exceeded
#define TCP stats.ip.tcp
#define UDP stats.ip.udp
#define TOS stats.ip.tos
#define ECN stats.ip.ecn
#define IRED ICMP4.types.redirect
#define FRAG stats.ip.frag

    if ( output_mode & OUT_ALL || output_mode & OUT_ETHERNET ) {
        print_header("Protocol (Ethernet)");
        print_line("IP:", stats.ip.cnt, stats.cnt, stats.ip.bs, stats.bs);
        print_line("ARP:", stats.arp.cnt, stats.cnt, stats.arp.bs, stats.bs);
        print_line("RARP", stats.rarp.cnt, stats.cnt, stats.rarp.bs, stats.bs);
        print_line("Other:", stats.other.cnt, stats.cnt, stats.other.bs, stats.bs);
        print_sep();
        print_line("Total:", stats.cnt, stats.cnt, stats.bs, stats.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_IP ) {
        print_header("Protocol L3 (IP)");
        print_line("IPv4:", IPV4.cnt, stats.ip.cnt, IPV4.bs, stats.ip.bs);
        print_line("IPv6:", IPV6.cnt, stats.ip.cnt, IPV6.bs, stats.ip.bs);
        print_sep();
        print_line("Total:", stats.ip.cnt, stats.ip.cnt, stats.ip.bs, stats.ip.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_TOS ) {
        print_header("TOS (IPv4)");
        print_line("Low-Delay:", TOS.lowdelay, IPV4.cnt, TOS.s_lowdelay, IPV4.bs);
        print_line("Throughput:", TOS.maxthroughput, IPV4.cnt, TOS.s_maxthroughput, IPV4.bs);
        print_line("Reliability:", TOS.reliability, IPV4.cnt, TOS.s_reliability, IPV4.bs);
        print_line("Low-Cost:", TOS.lowcost, IPV4.cnt, TOS.s_lowcost, IPV4.bs);
        print_line("None:", TOS.none, IPV4.cnt, TOS.s_none, IPV4.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_PROT_BREAK ) {
        print_header("Protocol L4 (IP)");
        print_line("TCP:", TCP.cnt, IPV4.cnt, TCP.bs, IPV4.bs);
        print_line("UDP:", UDP.cnt, IPV4.cnt, UDP.bs, IPV4.bs);
        print_line("ICMP:", ICMP4.cnt, IPV4.cnt, ICMP4.bs, IPV4.bs);
        print_line("Other:", stats.ip.other.cnt, IPV4.cnt, stats.ip.other.bs, IPV4.bs);
        print_sep();
        print_line("Total:", stats.ip.cnt, stats.ip.cnt, stats.ip.bs, stats.ip.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_TCP ) {
        print_header("TCP Options (IPv4)");
        print_line("URG:", TCP.opts.urg, TCP.cnt, TCP.opts.s_urg, TCP.bs);
        print_line("ACK:", TCP.opts.ack, TCP.cnt, TCP.opts.s_ack, TCP.bs);
        print_line("PSH:", TCP.opts.psh, TCP.cnt, TCP.opts.s_psh, TCP.bs);
        print_line("RST:", TCP.opts.rst, TCP.cnt, TCP.opts.s_rst, TCP.bs);
        print_line("SYN:", TCP.opts.syn, TCP.cnt, TCP.opts.s_syn, TCP.bs);
        print_line("FIN:", TCP.opts.fin, TCP.cnt, TCP.opts.s_fin, TCP.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_ICMP ) {
        print_header("ICMP Types (IPv4)");
        print_line("Echo-Request:",
            ICMP4.types.echo_request, ICMP4.cnt,
            ICMP4.types.s_echo_request, ICMP4.bs);
        print_line("Echo-Reply:",
            ICMP4.types.echo_reply, ICMP4.cnt,
            ICMP4.types.s_echo_reply, ICMP4.bs);
        print_line("Dest Unreach:",
            ICMP4DU.cnt, ICMP4.cnt,
            ICMP4DU.bs, ICMP4.bs);
        print_line("Source Quench:",
            ICMP4.types.source_quench, ICMP4.cnt,
                 ICMP4.types.s_source_quench, ICMP4.bs);
        print_line("Redirect:",
            ICMP4.types.redirect.cnt, ICMP4.cnt,
                 ICMP4.types.redirect.bs, ICMP4.bs);
        print_line("Time Exceeded:",
            ICMP4.types.time_exceeded.cnt, ICMP4.cnt,
                 ICMP4.types.time_exceeded.bs, ICMP4.bs);
        print_line("Parameter Prob:",
            ICMP4.types.para_prob, ICMP4.cnt,
                 ICMP4.types.s_para_prob, ICMP4.bs);
        print_line("Timetstamp:",
            ICMP4.types.timestamp, ICMP4.cnt,
                 ICMP4.types.s_timestamp, ICMP4.bs);
        print_line("Timetstamp Rpy:",
            ICMP4.types.timestamp_reply, ICMP4.cnt,
                 ICMP4.types.s_timestamp_reply, ICMP4.bs);
        print_line("Info Request:",
            ICMP4.types.info_request, ICMP4.cnt,
                 ICMP4.types.s_info_request, ICMP4.bs);
        print_line("Info Reply:",
            ICMP4.types.info_reply, ICMP4.cnt,
                 ICMP4.types.s_info_reply, ICMP4.bs);
        print_line("Address:",
            ICMP4.types.address, ICMP4.cnt,
                 ICMP4.types.s_address, ICMP4.bs);
        print_line("Address Reply:",
            ICMP4.types.address_reply, ICMP4.cnt,
                 ICMP4.types.s_address_reply, ICMP4.bs);
        print_line("Other:",
            ICMP4.types.other, ICMP4.cnt,
                 ICMP4.types.s_other, ICMP4.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_DESTU ) {
        print_header("ICMP Dest. Unreach.");
        print_line("Net Unreach:",
            ICMP4DU.net_unreach, ICMP4DU.cnt,
            ICMP4DU.s_net_unreach, ICMP4DU.bs);
        print_line("Host Unreach:",
            ICMP4DU.host_unreach, ICMP4DU.cnt,
             ICMP4DU.s_host_unreach, ICMP4DU.bs);
        print_line("Proto Unreach:",
            ICMP4DU.prot_unreach, ICMP4DU.cnt,
             ICMP4DU.s_prot_unreach, ICMP4DU.bs);
        print_line("Frag Needed:",
            ICMP4DU.frag_needed, ICMP4DU.cnt,
             ICMP4DU.s_frag_needed, ICMP4DU.bs);
        print_line("SR Failed:",
            ICMP4DU.sr_failed, ICMP4DU.cnt,
             ICMP4DU.s_sr_failed, ICMP4DU.bs);
        print_line("Net Unknown:",
            ICMP4DU.net_unknown, ICMP4DU.cnt,
             ICMP4DU.s_net_unknown, ICMP4DU.bs);
        print_line("Host Unknown:",
            ICMP4DU.host_unknown, ICMP4DU.cnt,
             ICMP4DU.s_host_unknown, ICMP4DU.bs);
        print_line("Host Isolated:",
            ICMP4DU.host_isolated, ICMP4DU.cnt,
             ICMP4DU.s_host_isolated, ICMP4DU.bs);
        print_line("Net Ano:",
            ICMP4DU.net_ano, ICMP4DU.cnt,
             ICMP4DU.s_net_ano, ICMP4DU.bs);
        print_line("Host Ano:",
            ICMP4DU.host_ano, ICMP4DU.cnt,
             ICMP4DU.s_host_ano, ICMP4DU.bs);
        print_line("Net Unr TOS:",
            ICMP4DU.net_unr_tos, ICMP4DU.cnt,
             ICMP4DU.s_net_unr_tos, ICMP4DU.bs);
        print_line("Host Unr TOS:",
            ICMP4DU.host_unr_tos, ICMP4DU.cnt,
             ICMP4DU.s_host_unr_tos, ICMP4DU.bs);
        print_line("Packet Filtered:",
            ICMP4DU.pkt_filtered, ICMP4DU.cnt,
             ICMP4DU.s_pkt_filtered, ICMP4DU.bs);
        print_line("Prec Violation:",
            ICMP4DU.prec_violation, ICMP4DU.cnt,
             ICMP4DU.s_prec_violation, ICMP4DU.bs);
        print_line("Prec Cutoff:",
            ICMP4DU.prec_cutoff, ICMP4DU.cnt,
             ICMP4DU.s_prec_cutoff, ICMP4DU.bs);
        print_line("Other:",
            ICMP4DU.other, ICMP4DU.cnt,
             ICMP4DU.s_other, ICMP4DU.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_REDIR ) {
        print_header("ICMP Redir");
        print_line("Network:", IRED.net, IRED.cnt, IRED.s_net, IRED.bs);
        print_line("Host:", IRED.host, IRED.cnt, IRED.s_host, IRED.bs);
        print_line("Network TOS:", IRED.nettos, IRED.cnt, IRED.s_nettos, IRED.bs);
        print_line("Host TOS:", IRED.hosttos, IRED.cnt, IRED.s_hosttos, IRED.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_TEXC ) {
        print_header("ICMP Time Exceeded");
        print_line("TTL:", TEX.ttl, TEX.cnt, TEX.s_ttl, TEX.bs);
        print_line("Fragtime:", TEX.fragtime, TEX.cnt, TEX.s_fragtime, TEX.bs);
        print_line("Other:", TEX.other, TEX.cnt, TEX.s_other, TEX.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_IP6 ) {
        printf("Warning: Experimental code!\n");
        print_header("IPv6");
        print_line("Flow-Label:", IPV6.fl, IPV6.cnt, IPV6.s_fl, IPV6.bs);
        print_line("Traffic-Class:", IPV6.tc, IPV6.cnt, IPV6.s_tc, IPV6.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_ECN ) {
        print_header("ECN");
        print_line("Not-ECT:", ECN.not_ect, stats.ip.cnt, ECN.s_not_ect, stats.ip.bs);
        print_line("ECT(1):", ECN.ect_1, stats.ip.cnt, ECN.s_ect_1, stats.ip.bs);
        print_line("ECT(0):", ECN.ect_0, stats.ip.cnt, ECN.s_ect_0, stats.ip.bs);
        print_line("CE:", ECN.ce, stats.ip.cnt, ECN.s_ce, stats.ip.bs);
        print_line("CWR (TCP):", ECN.cwr, stats.ip.cnt, ECN.s_cwr, stats.ip.bs);
        print_line("ECE (TCP):", ECN.ece, stats.ip.cnt, ECN.s_ece, stats.ip.bs);
        print_line("ECN-Setup (TCP):", ECN.ecn_setup, stats.ip.cnt, ECN.s_ecn_setup,
            stats.ip.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_ICMP6 ) {
        print_header("ICMP6 (IPv6)");
        print_line("Dest. Unreach.", ICMP6.dst_unreach.cnt, ICMP6.cnt,
                ICMP6.dst_unreach.bs, ICMP6.bs);
        print_line("Time Exceeded:", ICMP6.time_excd.cnt, ICMP6.cnt,
                ICMP6.time_excd.bs, ICMP6.bs);
        print_line("Param. Prob:", ICMP6.param_prob.cnt, ICMP6.cnt,
                ICMP6.param_prob.bs, ICMP6.bs);
        print_line("Pkt too big:", ICMP6.pkt_too_big, ICMP6.cnt,
                ICMP6.s_pkt_too_big, ICMP6.bs);
        print_line("Echo Request:", ICMP6.echo_request, ICMP6.cnt,
                ICMP6.s_echo_request, ICMP6.bs);
        print_line("Echo Reply:", ICMP6.echo_reply, ICMP6.cnt,
                ICMP6.s_echo_reply, ICMP6.bs);
        print_line("MbrShip Query:", ICMP6.mbrship_query, ICMP6.cnt,
                ICMP6.s_mbrship_query, ICMP6.bs);
        print_line("MbrShip Report:", ICMP6.mbrship_report, ICMP6.cnt,
                ICMP6.s_mbrship_report, ICMP6.bs);
        print_line("MbrShip Reduct.:", ICMP6.mbrship_reduction, ICMP6.cnt,
                ICMP6.s_mbrship_reduction, ICMP6.bs);
        print_line("ND Rtr Solicit:", ICMP6.nd_router_solicit, ICMP6.cnt,
                ICMP6.s_nd_router_solicit, ICMP6.bs);
        print_line("ND Rtr Advert:", ICMP6.nd_router_advert, ICMP6.cnt,
                ICMP6.s_nd_router_advert, ICMP6.bs);
        print_line("ND Neig Solicit:", ICMP6.nd_neighbor_solicit, ICMP6.cnt,
                ICMP6.s_nd_neighbor_solicit, ICMP6.bs);
        print_line("ND Neig Advert:", ICMP6.nd_neighbor_advert, ICMP6.cnt,
                ICMP6.s_nd_neighbor_advert, ICMP6.bs);
        print_line("ND Redirect:", ICMP6.nd_redirect, ICMP6.cnt,
                ICMP6.s_nd_redirect, ICMP6.bs);
        print_line("Other:", ICMP6.other, ICMP6.cnt, ICMP6.s_other, ICMP6.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_DESTU6 ) {
        print_header("Dest Unreach. (ICMP6)");
        print_line("No Route:", DESTU6.noroute, DESTU6.cnt,
                DESTU6.s_noroute, DESTU6.bs);
        print_line("Admin:", DESTU6.admin, DESTU6.cnt,
                DESTU6.s_admin, DESTU6.bs);
        print_line("Not Neighbor:", DESTU6.notneighbor, DESTU6.cnt,
                DESTU6.s_notneighbor, DESTU6.bs);
        print_line("Addr:", DESTU6.addr, DESTU6.cnt,
                DESTU6.s_addr, DESTU6.bs);
        print_line("No Port:", DESTU6.noport, DESTU6.cnt,
                DESTU6.s_noport, DESTU6.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_TEXC6 ) {
        print_header("Time Exceeded (ICMP6)");
        print_line("Transit:", TEXC6.transit, TEXC6.cnt,
                TEXC6.s_transit, TEXC6.bs);
        print_line("Reassembly:", TEXC6.reassembly, TEXC6.cnt,
                TEXC6.s_reassembly, TEXC6.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_PARA6 ) {
        print_header("Parameter Problem (ICMP6)");
        print_line("Header:", PARA6.header, PARA6.cnt,
                PARA6.s_header, PARA6.bs);
        print_line("Next Header:", PARA6.nextheader, PARA6.cnt,
                PARA6.s_nextheader, PARA6.bs);
        print_line("Option:", PARA6.option, PARA6.cnt,
                PARA6.s_option, PARA6.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_FRAG ) {
        print_header("Fragmented packets");
        print_line("Frag.:", FRAG.frag, stats.ip.cnt, FRAG.s_frag, stats.ip.bs);
        print_line("Not Frag.:", FRAG.nfrag, stats.ip.cnt, FRAG.s_nfrag, stats.ip.bs);
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_CONNS) {

        clean_conns();
        sort_conns();

        printf("Source Address      Destination Address    SPort    DPort          #      bytes\n");
        printf("-------------------------------------------------------------------------------\n");

        for (c=stats.ip.conns,i=0; c && i < list_length; i++) {

            memset(serv1, 0, sizeof(serv1));
            memset(serv2, 0, sizeof(serv2));

            if ( (s = getservbyport(htons(c->sport), NULL)) )
                strncpy(serv1, s->s_name, sizeof(serv1));
            else
                snprintf(serv1, sizeof(serv1), "%d", c->sport);

            serv1[8] = '\0';

            if ( (s = getservbyport(htons(c->dport), NULL)) )
                strncpy(serv2, s->s_name, sizeof(serv2));
            else
                snprintf(serv2, sizeof(serv2), "%d", c->dport);

            serv2[8] = '\0';

            if (c->type == ETHERTYPE_IP) {

                resolvebyaddr(addrbuf, sizeof(addrbuf), (char *) &c->src_u.ip4,
                    sizeof(c->src_u.ip4), AF_INET);
                resolvebyaddr(addrbuf2, sizeof(addrbuf2), (char *) &c->dst_u.ip4,
                    sizeof(c->dst_u.ip4), AF_INET);

                printf("%-19s %-19s %8s %8s %10lld %10lld\n",
                        addrbuf, addrbuf2,
                        serv1, serv2,
                        c->cnt, c->bs);
            } else if (c->type == 0x86dd) {

                resolvebyaddr(addrbuf, sizeof(addrbuf), (char *) &c->src_u.ip6,
                    sizeof(c->src_u.ip6), AF_INET6);
                resolvebyaddr(addrbuf2, sizeof(addrbuf2), (char *) &c->dst_u.ip6,
                    sizeof(c->dst_u.ip6), AF_INET6);

                printf("%-19s %-19s %8s %8s %10lld %10lld\n",
                        addrbuf, addrbuf2,
                        serv1, serv2,
                        c->cnt, c->bs);
            }
            c = c->next;
        }
            
    }

    if ( output_mode & OUT_ALL || output_mode & OUT_PORT ) {
        print_header("Port (IP)");

        fill_ports();
        sort_ports();

        i = 0;
        setservent(1);
        for (ps=stats.ip.ports; ps && i < list_length; ps = ps->next) {

            if ( (s = getservbyport(htons(ps->port), NULL)) ) {
                print_line(s->s_name, ps->cnt, stats.ip.c_ports,
                           ps->bs, stats.ip.bs);
            } else {
                print_line(as_string(ps->port), ps->cnt, stats.ip.c_ports,
                           ps->bs, stats.ip.bs);
            }

            i++;
        }
        endservent();
    }
}
