"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bytes = exports.ByteSizeValue = void 0;
exports.ensureByteSizeValue = ensureByteSizeValue;
exports.tb = exports.mb = exports.kb = exports.gb = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const unitMultiplier = {
  b: Math.pow(1024, 0),
  gb: Math.pow(1024, 3),
  kb: Math.pow(1024, 1),
  mb: Math.pow(1024, 2)
};
function renderUnit(value, unit) {
  const prettyValue = Number(value.toFixed(2));
  return `${prettyValue}${unit}`;
}
class ByteSizeValue {
  static parse(text) {
    const match = /([1-9][0-9]*)(b|kb|mb|gb)/.exec(text);
    if (!match) {
      const number = Number(text);
      if (typeof number !== 'number' || isNaN(number)) {
        throw new Error(`Failed to parse value as byte value. Value must be either number of bytes, or follow the format <count>[b|kb|mb|gb] ` + `(e.g., '1024kb', '200mb', '1gb'), where the number is a safe positive integer.`);
      }
      return new ByteSizeValue(number);
    }
    const value = parseInt(match[1], 10);
    const unit = match[2];
    return new ByteSizeValue(value * unitMultiplier[unit]);
  }
  constructor(valueInBytes) {
    this.valueInBytes = valueInBytes;
    if (!Number.isSafeInteger(valueInBytes) || valueInBytes < 0) {
      throw new Error(`Value in bytes is expected to be a safe positive integer.`);
    }
  }
  isGreaterThan(other) {
    return this.valueInBytes > other.valueInBytes;
  }
  isLessThan(other) {
    return this.valueInBytes < other.valueInBytes;
  }
  isEqualTo(other) {
    return this.valueInBytes === other.valueInBytes;
  }
  getValueInBytes() {
    return this.valueInBytes;
  }
  toString(returnUnit) {
    let value = this.valueInBytes;
    let unit = `b`;
    for (const nextUnit of ['kb', 'mb', 'gb']) {
      if (unit === returnUnit || returnUnit == null && value < 1024) {
        return renderUnit(value, unit);
      }
      value = value / 1024;
      unit = nextUnit;
    }
    return renderUnit(value, unit);
  }
}
exports.ByteSizeValue = ByteSizeValue;
const bytes = value => new ByteSizeValue(value);
exports.bytes = bytes;
const kb = value => bytes(value * 1024);
exports.kb = kb;
const mb = value => kb(value * 1024);
exports.mb = mb;
const gb = value => mb(value * 1024);
exports.gb = gb;
const tb = value => gb(value * 1024);
exports.tb = tb;
function ensureByteSizeValue(value) {
  if (typeof value === 'string') {
    return ByteSizeValue.parse(value);
  }
  if (typeof value === 'number') {
    return new ByteSizeValue(value);
  }
  return value;
}