/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo, Marko Mattila

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __GAME_CLIENT_H
#define __GAME_CLIENT_H

#include <typeinfo>
#include <vector>
#include <string>
#include <algorithm>
#include <exception>

#include "SDL_ttf.h"
#include "SDL.h"
#include "SDL_net.h"
#include "msg.h"
#include "terrain.h"
#include <queue>
#include "gfx.h"
#include "player.h"
#include "game_obj.h"
#include "coord.h"
#include "block.h"
#include "particle_efects.h"
#include "animation.h"
#include "gfx_functions.h"
#include "gui.h"
#include "audio.h"
#include "game_statistic.h"
#include "mosaic.h"
#include "client_net.h"
#include "config.h"

using namespace std;

#define CURSOR_MOVE 0.1 // pixels to move / ms
#define CURSOR_MOVE_FAST 0.3 // pixels to move when pressed with FAST-mod / ms
#define DEFAULT_TEXT_DURATION 5000 // texts stay for 5s.

#define PI 3.141592654

static const double scatterprob[] = {
  0.5000000,
  0.5497382,
  0.5987063,
  0.6461698,
  0.6914625,
  0.7340145,
  0.7733726,
  0.8092130,
  0.8413447,
  0.8697055,
  0.8943502,
  0.9154343,
  0.9331928,
  0.9479187,
  0.9599408,
  0.9696036,
  0.9772499,
  0.9832067,
  0.9877755,
  1.0
};

class last_player_exception : public exception
{
 public:
  last_player_exception()
    { }
};

class esc_exception : public exception
{
 public:
  esc_exception()
    { }
};

/**
 * Class for the game client.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class game_client : public gui_screen
{
 public:
  game_client(string * names, IPaddress *_server, SDL_Surface *screen, TTF_Font *font, int _local_players, gfx *graphix, starfield_efect *se, audio *au, config *cfg);
  virtual ~game_client();

  void connect() throw (string &);
  // connect to local server, give out queues
  void connect_local(monitor_queue<msg *> **in, monitor_queue<msg*> ** out) throw (string &);
  void disconnect();
  virtual void activate(int signum);

  vector<string> get_errors();


 protected:
  client_net *serv_conn;
  terrain *terra;
  game_statistic *stats;
  IPaddress* server_ip;
  int player_index[MAX_PLAYERS];
  player players[MAX_PLAYERS];
  int mode;
  int cannon_size[MAX_PLAYERS];
  queue<msg *> outp_queue;
  vector<int> locked_choosers;
  coord cannon_placers[MAX_PLAYERS];
  coord character_choosers[MAX_PLAYERS];//character choosers
  coord cursors[MAX_PLAYERS];
  bool cursor_lock[MAX_PLAYERS];
  vector<cannon_in*> own_cannons[MAX_PLAYERS];//so that it can contain both cannon and big_cannon objects.
  block block_placers[MAX_PLAYERS];
  coord home_castles[MAX_PLAYERS];
  bool mode_change;
  bool game_on;
  vector<string> errors;

  //talkmode stuff
  string talk_txt[MAX_PLAYERS];
  int talker;
  bool talk;

  int wins;
  int round;
  int local_players;
  int keyA[MAX_PLAYERS], keyB[MAX_PLAYERS], keyUP[MAX_PLAYERS], keyDOWN[MAX_PLAYERS], keyRIGHT[MAX_PLAYERS], keyLEFT[MAX_PLAYERS], keyTALK[MAX_PLAYERS];
  /** current game state */
  enum game_state {
    ID_MODE_NOTACTIVE = 0,
    ID_MODE_PLACEC,
    ID_MODE_SHOOT,
    ID_MODE_REPAIR,
    ID_MODE_ENDR,
    ID_MODE_CSELECT,
    ID_MODE_CHSELECT,
    ID_MODE_ENDGAME
  };

  enum character {
    ROCMAN=0,
    ANUZ,
    FIREMAN,
    DOD,
    NOONE
  };

  static const char * character_names[4]; // array of 4 char names


  /**
   * Places block.
   * \param b Block to be placed.
   * \param owner Owner of the block to be placed.
   * \parma
   * \return False if the tiles occupied by the block are occupied, in which case the block is NOT placed. Returns
   * true if the block was placed succesfully.
   */
  bool place_block(block &b, int owner);
  /**
   * Shoots to the pixel-coordinates.
   * \param c Pixel-coordinates where to shoot.
   * \param shooting players index
   */
  void shoot (const coord &c, int pn);
  /**
   * Performs a shoot with given player (actual shooting routine).
   * \param from Pixel-coordinates where to shoot from.
   * \param to Pixel-coordinates where to shoot into.
   * \param shooter Player index of the shooter.
   * \param ammosize size of the ammo.
   */
  void shoot (const coord &from, const coord & to, int shooter, int ammosize);
  /**
   * Places a cannon into the place where the current cannonplacer is at the moment.
   * \param player number
   * \return True if the placement was succesful, false if not (if no cannons left to place or squares occupied).
   */
  bool place_cannon(int pn);
  /**
   * Places a big cannon into the place where the current big cannonplacer is at the moment.
   * \param player number
   * \return True if the placement was succesful, false if not (if no cannons left to place or squares occupied).
   */
  bool place_big_cannon(int pn);
  /**
   * Chooses a character for player
   * \param players number
   */
  void choose_character(int pn);
  /**
   * Move character chooser
   * \param sym Kaysym indicating how to move the character chooser
   * \param player number
   */
  void move_characterchooser (SDLKey sym, int pn);
  /**
   * Moves the cannonplacer.
   * \param sym Keysym indicating how to move the cannonplacer.
   * \param player number
   */
  void move_cannonplacer(SDLKey sym, int pn);
  /**
   * Moves the blockplacer.
   * \param sym Keysym indicating how to move the blockplacer.
   */
  void move_blockplacer(SDLKey sym);
  /**
   * function for communication
   */
  void data_transfer();
  /**
   * Static function for the actual game loop thread.
   */
  static int game_loop(void *data);
  /**
   * Clear all for next round
   */
  void clearallfornextround();
  /**
   * Inactive mode.
   */
  void mode_inactive() throw (exception);
  /**
   * Castle select mode.
   */
  void mode_character_select() throw (exception);
  /**
   * Castle select mode.
   */
  void mode_castle_select() throw (exception);
  /**
   * Shooting mode.
   */
  void mode_shooting() throw (exception);
  /**
   * Cannon placing mode.
   */
  void mode_place_cannons() throw (exception);
  /**
   * Repair mode.
   */
  void mode_repair() throw (exception);
  /**
   * Endround mode.
   * \return True if the game continues, false if it has ended.
   */
  bool mode_endround() throw (exception);
  /**
   * End game mode.
   */
  void mode_endgame() throw (exception);
  /**
   * Locks the castle chooser.
   */
  void chooser_lock (int player_index);
  /**
   * Gives information whether the players castlechooser is locked.
   * \return True if the chooser is locked, false otherwise.
   * \param player_number
   */
  bool own_chooser_locked (int player_number);
  /**
   * Builds default walls around the chosen player castles.
   */
  void build_default_walls();
  /**
   * Clears the event queue for events.
   */
  void clear_event_queue();
  /**
   * Processses castlechooser move msg from server.
   * \param owner Owner of the chooser.
   * \param co New terrain-coordinates of the chooser.
   */
  void process_castlechooser_move (int owner, const coord & co);
  /**
   * Handles player disconnect.
   * \param index Index of player that has disconnected.
   */
  void handle_player_disconnect(int index) throw (last_player_exception);
  /**
   * Sets keys
   * \param action key a
   * \param action key b
   * \param key for up
   * \param key for down
   * \param key for right
   * \param key for left
   * \param key for talk
   * \param player number
   */
  void set_keys(int _keyA, int _keyB, int _keyUP, int  _keyDOWN, int _keyRIGHT, int _keyLEFT, int _keyTALK, int player_num);

  bool handle_talk(string* txt, SDLKey syn, SDLMod mod, int index);

  virtual void process_objects();
  
  void draw_castleowners();

  void play_ended();

  void change_cannon(int owner);

  /* number of players connected to game */
  int get_no_players() const;

  void process_talkmessage(msg_talkmessage* mmc);

};


#endif
