/* get cddb-information of cdrom

   Written by Matthias Hensler
   Copyright WSPse 1999+2000
   eMail: wsp@gmx.de

Created: 1999/06/05
Updated: 2001/08/05
         2006/07/30: switch to cddbp protocol level 5
*/

/* Copying:
   This program is free software; you can redistribute it and/or modify it under
   the terms of the GNU Gerneral Public License as published by the Free Soft-
   ware Foundation; either version 2 of License, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILTY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 675 Mass
   Ave, Cambridge, MA 02139, USA.
   */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include "mp3creat.h"

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

/* fetch externals */
extern int track_last;

extern struct {
  int min;
  int sec;
  int frame;
} cdtoc[100];

extern char *mp3_genre[];
extern unsigned long cddb_discid(int tot_trks);
extern void setup_stat_win(int max_length);
extern void destroy_stat_win();
extern void print_stat_win(char *text);
extern void popup_error_win(char *tx);
extern void wuuush(int);
extern void free_field_select(field_select_typ **anchor);
extern int open_socket (const char *address);
extern int read_from_sock(char **line, int sock_fd, BOOL wait);
extern int write_to_sock(char *buf, int len, int sock_fd);
extern field_select_typ *select_field_box(field_select_typ *fs_anchor, int max_length,
					  char *stat_text, BOOL use_old, BOOL allow_space);
extern void filenm_generate(song_typ *track);
extern int create_sub_dirs(char *filename, BOOL mode);
extern char *copy_char_str(char *old);
extern char *def_mp3_dir;
extern char *def_comment;
extern BOOL def_on_fly;
extern BOOL config_cddb_enbl;

BOOL cddb_cddb_entry_is_clean = FALSE; /* if revision of entry is (remotely) CDDB clean */
int  cddb_cddb_entry_revision = -1;    /* number or entry revision */

int cddb_internet_server_lookup (char *addr, char *cddb_path, BOOL force);

/*---------------------------------------------------------------------------*/
/* look up entry locally */
int cddb_local_lookup(char *path, char **cddb_fn)
{
  char discid[9];
  field_select_typ *fs_anchor, *fs_new;
  char *local_filename;
  DIR *cddb_dir;
  struct dirent *de;
  struct stat st;
  int found_entries;
  FILE *cddb_file;
  char tmp_buf[256];
  char *pat;

  /* discid $00000000 is illegal! */
  if(cddb_discid(track_last) == 0L) {
    print_stat_win(_("illegal discid"));
    return -1;
  }

  snprintf(discid, 9, "%08lx", cddb_discid(track_last));

  print_stat_win(_("look in local CDDB"));
#ifdef DEBUG
  fprintf(stderr, "DEBUG: looking up \"%s\" locally.\n", discid);
#endif
  
  cddb_dir = opendir(path);
  if(cddb_dir == NULL)
    return -1;

  local_filename = (char *) malloc(sizeof(char) * (strlen(path)+1));
  if(local_filename == NULL) {
    perror("malloc");
    wuuush(1);
  }
  
  found_entries = 0;
  fs_anchor     = NULL;
  
  while((de = readdir(cddb_dir)) != NULL) {
    if(strcmp(de->d_name, ".") && strcmp(de->d_name, "..")) {
      local_filename = (char *) realloc(local_filename, sizeof(char) *
					(strlen(path)+strlen(de->d_name) +2));
      if(local_filename == NULL) {
	perror("realloc");
	wuuush(1);
      }
      strcpy(local_filename, path);
      strcat(local_filename, "/");
      strcat(local_filename, de->d_name);

      if((stat(local_filename, &st) ==0) && S_ISDIR(st.st_mode)) {
#ifdef DEBUG
	fprintf(stderr, "DEBUG: looking in directory \"%s\"\n", local_filename);
#endif

	local_filename = (char *) realloc(local_filename, sizeof(char) *
	  				  (strlen(local_filename) + 10));
	if(local_filename == NULL) {
	  perror("realloc");
	  wuuush(1);
	}
      
	strcat(local_filename, "/");
	strcat(local_filename, discid);

	if(access(local_filename, R_OK) ==0) {
	  fs_new = (field_select_typ *) malloc(sizeof(field_select_typ));
	  if(fs_new == NULL) {
	    perror("malloc");
	    wuuush(1);
	  }
	  fs_new->information = local_filename;
	  fs_new->dest        = NULL;
	  fs_new->next        = fs_anchor;
	  fs_new->prev        = NULL;
	  if(fs_anchor) fs_anchor->prev = fs_new;
	  fs_anchor           = fs_new;

	  cddb_file = fopen(local_filename, "r");
#ifdef DEBUG
	  if(cddb_file == NULL) {
	    fprintf(stderr, "DEBUG: weia, opening \"%s\" failed!\n",
		    local_filename);
	  }
#endif
	  if(cddb_file != NULL) {
	    local_filename = (char *) malloc(sizeof(char) *
		  			     strlen(de->d_name) +2);
	    if(local_filename == NULL) {
	      perror("malloc");
	      wuuush(1);
	    }
	    strcpy(local_filename, de->d_name);
	    strcat(local_filename, " ");
	    while(1) {
	      fgets(tmp_buf, 254, cddb_file);
	      if(feof(cddb_file)) break;
	      if(strncmp(tmp_buf, "DTITLE=", 7) == 0) {
		pat = strchr(tmp_buf, '\n');
		if(pat) *pat = 0;
		pat = strchr(tmp_buf, '\r');
		if(pat) *pat = 0;
		local_filename = (char *) realloc(local_filename, sizeof(char) *
			  			  (strlen(local_filename) +
			  			   strlen(tmp_buf)));
		if(local_filename == NULL) {
		  perror("realloc");
		  wuuush(1);
		}

		strcat(local_filename, "\"");
		strcat(local_filename, tmp_buf+7);
		strcat(local_filename, "\"");
	      }
	    }
	    fclose(cddb_file);
	  }  
	  fs_new->field = local_filename;
	  found_entries++;

#ifdef DEBUG
	  fprintf(stderr, "DEBUG: added \"%s\"\n", local_filename);
#endif

	  local_filename = (char *) malloc(sizeof(char) * (strlen(path)+1));
	  if(local_filename == NULL) {
	    perror("malloc");
	    wuuush(1);
	  }
	}
      }
    }
  }
  closedir(cddb_dir);
  
  free(local_filename);

  if(found_entries == 0) {
    free_field_select(&fs_anchor);
    return 1;
  }

  if(found_entries > 1) {
    fs_new = (field_select_typ *) select_field_box(fs_anchor, 70, _("select entry"), FALSE, FALSE);
    if(fs_new == NULL) fs_new = fs_anchor;
  } else fs_new = fs_anchor;

  local_filename = (char *) malloc(sizeof(char) * (strlen(fs_new->
							  information)+1));
  if(local_filename == NULL) {
    perror("malloc");
    wuuush(1);
  }
  strcpy(local_filename, fs_new->information);
  
  *cddb_fn = local_filename;
  free_field_select(&fs_anchor);
  return 0;
}

/*---------------------------------------------------------------------------*/
/* multi cddb-lookup */

int cddb_internet_lookup(char *addr, char *cddb_path, BOOL force)
{
  char *serv, *match;
  BOOL replaced;
  int status;
  
  if(! addr) {
    popup_error_win(_("no hostaddress specified"));
    return -1;
  }

  if(strcmp(addr, "0") == 0) {
    print_stat_win(_("remote cddb disabled"));
    return -1;
  }

  if(! cddb_path) {
    popup_error_win(_("no local cddb-database specified"));
    return -1;
  }

  if(access(cddb_path, F_OK)) {
    print_stat_win(_("no cddb-path, try to create"));
    create_sub_dirs(cddb_path, FALSE);
  }
  if(access(cddb_path, F_OK)) {
    popup_error_win(_("Uh, cannot receive entries without valid CDDB-path."));
    return -1;
  }
  if(access(cddb_path, W_OK)) {
    popup_error_win(_("STOP! No write permissions for CDDB-directory. <EMERGENCY ABORT>"));
    return -1;
  }

  if((!config_cddb_enbl) && (!force)) {
    popup_error_win(_("CDDB access is locked, use \"l\" to unlock"));
    return -1;
  }

  serv = addr;
  while(1) {
    replaced = FALSE;
    match = strchr(serv, ',');
    if(match) {
      *match   = 0;
      replaced = TRUE;
    }
    print_stat_win(serv);
    status = cddb_internet_server_lookup(serv, cddb_path, force);
    if(replaced) {
      *match = ',';
      serv   = match+1;
    }

    if((!status) || (!replaced) || (!(*serv))) break;
  }

  return 0;
}


/*---------------------------------------------------------------------------*/
/* open connection to cddb-server */
int cddb_open_cddbp(char *addr)
{
  int cddb_fd;
  char *buf, *tmp_str, *pat1, *pat2;
  int cddb_ver_cur, cddb_ver_sup, cddb_ver_sel;
  
  if(! addr) return -1;
  print_stat_win(_("connect to cddb-server"));
  cddb_fd = open_socket(addr);
  if(cddb_fd == -1) return -1;
  
#ifdef DEBUG
  fprintf(stderr, "DEBUG: connected to cddb-server \"%s\"\n", addr);
#endif
  
  /* CDDB-Server successfully connected, now wait for greeting text */
  print_stat_win(_("waiting for greeting"));
  while(1) {
    if(read_from_sock(&buf, cddb_fd, TRUE) != 0) {
#ifdef DEBUG
      fprintf(stderr, "DEBUG: error reading line from socket (cddb)\n");
#endif
      close(cddb_fd);
      return -1;
    }
    if(strncmp(buf, "20", 2)==0) break;
    if(strncmp(buf, "4", 1)==0) break;
    free(buf);
  }
#ifdef DEBUG
  fprintf(stderr, "DEBUG: greet: \"%s\"\n", buf);
#endif
  if(*buf == '4') {
    print_stat_win(_("too many users, try later"));
    sleep(1);
    close(cddb_fd);
    free(buf);
    return -1;
  }
  tmp_str = getenv("LOGNAME");
  if(! tmp_str) {
    tmp_str = getlogin();
  }
  if(! tmp_str) {
    buf = (char *) malloc(sizeof(char) * 17);
  } else {
    buf = (char *) malloc(sizeof(char) * (strlen(tmp_str) + 12));
  }
  if(! buf) {
    perror("malloc");
    wuuush(1);
  }
  
  strcpy(buf, "cddb hello ");
  if(tmp_str == NULL) strcat(buf, "WSPse");
  else                strcat(buf, tmp_str);
  
  tmp_str = getenv("HOSTNAME");
  if(tmp_str == NULL) {
    buf = (char *) realloc(buf, sizeof(char) * (strlen(buf) + 28 +
  						strlen(PRG_VERSION)));
  } else {
    buf = (char *) realloc(buf, sizeof(char) * (strlen(buf) + 21 +
  						strlen(PRG_VERSION) +
  						strlen(tmp_str)));
  }
  if(buf == NULL) {
    perror("malloc");
    wuuush(1);
  }
  
  strcat(buf, " ");
  if(tmp_str == NULL) strcat(buf, "unknown");
  else                strcat(buf, tmp_str);
  
  strcat(buf, " WSPse-MP3creat V" PRG_VERSION "\r\n");
  print_stat_win(_("saying hello to cddb-server"));
#ifdef DEBUG
  fprintf(stderr, "DEBUG: sending hello: \"%s\"\n", buf);
#endif
  
  if(write_to_sock(buf, strlen(buf), cddb_fd) != 0) {
#ifdef DEBUG
    fprintf(stderr, "DEBUG: hello failed\n");
#endif
    close(cddb_fd);
    free(buf);
    return -1;
  }
  free(buf);
  
  if(read_from_sock(&buf, cddb_fd, TRUE) != 0) {
    close(cddb_fd);
    return -1;
  }
#ifdef DEBUG
  fprintf(stderr, "DEBUG: answered: \"%s\"\n", buf);
#endif
  if(buf[0] != '2') {
    tmp_str = strchr(buf, '\n');
    if(tmp_str) *tmp_str = 0;
    popup_error_win(buf);
    free(buf);
    close(cddb_fd);
    return -1;
  }
  free(buf);

  /* ask server for protocol level and switch level if necessary */
  print_stat_win(_("checking supported protocol levels"));
  if(write_to_sock("proto\r\n", 7, cddb_fd)) {
#ifdef DEBUG
    fprintf(stderr, "DEBUG: proto failed\n");
#endif
    close(cddb_fd);
    return -1;
  }

  if(read_from_sock(&buf, cddb_fd, TRUE)) {
    close(cddb_fd);
    return -1;
  }
#ifdef DEBUG
  fprintf(stderr, "DEBUG: answered: \"%s\"\n", buf);
#endif
  if(buf[0]!='2') {
    tmp_str = strchr(buf, '\n');
    if(tmp_str) *tmp_str = 0;
    print_stat_win(buf);
    free(buf);
    /* this is no error, "proto" is not supported, so skip this */
  } else {
    cddb_ver_cur = 0;
    cddb_ver_sup = 0;
    tmp_str = strchr(buf, ':');
    if(tmp_str) {
      pat1 = strstr(tmp_str+1, "current ");
      pat2 = strstr(tmp_str+1, "supported ");
      if(pat1) cddb_ver_cur = atoi(pat1+8);
      if(pat2) cddb_ver_sup = atoi(pat2+10);
    }
    free(buf);
    if(cddb_ver_cur && cddb_ver_sup) {
#ifdef DEBUG
      fprintf(stderr, "DEBUG: version current=%d, supported=%d\n", cddb_ver_cur, cddb_ver_sup);
#endif
      cddb_ver_sel = (cddb_ver_sup>5?5:cddb_ver_sup); /* we only support up to 5 for now */
      if(cddb_ver_cur<cddb_ver_sel) {
	/* we need to switch */
#ifdef DEBUG
	fprintf(stderr, "DEBUG: switching to version %d\n", cddb_ver_sel);
#endif
	buf = (char *) malloc(16);
	if(!buf) {
	  perror("malloc");
	  wuuush(1);
	}
	snprintf(buf, 16, "proto %d\r\n", cddb_ver_sel);
	print_stat_win(_("switching protocol level"));
	if(write_to_sock(buf, strlen(buf), cddb_fd)) {
#ifdef DEBUG
	  fprintf(stderr, "DEBUG: proto switch failed\n");
#endif
	  close(cddb_fd);
	  return -1;
	}

	if(read_from_sock(&buf, cddb_fd, TRUE)) {
	  close(cddb_fd);
	  return -1;
	}
#ifdef DEBUG
	fprintf(stderr, "DEBUG: answered: \"%s\"\n", buf);
#endif
	if(buf[0]!='2') {
	  tmp_str = strchr(buf, '\n');
	  if(tmp_str) *tmp_str=0;
	  print_stat_win(buf);
	} else {
	  print_stat_win(_("protocol level changed"));
	}
	free(buf);
      }
    }
  }

  return cddb_fd;
}

/* ask cddb-server for entry */
int cddb_internet_server_lookup(char *addr, char *cddb_path, BOOL force)
{
  int cddb_fd;
  char *buf;
  char *tmp_str;
  int i,j;
  field_select_typ *cat_anchor, *cat_new;
  FILE *cddb_file, *sec_file;
  struct stat st;
  char cd_discid[9];

  cddb_fd = cddb_open_cddbp(addr);
  if(cddb_fd < 0) return cddb_fd;

  /* Wow, are here. We said hello to cddb-server and he greets us */

  buf = (char *) malloc(sizeof(char) * (7 * (track_last+1)) + 23);
  if(buf == NULL) {
    perror("malloc");
    wuuush(1);
  }
  snprintf(cd_discid, 9, "%08lx", cddb_discid(track_last));
  sprintf(buf, "cddb query %8s %d", cd_discid, track_last);

  tmp_str = (char *) malloc(sizeof(char) * 7);
  if(tmp_str == NULL) {
    perror("malloc");
    wuuush(1);
  }
  for(i=0;i<track_last;i++) {
    strcat(buf, " ");
    sprintf(tmp_str, "%d", cdtoc[i].frame + (75*(cdtoc[i].sec +
		    				 (60*(cdtoc[i].min)))));
    strcat(buf, tmp_str);
  }
  sprintf(tmp_str, "%d", cdtoc[track_last].sec + (60*cdtoc[i].min));
  strcat(buf, " ");
  strcat(buf, tmp_str);
  free(tmp_str);

/*  sprintf(buf, "cddb query 0f10ef12 18 150 2091 21925 52227 53010 70106 74118 95895 117252 133625 156892 180969 185602 208111 227229 248846 277178 301347 4337"); */

  strcat(buf, "\r\n");
  print_stat_win(_("query category of cd"));
#ifdef DEBUG
  fprintf(stderr, "DEBUG: ask for category \"%s\"\n", buf);
#endif

  if(write_to_sock(buf, strlen(buf), cddb_fd) != 0) {
#ifdef DEBUG
    fprintf(stderr, "DEBUG: error category ask\n");
#endif
    close(cddb_fd);
    free(buf);
    return -1;
  }
  free(buf);

  if(read_from_sock(&buf, cddb_fd, TRUE) != 0) {
    close(cddb_fd);
    return -1;
  }
#ifdef DEBUG
  fprintf(stderr, "DEBUG: answered: \"%s\"\n", buf);
#endif

  if((buf[0] != '2') || (buf[2] == '2')) {
    free(buf);
    print_stat_win(_("not found..."));
    write_to_sock("quit\r\n", 6, cddb_fd);
    if(read_from_sock(&buf, cddb_fd, TRUE) == 0) {
#ifdef DEBUG
      fprintf(stderr, "DEBUG: saying bye: \"%s\"\n", buf);
#endif
      free(buf);
    }
    close(cddb_fd);
    return 1;
  }

  /* buf: 200 category ....
          found exact match
     buf: 210 .....
     buf: 211 .....
          many matches
   */

  if(buf[1] == '0') {
    print_stat_win(_("found exact match"));
    cat_anchor = (field_select_typ *) malloc(sizeof(field_select_typ));
    if(cat_anchor == NULL) {
      perror("malloc");
      wuuush(1);
    }
    cat_anchor->next = NULL;
    cat_anchor->dest = NULL;
    for(i=0;i<strlen(buf);i++)   if(buf[i] == ' ') break;
    for(j=i+1;j<strlen(buf);j++) if(buf[j] == ' ') break;
    tmp_str = (char *) malloc(sizeof(char) * (j-i));
    if(tmp_str == NULL) {
      perror("malloc");
      wuuush(1);
    }
    memcpy(tmp_str, (buf+i+1), (j-i)-1);
    (*(tmp_str + (j-i)-1)) = 0;
    cat_anchor->field = tmp_str;

    tmp_str = (char *) malloc(sizeof(char) * (strlen(cddb_path) +
					      strlen(cat_anchor->field) +
					      11));
    if(tmp_str == NULL) {
      perror("malloc");
      wuuush(1);
    }

    (*(buf + j + 9)) = 0;
    sprintf(tmp_str, "%s/%s/%8s", cddb_path, cat_anchor->field, (buf+j+1));
    free(buf);
    cat_anchor->information = tmp_str;
  } else {
    /* receive more categories */
    print_stat_win(_("serveral matches for cd"));
    cat_anchor = NULL;
    free(buf);
    while(1) {
      if(read_from_sock(&buf, cddb_fd, TRUE) != 0) {
#ifdef DEBUG
	fprintf(stderr, "DEBUG: oops, cddb-server failed\n");
#endif
	close(cddb_fd);
	return -1;
      }
      if(buf[0] == '.') {
	free(buf);
	break;
      }
#ifdef DEBUG
      fprintf(stderr, "DEBUG: suggested entry: \"%s\n", buf);
#endif
      cat_new = (field_select_typ *) malloc(sizeof(field_select_typ));
      if(cat_new == NULL) {
	perror("malloc");
	wuuush(1);
      }
      cat_new->dest = NULL;
		 
      for(i=0;i<strlen(buf);i++) if(buf[i] == ' ') break;
      tmp_str = (char *) malloc(sizeof(char) * (i+1));
      if(tmp_str == NULL) {
	perror("malloc");
	wuuush(1);
      }
      memcpy(tmp_str, buf, i);
      (*(tmp_str + i)) = 0;
      cat_new->field = tmp_str;
      
      tmp_str = (char *) malloc(sizeof(char) * (strlen(cddb_path) +
						strlen(cat_new->field) +
						11));
      if(tmp_str == NULL) {
	perror("malloc");
	wuuush(1);
      }

      (*(buf + i + 9)) = 0;
      sprintf(tmp_str, "%s/%s/%8s", cddb_path, cat_new->field, (buf+i+1));
      free(buf);
      cat_new->information = tmp_str;
#ifdef DEBUG
      fprintf(stderr, "DEBUG: added: \"%s\"\n", cat_new->information);
#endif
      cat_new->next = cat_anchor;
      cat_anchor    = cat_new;
    }
  }

  /* linear list "cat_anchor" contains entries which we must read from
     cddb-server.
   */
  
  cat_new = cat_anchor;
  while(cat_new != NULL) {
    for(i=strlen(cat_new->information)-1;i>=0;i--)
      if(cat_new->information[i] == '/') break;
    tmp_str = (char *) malloc(sizeof(char) * (i+1));
    if(tmp_str == NULL) {
      perror("malloc");
      wuuush(1);
    }
    memcpy(tmp_str, cat_new->information, i);
    (*(tmp_str + i)) = 0;
    if(access(tmp_str, F_OK) != 0) {
      /* directory doesn't exist */
      mkdir(tmp_str, 0777);
    }
    if(stat(tmp_str, &st) != 0) {
#ifdef DEBUG
      fprintf(stderr, "DEBUG: stat failed for \"%s\"\n", tmp_str);
#endif
      free(tmp_str);
    } else if(! S_ISDIR(st.st_mode)) {
#ifdef DEBUG
      fprintf(stderr, "DEBUG: \"%s\" isn't directory\n", tmp_str);
#endif
      free(tmp_str);
    } else {
      free(tmp_str);
      if((access(cat_new->information, F_OK) == 0) && (! force)) {
#ifdef DEBUG
	fprintf(stderr, "DEBUG: cddb-entry already exist\n");
#endif
      } else {
	cddb_file = fopen(cat_new->information, "w");
	if(cddb_file == NULL) {
#ifdef DEBUG
	  fprintf(stderr, "DEBUG: error opening \"%s\" for write\n", 
		  cat_new->information);
#endif
	} else {
	  tmp_str = (char *) malloc(sizeof(char) * (30 + strlen(cat_new->
								field)));
	  if(tmp_str == NULL) {
	    perror("malloc");
	    wuuush(1);
	  }
	  sprintf(tmp_str, "cddb read %s %8s\r\n", cat_new->field,
		  (cat_new->information + (strlen(cat_new->information)-8)));
	  print_stat_win(_("send request for data transfer"));
#ifdef DEBUG
	  fprintf(stderr, "DEBUG: sending request \"%s\"\n", tmp_str);
#endif
	  if(write_to_sock(tmp_str, strlen(tmp_str), cddb_fd) != 0) {
#ifdef DEBUG
	    fprintf(stderr, "DEBUG: error sending request \"%s\"\n", tmp_str);
#endif
	    free(tmp_str);
	    fclose(cddb_file);
	    unlink(cat_new->information);
	    free_field_select(&cat_anchor);
	    /* must leave here, because remote-error occured. if there are
	       some more files in the queue, otherwise we must waste a lot
	       of time for waiting for timeout */
	    return -1;
	  }
	  free(tmp_str);
	  if(read_from_sock(&tmp_str, cddb_fd, TRUE) != 0) {
#ifdef DEBUG
	    fprintf(stderr, "DEBUG: no controll message from cddb-server\n");
#endif
	    fclose(cddb_file);
	    unlink(cat_new->information);
	    free_field_select(&cat_anchor);
	    return -1;
	  }
	  if(strncmp(tmp_str, "210", 3) != 0) {
#ifdef DEBUG
	    print_stat_win("not available");
	    fprintf(stderr, "DEBUG: weia, file not available? \"%s\"\n",
		    tmp_str);
#endif
	    free(tmp_str);
	    fclose(cddb_file);
	    unlink(cat_new->information);
	  } else {
	    print_stat_win(_("receiving CDDB entry"));
  	    while(1) {
  	      if(read_from_sock(&tmp_str, cddb_fd, TRUE) != 0) {
#ifdef DEBUG
  		fprintf(stderr, "DEBUG: error receiving cddb-entry, maybe"
  			"timeout?\n");
#endif
  		fclose(cddb_file);
  		unlink(cat_new->information);
  		free_field_select(&cat_anchor);
  		/* s.o ;-) */
  		return -1;
  	      }
  	      if(tmp_str[0] == '.') break;
  	      fputs(tmp_str, cddb_file);
  	      free(tmp_str);
  	    }
  	    free(tmp_str);  
  	    fclose(cddb_file);
	    tmp_str = (char *) malloc(sizeof(char) * (strlen(cat_new->
							     information)+1));
	    if(tmp_str == NULL) {
	      perror("malloc");
	      wuuush(1);
	    }
	    strcpy(tmp_str, cat_new->information);
	    memcpy((tmp_str + strlen(tmp_str) -8), cd_discid, 8);
	    if((strcmp(tmp_str, cat_new->information) != 0) &&
	       ((access(tmp_str, F_OK) != 0) || force)) {
	      cddb_file = fopen(cat_new->information, "r");
	      if(cddb_file != NULL) {
		sec_file = fopen(tmp_str, "w");
		if(sec_file != NULL) {
		  while(1) {
		    i = fgetc(cddb_file);
		    if(feof(cddb_file)) break;
		    fputc(i, sec_file);
		  }
		  fclose(sec_file);
		}
		fclose(cddb_file);
	      }
	    }
	    free(tmp_str);
	  }
	}
      }
    }
    cat_new = cat_new->next;
  }
  
  free_field_select(&cat_anchor);
  
  /* say bye to cddb-server... */
  print_stat_win(_("closing connection"));
  if(write_to_sock("quit\r\n", 6, cddb_fd) == 0) {
    if(read_from_sock(&buf, cddb_fd, TRUE) == 0) {
#ifdef DEBUG
      fprintf(stderr, "DEBUG: saying bye: \"%s\"\n", buf);
#endif
      free(buf);
    }
  }
  close(cddb_fd);

  return 0;
}  
 
/*---------------------------------------------------------------------------*/
/* prepare data-structure for one cd-track
   lookup title and artist for track (maybe there are varios artists,
   so we must look for the content of tracknames)
 */

int lookup_track_data (int track, int cddb_fd, song_typ **ret_struct, BOOL force_sampler)
{
  int i,j;
  song_typ *new_song;
  char *buf;
  char *album, *title;
  char *extented;
  char pattern[10];
  char pattern2[8];

  lseek(cddb_fd, 0L, SEEK_SET);

  new_song = (song_typ *) malloc(sizeof(song_typ));
  if(new_song == NULL) {
    perror("malloc");
    wuuush(1);
  }

  new_song->next    = NULL;
  new_song->prev    = NULL;
  new_song->convert = TRUE;
  new_song->on_fly  = def_on_fly;
  new_song->year    = 1999;
  new_song->genre   = TOT_GENRES;  /* unknown */
  new_song->cddb_id = cddb_discid(track_last);
  new_song->tmp_wav_file = NULL;
  new_song->sampler = FALSE;
  
  new_song->comment = (char *) malloc(sizeof(char) * (strlen(def_comment)+1));
  if(new_song->comment == NULL) {
    perror("malloc");
    wuuush(1);
  }
  strcpy(new_song->comment, def_comment);

  album    = (char *) malloc(sizeof(char));
  title    = (char *) malloc(sizeof(char));
  extented = (char *) malloc(sizeof(char));
  if((album == NULL) || (title == NULL) || (extented == NULL)) {
    perror("malloc");
    wuuush(1);
  }
  *album    = 0;
  *title    = 0;
  *extented = 0;
  sprintf(pattern, "TTITLE%d=", track-1);
  sprintf(pattern2, "EXTT%d=",  track-1);

  while(1) {
    if(read_from_sock(&buf, cddb_fd, FALSE) != 0) break;
    if((*(buf + strlen(buf) - 1) == 0x0a) || (*(buf + strlen(buf) - 1) == 0x0d))
      *(buf + strlen(buf) - 1) = 0;
    if((*(buf + strlen(buf) - 1) == 0x0a) || (*(buf + strlen(buf) - 1) == 0x0d))      *(buf + strlen(buf) - 1) = 0;
    if(strncmp(buf, "# Generated: ", 13) == 0) {
      j=0;
      for(i=13;i<strlen(buf);i++) {
	if(*(buf+i) == ' ') j++;
	if(j == 4) break;
      }
      i++;
      if(strlen(buf)-i >= 4) {
	if((strlen(buf)-i > 4) && (*(buf + i +4) == ' ')) *(buf+i+4)=0;
	if(strlen(buf)-i == 4) {
	  for(j=0;j<4;j++) {
	    if((*(buf+i+j) < '0') || (*(buf+i+j) > '9')) break;
	  }
	  if(j == 4) {
	    new_song->year = atoi((buf+i));
#ifdef DEBUG
	    fprintf(stderr, "DEBUG: find year: %d\n", new_song->year);
#endif
	  }
	}
      }
    } else if(strncmp(buf, "# Revision: ", 12) == 0) {
      cddb_cddb_entry_revision = atoi(buf + 12);
      if(cddb_cddb_entry_revision < 0) {
	cddb_cddb_entry_revision = 0;
      }
    } else if(strncmp(buf, "DTITLE=", 7) == 0) {
      album = (char *) realloc(album, sizeof(char) * (strlen(album) +
						      strlen(buf+6)));
      if(album == NULL) {
	perror("realloc");
	wuuush(1);
      }
      strcat(album, buf+7);
    } else if(!strncmp(buf, "DYEAR=", 6)) {
      new_song->year = atoi(buf+6);
    } else if(!strncmp(buf, "DGENRE=", 7)) {
      for(i=0;i<TOT_GENRES;i++) if(!strcasecmp(mp3_genre[i], buf+7)) break;
      new_song->genre = i;
    } else if(strncmp(buf, pattern, strlen(pattern)) == 0) {
      title = (char *) realloc(title, sizeof(char) * (strlen(title) +
					      strlen(buf+strlen(pattern))+ 1));
      if(title == NULL) {
	perror("realloc");
	wuuush(1);
      }
      strcat(title, buf+strlen(pattern));
    } else if(strncmp(buf, pattern2, strlen(pattern2)) == 0) {
      extented = (char *) realloc(extented, sizeof(char) * (strlen(extented) +
			    strlen(buf+strlen(pattern2))+ 1));
      if(extented == NULL) {
	perror("realloc");
	wuuush(1);
      }
      strcat(extented, buf+strlen(pattern2));
    }
    free(buf);
  }
  new_song->title = title;

  /* new: there are entries which do not use the conventions, so we maybe have
   *      to correct this now */
  j = 0;
  for(i=0;i<strlen(album);i++) {
    if(*(album+i) == '/') {
      if((i>0) && (i<strlen(album)-1) && (*(album + i - 1) == ' ') &&
	 (*(album + i + 1) == ' ')) j=i;
    }
  }
 
  if(j == 0) { 
    for(i=0;i<strlen(album);i++) if(*(album+i) == '/') break;
  } else {
    i = j;
  }
  if((i>0) && (i<strlen(album))) {
    for(j=i-1;j>=0;j--) if(*(album+j) != ' ') break;
    buf = (char *) malloc(sizeof(char) * (j+2));
    if(buf == NULL) {
      perror("malloc");
      wuuush(1);
    }
    memcpy(buf, album, j+1);
    *(buf + j+1) = 0;
    new_song->artist = buf;
    
    for(j=i+1;j<strlen(album);j++) if(*(album+j) != ' ') break;
    buf = (char *) malloc(sizeof(char) * (strlen(album+j)+1));
    if(buf == NULL) {
      perror("malloc");
      wuuush(1);
    }
    strcpy(buf, (album+j));
    new_song->album = buf;
    free(album);
  } else {
    new_song->album = album;
    new_song->artist = (char *) malloc(sizeof(char) * 8);
    if(new_song->artist == NULL) {
      perror("malloc");
      wuuush(1);
    }
    strcpy(new_song->artist, "unknown");
  }

  /* routine is ready, but now we have to test if there are various artists */
#ifdef DEBUG
  fprintf(stderr, "DEBUG: interpreted data for track %d:\n"
	  "Album: \"%s\"\n"
	  "Artist: \"%s\"\n"
	  "Title: \"%s\"\n"
	  "Year: %d\n", track, new_song->album, new_song->artist,
	  new_song->title, new_song->year);
#endif
  if((strncasecmp(new_song->artist, "various", 7) == 0) ||
     (strncasecmp(new_song->artist, "compilation", 11) == 0) ||
     (strncasecmp(new_song->artist, "varios", 6) ==0) ||
     (strncasecmp(new_song->artist, "sampler", 7) == 0) ||
     (strncasecmp(new_song->artist, "divers", 6) == 0) ||
     (strncasecmp(new_song->artist, "v.a.", 4) == 0) ||
     (strncasecmp(new_song->artist, "misc",4) == 0) ||
     force_sampler) {
#ifdef DEBUG
    fprintf(stderr, "DEBUG: various artists...\n");
#endif
    new_song->sampler = TRUE;
    
    if(strlen(extented) > 0) {
      new_song->artist = extented;
      extented = NULL;
    }
    for(i=0;i<strlen(new_song->title);i++) {
      if((*(new_song->title + i) == '/') ||
	 (*(new_song->title + i) == '-') ||
	 ((*(new_song->title + i) == '(') &&
	  (*(new_song->title + strlen(new_song->title) -1) == ')'))) break;
    }
    if((i>0) && (i<strlen(new_song->title))) {
      if(*(new_song->title + i) == '(')
	*(new_song->title + strlen(new_song->title) -1) = 0;
      for(j=i-1;j>=0;j--) if(*(new_song->title + j) != ' ') break;
      buf = (char *) malloc(sizeof(char) * (j+2));
      if(buf == NULL) {
	perror("malloc");
	wuuush(1);
      }
      memcpy(buf, new_song->title, j+1);
      *(buf + j+1) = 0;
      free(new_song->artist);
      new_song->artist = buf;

      for(j=i+1;j<strlen(new_song->title);j++) if(*(new_song->title + j) != ' ')
	break;
      buf = (char *) malloc(sizeof(char) * (strlen(new_song->title + j) + 1));
      if(buf == NULL) {
	perror("malloc");
	wuuush(1);
      }
      strcpy(buf, (new_song->title+j));
      free(new_song->title);
      new_song->title = buf;
    }
#ifdef DEBUG
    fprintf(stderr, "DEBUG: new datas:\nArtist: %s\nTitle: %s\n",
	    new_song->artist, new_song->title);
#endif
  }
  
  if(extented) free(extented);
  *ret_struct = new_song;
  return 0;
}

/*---------------------------------------------------------------------------*/
/* build up data tree for inserted cd. returns 1 in case of error
   if cddb-entry is not available, build empty tree
 */
int build_data_tree(char *cddb_server, char *local_cddb_db,
		    song_typ **ret_tree, BOOL force_sampler)
{
  char *cddb_file;
  song_typ *new, *anchor, *curr;
  int i,j;
  int cddb_fd;
  int genre;
  char *buf;

  setup_stat_win(50);
  print_stat_win("lookup CDDB entry");

  cddb_cddb_entry_is_clean = FALSE;
  cddb_cddb_entry_revision = -1;
  
  anchor = NULL;
  curr   = NULL;
  
  /* first look if we can find an entry locally */
  if(cddb_local_lookup(local_cddb_db, &cddb_file) != 0) {
    /* not successful. if we have internet connect, lookup cddb-server */
    if(cddb_server) cddb_internet_lookup(cddb_server, local_cddb_db, FALSE);
    /* maybe we got some entries now */
    if(cddb_local_lookup(local_cddb_db, &cddb_file) != 0) {
      /* have no information, build empty tree */
      for(i=0;i<track_last;i++) {
	new = (song_typ *) malloc(sizeof(song_typ));
	if(new == NULL) {
	  perror("malloc");
	  wuuush(1);
	}
	new->convert = TRUE;
	new->artist  = copy_char_str("unknown");
	new->title   = copy_char_str("unknown");
	new->album   = copy_char_str("unknown");
	new->comment = copy_char_str(def_comment);
	new->dirname = copy_char_str(def_mp3_dir);
	new->year  = 1999;
	new->genre = TOT_GENRES;
	new->on_fly = def_on_fly;
	new->toc = i;
	new->fn_auto = TRUE;
	new->filename = NULL;
	new->cddb_id = cddb_discid(track_last);
	new->frame_len = (cdtoc[i+1].frame + 75*((60*cdtoc[i+1].min) + cdtoc[i+1].sec)) -
	  (cdtoc[i].frame + 75*((60*cdtoc[i].min) + cdtoc[i].sec));
	new->tmp_wav_file = NULL;
	new->sampler = force_sampler;
	filenm_generate(new);

	new->next = NULL;
	new->prev = curr;
	if(curr) curr->next = new;
	if(! anchor) anchor = new;
	curr = new;
      }
      *ret_tree = anchor;
      destroy_stat_win();
      return 0;
    }
  }

  /* well, we have an entry, now we fill our tree with useful information */
  for(i=strlen(cddb_file)-1;i>=0;i--) if(*(cddb_file + i) == '/') break;
  for(j=i-1;j>=0;j--)                 if(*(cddb_file + j) == '/') break;
  buf = (char *) malloc(sizeof(char) * (i-j));
  if(buf == NULL) {
    perror("malloc");
    wuuush(1);
  }
  memcpy(buf, (cddb_file + j+1), (i-j));
  *(buf + (i-j) -1) = 0;
#ifdef DEBUG
  fprintf(stderr, "DEBUG: find match for genre \"%s\"\n", buf);
#endif
  for(i=0;i<TOT_GENRES;i++) if(strcasecmp(mp3_genre[i], buf)==0) break;
#ifdef DEBUG
  fprintf(stderr, "DEBUG: match: \"%s\"\n", mp3_genre[i]);
#endif
  free(buf);
  genre = i;
  
  cddb_fd = open(cddb_file, O_RDONLY);
  cddb_cddb_entry_is_clean = TRUE;
  for(i=1;i<=track_last;i++) {
    lookup_track_data(i, cddb_fd, &new, force_sampler);
    if(! new) {
      new = (song_typ *) malloc(sizeof(song_typ));
      if(new == NULL) {
	perror("malloc");
	wuuush(1);
      }
      new->convert = TRUE;
      new->year    = 1999;
      new->on_fly  = def_on_fly;
      new->sampler = force_sampler;
    }
    if(! new->artist) {
      new->artist = copy_char_str("unknown");
    }
    if(! new->title) {
      new->title = copy_char_str("unknown");
    }
    if(! new->album) {
      new->album = copy_char_str("unknown");
    }
    if(! new->comment) {
      new->comment = copy_char_str(def_comment);
    }
    new->dirname = copy_char_str(def_mp3_dir);

    if(new->genre==TOT_GENRES) new->genre = genre;
    new->toc = i-1;
    new->fn_auto = TRUE;
    new->filename = NULL;
    new->frame_len = (cdtoc[i].frame + 75*((60*cdtoc[i].min) + cdtoc[i].sec)) -
      (cdtoc[i-1].frame + 75*((60*cdtoc[i-1].min) + cdtoc[i-1].sec));
    new->tmp_wav_file = NULL;
    filenm_generate(new);

    new->next = NULL;
    new->prev = curr;
    if(curr != NULL) curr->next = new;
    if(anchor == NULL) anchor = new;
    curr = new;
  }
  
  close(cddb_fd);
  *ret_tree = anchor;
  destroy_stat_win();
  return 0;
}

