/*
 *  svaf.h: header file for the SVAF libraries.
 *
 *  Copyright (c) 2006, Michael C. Martin
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - Neither the name 'Bumbershoot Software' nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 *  CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 *  BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 *  TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 *  THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 */

#ifndef __SVAF_H__
#define __SVAF_H__

#include <SDL_opengl.h>

typedef struct {
	GLfloat diffuse[4];
	GLfloat specular[4];
	GLfloat emissive[4];
	GLfloat shininess;
} svaf_material;

typedef struct {
	int count;
	GLenum type;
	GLuint *indices;
} svaf_elements;

typedef enum {
	SVAF_CMD_NONE,
	SVAF_CMD_MATERIAL,
	SVAF_CMD_ELEMENTS,
	SVAF_CMD_MAX
} svaf_cmd;

typedef struct {
	svaf_cmd tag;
	union {
		svaf_material material;
		svaf_elements elements;
	} command;
} svaf_command;

typedef struct {
	int num_vertices, num_commands;
	GLfloat *vertices;
	GLfloat *normals;
	svaf_command *commands;
} svaf_model;

typedef struct {
	GLfloat *vertices, *normals;
	int num_vertices;
} svaf_amalgam;

/* Function prototypes */

#ifdef __cplusplus
extern "C" {
#endif

svaf_model *svaf_ReadFile (const char *fname);
svaf_model *svaf_ReadBuffer (const unsigned char *buffer, int len, const char *name);
void svaf_FreeModel (svaf_model *m);

void svaf_SelectModel (svaf_model *m);
void svaf_SelectAmalgam (svaf_amalgam *a);
void svaf_RenderCommands (svaf_model *m);

svaf_amalgam *svaf_NewAmalgam (void);
void svaf_FreeAmalgam (svaf_amalgam *a);

void svaf_RegisterGeometry (svaf_amalgam *a, svaf_model *m);
void svaf_SetAlpha (svaf_model *m, GLfloat alpha);

#ifdef __cplusplus
}
#endif

#endif
