/*
 *  utils.c  - zniper utilities
 *
 *  Copyright (c) 2002 Claes M. Nyberg <md0claes@mdstud.chalmers.se>
 *  All rights reserved, all wrongs reversed.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 * 
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. The name of author may not be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 *  THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *      
 * $Id: utils.c,v 1.1.1.1 2004/07/30 11:19:33 cmn Exp $ 
 */ 

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <sys/stat.h>
#include "zniper.h"

/*
 * Returns 1 if path is a regular file, 0 otherwise
 */
int
is_reg_file(const char *path)
{
    struct stat sb;

    if (path == NULL)
        return(0);

    memset(&sb, 0x00, sizeof(sb));
    if (stat(path, &sb) < 0)
        return(0);

    return(S_ISREG(sb.st_mode));
}

/*
 *  Returns size of regular file in bytes, -1 on error
 */
int
file_size(const char *path)
{
    struct stat sb;

    if ((path == NULL) || !is_reg_file(path))
        return(-1);

    memset(&sb, 0x00, sizeof(sb));
    if (stat(path, &sb) < 0)
        return(-1);

    return(sb.st_size);
}


/*
 * Allocate memory or die
 */
void *
memordie(size_t n)
{
	void *pt;

	if ( (pt = calloc(1, n)) == NULL) {
		writestatus(0, "** Error: Failed to allocate memory: %s", 
			strerror(errno));
		exit(EXIT_FAILURE);
	}

	return(pt);
}

/*
 * Returns time as a string
 */
u_char *
timestr(time_t time)
{
    static u_char tstr[256];
    struct tm *tm;

    memset(tstr, 0x00, sizeof(tstr));

    if ( (tm = localtime(&time)) == NULL) {
        snprintf(tstr, sizeof(tstr) -1, "localtime(): %s",
                strerror(errno));
    }

    else if (strftime(tstr, sizeof(tstr) -1, "%b %d %T", tm) == 0) {
        snprintf(tstr, sizeof(tstr) -1, "localtime(): %s",
            strerror(errno));
    }

    return(tstr);
}

