#  xawale.tcl -- graphical interface to play the african game.
#  Copyright (C) 2006 Laurent Le Bot and Alain Le Bot.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2, or (at your option)
#  any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
#  USA
#
# Version 1.6 last modified on August 22nd, 2009.
#
#!/bin/bash
# start wish \
exec wish "$0" ${1+"$@"}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                 Welcome window with copyright
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc copyrightDialog {first} {
global BIN
destroy .copyright
set img [image create photo -file [file join $BIN copyright.gif]]
toplevel .copyright -borderwidth 1 -relief sunken
label .copyright.image -image $img
pack .copyright.image -side top -anchor nw
if {$first=="TRUE"} {wm overrideredirect .copyright 1}
wm geometry .copyright 500x327+300+250
focus .copyright
raise .copyright .
}


#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                   Initialisation
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc initGame {} {
global hole score label start
global level depth depth0
global turn iturn

      set label "RUN"

      for {set i 0} {$i<12} {incr i} {
       set hole($i) 4
      }

      set score(0) 0
      set score(1) 0

      set depth $depth0($level)

#      if {$start=="PLAYER"} {set label "RUN"} else {set label "BEG"}

      set iturn 0
      set turn(0) "board=4-4-4-4-4-4-4-4-4-4-4-4 score=0-0 move=0 $label"

      displaySeeds NULL
      if {$start=="COMPUTER"} {play 0 1 1 4}

}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                     Chose a file to import
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc importGame {} {
global PATH LANG IFIL EFIL
global level depth start
global hole score label
global turn iturn

set resu [tk_getOpenFile -parent . -title $IFIL($LANG) -initialdir [file join $PATH]]
if {$resu!=""} {set file $resu} else {return 0}
if {[catch {open $resu r} fid]} {tk_messageBox -message "$EFIL($LANG) $resu"; return}
set iturn -1
set depth [expr 7+$level]
while {[gets $fid turn([incr iturn])]>0} {
  scan $turn($iturn) "board=%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u score=%u-%u move=%u %6s" \
                hole(0) hole(1) hole(2) hole(3) hole(4) hole(5) \
                hole(6) hole(7) hole(8) hole(9) hole(10) hole(11) \
  score(0) score(1) mvt label
}
incr iturn -1
close $fid
displaySeeds NULL
endGame
if {$start=="COMPUTER"&&$label=="RUN"} {play 0 1 1 4}
}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                      Chose a file to save
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc saveGame {} {
global PATH LANG SFIL EFIL
global turn iturn
set resu [tk_getSaveFile -parent . -title $SFIL($LANG) -initialdir [file join $PATH]]
if {$resu!=""} {set file $resu} else {return 0}
if {[catch {open $resu w} fid]} {tk_messageBox -message "$EFIL($LANG) $resu"; return}
for {set jturn 0} {$jturn<=$iturn} {incr jturn} {puts $fid $turn($jturn)}
close $fid
}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                     Open configure file
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc openConfigure {} {
global PATH LANG level nbeat noeat noall noatk start

if {![catch {open [file join ~ .xawale.ini] r} fid]} {
gets $fid line
scan $line "LANG=%2s" LANG
gets $fid line
scan $line "LEVEL=%1s" level
gets $fid line
scan $line "NBEAT=%1s" nbeat
gets $fid line
scan $line "NOEAT=%1s" noeat
gets $fid line
scan $line "NOALL=%1s" noall
gets $fid line
scan $line "NOATK=%1s" noatk
gets $fid line
scan $line "START=%s" start
close $fid
}

}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                      Save configure file
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc saveConfigure {} {
global PATH LANG level nbeat noeat noall noatk start

if {![catch {open [file join ~ .xawale.ini] w} fid]} {
puts $fid LANG=$LANG
puts $fid LEVEL=$level
puts $fid NBEAT=$nbeat
puts $fid NOEAT=$noeat
puts $fid NOALL=$noall
puts $fid NOATK=$noatk
puts $fid START=$start
close $fid
}

}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                     Undo : cancel a turn
# Last modified 6 june 2008
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc undo {} {
global hole score label level
global turn iturn
set undo no
if {$level=="H"&$iturn>=1} {incr iturn -1; set undo yes}
if {$level!="H"&$iturn>=2} {incr iturn -2; set undo yes}
if {$undo=="yes"} {
scan $turn($iturn) "board=%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u score=%u-%u move=%u %6s" \
                hole(0) hole(1) hole(2) hole(3) hole(4) hole(5) \
                hole(6) hole(7) hole(8) hole(9) hole(10) hole(11) \
                score(0) score(1) mvt label
displaySeeds NULL
}
}


#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#              Display the board and seeds
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc displaySeeds {mvt} {
global LANG WIN LOSE DRAW NWIN SWIN CSCO PSCO P1CO P2CO DEPT TURN
global level
global hole score label depth
global img idhole mvtcircle indication
global turn iturn

set a 74
set b 83
# width of highlighted circle
set c 4

for {set i 0} {$i<12} {incr i} {.cnv delete $idhole($i)}
.cnv delete $mvtcircle

set idhole(11)  [.cnv create image [expr 52+$b*0] [expr 135] -image $img($hole(11))]
set idhole(10)  [.cnv create image [expr 52+$b*1] [expr 133] -image $img($hole(10))]
set idhole(9)   [.cnv create image [expr 53+$b*2] [expr 131] -image $img($hole(9))]
set idhole(8)   [.cnv create image [expr 55+$b*3] [expr 129] -image $img($hole(8))]
set idhole(7)   [.cnv create image [expr 57+$b*4] [expr 130] -image $img($hole(7))]
set idhole(6)   [.cnv create image [expr 58+$b*5] [expr 131] -image $img($hole(6))]
set idhole(0)   [.cnv create image [expr 54+$b*0] [expr 219] -image $img($hole(0))]
set idhole(1)   [.cnv create image [expr 54+$b*1] [expr 219] -image $img($hole(1))]
set idhole(2)   [.cnv create image [expr 58+$b*2] [expr 215] -image $img($hole(2))]
set idhole(3)   [.cnv create image [expr 58+$b*3] [expr 214] -image $img($hole(3))]
set idhole(4)   [.cnv create image [expr 58+$b*4] [expr 214] -image $img($hole(4))]
set idhole(5)   [.cnv create image [expr 58+$b*5] [expr 212] -image $img($hole(5))]

set idhole(12)  [.cnv create image [expr 60+$b*2] [expr 43] -image $img($score(1))]
if {$score(1)>15} {
set idhole(13)  [.cnv create image [expr 63+$b*3] [expr 45] -image $img([expr $score(1)-15])]
} else {
set idhole(13)  [.cnv create image [expr 63+$b*3] [expr 45] -image $img(0)]
}
set idhole(14)  [.cnv create image [expr 68+$b*2] [expr 308] -image $img($score(0))]
if {$score(0)>15} {
set idhole(15)  [.cnv create image [expr 68+$b*3] [expr 305] -image $img([expr $score(0)-15])]
} else {
set idhole(15)  [.cnv create image [expr 68+$b*3] [expr 305] -image $img(0)]
}

.cnv delete $indication
if {$level=="H"} {
   set indication [.cnv create text 20 30 \
   -text "$P1CO($LANG) $score(1)\n$P2CO($LANG) $score(0)\n$DEPT($LANG) $depth $TURN($LANG) $iturn" -anchor w]
} else {
   set indication [.cnv create text 20 30 \
   -text "$CSCO($LANG) $score(1)\n$PSCO($LANG) $score(0)\n$DEPT($LANG) $depth $TURN($LANG) $iturn" -anchor w]
}

# highlights just played hole
if {$label!="ERR"&$mvt>=1&$mvt<=12} {
if {$mvt=="12"} {set mvtcircle [.cnv create oval [expr 15+$b*0] [expr  98] [expr 15+$b*0+$a] [expr  98+$a] -width $c]}
if {$mvt=="11"} {set mvtcircle [.cnv create oval [expr 15+$b*1] [expr  96] [expr 15+$b*1+$a] [expr  96+$a] -width $c]}
if {$mvt=="10"} {set mvtcircle [.cnv create oval [expr 16+$b*2] [expr  94] [expr 16+$b*2+$a] [expr  94+$a] -width $c]}
if {$mvt=="9"}  {set mvtcircle [.cnv create oval [expr 18+$b*3] [expr  92] [expr 18+$b*3+$a] [expr  92+$a] -width $c]}
if {$mvt=="8"}  {set mvtcircle [.cnv create oval [expr 20+$b*4] [expr  93] [expr 20+$b*4+$a] [expr  93+$a] -width $c]}
if {$mvt=="7"}  {set mvtcircle [.cnv create oval [expr 21+$b*5] [expr  94] [expr 21+$b*5+$a] [expr  94+$a] -width $c]}
if {$mvt=="6"}  {set mvtcircle [.cnv create oval [expr 22+$b*5] [expr 175] [expr 22+$b*5+$a] [expr 175+$a] -width $c]}
if {$mvt=="5"}  {set mvtcircle [.cnv create oval [expr 22+$b*4] [expr 176] [expr 22+$b*4+$a] [expr 176+$a] -width $c]}
if {$mvt=="4"}  {set mvtcircle [.cnv create oval [expr 22+$b*3] [expr 175] [expr 22+$b*3+$a] [expr 175+$a] -width $c]}
if {$mvt=="3"}  {set mvtcircle [.cnv create oval [expr 22+$b*2] [expr 176] [expr 22+$b*2+$a] [expr 176+$a] -width $c]}
if {$mvt=="2"}  {set mvtcircle [.cnv create oval [expr 17+$b*1] [expr 180] [expr 17+$b*1+$a] [expr 180+$a] -width $c]}
if {$mvt=="1"}  {set mvtcircle [.cnv create oval [expr 17+$b*0] [expr 180] [expr 17+$b*0+$a] [expr 180+$a] -width $c]}
}

update
after 400
.cnv delete $mvtcircle

}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#             Display the message of game over
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc endGame {} {
global label level score
global LANG LOSE WIN SWIN NWIN DRAW

if {$label=="END"} {
  if {$level!="H"} {
     set S $LOSE($LANG)
     set N $WIN($LANG)
  } else {
     set S $SWIN($LANG)
     set N $NWIN($LANG)
  }
  if {$score(0)>$score(1)} {
#  tk_messageBox -message $S
  set endmessage [.cnv create text 250 172 -text $S -font [list helvetica 36 bold]]
  } elseif {$score(0)<$score(1)} {
#  tk_messageBox -message $N
  set endmessage [.cnv create text 250 172 -text $N -font [list helvetica 36 bold]]
  } else {
#  tk_messageBox -message $DRAW($LANG)
  set endmessage [.cnv create text 250 172 -text $DRAW($LANG) -font [list helvetica 36 bold]]
  }
  after 4000 .cnv delete $endmessage
}

}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                        Play
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc play {x y w h} {
global BIN JOUER
global level depth Tmin Tmax nbeat noeat noall noatk
global hole score label start
global turn iturn

set ligne [format %.0f [expr floor(4*$y/$h)]]
set colon [format %.0f [expr floor(6*$x/$w)]]
if {$ligne==1} {set hole0 [expr 12-$colon]}
if {$ligne==2} {set hole0 [expr $colon+1]}
if {$ligne!=1&$ligne!=2} {return}

.cnv configure -cursor watch
update

# player move
if {$level!="H"&&$ligne==2&&$label=="RUN"||$level=="H"&&$label=="RUN"} {

  displayDistribution [exec $JOUER \
    -b $hole(0)-$hole(1)-$hole(2)-$hole(3)-$hole(4)-$hole(5)-$hole(6)-$hole(7)-$hole(8)-$hole(9)-$hole(10)-$hole(11) \
    -o $score(0)-$score(1) -p $hole0 -e $nbeat [string map {" " ""} -sx$noeat$noall$noatk]]

}

# computer move
if {$level!="H"&&$ligne==2&&$label=="RUN"||$level!="H"&&$label=="RUN"&&$start=="COMPUTER"} {

  set time0 [clock seconds]

  displayDistribution [exec $JOUER \
    -b $hole(0)-$hole(1)-$hole(2)-$hole(3)-$hole(4)-$hole(5)-$hole(6)-$hole(7)-$hole(8)-$hole(9)-$hole(10)-$hole(11) \
    -o $score(0)-$score(1) -c 2 -d $depth -e $nbeat [string map {" " ""} -sx$noeat$noall$noatk]]

  set time [expr [clock seconds]-$time0]
  if {$time==0}            {after 1000}
  if {$time>$Tmax($level)} {incr depth -1}
  if {$time<$Tmin($level)} {incr depth  1}

}

.cnv configure -cursor top_left_arrow
if {$label=="ERR"} {set label "RUN"}
}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#          Display the distribution of seeds
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc displayDistribution {awaleoutput} {
  global hole score label
  global turn iturn

  scan [string range $awaleoutput [string first "move=" $awaleoutput] end] "move=%d" mvt
  displaySeeds $mvt
  set lines [split $awaleoutput "\n"]
  foreach line $lines {
    scan $line "board=%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u-%u score=%u-%u move=%u %6s" \
                hole(0) hole(1) hole(2) hole(3) hole(4) hole(5) \
                hole(6) hole(7) hole(8) hole(9) hole(10) hole(11) \
                score(0) score(1) mvt label
    displaySeeds $mvt
  }
  if {$label!="ERR"} {set turn([incr iturn]) $line}
  if {[cyclic]==1&&$label!="ERR"} {set label "END"}
  endGame

}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                        Cyclic
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc cyclic {} {
global hole score label
global turn iturn

set n [string first " score" turn($iturn)]

for {set i 1} {$i<$iturn} {incr i} {
  if {[string compare -length $n $turn($i) $turn($iturn)]==0} {
#     tk_messageBox -message "same position found in turn $i"
     return 1
  }
}

return 0

}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                   Display the content of a hole
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc displayContent {x y w h} {
global level depth
global hole score label LANG SEED

set ligne [format %.0f [expr floor(4*$y/$h)]]
set colon [format %.0f [expr floor(6*$x/$w)]]

if {$ligne==0&$label=="RUN"} {
   after 1000 ".cnv delete [.cnv create text 38 70 -text "$score(1) $SEED($LANG)"]"
}

if {$ligne==1&$label=="RUN"} {
   after 1000 ".cnv delete [.cnv create text 38 70 -text "$hole([expr 11-$colon]) $SEED($LANG)"]"
}

if {$ligne==2&$label=="RUN"} {
   after 1000 ".cnv delete [.cnv create text 38 70 -text "$hole($colon) $SEED($LANG)"]"
}

if {$ligne==3&$label=="RUN"} {
   after 2000 ".cnv delete [.cnv create text 38 70 -text "$score(0) $SEED($LANG)"]"
}


}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                       Configure rules
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc configureRules {} {
global BIN LANG MAXI EATL EATN EATA NOAL ATKA NOAK YES NO OK DEFA DETA nbeat noeat noall noatk
destroy .configure
toplevel .configure -borderwidth 1 -relief sunken

labelframe .configure.frame1 -text $DETA($LANG) -labelanchor nw -padx 10 -borderwidth 2

frame  .configure.frame1.nbeat -borderwidth 10
label  .configure.frame1.nbeat.lbl -text $MAXI($LANG)
radiobutton .configure.frame1.nbeat.check1 -text "1" -variable nbeat -value "1"
radiobutton .configure.frame1.nbeat.check2 -text "2" -variable nbeat -value "2"
radiobutton .configure.frame1.nbeat.check3 -text "3" -variable nbeat -value "3"
radiobutton .configure.frame1.nbeat.check4 -text "4" -variable nbeat -value "4"
radiobutton .configure.frame1.nbeat.check5 -text "5" -variable nbeat -value "5"
radiobutton .configure.frame1.nbeat.check6 -text "6" -variable nbeat -value "6"
pack   .configure.frame1.nbeat.lbl -side left -anchor n
pack   .configure.frame1.nbeat.check1 -side left -anchor n
pack   .configure.frame1.nbeat.check2 -side left -anchor n
pack   .configure.frame1.nbeat.check3 -side left -anchor n
pack   .configure.frame1.nbeat.check4 -side left -anchor n
pack   .configure.frame1.nbeat.check5 -side left -anchor n
pack   .configure.frame1.nbeat.check6 -side left -anchor n

frame  .configure.frame1.eat -borderwidth 10
frame  .configure.frame1.eat.lbl
label  .configure.frame1.eat.lbl.1 -text $EATL($LANG)
label  .configure.frame1.eat.lbl.2 -text $EATN($LANG)
pack   .configure.frame1.eat.lbl.1 -side top -anchor w
pack   .configure.frame1.eat.lbl.2 -side top -anchor w

frame  .configure.frame1.eat.chk
radiobutton .configure.frame1.eat.chk.1 -variable noeat -value " "
radiobutton .configure.frame1.eat.chk.2 -variable noeat -value "m"
pack   .configure.frame1.eat.chk.1 -side top
pack   .configure.frame1.eat.chk.2 -side top

frame  .configure.frame1.atk -borderwidth 10
frame  .configure.frame1.atk.lbl
label  .configure.frame1.atk.lbl.1 -text $ATKA($LANG)
label  .configure.frame1.atk.lbl.2 -text $NOAK($LANG)
pack   .configure.frame1.atk.lbl.1 -side top -anchor w
pack   .configure.frame1.atk.lbl.2 -side top -anchor w

frame  .configure.frame1.atk.chk
radiobutton .configure.frame1.atk.chk.1 -variable noatk -value " "
radiobutton .configure.frame1.atk.chk.2 -variable noatk -value "k"
pack   .configure.frame1.atk.chk.1 -side top
pack   .configure.frame1.atk.chk.2 -side top

frame  .configure.frame1.all -borderwidth 10
frame  .configure.frame1.all.lbl
label  .configure.frame1.all.lbl.1 -text $EATA($LANG)
label  .configure.frame1.all.lbl.2 -text $NOAL($LANG)
pack   .configure.frame1.all.lbl.1 -side top -anchor w
pack   .configure.frame1.all.lbl.2 -side top -anchor w

frame  .configure.frame1.all.chk
radiobutton .configure.frame1.all.chk.1 -variable noall -value " "
radiobutton .configure.frame1.all.chk.2 -variable noall -value "a"
pack   .configure.frame1.all.chk.1 -side top
pack   .configure.frame1.all.chk.2 -side top

frame  .configure.button
button .configure.button.ok      -text $OK($LANG)   -command {destroy .configure}
button .configure.button.default -text $DEFA($LANG) -command {
                                                           global noeat nbeat noall noatk
                                                           set noeat " "
                                                           set nbeat 4
                                                           set noall " "
                                                           set noatk " "
                                                          }
#button .configure.button.cancel -text "CANCEL" -command {destroy .configure}
pack .configure.button.ok      -side left  -ipadx 18 -padx 5 -pady 5
pack .configure.button.default -side right -padx 5 -pady 5
#pack .configure.button.cancel -side right -padx 5 -pady 5


pack .configure.frame1         -side top
pack .configure.frame1.nbeat   -side top   -anchor w
pack .configure.frame1.eat     -side top   -anchor w
pack .configure.frame1.eat.chk -side left  -anchor w
pack .configure.frame1.eat.lbl -side left  -anchor w
pack .configure.frame1.atk     -side top   -anchor w
pack .configure.frame1.atk.chk -side left  -anchor w
pack .configure.frame1.atk.lbl -side left  -anchor w
pack .configure.frame1.all     -side top   -anchor w
pack .configure.frame1.all.chk -side left  -anchor w
pack .configure.frame1.all.lbl -side left  -anchor w
pack .configure.button -side bottom -anchor s

}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                          Display rule
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc helpDialog {} {
global BIN LANG
destroy .rule
toplevel .rule -borderwidth 1 -relief sunken
set flag_uk [image create photo -file [file join $BIN uk.gif]]
set flag_fr [image create photo -file [file join $BIN fr.gif]]
set flag_sp [image create photo -file [file join $BIN sp.gif]]
set flag_de [image create photo -file [file join $BIN de.gif]]
frame .rule.flag
button .rule.flag.uk -image $flag_uk -command rule_uk
button .rule.flag.fr -image $flag_fr -command rule_fr
button .rule.flag.sp -image $flag_sp -command rule_sp
button .rule.flag.de -image $flag_de -command rule_de
text .rule.txt -relief sunken -bd 2 -yscrollcommand ".rule.scroll set" -setgrid 1 \
 -height 37 -width 100 -spacing2 5;
scrollbar .rule.scroll -command ".rule.txt yview"
pack .rule.flag.uk -side left -padx 5 -pady 5
pack .rule.flag.fr -side left -padx 5 -pady 5
pack .rule.flag.sp -side left -padx 5 -pady 5
pack .rule.flag.de -side left -padx 5 -pady 5
pack .rule.flag -side top -anchor w
pack .rule.scroll -side right -fill y
pack .rule.txt -expand yes -fill both
rule_$LANG
}


proc rule_sp {} {
.rule.txt delete 0.0 end
.rule.txt insert 0.0 \
{Reglas del awale.

El juego del awale se compone de dos lneas de seis agujeros. La lnea de arriba se atribuye \
al ordenador mientras que la lnea de abajo se atribuye al jugador. Los otros agujeros sirven \
para almacenar la semillas comidas durante la partida.

Cada agujero contiene cuatro semillas al principio de la partida.

Los jugadores juegan cada uno su turno. No se puede jugar dos veces seguidas. No se puede \
saltar su turno.

El jugador que juega coge todas las semillas de uno de sus agujeros y las distribuye una a \
una en cada uno de los agujeros siguientes en el sentido contrario a un reloj. No se ponen \
semillas en el agujero inicial (ese caso se encuentra cuando el jugador distribuye ms de \
doce semillas).

Se comen las semillas del ltimo agujero al que se llega cuando este est situado en el \
campo adverso y contiene dos o tres semillas contando la ltima semilla puesta. Se comen \
tambin los agujeros consecutivos precedentes que responden a esta condicin. No se pueden \
comer ms de cuatro agujeros consecutivos. Solo se comen los cuatro ltimos agujeros cuando \
cinco agujeros o ms se pueden comer.

El jugador tiene entonces que dar semillas al adversario cuando no le quedan ms a este. \
Si no se pueden dar ms semillas, la partida se para.

El jugador que ha comido ms semillas ha ganado.

VARIANTES :
Existen muchas variantes en Africa. La mayora se pueden jugar configurando la regla con los\
cuatro parmetros siguientes.
- el nmero mximo E de agujeros que se pueden comer puede ser modificado (E=1 a 6).
- no se come nada cuando la cantidad de agujeros que se pueden comer es superior al nmero mximo E.
- se puede prohibir atacar todas las semillas adversas menos si no hay otra alternativa.
- cuando se atacan todas las semillas adversas, no se come nada.
}
}

proc rule_fr {} {
.rule.txt delete 0.0 end
.rule.txt insert 0.0 \
{Rgles de l'awal.

Le jeu d'awal se dcompose en deux lignes de six trous. La ligne du haut est affecte \
l'ordinateur tandis que la ligne du bas est affecte au joueur. Les autres trous servent  stocker \
les graines "manges" au cours de la partie.

Chaque trou contient quatre graines en dbut de partie.

Les joueurs jouent chacun leur tour. On ne peut pas jouer deux fois de suite. On ne peut pas sauter son tour.

Le joueur qui joue prend toutes les graines de l'un de ses trous et les distribue une  une \
dans les trous suivants en tournant dans le sens anti-horaire. On ne pose pas de graine \
dans le trou de dpart (ce cas de figure se prsente lorsque le joueur distribue plus de douze \
graines).

On "mange" les graines du dernier trou atteint lorsque celui se situe chez l'adversaire et contient deux ou trois graines \
en comptant la dernire graine pose. On "mange" aussi les trous conscutifs prcdents qui respectent \
cette condition. On ne peut pas "manger" plus de quatre trous conscutifs. Seuls les quatre derniers trous sont \
"mangs" lorsque cinq trous ou plus sont "mangeables".

Le joueur est tenu de fournir des graines  l'adversaire lorsque celui-ci n'en possde plus. \
S'il ne peut pas fournir de graines, la partie s'arrte.

Le joueur qui a "mang" le plus de graines a gagn.

VARIANTES :
Il existe de nombreuses variantes en Afrique. La plupart d'entre elles peuvent tre joues en\
configurant la rgle avec les quatre paramtres suivants.
- le nombre maximum E de trous pouvant tre mangs peut tre modifi (E=1  6).
- on ne mange rien lorsque plus de E trous sont "mangeables".
- vous pouvez interdire d'attaquer toutes les graines adverses, sauf si c'est le seul coup jouable.
- lorsque toutes les graines adverses sont attaques, rien n'est mang.
}
}

proc rule_de {} {
.rule.txt delete 0.0 end
.rule.txt insert 0.0 \
{Awale-Regeln

Das Awale-Spielbrett besteht aus zwei Reihen mit je sechs Mulden. Jedem Spieler gehrt eine Reihe, die obere dem Computer, die untere dem menschlichen Spieler. Noch je zwei Mulden sind fr die gefangenen Samen.

Jede Mulde enthlt am Anfang des Spiels vier Samen.

Die Spieler spielen je einen Zug abwechselnd, ohne zwei Zge hintereinander zu machen oder auszusetzen.

Ein Zug besteht darin, alle Samen aus einer eigenen Mulde zu nehmen und sie im Gegenuhrzeigersinn einzeln in die darauffolgenden Mulden zu legen, wobei kein Samen in die Startmulde abgelegt werden darf (dies tritt auf, wenn in der Startmulde mindestens zulf Samen sind).

Die Samen der Endmulde werden gefangen, wenn sie auf der Seite des Gegners liegt und nach Verteilung aller Samen zwei oder drei Samen beinhaltet. Die Samen aller vorausgehenden Mulden, die ebenfalls diese Bedingungen erfllen, werden ebenfalls gefangen. Man kann jeweils hchstens die Samen der letzten vier Mulden fangen. Wenn es mehr sind, werden nur die letzten vier gefangen.

Man darf nur so spielen, da dem Gegner mindestens ein Stein verbleibt. Ist dies unmglich, hrt das Spiel auf.

Der Sieger ist derjenige mit den meisten gefangenen Samen.


VARIANTEN

In Afrika gibt es viele Varianten dieses Spiels. Die meisten von ihnen knnen dadurch gespielt werden, da man folgende Parameter einstellt:
- Die maximale Anzahl E der Mulden mit gefangenen Samen (1-6)
- Gar keinen Samen fangen, anstatt die Samen der letzten E Mulden zu fangen, wenn mehr als eine angegriffen wird
- Man kann verbieten, da alle gegnerischen Samen angegriffen werden, es sei denn, dies ist der einzig mgliche Zug
- Wenn alle gegnerischen Samen angegriffen werden, wird kein Samen gefangen.
}
}

proc rule_uk {} {
.rule.txt delete 0.0 end
.rule.txt insert 0.0 \
{Rules of awale.

Awale game contains two lines of six holes. Each player plays a single line, the \
upper line for the computer and the lower line for the player. Other pair of holes \
are used for eaten seeds.

Every holes contain four seeds at the beginning of the game.

One plays alternately. One never play two turns, one cannot pass his turn.

When playing, take all the seeds of a hole in your line and distribute it \
in successive holes turning counterclock wise, one seed per hole. Do not \
put a seed in the starting hole (this case arrives when the starting hole \
contains more than twelve seeds.)

The seeds of the arrival hole are eaten when the arrival hole lies in the \
opponent line and contains two or three seeds (after all seeds are distributed). \
All contiguous previous holes which meet this condition are also \
eaten. It is possible to eat a maximum of four holes, when five holes are \
eatable, only the last four ones are eaten.

Players have to play such that the opponent always has at least one seed. \
In case it is impossible, the game stops.

The winner is the player who has eaten the most seeds.

VARIANTS:
It exists many variants in Africa. Most of them can be played\
by configuring the rule with the four following parameters.
- the maximum number E of holes that can be eaten can be modified (E=1 to 6).
- instead of eating the last E holes when attacking more, you eat nothing.
- you can forbid any move attacking all opponent seeds excepted if this is the only possible move.
- in case all opponent seeds are attacked, nothing is eaten.
}
}

#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                         Change of language
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

proc changeLang {} {
global PLAY NEW OPEN SAVE UNDO QUIT PARA LEVE TWOP OPPO STAR BOTT TOP CONF LING UK FR SP HELP RULE ABOU LANG
global label

.menu entryconfigure  0 -label $PLAY($LANG)
.menu.play entryconfigure  0 -label $NEW($LANG)
.menu.play entryconfigure  1 -label $OPEN($LANG)
.menu.play entryconfigure  2 -label $SAVE($LANG)
.menu.play entryconfigure  3 -label $UNDO($LANG)
.menu.play entryconfigure  4 -label $QUIT($LANG)
.menu entryconfigure  1 -label $PARA($LANG)
.menu.para entryconfigure  0 -label $LEVE($LANG)
.menu.para.level entryconfigure  0 -label $TWOP($LANG)
.menu.para.level entryconfigure  1 -label "$OPPO($LANG) 0"
.menu.para.level entryconfigure  2 -label "$OPPO($LANG) 1"
.menu.para.level entryconfigure  3 -label "$OPPO($LANG) 2"
.menu.para.level entryconfigure  4 -label "$OPPO($LANG) 3"
.menu.para entryconfigure  1 -label $STAR($LANG)
.menu.para.start entryconfigure  0 -label $BOTT($LANG)
.menu.para.start entryconfigure  1 -label $TOP($LANG)
.menu.para entryconfigure  3 -label $CONF($LANG)
.menu.para entryconfigure  2 -label $LING($LANG)
.menu.para.lang entryconfigure 0 -label $UK($LANG)
.menu.para.lang entryconfigure 1 -label $FR($LANG)
.menu.para.lang entryconfigure 2 -label $SP($LANG)
.menu entryconfigure  2 -label $HELP($LANG)
.menu.help entryconfigure  0 -label $RULE($LANG)
.menu.help entryconfigure  1 -label $ABOU($LANG)

if {$label=="BEG"|$label=="RUN"} {displaySeeds NULL}
}


#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
#                           Begin of main
#ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

switch $tcl_platform(platform) {
"unix"      {
    set JOUER [file join [file dirname [info script]] .. .. bin awale] 
            }
"macintosh" {
    set JOUER [file join [file dirname [info script]] .. .. bin awale]
            }
"windows"   {
    set JOUER [file join [file dirname [info script]] .. .. bin awale.exe]
            }
}

if {$argc==1&([lindex $argv 0]=="uk"|[lindex $argv 0]=="sp"|[lindex $argv 0]=="fr")} {set LANG [lindex $argv 0]} else {set LANG "uk"}

set BIN [file dirname [info script]]
set HOME $env(HOME)
set PATH $HOME
set level 0
array set depth0 [list H 0 0 1 1 7 2 9 3 10]
set start PLAYER
set nbeat 4
set noeat " "
set noall " "
set noatk " "
set label END
for {set i 0} {$i<12} {incr i} {set idhole($i) 0}
set mvtcircle 0
set iturn 0
set indication 0
array set Tmin [list 0 0   1 1 2 3  3 15]
array set Tmax [list 0 999 1 2 2 20 3 100]

array set SEED [list uk "seed(s)"        fr "graine(s)"          sp "semilla(s)"             de "Samen"]
array set PLAY [list uk "Play"           fr "Jeu"                sp "Jugar"                  de "Spielen"]
array set NEW  [list uk "New game"       fr "Nouvelle partie"    sp "Nueva partida"          de "Neues Spiel"]
array set OPEN [list uk "Open"           fr "Ouvrir"             sp "Abrir"                  de "ffnen"]
array set SAVE [list uk "Save"           fr "Sauver"             sp "Guardar"                de "Speichern"]
array set UNDO [list uk "Undo"           fr "Annuler"            sp "Anular"                 de "Rckgngig"]
array set QUIT [list uk "Quit"           fr "Quitter"            sp "Salir"                  de "Beenden"]
array set PARA [list uk "Parameters"     fr "Paramtres"         sp "Parmetros"	     de "Parameter"]
array set LEVE [list uk "Level"          fr "Niveau"             sp "Nivel"		     de "Spielniveau"]
array set TWOP [list uk "Two players"    fr "Deux joueurs"       sp "Dos jugadores"	     de "Zwei Spieler"]
array set OPPO [list uk "Computer level" fr "Ordinateur niveau"  sp "Ordenador nivel"	     de "Computer-Niveau"]
array set STAR [list uk "First turn"     fr "Premier coup"       sp "Primer turno"	     de "Erster Zug"]
array set LING [list uk "Language"       fr "Language"           sp "Lengua"		     de "Sprache"]
array set BOTT [list uk "Bottom"         fr "Bas"                sp "Abajo"		     de "Unten"]
array set TOP  [list uk "Top"            fr "Haut"               sp "Arriba"		     de "Oben"]
array set CONF [list uk "Configure rule" fr "Configurer la rgle" sp "Configurar las reglas" de "Regeln konfigurieren"]
array set DETA [list uk "Detailed rule"  fr "Rgle dtaille"    sp "Reglas detalladas"	     de "Detaillierte Regeln"]
array set HELP [list uk "Help"           fr "Aide"               sp "Ayuda" 		     de "Hilfe"]
array set RULE [list uk "Rules"          fr "Rgles"             sp "Reglas"		     de "Regeln"]
array set ABOU [list uk "About"          fr "A propos"           sp "Sobre"		     de "ber"]
array set UK   [list uk "English"        fr "English"            sp "English"	             de "English"]
array set FR   [list uk "Franais"       fr "Franais"           sp "Franais"		     de "Franais"]
array set SP   [list uk "Castellano"     fr "Castellano"         sp "Castellano"	     de "Castellano"]
array set DE   [list uk "Deutsch"        fr "Deutsch"            sp "Deutsch"   	     de "Deutsch"]
array set WIN  [list uk "You lose"       fr "Je t'ai bien gagn" sp "He ganado"		     de "Sie haben verloren"]
array set LOSE [list uk "You win"        fr "Tu m'as bien gagn" sp "Me has ganado"	     de "Sie haben gewonnen"]
array set NWIN [list uk "North wins"     fr "Nord a bien gagn"  sp "Norte ha ganado"	     de "Nord hat gewonnen"]
array set SWIN [list uk "South wins"     fr "Sud a bien gagn"   sp "Sur ha ganado"	     de "Sd hat gewonnen"]
array set DRAW [list uk "It's a draw"    fr "C'est mme score"   sp "Igualdad"		     de "Unentschieden"]
array set CSCO [list uk "Computer score" fr "Score ordinateur"   sp "Score del ordenador"    de "Punkte des Computers"]
array set PSCO [list uk "Player score"   fr "Score joueur"       sp "Score del jugador"	     de "Punkte des Spielers"]
array set P1CO [list uk "North score"    fr "Score nord"         sp "Score del norte"	     de "Punkte von Nord"]
array set P2CO [list uk "South score"    fr "Score sud"          sp "Score del sur"	     de "Punkte von Sd"]
array set DEPT [list uk "Depth"          fr "Profondeur"         sp "Profundidad"	     de "Tiefe"]
array set TURN [list uk "Turn"           fr "Tour"               sp "Turno"		     de "Runde"]
array set YES  [list uk "Yes"            fr "Oui"                sp "Si"		     de "Ja"]
array set NO   [list uk "No"             fr "Non"                sp "No"		     de "Nein"]
array set OK   [list uk "OK"             fr "OK"                 sp "VALE"		     de "OK"]
array set DEFA [list uk "DEFAULT" fr "DEFAUT" sp "DEFECTO" de "VOREINSTELLUNG"]
array set MAXI [list uk "Maximum number E of holes that can be eaten"\
                     fr "Nombre maximum E de trous pouvant tre mangs"\
                     sp "Cantidad mxima E de agujeros que comer"\
		     de "Maximale Anzahl E der Mulden mit Samen, die gefangen werden knnen"]
array set EATL [list uk "Eat the last E holes when attacking more than E holes"\
                     fr "Mange les E derniers trous lorsqu'on attaque plus de E trous"\
                     sp "Se comen los E ltimos agujeros cuando se pasa del mximo E"\
		     de "Wenn mehr als E Mulden angegriffen werden, nur die Samen der letzten E Mulden gefangen nehmen"]
array set EATN [list uk "Eat nothing when attacking more than E holes"\
                     fr "Ne mange rien lorsqu'on attaque plus de E trous"\
                     sp "No se come nada cuando se pasa del mximo E"\
		     de "Gar keinen Samen fangen, wenn mehr als E angegriffen werden"]
array set EATA [list uk "Can eat all opponent seeds"\
                     fr "Peut manger toutes les graines adverses"\
                     sp "Se puede comer todos los agujeros del adversario"\
		     de "Man kann smtliche gegnerischen Samen fangen"]
array set NOAL [list uk "Eat nothing when trying to eat all opponent seeds"\
                     fr "Ne mange rien lorsqu'on attaque toutes les graines adverses"\
                     sp "No se come nada cuando se intenta comer todas las semillas adversas"\
		     de "Gar keinen Samen fangen, wenn alle gegnerischen Samen angegriffen werden"]
array set ATKA [list uk "Allow to play a move which attacks all opponent seeds"\
                     fr "Peut jouer un coup qui attaque toutes les graines adverses"\
                     sp "Se puede atacar todos los agujeros del adversario"\
		     de "Man kann smtliche gegnerischen Samen angreifen"]
array set NOAK [list uk "Forbidden to attack all opponent seeds excepted if this is the only possible move"\
                     fr "Interdiction d'attaquer toutes les graines adverses sauf si c'est le seul coup jouable"\
                     sp "No se puede atacar todas las semillas adversas excepto si es la sola jugada posible"\
		     de "Man kann nicht smtliche gegnerischen Samen angreifen, es sei denn, es ist der einzige mgliche Zug"]
array set IFIL [list uk "Select file to import" \
                     fr "Choisir le fichier  importer" \
                     sp "Elegir el fichero para importar"\
		     de "Datei zum Importieren whlen"]
array set SFIL [list uk "Select file to save" \
                     fr "Choisir le fichier  importer" \
                     sp "Qu fichero quieres ?"\
		     de "Datei zum Speichern whlen"]
array set EFIL [list uk "ERROR when opening file" \
                     fr "ERREUR lors de l'ouverture du fichier" \
                     sp "ERROR cuando se abre el fichero"\
		     de "FEHLER beim ffnen der Datei"]

openConfigure

wm title . "Free Awale"

# creation of menu

menu .menu -tearoff 0

set m .menu.play
menu $m -tearoff 0
.menu add cascade -label $PLAY($LANG) -menu $m -underline 0
$m add command -label $NEW($LANG)  -accelerator Crtl+N -command {initGame}
$m add command -label $OPEN($LANG) -accelerator Crtl+O -command {importGame}
$m add command -label $SAVE($LANG) -accelerator Crtl+S -command {saveGame}
$m add command -label $UNDO($LANG) -accelerator Crtl+Z -command {undo}
$m add command -label $QUIT($LANG) -accelerator Crtl+Q -command {saveConfigure; destroy .}
bind . <Control-n> {initGame}
bind . <Control-o> {importGame}
bind . <Control-s> {saveGame}
bind . <Control-z> {undo}
bind . <Control-q> {destroy .}


set m .menu.para
menu $m -tearoff 0
.menu add cascade -label $PARA($LANG) -menu $m -underline 0
$m add cascade -label $LEVE($LANG) -menu $m.level
menu $m.level -tearoff 0
$m.level add radio -label "$TWOP($LANG)"   -variable level -value "H" -command {changeLang}
$m.level add radio -label "$OPPO($LANG) 0" -variable level -value 0 \
                                           -command "set depth $depth0(0); changeLang"
$m.level add radio -label "$OPPO($LANG) 1" -variable level -value 1 -command {changeLang}
$m.level add radio -label "$OPPO($LANG) 2" -variable level -value 2 -command {changeLang}
$m.level add radio -label "$OPPO($LANG) 3" -variable level -value 3 -command {changeLang}
$m add cascade -label $STAR($LANG) -menu $m.start
menu $m.start -tearoff 0
$m.start add radio -label $BOTT($LANG) -variable start -value "PLAYER"
$m.start add radio -label $TOP($LANG)  -variable start -value "COMPUTER"
$m add cascade -label $LING($LANG) -menu $m.lang
menu $m.lang -tearoff 0
$m.lang add radio -label $UK($LANG) -variable LANG -value "uk" -command {changeLang}
$m.lang add radio -label $FR($LANG) -variable LANG -value "fr" -command {changeLang}
$m.lang add radio -label $SP($LANG) -variable LANG -value "sp" -command {changeLang}
$m.lang add radio -label $DE($LANG) -variable LANG -value "de" -command {changeLang}
$m add command -label $CONF($LANG) -command {configureRules}

set m .menu.help
menu $m -tearoff 0
.menu add cascade -label $HELP($LANG) -menu $m -underline 0
$m add command -label $RULE($LANG) -command {helpDialog}
$m add command -label $ABOU($LANG) -command {copyrightDialog FALSE}

. configure -menu .menu


# creation of the main image

set fond [image create photo -file [file join $BIN board.gif]]
set largeur [image width $fond]
set hauteur [image height $fond]
for {set i  0} {$i<10} {incr i} {set img($i) [image create photo -file [file join $BIN Trou0$i.gif]]}
for {set i 10} {$i<16} {incr i} {set img($i) [image create photo -file [file join $BIN Trou$i.gif]]}
for {set i 16} {$i<49} {incr i} {set img($i) [image create photo -file [file join $BIN Trou15.gif]]}

canvas .cnv -width $largeur -height $hauteur -cursor top_left_arrow
bind .cnv <ButtonPress-1> "play %x %y $largeur $hauteur"
bind .cnv <ButtonPress-2> "displayContent %x %y $largeur $hauteur"
bind .cnv <ButtonPress-3> "displayContent %x %y $largeur $hauteur"
pack .cnv -side top -anchor nw

wm geometry . +400+100

.cnv create image [expr $largeur/2] [expr $hauteur/2] -image $fond

# welcome window

copyrightDialog TRUE
after 2000 {destroy .copyright; focus .}
