//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cGameCore.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cGameCore.hpp"

#include <stdexcept>
#include <fstream>
#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include "cTextureManager.hpp"
#include "cAnimationManager.hpp"
#include "cMixer.hpp"
#include "cGameMode.hpp"
#include "cMainMenu.hpp"
#include "cDisplayManager.hpp"
#include "cOptions.hpp"
#include "Debug.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================

#ifndef DEFAULT_DATA_DIR
	#define DEFAULT_DATA_DIR "./data/"
#endif

#ifndef OPTIONS_FILE
	#define OPTIONS_FILE "shootingstar.cfg"
#endif


//==============================================================================
//! Constructor
//------------------------------------------------------------------------------
cGameCore::cGameCore (void):
mRunning (false),
mpMainMenu (NULL),
mpGameMode (NULL),
mDisplayManager (cDisplayManager::GetInstance ()),
mOptions (cOptions::GetInstance ()),
mDataDir (DEFAULT_DATA_DIR),
mOptionsFile (OPTIONS_FILE)
{
	// Empty
};
//==============================================================================

//==============================================================================
//! Destructor
//------------------------------------------------------------------------------
cGameCore::~cGameCore (void)
{
	// Quit SDL
	SDL_Quit ();

	delete mpMainMenu;
	delete mpGameMode;
};
//==============================================================================

//==============================================================================
//! Initialize the game
//------------------------------------------------------------------------------
void 
cGameCore::Initialize (int argc, char **argv)
{
	Uint32 videoFlags = SDL_OPENGL|SDL_GL_DOUBLEBUFFER;
	int videoW = 800, videoH = 600;

	// Windows version should be compiled with NO_HOME_CONFIG
	#ifndef NO_HOME_CONFIG
	char *pTemp = getenv ("HOME");
	if ( pTemp != NULL )
	{
		string optionsFile = pTemp;
		optionsFile.append ("/." + string (OPTIONS_FILE));
		
		string systemWide (string (DEFAULT_DATA_DIR) + string (OPTIONS_FILE));

		ifstream ftest (optionsFile.c_str ());
		if ( !ftest )
			CopyFile (systemWide, optionsFile);
		else
			ftest.close ();
		
		mOptionsFile = optionsFile;
		mOptions.Load (optionsFile);

	}
	else
	{
		dbgError () << "Unable to get home dir\n";
		throw runtime_error ("Unable to load config");
	}
	#else
		mOptions.Load (OPTIONS_FILE);
	#endif

	if ( mOptions.mLowRes )
	{
		videoW = 640;
		videoH = 480;
	}
	if ( mOptions.mFullscreen )
		videoFlags |= SDL_FULLSCREEN;
	
	for ( int i = 1; i < argc; i++ )
	{
		if ( strcmp (argv[i], "--fullscreen") == 0 )
		{
			videoFlags |= SDL_FULLSCREEN;
			continue;
		}
		if ( strcmp (argv[i], "--800x600") == 0 )
		{
			videoW = 800;
			videoH = 600;
			continue;
		}	
		if ( strcmp (argv[i], "--640x480") == 0 )
		{
			videoW = 640;
			videoH = 480;
			continue;
		}
		dbgError () << "Unknown command line argument \'" << argv[i] << "\'\n";
	}	
	
	dbgInfo () << "Initializing game" <<endl;
	
	// Random seed
	time_t seed = time (NULL);
	dbgInfo () << "Random seed: " << seed << '\n';
	srand (seed);
	
	// Initialize SDL
	if ( SDL_Init (SDL_INIT_VIDEO) != 0 )
	{
		dbgError () << "Unable to initialize SDL: " << SDL_GetError () << endl;
		throw runtime_error ("Initialization failed");
	}
	
	// Set caption
	SDL_WM_SetCaption ("Shooting Star", NULL);
	
	// Set video mode
	SDL_GL_SetAttribute (SDL_GL_DOUBLEBUFFER, 1);
	SDL_GL_SetAttribute (SDL_GL_DEPTH_SIZE,   0);
	SDL_GL_SetAttribute (SDL_GL_STENCIL_SIZE, 0);
	SDL_GL_SetAttribute (SDL_GL_ACCUM_RED_SIZE, 0);
	SDL_GL_SetAttribute (SDL_GL_ACCUM_GREEN_SIZE, 0);
	SDL_GL_SetAttribute (SDL_GL_ACCUM_BLUE_SIZE, 0);
	SDL_GL_SetAttribute (SDL_GL_ACCUM_ALPHA_SIZE, 0);
	SDL_ShowCursor (false);
	mDisplayManager.SetVideoMode (videoW, videoH, 32, videoFlags);
	
	// Initialize viewport
	mDisplayManager.SetViewport (0, 0, mDisplayManager.GetScreenWidth(), mDisplayManager.GetScreenHeight());

	// Initialize GL_PROJECTION matrix
	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
	mDisplayManager.SetOrtho (0, 800, 600, 0 );
	
	// Initialize modelview matrix
	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();
	
	// Setup shading
	glShadeModel (GL_SMOOTH);
	
	// Setup background color
	glClearColor (0.0f, 0.0f, 0.0f, 0.0f);
	
	// Setup blending
	glEnable (GL_BLEND);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	dbgInfo () << "Initializing done" <<endl;
	
	cTextureManager::GetInstance ().SetDataDir (mDataDir + string ("textures/"));
	cAnimationManager::GetInstance ().SetDataDir (mDataDir + string ("animations/"));
	cMixer::GetInstance ().SetDataDir (mDataDir + string ("sounds/"));

	GLenum error = glGetError ();
	while ( error != GL_NO_ERROR )
	{
		dbgError () << "cGameCore: OpenGL error: " << gluErrorString (error) << '\n';
		error = glGetError ();
	}
	
	mpMainMenu = new cMainMenu (*this);
	mpGameMode = new cGameMode (*this);
}
//==============================================================================

//==============================================================================
//! Run the game
//------------------------------------------------------------------------------
void 
cGameCore::Run (void)
{
	// Play music
	cMixer::GetInstance ().PlayMusic ("menu.ogg");

	mRunning = true;
	while ( mRunning )
	{
		switch ( mpMainMenu->Run () )
		{
			case cMainMenu::QuitGame:
				mRunning = false;
				break;
			case cMainMenu::SplitScreen:
				cMixer::GetInstance ().FadeMusic ();
				cMixer::GetInstance ().PlayMusic ("dead.ogg");
				mpGameMode->Run (cGameMode::GameType_Split);
				break;
			case cMainMenu::SinglePlayer:
				cMixer::GetInstance ().FadeMusic ();
				cMixer::GetInstance ().PlayMusic ("dead.ogg");
				mpGameMode->Run (cGameMode::GameType_Single);
				break;
		}
		GLenum error = glGetError ();
		while ( error != GL_NO_ERROR )
		{
			dbgError () << "cGameCore::Run: OpenGL error: " << gluErrorString (error) << '\n';
			error = glGetError ();
		}
	}
	
	cMixer::GetInstance ().FadeMusic ();
}

void 
cGameCore::CopyFile (string src, string dst)
{
	dbgInfo () << "Copying " << src << " to " << dst << '\n';
	
	ifstream fin (src.c_str ());
	if ( !fin )
	{
		dbgError () << "Unable to open file " << src << " for reading\n";
		throw ("Unable to copy files");
	}
	
	ofstream fout (dst.c_str ());
	if ( !fout )
	{
		dbgError () << "Unable to open file " << dst << " for writing\n";
		throw ("Unable to copy files");
	}
	
	fout << fin.rdbuf ();
	fin.close ();
	fout.close ();
}

//==============================================================================
// EOF
//==============================================================================
