//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cGameView.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarsk@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cGameView.hpp"

#include <GL/gl.h>
#include <GL/glu.h>
#include "cGameMode.hpp"
#include "cWorld.hpp"
#include "cPlayer.hpp"
#include "cDisplayManager.hpp"
#include "Debug.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================


//! Constructor
cGameView::cGameView (cGameMode &game):
mGameMode (game),
mSliding (0.0f)
{
	mScreenW = cDisplayManager::GetInstance ().GetScreenWidth ();
	mScreenW2 = mScreenW / 2;
	mScreenH = cDisplayManager::GetInstance ().GetScreenHeight ();
	mScreenH2 = mScreenH / 2;
};

//! Destructor
cGameView::~cGameView (void)
{
	// Empty
};

void 
cGameView::Update (Uint32 deltaTime)
{
	if ( mGameMode.mGameType == cGameMode::GameType_Single )
		return;

	if ( !mGameMode.mPlayer2->IsAlive () && !mGameMode.mPlayer1->IsAlive () && mSliding != 0.0f )
	{
		if ( mSliding > 0.0f )
		{
			mSliding -= 0.0006f * deltaTime;
			if ( mSliding < 0.0f )
				mSliding = 0.0f;
		}
		else if ( mSliding < 0.0f )
		{
			mSliding += 0.0006f * deltaTime;
			if ( mSliding > 0.0f )
				mSliding = 0.0f;
		}

	}
	else if ( !mGameMode.mPlayer2->IsAlive () && mGameMode.mPlayer1->IsAlive () && mSliding > -1.0f )
	{
		mSliding -= 0.0003f * deltaTime;
		if ( mSliding < -1.0f )
			mSliding = -1.0f;
	}
	else if ( mGameMode.mPlayer2->IsAlive () && !mGameMode.mPlayer1->IsAlive () && mSliding < 1.0f )
	{
		mSliding += 0.0003f * deltaTime;
		if ( mSliding > 1.0f )
			mSliding = 1.0f;
	}
}

//! Render the game
void 
cGameView::Render (Uint32 deltaTime)
{
	switch ( mGameMode.mGameType )
	{
		case cGameMode::GameType_Single:
			RenderSingle (deltaTime);
			break;
		case cGameMode::GameType_Split:
			RenderSplit (deltaTime);
			break;
	}
}

void 
cGameView::RenderSingle (Uint32 deltaTime)
{
	RenderGame (0, 0, mScreenW, mScreenH, mGameMode.mPlayer1, deltaTime);
}

void 
cGameView::RenderSplit (Uint32 deltaTime)
{
	if ( mSliding > -1.0f )
		RenderGame (0, 0, mScreenW2 + int (mSliding * mScreenW2), mScreenH, mGameMode.mPlayer2, deltaTime);
	if ( mSliding < 1.0f )
		RenderGame (mScreenW2 + int (mSliding * mScreenW2), 0, mScreenW2 
			- int (mSliding * mScreenW2), mScreenH, mGameMode.mPlayer1, deltaTime);
}

void 
cGameView::RenderGame (int x, int y, int w, int h, cPlayer *pPlayer, Uint32 deltaTime)
{
	glPushMatrix ();
	glScissor (x, y, w, h);
	glEnable (GL_SCISSOR_TEST);

	float xx = float (x) / float (mScreenW) * 800.0f;
	float yy = float (y) / float (mScreenH) * 600.0f;
	float width = float (w) / float (mScreenW) * 800.0f;
	float height = float (h) / float (mScreenH) * 600.0f;

	cVector2f camera = pPlayer->GetPosition ();
	if ( camera.mX - width * 0.5f < 0.0f )
		camera.mX = width * 0.5f;
	else if ( camera.mX + width * 0.5f > mGameMode.mMapWidth )
		camera.mX = mGameMode.mMapWidth - width * 0.5f;
	if ( camera.mY - height * 0.5f < 0.0f )
		camera.mY = height * 0.5f;
	else if ( camera.mY + height * 0.5f > mGameMode.mMapHeight )
		camera.mY = mGameMode.mMapHeight - height * 0.5f;
		
	glTranslatef (-camera.mX + width * 0.5f + xx, -camera.mY + height * 0.5f + yy, 0.0f);
	
	mGameMode.mWorld.Render (deltaTime);
	
	glPopMatrix ();
	
	if ( !pPlayer->IsAlive () )
	{
		glDisable (GL_TEXTURE_2D);
		glColor4f (1.0f, 0.0f, 0.0f, 0.5f);
		glBegin (GL_QUADS);
			glVertex2i (0, 0);
			glVertex2i (0, 600);
			glVertex2i (800, 600);
			glVertex2i (800, 0);
		glEnd ();
		glEnable (GL_TEXTURE_2D);
	}

	glDisable (GL_SCISSOR_TEST);
}


//==============================================================================
// EOF
//==============================================================================
