//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cPlayer.cpp
// Project: Shooting Star
// Author: 
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cPlayer.hpp"
#include <GL/gl.h>
#include "Actions.hpp"
#include "Debug.hpp"
#include "cPistol.hpp"
#include "cWorld.hpp"
#include "cBurningEffect.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================


//! Constructor
cPlayer::cPlayer (string animationPrefix):
cSoldier (animationPrefix)
{
	// Starting weapon
	cWeapon *pWeapon = new cPistol ();
	cWorld::GetInstance ().SpawnObject (pWeapon);
	AddWeapon (pWeapon);
};

//! Destructor
cPlayer::~cPlayer (void)
{
	// Empty
};

//! Rendering interface
void 
cPlayer::Render (Uint32 deltaTime)
{
	// Call base Render
	cSoldier::Render (deltaTime);

	// Render health bar
	glPushMatrix ();
	glPushAttrib (GL_ENABLE_BIT|GL_CURRENT_BIT);
	glDisable (GL_TEXTURE_2D);
	glRotatef (-GetRotation () - 90.0f, 0.0f, 0.0f, 1.0f);
	glTranslatef (-32.0f, 40.0f, 0.0f);
	float health = float (GetHealth ()) / float (GetMaxHealth ());
	glBegin (GL_QUADS);
		if ( GetMaxArmor () != 0 )
		{
			float armor = float (GetArmor ()) / float (GetMaxArmor ());
			glColor4f (0.0f, 1.0f, 0.0f, 0.5f);
			glVertex2f (0.0f,-5.0f);
			glVertex2f (0.0f,-1.0f);
			glColor4f (0.0f, 1.0f - armor, armor, 0.5f);
			glVertex2f (armor * 64.0f,-1.0f);
			glVertex2f (armor * 64.0f,-5.0f);
		}
	
		glColor4f (1.0f, 0.0f, 0.0f, 0.5f);
		glVertex2f (0.0f, 0.0f);
		glVertex2f (0.0f, 4.0f);
		glColor4f (1.0f - health, health, 0.0f, 0.5f);
		glVertex2f (health * 64.0f, 4.0f);
		glVertex2f (health * 64.0f, 0.0f);
	
	
		cWeapon *pCurrentWeapon = GetCurrentWeapon ();
		if ( pCurrentWeapon != NULL )
		{
			float status = 0.0f;
			if ( pCurrentWeapon->GetState () == cWeapon::State_Reloading )
			{
				status = pCurrentWeapon->GetReloadStatus ();
				glColor4f (0.0f, 0.0f, 1.0f, 0.5f);
			}
			else
			{
				status = pCurrentWeapon->GetClipStatus ();
				glColor4f (1.0f, 1.0f, 0.0f, 0.5f);
			}
			glVertex2f (0.0f, 5.0f);
			glVertex2f (0.0f, 9.0f);
			glVertex2f (status * 64.0f, 9.0f);
			glVertex2f (status * 64.0f, 5.0f);
		}
	glEnd ();
	glPopAttrib ();	
	glPopMatrix ();
		
	glPopMatrix ();
	glPushMatrix ();

	glDisable (GL_TEXTURE_2D);
	
	cVector2f begin = GetEmittingPosition ();
	cVector2f end = begin + GetDirection (0) * 200.0f;

	// Aim line
	/*glLineWidth (3.0f);
	glColor4f (1.0f, 0.0f, 0.0f, 0.25f);
	glBegin (GL_LINES);
		glVertex2fv ((float *)&begin);
		glVertex2fv ((float *)&end);
	glEnd ();
	glLineWidth (1.0f);*/

	// Crosshair
	glColor4f (1.0f, 0.0f, 0.0f, 1.0f);
	glPointSize (3.0f);
	glBegin (GL_POINTS);
		glVertex2fv ((float *)&end);
	glEnd ();
	glPointSize (1.0f);

	glEnable (GL_TEXTURE_2D);
}

void 
cPlayer::HandleAction (int action)
{
	switch ( action )
	{
		case PlayerAction_PullTrigger:
			PullTrigger ();
			break;
		case PlayerAction_ReleaseTrigger:
			ReleaseTrigger ();
			break;
		case PlayerAction_NextWeapon:
			NextWeapon ();
			break;
		case PlayerAction_PreviousWeapon:
			PreviousWeapon ();
			break;
		case PlayerAction_EnableFineAim:
			EnableFineAim ();
			break;
		case PlayerAction_DisableFineAim:
			DisableFineAim ();
			break;
		case PlayerAction_EnableStrafe:
			EnableStrafe ();
			break;
		case PlayerAction_DisableStrafe:
			DisableStrafe ();
			break;
		default:
			break;		
	}
};

void 
cPlayer::OnLevelChanged ()
{
	// This is called when level changes

	// Call base
	cSoldier::OnLevelChanged ();

	// Reset healt & armor
	SetHealth (GetMaxHealth ());
	SetArmor (GetMaxArmor ());

	StopWalking ();
	StopTurning ();
	ReleaseTrigger ();
	DisableFineAim ();
	DisableStrafe ();

	// We have to respawn all weapons because all objects are killed 
	// when level changes
	for ( unsigned int i = 0; i < mWeapons.size (); i++ )
	{
		mWeapons[i]->SetParent (this);
		mWeapons[i]->Reset ();
		if ( !mWeapons[i]->IsAlive () )
			cWorld::GetInstance ().SpawnObject (mWeapons[i]);
	}
	
	cWeapon *pCurrentWeapon = GetCurrentWeapon ();	
	if ( pCurrentWeapon != NULL )
		pCurrentWeapon->SetActive (true);
}

void 
cPlayer::SaveWeapons (void)
{
	mStoredWeapons.clear ();
	for ( unsigned int i = 0; i < mWeapons.size (); i++ )
	{
		tWeaponInfo weapon;
		weapon.name = mWeapons[i]->GetName ();
		mWeapons[i]->GetAmmo (weapon.ammo, weapon.clips);
		mStoredWeapons.push_back (weapon);
	}
	//dbgInfo () << mStoredWeapons.size () << " weapons saved\n";
}

void 
cPlayer::RestoreWeapons (void)
{
	mCurrentWeapon = -1;

	vector<cPointer<cWeapon> > newWeapons;

	//dbgInfo () << "Size of weapons " << mWeapons.size ();

	bool remove;
	vector<cPointer<cWeapon> >::iterator weapon = mWeapons.begin ();
	while ( weapon != mWeapons.end () )
	{
		for ( unsigned int i = 0; i < mStoredWeapons.size (); i++ )
		{
			if ( (*weapon)->GetName () == mStoredWeapons[i].name )
			{
				//dbgInfo () << "Restoring " << (*weapon)->GetName () << '\n';
				(*weapon)->SetAmmo (mStoredWeapons[i].ammo, mStoredWeapons[i].clips);
				newWeapons.push_back (*weapon);
			}
		}
		weapon++;
	}
	
	mWeapons.clear ();
	mWeapons = newWeapons;
	
	//dbgInfo () << mWeapons.size () << " weapons restored\n";
	
	dbg::assertion (DBG_ASSERTION (mWeapons.size () == mStoredWeapons.size ()));
	NextWeapon ();
}

//==============================================================================
// EOF
//==============================================================================
