/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.encrypted;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.NoSuchFileException;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.function.Supplier;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.CheckedSupplier;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetadata;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.io.stream.ReleasableBytesStreamOutput;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.CheckedFunction;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.core.internal.io.Streams;
import org.elasticsearch.indices.recovery.RecoverySettings;
import org.elasticsearch.license.LicenseUtils;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.repositories.FinalizeSnapshotContext;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.RepositoryStats;
import org.elasticsearch.repositories.SnapshotShardContext;
import org.elasticsearch.repositories.blobstore.BlobStoreRepository;
import org.elasticsearch.repositories.encrypted.AESKeyUtils;
import org.elasticsearch.repositories.encrypted.ChainingInputStream;
import org.elasticsearch.repositories.encrypted.DecryptionPacketsInputStream;
import org.elasticsearch.repositories.encrypted.EncryptedRepositoryPlugin;
import org.elasticsearch.repositories.encrypted.EncryptionPacketsInputStream;
import org.elasticsearch.repositories.encrypted.SingleUseKey;
import org.elasticsearch.snapshots.SnapshotInfo;
import org.elasticsearch.xcontent.NamedXContentRegistry;

public class EncryptedRepository
extends BlobStoreRepository {
    static final Logger logger = LogManager.getLogger(EncryptedRepository.class);
    static final int GCM_TAG_LENGTH_IN_BYTES = 16;
    static final int GCM_IV_LENGTH_IN_BYTES = 12;
    static final int AES_BLOCK_LENGTH_IN_BYTES = 128;
    static final String DATA_ENCRYPTION_SCHEME = "AES/GCM/NoPadding";
    static final long PACKET_START_COUNTER = Long.MIN_VALUE;
    static final int MAX_PACKET_LENGTH_IN_BYTES = 0x800000;
    static final int PACKET_LENGTH_IN_BYTES = 65536;
    static final String DEK_ROOT_CONTAINER = ".encryption-metadata";
    static final int DEK_ID_LENGTH = 22;
    static final String PASSWORD_HASH_USER_METADATA_KEY = EncryptedRepository.class.getName() + ".repositoryPasswordHash";
    static final String PASSWORD_SALT_USER_METADATA_KEY = EncryptedRepository.class.getName() + ".repositoryPasswordSalt";
    private static final int DEK_CACHE_WEIGHT = 2048;
    private final BlobStoreRepository delegatedRepository;
    private final Supplier<Tuple<BytesReference, SecretKey>> dekGenerator;
    protected Supplier<XPackLicenseState> licenseStateSupplier;
    private final SecureString repositoryPassword;
    private final String localRepositoryPasswordHash;
    private final String localRepositoryPasswordSalt;
    private volatile String validatedLocalRepositoryPasswordHash;
    private final Cache<String, SecretKey> dekCache;
    static final Setting<Boolean> COMPRESS_SETTING = Setting.boolSetting((String)"compress", (boolean)false, (Setting.Property[])new Setting.Property[0]);

    public static long getEncryptedBlobByteLength(long plaintextBlobByteLength) {
        return 22L + EncryptionPacketsInputStream.getEncryptionLength(plaintextBlobByteLength, 65536);
    }

    protected EncryptedRepository(RepositoryMetadata metadata, NamedXContentRegistry namedXContentRegistry, ClusterService clusterService, BigArrays bigArrays, RecoverySettings recoverySettings, BlobStoreRepository delegatedRepository, Supplier<XPackLicenseState> licenseStateSupplier, SecureString repositoryPassword) throws GeneralSecurityException {
        super(metadata, ((Boolean)COMPRESS_SETTING.get(metadata.settings())).booleanValue(), namedXContentRegistry, clusterService, bigArrays, recoverySettings);
        this.delegatedRepository = delegatedRepository;
        this.dekGenerator = this.createDEKGenerator();
        this.licenseStateSupplier = licenseStateSupplier;
        this.repositoryPassword = repositoryPassword;
        this.localRepositoryPasswordSalt = UUIDs.randomBase64UUID();
        this.validatedLocalRepositoryPasswordHash = this.localRepositoryPasswordHash = AESKeyUtils.computeId(AESKeyUtils.generatePasswordBasedKey(repositoryPassword, this.localRepositoryPasswordSalt));
        this.dekCache = CacheBuilder.builder().setMaximumWeight(2048L).build();
        if (this.isReadOnly() != delegatedRepository.isReadOnly()) {
            throw new RepositoryException(metadata.name(), "Unexpected fatal internal error", (Throwable)new IllegalStateException("The encrypted repository must be read-only iff the delegate repository is read-only"));
        }
    }

    public RepositoryStats stats() {
        return this.delegatedRepository.stats();
    }

    public Map<String, Object> adaptUserMetadata(Map<String, Object> userMetadata) {
        if (!EncryptedRepositoryPlugin.ENCRYPTED_SNAPSHOT_FEATURE.checkWithoutTracking(this.licenseStateSupplier.get())) {
            throw LicenseUtils.newComplianceException((String)"encrypted snapshots");
        }
        HashMap<String, Object> snapshotUserMetadata = new HashMap<String, Object>();
        if (userMetadata != null) {
            snapshotUserMetadata.putAll(userMetadata);
        }
        snapshotUserMetadata.put(PASSWORD_SALT_USER_METADATA_KEY, this.localRepositoryPasswordSalt);
        snapshotUserMetadata.put(PASSWORD_HASH_USER_METADATA_KEY, this.localRepositoryPasswordHash);
        logger.trace("Snapshot metadata for local repository password  [{}] and [{}]", (Object)this.localRepositoryPasswordSalt, (Object)this.localRepositoryPasswordHash);
        return org.elasticsearch.core.Map.copyOf(snapshotUserMetadata);
    }

    public void finalizeSnapshot(FinalizeSnapshotContext finalizeSnapshotContext) {
        SnapshotInfo snapshotInfo = finalizeSnapshotContext.snapshotInfo();
        try {
            this.validateLocalRepositorySecret(snapshotInfo.userMetadata());
        }
        catch (RepositoryException passwordValidationException) {
            finalizeSnapshotContext.onFailure((Exception)((Object)passwordValidationException));
            return;
        }
        HashMap updatedUserMetadata = new HashMap(snapshotInfo.userMetadata());
        updatedUserMetadata.remove(PASSWORD_HASH_USER_METADATA_KEY);
        updatedUserMetadata.remove(PASSWORD_SALT_USER_METADATA_KEY);
        SnapshotInfo updatedSnapshotInfo = new SnapshotInfo(snapshotInfo.snapshot(), snapshotInfo.indices(), snapshotInfo.dataStreams(), snapshotInfo.featureStates(), snapshotInfo.reason(), snapshotInfo.version(), snapshotInfo.startTime(), snapshotInfo.endTime(), snapshotInfo.totalShards(), snapshotInfo.successfulShards(), snapshotInfo.shardFailures(), snapshotInfo.includeGlobalState(), updatedUserMetadata, snapshotInfo.state(), snapshotInfo.indexSnapshotDetails());
        super.finalizeSnapshot(new FinalizeSnapshotContext(finalizeSnapshotContext.updatedShardGenerations(), finalizeSnapshotContext.repositoryStateId(), finalizeSnapshotContext.clusterMetadata(), updatedSnapshotInfo, finalizeSnapshotContext.repositoryMetaVersion(), (ActionListener)finalizeSnapshotContext));
    }

    public void snapshotShard(SnapshotShardContext context) {
        try {
            this.validateLocalRepositorySecret(context.userMetadata());
        }
        catch (RepositoryException passwordValidationException) {
            context.onFailure((Exception)((Object)passwordValidationException));
            return;
        }
        super.snapshotShard(context);
    }

    protected BlobStore createBlobStore() {
        Supplier<Tuple<BytesReference, SecretKey>> blobStoreDEKGenerator = this.isReadOnly() ? () -> {
            throw new RepositoryException(this.metadata.name(), "Unexpected fatal internal error", (Throwable)new IllegalStateException("DEKs are required for encryption but this is a read-only repository"));
        } : this.dekGenerator;
        return new EncryptedBlobStore(this.delegatedRepository.blobStore(), this.delegatedRepository.basePath(), this.metadata.name(), this::generateKEK, blobStoreDEKGenerator, this.dekCache);
    }

    public BlobPath basePath() {
        return BlobPath.EMPTY;
    }

    protected void doStart() {
        this.delegatedRepository.start();
        super.doStart();
    }

    protected void doStop() {
        super.doStop();
        this.delegatedRepository.stop();
    }

    protected void doClose() {
        super.doClose();
        this.delegatedRepository.close();
    }

    private Supplier<Tuple<BytesReference, SecretKey>> createDEKGenerator() throws GeneralSecurityException {
        SecureRandom dekSecureRandom = new SecureRandom();
        SecureRandom dekIdSecureRandom = new SecureRandom();
        KeyGenerator dekGenerator = KeyGenerator.getInstance(DATA_ENCRYPTION_SCHEME.split("/")[0]);
        dekGenerator.init(256, dekSecureRandom);
        return () -> {
            BytesArray dekId = new BytesArray(UUIDs.randomBase64UUID((Random)dekIdSecureRandom));
            SecretKey dek = dekGenerator.generateKey();
            logger.debug("Repository [{}] generated new DEK [{}]", (Object)this.metadata.name(), (Object)dekId);
            return new Tuple((Object)dekId, (Object)dek);
        };
    }

    Tuple<String, SecretKey> generateKEK(String dekId) {
        try {
            SecretKey kek = AESKeyUtils.generatePasswordBasedKey(this.repositoryPassword, dekId);
            String kekId = AESKeyUtils.computeId(kek);
            logger.debug("Repository [{}] computed KEK [{}] for DEK [{}]", (Object)this.metadata.name(), (Object)kekId, (Object)dekId);
            return new Tuple((Object)kekId, (Object)kek);
        }
        catch (GeneralSecurityException e) {
            throw new RepositoryException(this.metadata.name(), "Failure to generate KEK to wrap the DEK [" + dekId + "]", (Throwable)e);
        }
    }

    private void validateLocalRepositorySecret(Map<String, Object> snapshotUserMetadata) throws RepositoryException {
        assert (snapshotUserMetadata != null);
        assert (snapshotUserMetadata.get(PASSWORD_HASH_USER_METADATA_KEY) instanceof String);
        String masterRepositoryPasswordId = (String)snapshotUserMetadata.get(PASSWORD_HASH_USER_METADATA_KEY);
        if (!masterRepositoryPasswordId.equals(this.validatedLocalRepositoryPasswordHash)) {
            String computedRepositoryPasswordId;
            assert (snapshotUserMetadata.get(PASSWORD_SALT_USER_METADATA_KEY) instanceof String);
            String masterRepositoryPasswordIdSalt = (String)snapshotUserMetadata.get(PASSWORD_SALT_USER_METADATA_KEY);
            try {
                computedRepositoryPasswordId = AESKeyUtils.computeId(AESKeyUtils.generatePasswordBasedKey(this.repositoryPassword, masterRepositoryPasswordIdSalt));
            }
            catch (Exception e) {
                throw new RepositoryException(this.metadata.name(), "Unexpected fatal internal error", (Throwable)e);
            }
            if (computedRepositoryPasswordId.equals(masterRepositoryPasswordId)) {
                this.validatedLocalRepositoryPasswordHash = computedRepositoryPasswordId;
            } else {
                throw new RepositoryException(this.metadata.name(), "Repository password mismatch. The local node's repository password, from the keystore setting [" + EncryptedRepositoryPlugin.ENCRYPTION_PASSWORD_SETTING.getConcreteSettingForNamespace((String)EncryptedRepositoryPlugin.PASSWORD_NAME_SETTING.get(this.metadata.settings())).getKey() + "], is different compared to the elected master node's which started the snapshot operation");
            }
        }
    }

    public boolean hasAtomicOverwrites() {
        return this.delegatedRepository.hasAtomicOverwrites();
    }

    class EncryptedBlobStore
    implements BlobStore {
        private final BlobStore delegatedBlobStore;
        private final BlobPath delegatedBasePath;
        private final String repositoryName;
        private final Function<String, Tuple<String, SecretKey>> getKEKforDEK;
        private final Cache<String, SecretKey> dekCache;
        private final CheckedSupplier<SingleUseKey, IOException> singleUseDEKSupplier;

        EncryptedBlobStore(BlobStore delegatedBlobStore, BlobPath delegatedBasePath, String repositoryName, Function<String, Tuple<String, SecretKey>> getKEKforDEK, Supplier<Tuple<BytesReference, SecretKey>> dekGenerator, Cache<String, SecretKey> dekCache) {
            this.delegatedBlobStore = delegatedBlobStore;
            this.delegatedBasePath = delegatedBasePath;
            this.repositoryName = repositoryName;
            this.getKEKforDEK = getKEKforDEK;
            this.dekCache = dekCache;
            this.singleUseDEKSupplier = SingleUseKey.createSingleUseKeySupplier(() -> {
                Tuple newDEK = (Tuple)dekGenerator.get();
                this.storeDEK(((BytesReference)newDEK.v1()).utf8ToString(), (SecretKey)newDEK.v2());
                return newDEK;
            });
        }

        SecretKey getDEKById(String dekId) throws IOException {
            try {
                return (SecretKey)this.dekCache.computeIfAbsent((Object)dekId, ignored -> this.loadDEK(dekId));
            }
            catch (ExecutionException e) {
                if (e.getCause() instanceof IOException) {
                    throw (IOException)e.getCause();
                }
                if (e.getCause() instanceof ElasticsearchException) {
                    throw (ElasticsearchException)e.getCause();
                }
                throw new RepositoryException(this.repositoryName, "Unexpected exception retrieving DEK [" + dekId + "]", (Throwable)e);
            }
        }

        private SecretKey loadDEK(String dekId) throws IOException {
            BlobPath dekBlobPath = this.delegatedBasePath.add(EncryptedRepository.DEK_ROOT_CONTAINER).add(dekId);
            logger.debug("Repository [{}] loading wrapped DEK [{}] from blob path {}", (Object)this.repositoryName, (Object)dekId, (Object)dekBlobPath);
            BlobContainer dekBlobContainer = this.delegatedBlobStore.blobContainer(dekBlobPath);
            Tuple<String, SecretKey> kekTuple = this.getKEKforDEK.apply(dekId);
            String kekId = (String)kekTuple.v1();
            SecretKey kek = (SecretKey)kekTuple.v2();
            logger.trace("Repository [{}] using KEK [{}] to unwrap DEK [{}]", (Object)this.repositoryName, (Object)kekId, (Object)dekId);
            byte[] encryptedDEKBytes = new byte[40];
            try (InputStream encryptedDEKInputStream = dekBlobContainer.readBlob(kekId);){
                int bytesRead = org.elasticsearch.common.io.Streams.readFully((InputStream)encryptedDEKInputStream, (byte[])encryptedDEKBytes);
                if (bytesRead != 40) {
                    throw new RepositoryException(this.repositoryName, "Wrapped DEK [" + dekId + "] has smaller length [" + bytesRead + "] than expected");
                }
                if (encryptedDEKInputStream.read() != -1) {
                    throw new RepositoryException(this.repositoryName, "Wrapped DEK [" + dekId + "] is larger than expected");
                }
            }
            catch (NoSuchFileException e) {
                throw new ElasticsearchException("Failure to read and decrypt DEK [" + dekId + "] from " + dekBlobContainer.path() + ". Most likely the repository password is incorrect, where previous snapshots have used a different password.", (Throwable)e, new Object[0]);
            }
            logger.trace("Repository [{}] successfully read DEK [{}] from path {} {}", (Object)this.repositoryName, (Object)dekId, (Object)dekBlobPath, (Object)kekId);
            try {
                SecretKey dek = AESKeyUtils.unwrap(kek, encryptedDEKBytes);
                logger.debug("Repository [{}] successfully loaded DEK [{}] from path {} {}", (Object)this.repositoryName, (Object)dekId, (Object)dekBlobPath, (Object)kekId);
                return dek;
            }
            catch (GeneralSecurityException e) {
                throw new RepositoryException(this.repositoryName, "Failure to AES unwrap the DEK [" + dekId + "]. Most likely the encryption metadata in the repository has been corrupted", (Throwable)e);
            }
        }

        void storeDEK(String dekId, SecretKey dek) throws IOException {
            byte[] encryptedDEKBytes;
            BlobPath dekBlobPath = this.delegatedBasePath.add(EncryptedRepository.DEK_ROOT_CONTAINER).add(dekId);
            logger.debug("Repository [{}] storing wrapped DEK [{}] under blob path {}", (Object)this.repositoryName, (Object)dekId, (Object)dekBlobPath);
            BlobContainer dekBlobContainer = this.delegatedBlobStore.blobContainer(dekBlobPath);
            Tuple<String, SecretKey> kek = this.getKEKforDEK.apply(dekId);
            logger.trace("Repository [{}] using KEK [{}] to wrap DEK [{}]", (Object)this.repositoryName, kek.v1(), (Object)dekId);
            try {
                encryptedDEKBytes = AESKeyUtils.wrap((SecretKey)kek.v2(), dek);
                if (encryptedDEKBytes.length != 40) {
                    throw new RepositoryException(this.repositoryName, "Wrapped DEK [" + dekId + "] has unexpected length [" + encryptedDEKBytes.length + "]");
                }
            }
            catch (GeneralSecurityException e) {
                throw new RepositoryException(this.repositoryName, "Failure to AES wrap the DEK [" + dekId + "]", (Throwable)e);
            }
            logger.trace("Repository [{}] successfully wrapped DEK [{}]", (Object)this.repositoryName, (Object)dekId);
            dekBlobContainer.writeBlobAtomic((String)kek.v1(), (BytesReference)new BytesArray(encryptedDEKBytes), true);
            logger.debug("Repository [{}] successfully stored DEK [{}] under path {} {}", (Object)this.repositoryName, (Object)dekId, (Object)dekBlobPath, kek.v1());
        }

        public BlobContainer blobContainer(BlobPath path) {
            BlobPath delegatedBlobContainerPath = this.delegatedBasePath;
            for (String s : path.parts()) {
                delegatedBlobContainerPath = delegatedBlobContainerPath.add(s);
            }
            BlobContainer delegatedBlobContainer = this.delegatedBlobStore.blobContainer(delegatedBlobContainerPath);
            return new EncryptedBlobContainer(path, this.repositoryName, delegatedBlobContainer, this.singleUseDEKSupplier, (CheckedFunction<String, SecretKey, IOException>)((CheckedFunction)this::getDEKById));
        }

        public void close() {
        }
    }

    private final class EncryptedBlobContainer
    extends AbstractBlobContainer {
        private final String repositoryName;
        private final BlobContainer delegatedBlobContainer;
        private final CheckedSupplier<SingleUseKey, IOException> singleUseDEKSupplier;
        private final CheckedFunction<String, SecretKey, IOException> getDEKById;

        EncryptedBlobContainer(BlobPath path, String repositoryName, BlobContainer delegatedBlobContainer, CheckedSupplier<SingleUseKey, IOException> singleUseDEKSupplier, CheckedFunction<String, SecretKey, IOException> getDEKById) {
            String rootPathElement;
            super(path);
            this.repositoryName = repositoryName;
            String string = rootPathElement = path.parts().isEmpty() ? null : (String)path.parts().get(0);
            if (EncryptedRepository.DEK_ROOT_CONTAINER.equals(rootPathElement)) {
                throw new RepositoryException(repositoryName, "Cannot descend into the DEK blob container " + path);
            }
            this.delegatedBlobContainer = delegatedBlobContainer;
            this.singleUseDEKSupplier = singleUseDEKSupplier;
            this.getDEKById = getDEKById;
        }

        public boolean blobExists(String blobName) throws IOException {
            return this.delegatedBlobContainer.blobExists(blobName);
        }

        public InputStream readBlob(String blobName) throws IOException {
            InputStream encryptedDataInputStream = this.delegatedBlobContainer.readBlob(blobName);
            try {
                byte[] dekIdBytes = new byte[22];
                int bytesRead = org.elasticsearch.common.io.Streams.readFully((InputStream)encryptedDataInputStream, (byte[])dekIdBytes);
                if (bytesRead != 22) {
                    throw new RepositoryException(this.repositoryName, "The encrypted blob [" + blobName + "] is too small [" + bytesRead + "]");
                }
                String dekId = new String(dekIdBytes, StandardCharsets.UTF_8);
                SecretKey dek = (SecretKey)this.getDEKById.apply((Object)dekId);
                return new DecryptionPacketsInputStream(encryptedDataInputStream, dek, 65536);
            }
            catch (Exception e) {
                try {
                    encryptedDataInputStream.close();
                }
                catch (IOException closeEx) {
                    e.addSuppressed(closeEx);
                }
                throw e;
            }
        }

        public InputStream readBlob(String blobName, long position, long length) throws IOException {
            throw new UnsupportedOperationException("Not yet implemented");
        }

        public void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
            SingleUseKey singleUseNonceAndDEK = (SingleUseKey)this.singleUseDEKSupplier.get();
            BytesReference dekIdBytes = this.getDEKBytes(singleUseNonceAndDEK);
            long encryptedBlobSize = EncryptedRepository.getEncryptedBlobByteLength(blobSize);
            try (ChainingInputStream encryptedInputStream = this.encryptedInput(org.elasticsearch.common.io.Streams.noCloseStream((InputStream)inputStream), singleUseNonceAndDEK, dekIdBytes);){
                this.delegatedBlobContainer.writeBlob(blobName, (InputStream)encryptedInputStream, encryptedBlobSize, failIfAlreadyExists);
            }
        }

        public void writeBlob(String blobName, BytesReference bytes, boolean failIfAlreadyExists) throws IOException {
            SingleUseKey singleUseNonceAndDEK = (SingleUseKey)this.singleUseDEKSupplier.get();
            BytesReference dekIdBytes = this.getDEKBytes(singleUseNonceAndDEK);
            try (ReleasableBytesStreamOutput tmp = new ReleasableBytesStreamOutput(Math.toIntExact(EncryptedRepository.getEncryptedBlobByteLength(bytes.length())), EncryptedRepository.this.bigArrays);){
                try (ChainingInputStream encryptedInputStream = this.encryptedInput((InputStream)bytes.streamInput(), singleUseNonceAndDEK, dekIdBytes);){
                    Streams.copy((InputStream)encryptedInputStream, (OutputStream)tmp, (boolean)false);
                }
                this.delegatedBlobContainer.writeBlob(blobName, tmp.bytes(), failIfAlreadyExists);
            }
        }

        public void writeBlob(String blobName, boolean failIfAlreadyExists, boolean atomic, CheckedConsumer<OutputStream, IOException> writer) throws IOException {
            try (ReleasableBytesStreamOutput out = new ReleasableBytesStreamOutput(EncryptedRepository.this.bigArrays);){
                writer.accept((Object)out);
                if (atomic) {
                    this.writeBlobAtomic(blobName, out.bytes(), failIfAlreadyExists);
                } else {
                    this.writeBlob(blobName, out.bytes(), failIfAlreadyExists);
                }
            }
        }

        private ChainingInputStream encryptedInput(InputStream inputStream, SingleUseKey singleUseNonceAndDEK, BytesReference dekIdBytes) throws IOException {
            return ChainingInputStream.chain((InputStream)dekIdBytes.streamInput(), new EncryptionPacketsInputStream(inputStream, singleUseNonceAndDEK.getKey(), singleUseNonceAndDEK.getNonce(), 65536));
        }

        private BytesReference getDEKBytes(SingleUseKey singleUseNonceAndDEK) {
            BytesReference dekIdBytes = singleUseNonceAndDEK.getKeyId();
            if (dekIdBytes.length() != 22) {
                throw new RepositoryException(this.repositoryName, "Unexpected fatal internal error", (Throwable)new IllegalStateException("Unexpected DEK Id length [" + dekIdBytes.length() + "]"));
            }
            return dekIdBytes;
        }

        public void writeBlobAtomic(String blobName, BytesReference bytes, boolean failIfAlreadyExists) throws IOException {
            this.writeBlob(blobName, bytes, failIfAlreadyExists);
        }

        public DeleteResult delete() throws IOException {
            return this.delegatedBlobContainer.delete();
        }

        public void deleteBlobsIgnoringIfNotExists(Iterator<String> blobNames) throws IOException {
            this.delegatedBlobContainer.deleteBlobsIgnoringIfNotExists(blobNames);
        }

        public Map<String, BlobMetadata> listBlobs() throws IOException {
            return this.delegatedBlobContainer.listBlobs();
        }

        public Map<String, BlobMetadata> listBlobsByPrefix(String blobNamePrefix) throws IOException {
            return this.delegatedBlobContainer.listBlobsByPrefix(blobNamePrefix);
        }

        public Map<String, BlobContainer> children() throws IOException {
            Map childEncryptedBlobContainers = this.delegatedBlobContainer.children();
            HashMap<String, BlobContainer> resultBuilder = new HashMap<String, BlobContainer>(childEncryptedBlobContainers.size());
            for (Map.Entry childBlobContainer : childEncryptedBlobContainers.entrySet()) {
                if (((String)childBlobContainer.getKey()).equals(EncryptedRepository.DEK_ROOT_CONTAINER) && this.path().parts().isEmpty()) continue;
                resultBuilder.put((String)childBlobContainer.getKey(), (BlobContainer)new EncryptedBlobContainer(this.path().add((String)childBlobContainer.getKey()), this.repositoryName, (BlobContainer)childBlobContainer.getValue(), this.singleUseDEKSupplier, this.getDEKById));
            }
            return resultBuilder;
        }
    }
}

