/*

Copyright 2006 Suzanne Skinner, John Spray

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/



#ifndef GTKLIFE_H
#define GTKLIFE_H

#include <gtk/gtk.h>

#include "life.h"
#include "util.h"

#include <config.h>

/* Defines */

#define PROG      PACKAGE
#define TITLE     PACKAGE_NAME
#define USER_DIR  "." PACKAGE
// VERSION already defined in config.h
/*#define VERSION   PACKAGE_VERSION*/

#define INIT_X_CENTER             (WORLD_SIZE / 2)
#define INIT_Y_CENTER             (WORLD_SIZE / 2)
#define MAX_RECENT_FILES          5
#define MAX_SKIPPED_FRAMES        10
#define MAX_SPEED                 10000
#define MAX_ZOOM                  16
#define MIN_ZOOM                  1
#define RECENT_FILES_INSERT_POS   11
#define STATUS_MESSAGE_COLOR      0x007000
#define TABLE_OPTS                (GTK_SHRINK | GTK_FILL)
#define TEMP_MESSAGE_INTERVAL     5000      /* in milliseconds */

#define FILE_LOADED_MESSAGE       "File loaded successfully"
#define FILE_SAVED_MESSAGE        "File saved successfully"
#define FULLSCREEN_MESSAGE        "Press F11 to leave fullscreen mode"
#define PASTING_MESSAGE           "Pasting, Ctrl-G to cancel"
#define MOVING_MESSAGE            "Moving, Ctrl-G to cancel"

#define SCROLL_STEP  -1
#define SCROLL_PAGE  -2
#define SPEED_INCREMENT 5
#define SPEED_MIN 1

/* Configuration Defaults */

#define DEFAULT_COLLECTION        COLL_LPA
#define DEFAULT_WINDOW_WIDTH      800
#define DEFAULT_WINDOW_HEIGHT     600
#define DEFAULT_ZOOM              1
#define DEFAULT_SHOW_TOOLBAR      TRUE
#define DEFAULT_SHOW_SIDEBAR      TRUE
#define DEFAULT_SHOW_SCROLLBARS   TRUE
#define DEFAULT_SHOW_STATUSBAR    TRUE
#define DEFAULT_FULLSCREEN        FALSE
#define DEFAULT_BG_COLOR          0xC0C0C0
#define DEFAULT_CELL_COLOR        0x000080
#define DEFAULT_GRID_COLOR        0x808080
#define DEFAULT_SELECT_COLOR      0xFF0000
#define DEFAULT_SPEED             20
#define DEFAULT_SPEED_MAX 500
#define DEFAULT_SKIP_FRAMES       TRUE

/* Enumerations */

enum color_index_enum  {BG_COLOR_INDEX, CELL_COLOR_INDEX, GRID_COLOR_INDEX, SELECT_COLOR_INDEX,
                        NUM_COLORS};

typedef enum tracking_mouse_enum {NORMAL, DRAWING, SELECTING, PASTING, GRABBING} tracking_mouse_type;

typedef enum coll_status_enum {COLL_OKAY, COLL_BAD, COLL_DEFAULT_BAD} coll_status_type;

typedef enum component_enum {
    COMPONENT_TOOLBAR,
    COMPONENT_SIDEBAR,
    COMPONENT_SCROLLBARS,
    COMPONENT_STATUSBAR,

    NUM_COMPONENTS
} component_type;

typedef enum pattern_coll_enum {
    COLL_LPA,
    COLL_JSLIFE,

    NUM_COLLECTIONS
} pattern_coll_id;

typedef enum tool_id_enum {
	TOOL_DRAW,
	TOOL_SELECT,
	TOOL_GRAB
} tool_id;

/* Macros */

#define DRAW_GRID(zoom_level)  ((zoom_level) > 2)
#define SELECTION_ACTIVE()     (!rects_identical(&state.selection, &null_rect))
#define COPY_BUFFER_ACTIVE()   (!rects_identical(&state.copy_rect, &null_rect))
#define SWAP(a,b)              {temp=a; a=b; b=temp;}

#define LEFT_ALIGN(w)   alignment = gtk_alignment_new(0, 0.5, 0, 0); \
                        gtk_container_add(GTK_CONTAINER(alignment), (w))
#define RIGHT_ALIGN(w)  alignment = gtk_alignment_new(1, 0.5, 0, 0); \
                        gtk_container_add(GTK_CONTAINER(alignment), (w))
#define CENTER_ALIGN(w) alignment = gtk_alignment_new(0.5, 0.5, 0, 0); \
                        gtk_container_add(GTK_CONTAINER(alignment), (w))
#define FRAME_ALIGN(w)  alignment = gtk_alignment_new(0, 0, 1, 0); \
                        gtk_container_add(GTK_CONTAINER(alignment), (w))

/* Types */

typedef void (*bound_function)(void);

typedef struct dimension_struct {
    int32 width;
    int32 height;
} dimension;

typedef struct point_struct {
    int32 x;
    int32 y;
} point;

typedef struct rect_struct {
    point start;
    point end;
} rect;

typedef struct command_widgets_struct {
    GtkWidget*  menu_item;
    GtkWidget*  toolbar_button;
} command_widgets_type;

typedef struct pattern_file_struct {
    char*    filename;
    char*    path;
    char*    title;
    boolean  is_directory;
} pattern_file;

typedef struct recent_file_struct {
    char*       filename;
    char*       full_path;
    GtkWidget*  menu_item;
    GtkWidget*  label;
} recent_file;

typedef struct pattern_coll_struct {
    const char*  dir;
    const char*  title;
} pattern_coll;

/* Prototypes */

void  usage_abort(void);

/* Main loop functions */

void  main_loop(void);
void  tick_and_update(void);

/* Functions for reading/saving user settings */

void  create_user_dir(void);
void  load_preferences(void);
void  state_from_config(void);
void  config_from_state(void);
void  load_recent_files_list(void);
void  save_preferences(void);
void  save_recent_files_list(void);

/* Bound Functions */

void  file_new(void);
void  file_open(void);
void  file_reopen(void);
void  file_save(void);
void  file_save_as(void);
void  file_description(void);
void  file_change_collection(void);
void  file_recent(GtkMenuItem* menu_item, gpointer user_data);
void  file_quit(void);

void  view_zoom_in(void);
void  view_zoom_out(void);
void  view_recenter(void);
void  view_find_active_cells(void);

void  position_scroll_left(void);
void  position_scroll_right(void);
void  position_scroll_up(void);
void  position_scroll_down(void);
void  position_scroll_nw(void);
void  position_scroll_ne(void);
void  position_scroll_sw(void);
void  position_scroll_se(void);
void  position_page_left(void);
void  position_page_right(void);
void  position_page_up(void);
void  position_page_down(void);
void  position_page_nw(void);
void  position_page_ne(void);
void  position_page_sw(void);
void  position_page_se(void);

void  edit_cut(void);
void  edit_copy(void);
void  edit_clear(void);
void  edit_paste(void);
void  edit_move(void);
void  edit_cancel_paste(void);
void  edit_preferences(void);

void  run_start_stop(void);
void  run_step(void);
void  run_jump(void);
void  run_faster(void);
void  run_slower(void);
void  run_speed(void);

void  help_help(void);
void  help_pattern_archive(void);
void  help_glf_file_format(void);
void  help_about(void);

/* Bound Function Helpers */

void     view_zoom(int32 new_zoom);
void     view_show(component_type component, boolean is_visible);
void     position_scroll_generic(int32 xdir, int32 ydir, int32 how_far);
void     edit_paste_win_style(const point* pt);
void     edit_paste_unix_style(const point* pt);
boolean  edit_paste_verify_target(const point* pt);
void     edit_paste_perform(const point* pt);
void     help_view_page(const char* filename);

/* GUI initialization functions */

void  init_gui(void);
void  init_stock_icons(void);
void  init_rgb(void);
void  init_window(void);
void  init_menubar(GtkWidget* containing_box);
void  init_toolbar(GtkWidget* containing_box);
void  init_sensitivities(void);
void  init_speed_box(GtkWidget* containing_box);
void  init_sidebar(GtkWidget* containing_box);
void  init_canvas(GtkWidget* containing_box);
void  init_statusbar(GtkWidget* containing_box);
void  init_tick_display(GtkWidget* containing_box);
void  init_population_display(GtkWidget* containing_box);

/* Event handlers */

void      activate_url(GtkAboutDialog *about, const gchar *url, gpointer data);

gboolean  handle_canvas_expose(GtkWidget* widget, GdkEventExpose* event, gpointer user_data);
gboolean  handle_canvas_resize(GtkWidget* widget, GdkEventConfigure* event, gpointer user_data);
void      handle_child_window_realize(GtkWidget* child, gpointer user_data);
void      handle_main_window_destroy(GtkObject* object, gpointer user_data);
gboolean handle_mouse_movement (GtkWidget *canvas,
                                GdkEventMotion *event,
                                gpointer user_data);
gboolean  handle_mouse_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data);
gboolean  handle_mouse_scroll(GtkWidget* widget, GdkEventScroll* event, gpointer user_data);
gboolean  handle_mouse_release(GtkWidget* widget, GdkEventButton* event, gpointer user_data);
void      handle_hscrollbar_change(GtkAdjustment* adjustment, gpointer user_data);
void      handle_vscrollbar_change(GtkAdjustment* adjustment, gpointer user_data);
void      handle_sidebar_select(GtkCList* clist, gint row, gint column, GdkEventButton* event,
                                gpointer user_data);
void      handle_sidebar_header_click(GtkCList* clist, gint column, gpointer user_data);
void      handle_speed_changed(GtkAdjustment* adjustment, gpointer user_data);

void      file_save_as_confirm_ok(GtkButton* button, gpointer user_data);
void      file_change_coll_toggle_custom(GtkToggleButton* toggle, gpointer user_data);
void      file_description_changed(GtkTextBuffer *textbuffer, gpointer user_data);
void      view_zoom_changed(GtkRadioAction *action, GtkRadioAction *current, gpointer user_data);
void      view_showtoolbar_cb(GtkToggleAction *toggleaction, gpointer user_data);
void      view_showsidebar_cb(GtkToggleAction *toggleaction, gpointer user_data);
void      view_showstatusbar_cb(GtkToggleAction *toggleaction, gpointer user_data);
void      view_showscrollbars_cb(GtkToggleAction *toggleaction, gpointer user_data);
void      view_fullscreen_cb(GtkToggleAction *toggleaction, gpointer user_data);
void      edit_tool_changed(GtkRadioAction *action, GtkRadioAction *current, gpointer user_data);
void      prefs_skip_cb (GtkWidget *widget, gpointer user_data);
void      prefs_color_cb (GtkColorButton *button, gpointer user_data);
void      prefs_speed_max_cb (GtkAdjustment *adj, gpointer user_data);
void      run_play_cb(GtkToggleAction *toggleaction, gpointer user_data);
void      run_jump_execute(gint);
void      run_jump_abort(GtkDialog* dialog, gint arg1, gpointer user_data);

/* Functions for drawing the offscreen pixmap */

void  draw_life_pixmap(void);
void  draw_grid_and_boxes(void);
void  draw_screen_box(const rect* box);
void  draw_life_block(uint16 block, int32 xstart, int32 ystart, boolean full_update);
void  user_draw(const point* pt);
void  draw_from_to(const point* start, const point* end);

/* Misc. GUI functions */

void       adjust_scrollbars(void);
void       adjust_scrollbar_values(void);
void       center_child_window(GtkWidget* child);
void       error_dialog(const char* msg, ...);
void       force_sidebar_resize(void);
void       full_canvas_redraw(void);
void       put_filename_in_window_title(void);
void       restore_status_message(void);
void       set_action_sensitive (gchar *action, gboolean sensitive);
void       set_active_tool (tool_id newtool);
void       set_toggle_action_active (gchar *action, gboolean active);
void       set_effective_canvas_size(void);
void       set_speed_label_size(int32 max_speed);
void       set_status_message(char* msg, boolean is_temporary);
void       set_viewport_position(const point* pt, boolean center_around);
void       set_zoom_sensitivities(void);
void       trigger_canvas_update(void);
void       update_description_textbox(boolean first_time);
void       update_hover_point_label(void);
void       update_population_label(void);
void       update_sidebar_contents(void);
void       update_sub_sidebar_contents(int32 sidebar_selection);
void       update_sidebar_generic(boolean is_sub_sidebar, const char* new_dir);
void       update_tick_label(void);
GtkWidget* create_spinbox(int32 init_value, int32 min, int32 max);

/* Miscellanous Functions */

void     add_to_recent_files(void);
boolean  attempt_load_pattern(const char* path);
boolean  attempt_save_pattern(const char* path, file_format_type format);
void     activate_selection(const point* pt);
void     clear_pattern_list(pattern_file** patterns, int32* num_patterns);
void     deactivate_selection(boolean redraw);
void     deactivate_paste(boolean redraw);
char*    find_life_file(const char* path);
int32    get_component_by_short_name(const char* name);
void     get_component_widgets(component_type component, GtkWidget** widget1,
                               GtkWidget** widget2);
void     get_logical_coords(const point* p, point* logical_p);
void     get_screen_coords(const point* p, point* screen_p);
void     get_screen_rectangle(const rect* r, rect* screen_r);
void     load_pattern_directory(const char* dir, boolean include_subdirs, pattern_file** patterns,
                                int32* num_patterns);
void     reset_fps_measure(void);
void     screen_box_update(rect* oldr, const rect* newr, boolean redraw);
void     selection_copy_clear(boolean copy, boolean clear);
void     set_collection_dir(const char* path, boolean have_gui);
void     set_current_dir_from_file(const char* filepath);
void     set_pattern_title(pattern_file* file);
void     set_speed(int32 new_speed);
void     sidebar_unselect(void);
void     start_stop(void);
char*    validate_collection_dir(const char* path, boolean have_gui);
boolean  validate_and_set_collection_dir(const char* path, boolean have_gui);

/* Utility Functions */

char*      append_trailing_slash(const char* path);
char*      get_canonical_path(const char* path);
char*      get_home_directory(void);
void       get_rect_dimensions(const rect* r, dimension* dim);
uint64     get_time_milliseconds(void);
char*      join_path(const char* dir, const char* file);
void       normalize_rectangle(rect* r);
boolean    points_identical(const point* p1, const point* p2);
int32      qsort_pattern_files(const void* pat1, const void* pat2);
boolean    rects_identical(const rect* r1, const rect* r2);
char*      remove_underscore(const char* str);
void       split_path(const char* path, char** dir, char** file);

#endif
