/*-
 * Copyright (c) 2001, 2002, 2003 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: psrc-bpf.c,v 1.18 2006/04/07 18:58:36 vlm Exp $
 */

#include "ipcad.h"
#include "opt.h"
#include "cfgvar.h"
#include <assert.h>

static int apply_bpf_filter(packet_source_t *ps, int bpf, int dlt, char *filter);

int
init_packet_source_bpf(packet_source_t *ps, int retry_mode) {
	int bpf_fd;
	struct ifreq ifr;

	assert(ps->state != PST_INVALID);	/* Embryonic or Ready */
	assert(ps->iface_type == IFACE_BPF);	/* Don't cook crap */

	/*
	 * Open BPF file descriptor.
	 */
	bpf_fd = ps->fd;
	ps->fd = -1;
	if(bpf_fd == -1) {
		char buf1[32];			/* The very first try */
		char buf2[sizeof(buf1)];	/* Current try */
		char *buf = buf1;
		int iteration;

		/*
		 * Open spare BPF device
		 */
		iteration = 0;
		do {
			snprintf(buf, sizeof(buf1), "/dev/bpf%d", iteration++);
			bpf_fd = open(buf, O_RDONLY
#ifdef	O_NONBLOCK
			| O_NONBLOCK
#endif
			);
		} while( bpf_fd == -1
			&& (
				/* Choose the first available bpf device */
				errno == EBUSY
				/* Skip the gaps in the names */
				|| (errno == ENOENT
					&& iteration < 10
					&& (buf = buf2) /* Preserve original */)
			)
		);

		if(bpf_fd == -1) {
			if(!retry_mode) {
				int tmp_errno = errno;
				fprintf(stderr, "[%s%s] ",
					conf->chroot_to?conf->chroot_to:"",
					buf1);
				errno = tmp_errno;
			}
			return -1;
		}
	} else {
		ps->state = PST_EMBRYONIC;
	}


	/*
	 * Set up buffer size.
	 */
	if(conf->bufsize) {
		int bufsize = conf->bufsize;
		if(bufsize < 128)
			bufsize = 128;
		if(ioctl(bpf_fd, BIOCSBLEN, &bufsize) == -1) {
			if(!retry_mode) {
				fprintf(stderr,
					"[can't set up %d bytes buffer: %s] ",
					bufsize, strerror(errno));
			}
			/*
			 * This is a relatively safe error to continue.
			close(bpf_fd);
			return -1;
			*/
		}
	}

	/* Get buffer length */
	if ( ioctl( bpf_fd, BIOCGBLEN, &ps->bufsize ) == -1 ) {
		close(bpf_fd);
		return -1;
	} else {
		if(!retry_mode)
			fprintf(stderr, "[%d] ", (int)ps->bufsize);
	}

	/*
	 * Attach an interface to the BPF.
	 */

	memset(&ifr, 0, sizeof(ifr));
	strncpy( ifr.ifr_name, IFNameBySource(ps), sizeof(ifr.ifr_name) );

	/* Bind to a specified interface */
	if( ioctl(bpf_fd, BIOCSETIF, &ifr) == -1 ) {
		if(errno == ENETDOWN) {
			ps->fd = bpf_fd;
		} else {
			close(bpf_fd);
		}
		return -1;
	}

	/* Enable promiscuous mode */
	if(ps->iflags & IFLAG_PROMISC) {
		if( ioctl(bpf_fd, BIOCPROMISC) == -1 ) {
			fprintf(stderr, "[Can't put into promiscuous mode!] ");
			close(bpf_fd);
			return -1;
		}
	}

	/* Getting type of underlying link */
	if( ioctl(bpf_fd, BIOCGDLT, &ps->dlt) == -1 ) {
		close(bpf_fd);
		return -1;
	}

	/* Initialize packet filter for optimized throughput */
	if ( apply_bpf_filter(ps, bpf_fd, ps->dlt, ps->custom_filter) ) {
		fprintf(stderr, "[Error: Can't initialize filter!] ");
		close(bpf_fd);
		errno = ENODEV;
		return -1;
	}

	if(ps->iflags & IFLAG_INONLY) {
#ifdef	BIOCSSEESENT
		long l = 0;
		if( ioctl(bpf_fd, BIOCSSEESENT, &l) == -1 ) {
			close(bpf_fd);
			return -1;
		}
#else	/* BIOCSSEESENT */
#warning "Ignore sent" interface feature is not supported by this BPF implementation. Minor warning.

		fprintf(stderr, "\"Ignore sent\" interface feature is not supported by kernel.\n");
#ifndef	EOSERR
#define	EOSERR	EINVAL
#endif
		errno = EOSERR;
		close(bpf_fd);
		return -1;
#endif	/* BIOCSSEESENT */
	}

	/* Don't do buffering in interactive mode */;
	if(daemon_mode == 0) {
		unsigned int on = 1;
		if(ioctl(bpf_fd, BIOCIMMEDIATE, &on) == 0) {
			fprintf(stderr,
			"[%s/interactive] ",
			IFNameBySource(ps));
		}
	}

	/*
	 * Finish initialization of the structure.
	 */
	if(ps->buf) free(ps->buf);
	ps->buf = (char *)malloc(ps->bufsize);
	if(ps->buf == NULL) {
		close(bpf_fd);
		/* ENOMEM */
		return -1;
	}
	ps->fd = bpf_fd;
	ps->state = PST_READY;

	return 0;
}

#ifndef	RAW_HDR_LEN
#define	RAW_HDR_LEN	0
#endif

#ifndef	NULL_HDR_LEN
#define	NULL_HDR_LEN	4
#endif

#ifndef	PPP_HDR_LEN
#define	PPP_HDR_LEN	4
#endif

#ifndef	PPP_IP
#define	PPP_IP	0x0021
#endif

static int
apply_RAW_filter(int bpf, int snaplen) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_RET+BPF_K, snaplen),
	};
	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	assert(snaplen >= (int)(RAW_HDR_LEN + sizeof(struct ip)));

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	}

	return 0;
}

static int
apply_NULL_filter(int bpf, int snaplen) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_LD+BPF_W+BPF_ABS, 0),
#ifdef	WORDS_BIGENDIAN
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, 0x00000002, 0, 1),
#else
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, 0x02000000, 0, 1),
#endif
		BPF_STMT(BPF_RET+BPF_K, snaplen),
		BPF_STMT(BPF_RET+BPF_K, 0),
	};
	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	assert(snaplen >= (int)(NULL_HDR_LEN + sizeof(struct ip)));

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	}

	return 0;
}

#ifdef DLT_LOOP
static int
apply_LOOP_filter(int bpf, int snaplen) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_LD+BPF_W+BPF_ABS, 0),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, 0x00000002, 0, 1),
		BPF_STMT(BPF_RET+BPF_K, snaplen),
		BPF_STMT(BPF_RET+BPF_K, 0),
	};
	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	assert(snaplen >= (int)(NULL_HDR_LEN + sizeof(struct ip)));

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	}

	return 0;
}
#endif	/* DLT_LOOP */

static int
apply_PPP_filter(int bpf, int snaplen) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_LD + BPF_ABS + BPF_H, 0x0002),
		BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, PPP_IP, 0, 1),
		BPF_STMT(BPF_RET+BPF_K, snaplen),
		BPF_STMT(BPF_RET+BPF_K, 0),
	};
	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	assert(snaplen >= (int)(PPP_HDR_LEN + sizeof(struct ip)));

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	}

	return 0;
}

static int
apply_EN10MB_filter(int bpf, int snaplen) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 12),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, ETHERTYPE_IP, 0, 1),
		BPF_STMT(BPF_RET+BPF_K, snaplen),
		BPF_STMT(BPF_RET+BPF_K, 0),
	};
	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	assert(snaplen >= (int)(ETHER_HDR_LEN + sizeof(struct ip)));

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	}

	return 0;
}

static int
apply_bpf_filter(packet_source_t *ps, int bpf, int dlt, char *filter) {
	int snaplen;

	switch(dlt) {
		case DLT_RAW:	/* Some PPP implementations, etc. */
			snaplen = RAW_HDR_LEN;
			break;
#ifdef DLT_LOOP
		case DLT_LOOP:	/* Loopback */
#endif
		case DLT_NULL:	/* Loopback */
			snaplen = NULL_HDR_LEN;
			break;
		case DLT_EN10MB:	/* Ethernet */
			snaplen = ETHER_HDR_LEN;
			break;
		case DLT_PPP:
			snaplen = PPP_HDR_LEN;
			break;
		default:
			return -1;
	};

	if((ps->iflags & IFLAG_LARGE_CAP)) {
		snaplen += sizeof(struct ip);
		snaplen += 5 * 4;	/* Allow a few IP options */
		snaplen += sizeof(struct tcphdr);
	} else {
		snaplen += sizeof(struct ip);
	}

#if HAVE_LIBPCAP && HAVE_PCAP_H
	if(filter) {
		struct bpf_program bp;

		if(pcap_compile_nopcap(snaplen, dlt, &bp, filter, 1, -1)) {
			fprintf(stderr, "[custom filter compile error] ");
			return -1;
		}

		if(ioctl(bpf, BIOCSETF, &bp) == -1) {
			fprintf(stderr, "[custom filter install error] ");
			return -1;
		}

		return 0;
	}
#else
	if(filter) {
		fprintf(stderr, "[libpcap required for custom filters] ");
		return -1;
	}
#endif


	switch(dlt) {
		case DLT_RAW:	/* Some PPP implementations, etc. */
			return apply_RAW_filter(bpf, snaplen);
#ifdef DLT_LOOP
		case DLT_LOOP:
			return apply_LOOP_filter(bpf, snaplen);
#endif
		case DLT_NULL:	/* Loopback */
			return apply_NULL_filter(bpf, snaplen);
		case DLT_EN10MB:	/* Ethernet */
			return apply_EN10MB_filter(bpf, snaplen);
		case DLT_PPP:
			return apply_PPP_filter(bpf, snaplen);
		default:
			return -1;
	}

	return -1;
}


void
print_stats_bpf(FILE *f, packet_source_t *ps) {
	struct bpf_stat bs;

	assert(ps->iface_type == IFACE_BPF);

	/*
	 * Interface is unavailable.
	 */
	if(ioctl(ps->fd, BIOCGSTATS, &bs) == -1) {
		if(ps->state == PST_EMBRYONIC
		|| ps->fd == -1)
			fprintf(f, "Interface %s: DOWN\n",
				IFNameBySource(ps));
		else
			fprintf(f, "Interface %s: %s\n",
				IFNameBySource(ps), strerror(errno));
	} else {
		fprintf(f, "Interface %s: received %u",
			IFNameBySource(ps), bs.bs_recv );

		fprintf(f, ", %.0f m average %lld bytes/sec, %lld pkts/sec",
			ps->avg_period / 60,
			ps->bps_lp,
			ps->pps_lp
		);

		fprintf(f, ", dropped %u",
			bs.bs_drop);

		fprintf(f, "\n");
	}
}
