/*-
 * Copyright (c) 2001, 2002, 2005 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: psrc-file.c,v 1.2 2006/04/07 18:58:02 vlm Exp $
 */

#include "ipcad.h"
#include "cfgvar.h"
#include "opt.h"


#warning "Ignore sent" interface feature is not supported by PCAP file. Minor warning.

static int apply_filter(pcap_t *dev, char *filter);

int
reopen_packet_source_file(packet_source_t *ps) {
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_t *dev;

	assert(ps->state != PST_INVALID);	/* Embryonic or Ready */
	assert(ps->iface_type == IFACE_FILE);	/* Don't cook crap */

	dev = pcap_open_offline(IFNameBySource(ps), errbuf);
	if(dev == NULL) {
		fprintf(stderr, "[%s] ", errbuf);
		ps->state = PST_FINISHED;
		errno = EINVAL;
		return -1;
	}

	/* Get device type */
	ps->dlt = pcap_datalink(dev);

	if( 0 && apply_filter(dev, ps->custom_filter) ) {
		pcap_close(dev);

		errno = ENODEV;
		fprintf(stderr,
			"[Warning: Can't initialize filter!] ");
		return -1;
	} else {
		ps->iface.file.dev = dev;
		ps->state = PST_READY;
		return 0;
	}
}

int
init_packet_source_file(packet_source_t *ps) {

        if(ps->iface_type != IFACE_FILE) return -1;

	if(ps->iflags & IFLAG_INONLY) {
		printf("%s: Input-only feature not supported by PCAP file.\n",
			IFNameBySource(ps));
		errno = EPERM;
		return -1;
	}

	if(reopen_packet_source_file(ps))
		return -1;

	/* Complain about unknown devices. */
	switch(ps->dlt) {
#ifdef	DLT_LOOP
	case DLT_LOOP:
#endif	/* DLT_LOOP */
	case DLT_NULL:    /* Loopback */
	case DLT_RAW:     /* Some PPP implementations, etc. */
	case DLT_EN10MB:  /* Generic Ethernet-compatible */
	case DLT_PPP:     /* Point-to point interface */
#ifdef	DLT_C_HDLC
	case DLT_C_HDLC:  /* BSD/OS Cisco HDLC */
#endif
	case DLT_IEEE802: /* Token Ring */
#ifdef	DLT_LINUX_SLL
	case DLT_LINUX_SLL:	/* fake header for Linux cooked socket */
#endif
		break;
	default:
		fprintf(stderr, "[Unknown interface type] ");
		pcap_close(ps->iface.file.dev);
		ps->iface.file.dev = NULL;
		ps->state = PST_FINISHED;
		errno = EINVAL;
		return -1;
	};

	return 0;
}

static int
apply_filter(pcap_t *dev, char *filter) {
	struct bpf_program fp;

	if(filter == NULL)
		filter = "ip";

	if( pcap_compile(dev, &fp, filter, 1, -1) )
		return -1;

	if( pcap_setfilter(dev, &fp) )
		return -1;

	return 0;
}



void
print_stats_file(FILE *f, packet_source_t *ps) {
	assert(ps->iface_type == IFACE_FILE);

	if(ps->state == PST_FINISHED) {
		fprintf(f,"Interface %s: FINISHED\n", IFNameBySource(ps));
	} else {
		fprintf(f, "Interface %s: IN-PROGRESS\n", IFNameBySource(ps));
	}
}
