; ABNF (Augmented Backus-Naur Form) Library
;
; Copyright (C) 2024 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (www.alessandrocoglio.info)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ABNF")

(include-book "../grammar-parser/executable")
(include-book "../grammar-printer/executable")
(include-book "../notation/syntax-abstraction")
(include-book "../operations/closure")
(include-book "../operations/well-formedness")

(include-book "kestrel/utilities/er-soft-plus" :dir :system)
(include-book "kestrel/std/system/constant-namep" :dir :system)
(include-book "kestrel/std/system/constant-value" :dir :system)
(include-book "kestrel/std/system/table-alist-plus" :dir :system)
(include-book "kestrel/std/util/error-value-tuples" :dir :system)
(include-book "kestrel/utilities/true-list-listp-theorems" :dir :system)
(include-book "std/alists/assoc" :dir :system)
(include-book "std/typed-alists/string-symbol-alistp" :dir :system)
(include-book "std/typed-alists/string-symbollist-alistp" :dir :system)

(local (include-book "kestrel/std/system/partition-rest-and-keyword-args" :dir :system))
(local (include-book "std/lists/len" :dir :system))
(local (include-book "std/typed-alists/symbol-alistp" :dir :system))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(xdoc::evmac-topic-implementation

 deftreeops

 :items

 (xdoc::*evmac-topic-implementation-item-state*

  xdoc::*evmac-topic-implementation-item-wrld*

  xdoc::*evmac-topic-implementation-item-ctx*

  (xdoc::evmac-topic-implementation-item-input "grammar")

  (xdoc::evmac-topic-implementation-item-input "prefix"))

 :additional

 ((xdoc::p
   "The generation of the functions and theorems happens in two passes:")
  (xdoc::ol
   (xdoc::li
    "In the first pass, we go through all the rules of the grammar
     and generate a @(tsee deftreeops-rulename-info-alist),
     which mainly contains information about
     the names of the functions and theorems to be generated,
     along with some additional information.")
   (xdoc::li
    "In the second pass, we go through the alist built in the first pass,
     and we generate all the events for the functions and theorems.
     The names and additional information in the alist
     provide the means for the events to reference each other as needed.
     Care is taken to generate the events so that
     there are no forward references.
     The generated events are returned as separate sequences,
     in order to put analogous events next to each other."))
  (xdoc::p
   "The alist of information about rule names is also stored
    in the @(tsee deftreeops) table,
    via an event generated along with the functions and theorems.
    This way, the information about the generated functions and theorems
    can be easily accessed, interactively or programmatically.")
  (xdoc::p
   "In the documentation below,
    we say `defining alternation of a rule name'
    instead of `defining alternation of a rule'.
    In ABNF, a "
   (xdoc::seetopic "well-formedness" "well-formed grammar")
   " may have more than one rule for the same rule name,
    with the rules after the first one being incremental ones.
    Thus, by `defining alternation of a rule name' we mean
    the alternation consisting of all the concatenations in
    the alternations in the rules for the rule name,
    in the order in which they appear in the grammar,
    as returned by @(tsee lookup-rulename).")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ deftreeops-info
  :parents (deftreeops-implementation)
  :short "Information about the events generated by @(tsee deftreeops)."
  :long
  (xdoc::topstring
   (xdoc::p
    "As discussed in @(see deftreeops-implementation),
     this is generated in the first pass."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defprod deftreeops-rep-info
  :short "Fixtype of @(tsee deftreeops) information about
          a repetition in a concatenation
          in the defining alternation of a rule name."
  :long
  (xdoc::topstring
   (xdoc::p
    "This information consists of:")
   (xdoc::ul
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-conc<i>-rep<j>-matching') theorem
      described in @(tsee deftreeops).
      This is @('nil') if the theorem is not generated,
      i.e. if the repetition does not have range 1
      or the concatenation of which the repetition is part
      is not a singleton.")))
  ((matching-thm acl2::symbol))
  :pred deftreeops-rep-infop)

;;;;;;;;;;;;;;;;;;;;

(fty::deflist deftreeops-rep-info-list
  :short "Fixtype of @(tsee deftreeops) information about
          the list of repetitions that form a concatenation
          in the defining alternation of a rule name."
  :elt-type deftreeops-rep-info
  :true-listp t
  :elementp-of-nil nil
  :pred deftreeops-rep-info-listp)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defprod deftreeops-conc-info
  :short "Fixtype of @(tsee deftreeops) information about a concatenation
          in the defining alternation of a rule name."
  :long
  (xdoc::topstring
   (xdoc::p
    "This information consists of:")
   (xdoc::ul
    (xdoc::li
     "The concatenation.")
    (xdoc::li
     "The discriminant term used in
      the @('<prefix>-<rulename>-conc-equivs') theorem
      described in @(tsee deftreeops).
      This is @('nil') if the rule name is defined by
      an alternation of just one concatenation.")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-conc<i>-matching') theorem
      described in @(tsee deftreeops).
      This is @('nil') if the theorem is not generated,
      i.e. if the concatenation is not a singleton.")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-conc?-<i>-iff-match-conc') theorem
      described in @(tsee deftreeops).
      This is @('nil') if the theorem is not generated,
      i.e. if the rule name is defined by
      an alternation of just one concatenation.")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-conc')
      or @('<prefix>-<rulename>-conc<i>') function
      described in @(tsee deftreeops).")
    (xdoc::li
     "The information about the repetitions that form the concatenation.
      This is @('nil') if the concatenation is not a singleton,
      because in that case the matching theorems for the repetitions
      are not generated (see @(tsee deftreeops-rep-info)).")))
  ((conc concatenationp)
   (discriminant-term "A term.")
   (matching-thm acl2::symbol)
   (check-conc-fn-equiv-thm acl2::symbol)
   (get-tree-list-list-fn acl2::symbol)
   (rep-infos deftreeops-rep-info-list))
  :pred deftreeops-conc-infop)

;;;;;;;;;;;;;;;;;;;;

(fty::deflist deftreeops-conc-info-list
  :short "Fixtype of @(tsee deftreeops) information about
          the list of concatenations that form
          the defining alternation of a rule name."
  :elt-type deftreeops-conc-info
  :true-listp t
  :elementp-of-nil nil
  :pred deftreeops-conc-info-listp)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defprod deftreeops-rulename-info
  :short "Fixtype of @(tsee deftreeops) information about a rule name."
  :long
  (xdoc::topstring
   (xdoc::p
    "This information consists of:")
   (xdoc::ul
    (xdoc::li
     "The alternation that defines the rule name.")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-nonleaf') theorem
      described in @(tsee deftreeops).")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-rulename') theorem
      described in @(tsee deftreeops).")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-branches-match-alt') theorem
      described in @(tsee deftreeops).")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-concs') theorem
      described in @(tsee deftreeops).")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-conc-equivs') theorem
      described in @(tsee deftreeops).
      This is @('nil') if the theorem is not generated,
      i.e. if some concatenation in the alternation that defines the rule name
      is not a singleton concatenation
      consisting of a singleton repetition of a rulename element.")
    (xdoc::li
     "The name of the @('<prefix>-<rulename>-conc?') function
      described in @(tsee deftreeops).
      This is @('nil') if the function is not generated,
      i.e. if the rule name is defined by
      an alternation of just one concatenation.")
    (xdoc::li
     "The information about the concatenations that form
      the alternation that defines the rule name.")))
  ((alt alternationp)
   (nonleaf-thm acl2::symbol)
   (rulename-thm acl2::symbol)
   (match-thm acl2::symbol)
   (concs-thm acl2::symbol)
   (conc-equivs-thm acl2::symbol)
   (check-conc-fn acl2::symbol)
   (conc-infos deftreeops-conc-info-list))
  :pred deftreeops-rulename-infop)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defalist deftreeops-rulename-info-alist
  :short "Fixtype of alists from rule names to information about rule names."
  :key-type rulename
  :val-type deftreeops-rulename-info
  :true-listp t
  :keyp-of-nil nil
  :valp-of-nil nil
  :pred deftreeops-rulename-info-alistp
  ///

  (defrule deftreeops-rulename-infop-when-deftreeops-rulename-info-alistp
    (implies (deftreeops-rulename-info-alistp alist)
             (iff (deftreeops-rulename-infop (cdr (assoc-equal key alist)))
                  (cdr (assoc-equal key alist))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ deftreeops-table
  :parents (deftreeops-implementation)
  :short "Table of @(tsee deftreeops) calls."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is for detecting redundant calls,
     and for storing the information about the rule names."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defprod deftreeops-table-value
  :short "Fixtype of values of the table of @(tsee deftreeops) calls."
  :long
  (xdoc::topstring
   (xdoc::p
    "This consists of the alist from rule names to rule name information.
     We put it into a one-component product type for future extensibility,
     and also so that we can define the option type based on this."))
  ((rulename-info-alist deftreeops-rulename-info-alist))
  :pred deftreeops-table-valuep)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defoption deftreeops-table-value-option
  deftreeops-table-value
  :short "Fixtype of optional values of the table of @(tsee deftreeops) calls."
  :pred deftreeops-table-value-optionp)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection deftreeops-table-definition
  :short "Definition of the table of @(tsee deftreeops) calls."
  :long
  (xdoc::topstring
   (xdoc::p
    "We use the calls themselves as keys."))

  (table deftreeops-table nil nil
    :guard (and (pseudo-event-formp acl2::key)
                (deftreeops-table-valuep acl2::val))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-table-lookup ((call pseudo-event-formp) (wrld plist-worldp))
  :returns (info? deftreeops-table-value-optionp)
  :short "Look up a @(tsee deftreeops) call in the table."
  :long
  (xdoc::topstring
   (xdoc::p
    "Returns a boolean, saying whether the call is in the table or not."))
  (b* ((info? (cdr (assoc-equal call (table-alist+ 'deftreeops-table wrld)))))
    (and (deftreeops-table-valuep info?)
         info?)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-table-add ((call pseudo-event-formp)
                              (info deftreeops-table-valuep))
  :returns (event pseudo-event-formp)
  :short "Event to record a @(tsee deftreeops) call in the table."
  `(table deftreeops-table ',call ',info))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(xdoc::evmac-topic-input-processing deftreeops)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-process-grammar (grammar (wrld plist-worldp))
  :returns (mv erp
               (grammar acl2::symbolp)
               (rules rulelistp))
  :short "Process the @('*grammar*') input."
  (b* (((reterr) nil nil)
       ((unless (constant-namep grammar wrld))
        (reterr (msg "The *GRAMMAR* input ~x0 must be the name of a constant."
                     grammar)))
       (rules (constant-value grammar wrld))
       ((unless (and (rulelistp rules)
                     (consp rules)))
        (reterr (msg "The *GRAMMAR* input is the name of a constant, ~
                      but its value ~x0 is not a non-empty ABNF grammar."
                     rules)))
       ((unless (rulelist-wfp rules))
        (reterr (msg "The *GRAMMAR* input denotes and ABNF grammar, ~
                      but the grammar is not well-formed
                      (see :DOC ABNF::WELL-FORMEDNESS).")))
       ((unless (rulelist-closedp rules))
        (reterr (msg "The *GRAMMAR* input denotes an ABNF grammar, ~
                      but the grammar is not closed
                      (see :DOC ABNF::CLOSURE)."))))
    (retok grammar rules)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-process-prefix (prefix)
  :returns (mv erp (prefix acl2::symbolp))
  :short "Process the @(':prefix') input."
  (b* (((reterr) nil)
       ((unless (acl2::symbolp prefix))
        (reterr (msg "The :PREFIX input ~x0 must be a symbol." prefix))))
    (retok prefix)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defval *deftreeops-allowed-options*
  :short "Keyword options accepted by @(tsee deftreeops)."
  (list :prefix)
  ///
  (assert-event (keyword-listp *deftreeops-allowed-options*))
  (assert-event (no-duplicatesp-eq *deftreeops-allowed-options*)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-process-inputs ((args true-listp) (wrld plist-worldp))
  :returns (mv erp
               (grammar acl2::symbolp)
               (rules rulelistp)
               (prefix acl2::symbolp))
  :short "Process all the inputs."
  (b* (((reterr) nil nil nil)
       ((mv erp grammar options)
        (partition-rest-and-keyword-args args *deftreeops-allowed-options*))
       ((when (or erp
                  (not (consp grammar))
                  (not (endp (cdr grammar)))))
        (reterr (msg "The inputs must be the constant name for the grammar ~
                      followed by the options ~&0."
                     *deftreeops-allowed-options*)))
       (grammar (car grammar))
       ((erp grammar rules) (deftreeops-process-grammar grammar wrld))
       (prefix-option (assoc-eq :prefix options))
       ((unless (consp prefix-option))
        (reterr (msg "The :PREFIX input must be supplied.")))
       (prefix (cdr prefix-option))
       ((erp prefix) (deftreeops-process-prefix prefix)))
    (retok grammar rules prefix))
  :guard-hints (("Goal" :in-theory (enable acl2::alistp-when-symbol-alistp))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(xdoc::evmac-topic-event-generation deftreeops)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-match-pred ((prefix acl2::symbolp))
  :returns (pred acl2::symbolp)
  :short "Name of the @('<prefix>-matchp') predicate."
  (add-suffix-to-fn prefix "-MATCHP"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-elem-match-pred ((prefix acl2::symbolp))
  :returns (pred acl2::symbolp)
  :short "Name of the @('<prefix>-list-elem-matchp') predicate."
  (add-suffix-to-fn prefix "-LIST-ELEM-MATCHP"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-rep-match-pred ((prefix acl2::symbolp))
  :returns (pred acl2::symbolp)
  :short "Name of the @('<prefix>-list-rep-matchp') predicate."
  (add-suffix-to-fn prefix "-LIST-REP-MATCHP"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-conc-match-pred ((prefix acl2::symbolp))
  :returns (pred acl2::symbolp)
  :short "Name of the @('<prefix>-list-list-conc-matchp') predicate."
  (add-suffix-to-fn prefix "-LIST-LIST-CONC-MATCHP"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-alt-match-pred ((prefix acl2::symbolp))
  :returns (pred acl2::symbolp)
  :short "Name of the @('<prefix>-list-list-alt-matchp') predicate."
  (add-suffix-to-fn prefix "-LIST-LIST-ALT-MATCHP"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-cst-match ((grammar acl2::symbolp)
                                  (prefix acl2::symbolp))
  :returns (events pseudo-event-form-listp)
  :short "Generate the first of the specialized matching predicates."
  (b* ((cst-matchp (deftreeops-match-pred prefix))
       (cst-matchp$ (add-suffix-to-fn cst-matchp "$")))
    `((define ,cst-matchp$ ((tree treep) (elem elementp))
        :returns (yes/no booleanp)
        (and (tree-terminatedp tree)
             (tree-match-element-p tree elem ,grammar))
        :hooks (:fix))
      (defmacro ,cst-matchp (tree elem)
        (declare (xargs :guard (acl2::stringp elem)))
        (b* (((mv err elem rest)
              (parse-element (string=>nats elem)))
             ((when err) (er hard ',cst-matchp "~@0" err))
             ((when (consp rest))
              (er hard ',cst-matchp
                  "Extra: ~s0" (nats=>string rest)))
             (elem (abstract-element elem)))
          `(,',cst-matchp$ ,tree ',elem)))
      (table acl2::macro-aliases-table
        ',cst-matchp
        ',cst-matchp$))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-cst-list-elem-match ((grammar acl2::symbolp)
                                            (prefix acl2::symbolp))
  :returns (events pseudo-event-form-listp)
  :short "Generate the second of the specialized matching predicates."
  (b* ((cst-list-elem-matchp (deftreeops-elem-match-pred prefix))
       (cst-list-elem-matchp$ (add-suffix-to-fn cst-list-elem-matchp "$")))
    `((define ,cst-list-elem-matchp$ ((trees tree-listp) (elem elementp))
        :returns (yes/no booleanp)
        (and (tree-list-terminatedp trees)
             (tree-list-match-element-p trees elem ,grammar))
        :hooks (:fix))
      (defmacro ,cst-list-elem-matchp (trees elem)
        (declare (xargs :guard (acl2::stringp elem)))
        (b* (((mv err elem rest)
              (parse-element (string=>nats elem)))
             ((when err) (er hard ',cst-list-elem-matchp "~@0" err))
             ((when (consp rest))
              (er hard ',cst-list-elem-matchp
                  "Extra: ~s0" (nats=>string rest)))
             (elem (abstract-element elem)))
          `(,',cst-list-elem-matchp$ ,trees ',elem)))
      (table acl2::macro-aliases-table
        ',cst-list-elem-matchp
        ',cst-list-elem-matchp$))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-cst-list-rep-match ((grammar acl2::symbolp)
                                           (prefix acl2::symbolp))
  :returns (events pseudo-event-form-listp)
  :short "Generate the third of the specialized matching predicates."
  (b* ((cst-list-rep-matchp (deftreeops-rep-match-pred prefix))
       (cst-list-rep-matchp$ (add-suffix-to-fn cst-list-rep-matchp "$")))
    `((define ,cst-list-rep-matchp$ ((trees tree-listp) (rep repetitionp))
        :returns (yes/no booleanp)
        (and (tree-list-terminatedp trees)
             (tree-list-match-repetition-p trees rep ,grammar))
        :hooks (:fix))
      (defmacro ,cst-list-rep-matchp (trees rep)
        (declare (xargs :guard (acl2::stringp rep)))
        (b* (((mv err rep rest)
              (parse-repetition (string=>nats rep)))
             ((when err) (er hard ',cst-list-rep-matchp "~@0" err))
             ((when (consp rest))
              (er hard ',cst-list-rep-matchp
                  "Extra: ~s0" (nats=>string rest)))
             (rep (abstract-repetition rep)))
          `(,',cst-list-rep-matchp$ ,trees ',rep)))
      (table acl2::macro-aliases-table
        ',cst-list-rep-matchp
        ',cst-list-rep-matchp$))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-cst-list-list-conc-match ((grammar acl2::symbolp)
                                                 (prefix acl2::symbolp))
  :returns (events pseudo-event-form-listp)
  :short "Generate the fourth of the specialized matching predicates."
  (b* ((cst-list-list-conc-matchp (deftreeops-conc-match-pred prefix))
       (cst-list-list-conc-matchp$
        (add-suffix-to-fn cst-list-list-conc-matchp "$")))
    `((define ,cst-list-list-conc-matchp$ ((treess tree-list-listp)
                                           (conc concatenationp))
        :returns (yes/no booleanp)
        (and (tree-list-list-terminatedp treess)
             (tree-list-list-match-concatenation-p treess conc ,grammar))
        :hooks (:fix))
      (defmacro ,cst-list-list-conc-matchp (treess conc)
        (declare (xargs :guard (acl2::stringp conc)))
        (b* (((mv err conc rest)
              (parse-concatenation (string=>nats conc)))
             ((when err) (er hard ',cst-list-list-conc-matchp "~@0" err))
             ((when (consp rest))
              (er hard ',cst-list-list-conc-matchp
                  "Extra: ~s0" (nats=>string rest)))
             (conc (abstract-concatenation conc)))
          `(,',cst-list-list-conc-matchp$ ,treess ',conc)))
      (table acl2::macro-aliases-table
        ',cst-list-list-conc-matchp
        ',cst-list-list-conc-matchp$))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-cst-list-list-alt-match ((grammar acl2::symbolp)
                                                (prefix acl2::symbolp))
  :returns (events pseudo-event-form-listp)
  :short "Generate the fifth of the specialized matching predicates."
  (b* ((cst-list-list-alt-matchp (deftreeops-alt-match-pred prefix))
       (cst-list-list-alt-matchp$
        (add-suffix-to-fn cst-list-list-alt-matchp "$")))
    `((define ,cst-list-list-alt-matchp$ ((treess tree-list-listp)
                                          (alt alternationp))
        :returns (yes/no booleanp)
        (and (tree-list-list-terminatedp treess)
             (tree-list-list-match-alternation-p treess alt ,grammar))
        :hooks (:fix))
      (defmacro ,cst-list-list-alt-matchp (treess alt)
        (declare (xargs :guard (acl2::stringp alt)))
        (b* (((mv err alt rest)
              (parse-alternation (string=>nats alt)))
             ((when err) (er hard ',cst-list-list-alt-matchp "~@0" err))
             ((when (consp rest))
              (er hard ',cst-list-list-alt-matchp
                  "Extra: ~s0" (nats=>string rest)))
             (alt (abstract-alternation alt)))
          `(,',cst-list-list-alt-matchp$ ,treess ',alt)))
      (table acl2::macro-aliases-table
        ',cst-list-list-alt-matchp
        ',cst-list-list-alt-matchp$))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-matchers ((grammar acl2::symbolp)
                                 (prefix acl2::symbolp))
  :returns (events pseudo-event-form-listp)
  :short "Generate the specialized matching predicates."
  (append (deftreeops-gen-cst-match grammar prefix)
          (deftreeops-gen-cst-list-elem-match grammar prefix)
          (deftreeops-gen-cst-list-rep-match grammar prefix)
          (deftreeops-gen-cst-list-list-conc-match grammar prefix)
          (deftreeops-gen-cst-list-list-alt-match grammar prefix)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-discriminant-terms ((alt alternationp))
  :returns (mv (okp booleanp) (terms pseudo-term-listp))
  :short "Generate the terms to discriminate among
          two or more concatenations that form
          the alternation that defines a rule name."
  :long
  (xdoc::topstring
   (xdoc::p
    "These are the terms used in
     the @('<prefix>-<rulename>-conc-equivs') theorem
     described in @(tsee deftreeops).")
   (xdoc::p
    "For now we only support alternations of certain forms.
     The first result of this function returns
     @('t') if terms are generated, @('nil') otherwise.
     The second result is the list of terms,
     of the same length as the alternation,
     or @('nil') if the first result is @('nil').
     If the alternation consists of just one concatenation,
     we return a single term @('t'),
     which makes sense since the concatenation must be always that only one.")
   (xdoc::p
    "For now we only support alternations
     each of whose concatenations are singletons
     each consisting of a repetition with range 1
     whose element is a rule name."))
  (b* (((when (and (consp alt)
                   (endp (cdr alt))))
        (mv t (list acl2::*t*))))
    (deftreeops-gen-discriminant-terms-aux alt))

  :prepwork
  ((define deftreeops-gen-discriminant-terms-aux ((alt alternationp))
     :returns (mv (okp booleanp) (terms pseudo-term-listp))
     :parents nil
     (b* (((when (endp alt)) (mv t nil))
          (conc (car alt))
          ((unless (and (consp conc)
                        (endp (cdr conc))))
           (mv nil nil))
          (rep (car conc))
          ((unless (equal (repetition->range rep)
                          (make-repeat-range :min 1
                                             :max (nati-finite 1))))
           (mv nil nil))
          (elem (repetition->element rep))
          ((unless (element-case elem :rulename))
           (mv nil nil))
          (rulename (element-rulename->get elem))
          (term `(equal (tree-nonleaf->rulename?
                         (nth '0 (nth '0 (tree-nonleaf->branches cst))))
                        ',rulename))
          ((mv okp terms) (deftreeops-gen-discriminant-terms-aux (cdr alt)))
          ((unless okp) (mv nil nil)))
       (mv t (cons term terms)))
     ///
     (defret len-of-deftreeops-gen-discriminant-terms-aux
       (implies okp
                (equal (len terms)
                       (len alt))))))

  ///

  (defret len-of-deftreeops-gen-discriminant-terms
    (implies okp
             (equal (len terms)
                    (len alt)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-rep-info
  ((rep repetitionp)
   (i posp "Indentifies the concatenation that this repetition is part of,
            starting from 1.")
   (rulename-upstring acl2::stringp "Rule name normalized in uppercase.")
   (prefix acl2::symbolp))
  :returns (info deftreeops-rep-infop)
  :short "Generate the information for
          a repetition in a concatenation in
          the alternation that defines a rule name."
  (b* ((matching-thm
        (and (equal (repetition->range rep)
                    (make-repeat-range :min 1 :max (nati-finite 1)))
             (packn-pos (list prefix
                              '-
                              rulename-upstring
                              '-conc
                              i
                              '-rep1-matching)
                        prefix))))
    (make-deftreeops-rep-info :matching-thm matching-thm)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-rep-info-list ((conc concatenationp)
                                      (i posp)
                                      (rulename-upstring acl2::stringp)
                                      (prefix acl2::symbolp))
  :returns (infos deftreeops-rep-info-listp)
  :short "Lift @(tsee deftreeops-gen-rep-info) to
          lists of repetitions, i.e. to concatenations."
  (b* (((when (endp conc)) nil)
       (info (deftreeops-gen-rep-info
               (car conc) i rulename-upstring prefix))
       (more-info (deftreeops-gen-rep-info-list
                    (cdr conc) i rulename-upstring prefix)))
    (cons info more-info)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-conc-info
  ((conc concatenationp)
   (i posp "Indentifies the concatenation, starting from 1.")
   (discriminant-term "The discriminant term for the concatenation.")
   (check-conc-fn acl2::symbolp
                  "The @('check-conc-fn') component of
                   @(tsee deftreeops-rulename-info).")
   (alt-singletonp booleanp "Whether the alternation that defines the rule name
                             consists of a single concatenation or not.")
   (rulename-upstring acl2::stringp "Rule name normalized in uppercase.")
   (prefix acl2::symbolp))
  :returns (info deftreeops-conc-infop)
  :short "Generate the information for a concatenation in
          the alternation that defines a rule name."
  (b* ((conc-singletonp (and (consp conc)
                             (not (consp (cdr conc)))))
       (matching-thm
        (and conc-singletonp
             (packn-pos (list prefix '- rulename-upstring '-conc i '-matching)
                        prefix)))
       (check-conc-fn-equiv-thm
        (and check-conc-fn
             (packn-pos (list check-conc-fn '- i '-iff-match-conc)
                        check-conc-fn)))
       (get-tree-list-list-fn
        (if alt-singletonp
            (packn-pos (list prefix '- rulename-upstring '-conc)
                       prefix)
          (and check-conc-fn
               (packn-pos (list prefix '- rulename-upstring '-conc i)
                          prefix))))
       (rep-infos
        (and conc-singletonp
             (deftreeops-gen-rep-info-list conc i rulename-upstring prefix)))
       (info (make-deftreeops-conc-info
              :conc conc
              :discriminant-term discriminant-term
              :matching-thm matching-thm
              :check-conc-fn-equiv-thm check-conc-fn-equiv-thm
              :get-tree-list-list-fn get-tree-list-list-fn
              :rep-infos rep-infos)))
    info))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-conc-info-list
  ((alt alternationp)
   (discriminant-terms "A list of terms.")
   (check-conc-fn acl2::symbolp
                  "The @('check-conc-fn') component of
                   @(tsee deftreeops-rulename-info).")
   (alt-singletonp booleanp "Whether the alternation that defines the rule name
                             consists of a single concatenation or not.")
   (rulename-upstring acl2::stringp)
   (prefix acl2::symbolp))
  :guard (equal (len discriminant-terms) (len alt))
  :returns (infos deftreeops-conc-info-listp)
  :short "Lift @(tsee deftreeops-gen-conc-info)
          to lists of concatenations, i.e. to alternations."
  (deftreeops-gen-conc-info-list-aux
    alt 1 discriminant-terms check-conc-fn alt-singletonp
    rulename-upstring prefix)

  :prepwork
  ((define deftreeops-gen-conc-info-list-aux
     ((alt alternationp)
      (i posp)
      (discriminant-terms "A list of terms.")
      (check-conc-fn acl2::symbolp)
      (alt-singletonp booleanp)
      (rulename-upstring acl2::stringp)
      (prefix acl2::symbolp))
     :guard (equal (len discriminant-terms) (len alt))
     :returns (infos deftreeops-conc-info-listp)
     :parents nil
     (b* (((when (endp alt)) nil)
          (info
           (deftreeops-gen-conc-info
             (car alt)
             i
             (car discriminant-terms)
             check-conc-fn
             alt-singletonp
             rulename-upstring
             prefix))
          (more-info
           (deftreeops-gen-conc-info-list-aux
             (cdr alt)
             (1+ i)
             (cdr discriminant-terms)
             check-conc-fn
             alt-singletonp
             rulename-upstring
             prefix)))
       (cons info more-info)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-rulename-info
  ((rulename rulenamep)
   (alt alternationp "The alternation that define @('rulename').")
   (prefix acl2::symbolp))
  :returns (info deftreeops-rulename-infop)
  :short "Generate the information for a rule name."
  (b* ((rulename-string (rulename->get rulename))
       (rulename-upstring (str::upcase-string rulename-string))
       (nonleaf-thm
        (packn-pos (list prefix '- rulename-upstring '-nonleaf)
                   prefix))
       (rulename-thm
        (packn-pos (list prefix '- rulename-upstring '-rulename)
                   prefix))
       (match-thm
        (packn-pos (list prefix '- rulename-upstring '-branches-match-alt)
                   prefix))
       (concs-thm
        (packn-pos (list prefix '- rulename-upstring '-concs)
                   prefix))
       ((mv okp terms) (deftreeops-gen-discriminant-terms alt))
       (terms (if okp terms (repeat (len alt) nil)))
       (alt-singletonp (and (consp alt)
                            (endp (cdr alt))))
       (conc-equivs-thm
        (and (not alt-singletonp)
             okp
             (packn-pos (list prefix '- rulename-upstring '-conc-equivs)
                        prefix)))
       (check-conc-fn
        (and (not alt-singletonp)
             okp
             (packn-pos (list prefix '- rulename-upstring '-conc?)
                        prefix)))
       (conc-infos (deftreeops-gen-conc-info-list
                     alt terms check-conc-fn alt-singletonp
                     rulename-upstring prefix))
       (info (make-deftreeops-rulename-info
              :alt alt
              :nonleaf-thm nonleaf-thm
              :rulename-thm rulename-thm
              :match-thm match-thm
              :concs-thm concs-thm
              :conc-equivs-thm conc-equivs-thm
              :check-conc-fn check-conc-fn
              :conc-infos conc-infos)))
    info))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-rulename-info-alist
  ((rules rulelistp)
   (prefix acl2::symbolp))
  :returns (info deftreeops-rulename-info-alistp)
  :short "Generate the alist from rule names to rule name information,
          from a list of rules."
  :long
  (xdoc::topstring
   (xdoc::p
    "Note that we generate an alist entry for each rule name, not rule:
     each rule name may be defined by multiple rules
     (the ones after the first incremental ones, in well-formed grammars),
     so we keep track of which rule names we have encountered already,
     so we can skip them when encountered again;
     when encountered the first time,
     we obtain its defining alternation (via @(tsee lookup-rulename))
     and use it to generate the alist entry.")
   (xdoc::p
    "The generated alist has unique keys."))
  (deftreeops-gen-rulename-info-alist-aux rules nil prefix)

  :prepwork
  ((define deftreeops-gen-rulename-info-alist-aux
     ((rules rulelistp)
      (done rulename-listp)
      (prefix acl2::symbolp))
     :returns (info deftreeops-rulename-info-alistp)
     :parents nil
     (b* (((when (endp rules)) nil)
          (rule (car rules))
          (rulename (rule->name rule))
          ((when (member-equal rulename done))
           (deftreeops-gen-rulename-info-alist-aux (cdr rules) done prefix))
          (alt (lookup-rulename rulename rules))
          (info (deftreeops-gen-rulename-info rulename alt prefix))
          (more-info (deftreeops-gen-rulename-info-alist-aux
                       (cdr rules) (cons rulename done) prefix)))
       (acons rulename info more-info))
     :verify-guards :after-returns)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-rep-events ((rep repetitionp)
                                   (info deftreeops-rep-infop)
                                   (prefix acl2::symbolp))
  :returns (events pseudo-event-form-listp)
  :short "Generate the functions and theorems for
          a repetition in a concatenation in
          the alternation that defines a rule name."
  (b* (((deftreeops-rep-info info) info)
       (matchp (deftreeops-match-pred prefix))
       (rep-matchp (deftreeops-rep-match-pred prefix))
       (elem (repetition->element rep))
       (matching-thm-event?
        (and info.matching-thm
             `((defruled ,info.matching-thm
                 (implies (,rep-matchp csts
                                       ,(pretty-print-repetition rep))
                          (and (equal (len csts) 1)
                               (,matchp (nth 0 csts)
                                        ,(pretty-print-element elem))))
                 :in-theory
                 '(,rep-matchp
                   ,matchp
                   tree-list-match-repetition-p-of-1-repetition
                   tree-terminatedp-of-car-when-tree-list-terminatedp
                   (:e nati-finite)
                   (:e repeat-range)
                   (:e repetition->element)
                   (:e repetition->range)
                   nth
                   (:e zp)
                   len))))))
    matching-thm-event?))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-rep-list-events ((conc concatenationp)
                                        (infos deftreeops-rep-info-listp)
                                        (prefix acl2::symbolp))
  :guard (equal (len infos) (len conc))
  :returns (events pseudo-event-form-listp)
  :short "Lift @(tsee deftreeops-gen-rep-events) to lists."
  (b* (((when (endp conc)) nil)
       (events (deftreeops-gen-rep-events (car conc) (car infos) prefix))
       (more-events
        (deftreeops-gen-rep-list-events (cdr conc) (cdr infos) prefix)))
    (append events more-events)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-conc-events ((conc concatenationp)
                                    (info deftreeops-conc-infop)
                                    (i posp)
                                    (conc-equivs-thm acl2::symbolp)
                                    (check-conc-fn acl2::symbolp)
                                    (nonleaf-thm acl2::symbolp)
                                    (alt-match-thm acl2::symbolp)
                                    (alt-singletonp booleanp)
                                    (rulename rulenamep)
                                    (prefix acl2::symbolp))
  :returns (mv (matching-thm-event? pseudo-event-form-listp
                                    "A list of 0 or 1 elements.")
               (check-conc-fn-equiv-thm-event? pseudo-event-form-listp
                                               "A list of 0 or 1 elements.")
               (get-tree-list-list-fn-event pseudo-event-form-listp
                                            "A list of 0 or 1 elements.")
               (rep-events pseudo-event-form-listp))
  :short "Generate the functions and theorems for a concatenation in
          the alternation that defines a rule name."
  (b* (((deftreeops-conc-info info) info)
       (matchp (deftreeops-match-pred prefix))
       (alt-matchp (deftreeops-alt-match-pred prefix))
       (conc-matchp (deftreeops-conc-match-pred prefix))
       (rep-matchp (deftreeops-rep-match-pred prefix))
       (rulename-string (rulename->get rulename))
       (matching-thm-event?
        (and
         info.matching-thm
         (b* (((unless (consp conc))
               (raise "Internal error: empty concatenation."))
              (rep (car conc)))
           `((defruled ,info.matching-thm
               (implies (,conc-matchp cstss
                                      ,(pretty-print-concatenation conc))
                        (and (equal (len cstss) 1)
                             (,rep-matchp (nth 0 cstss)
                                          ,(pretty-print-repetition rep))))
               :in-theory
               '(,conc-matchp
                 ,rep-matchp
                 tree-list-list-match-concatenation-p-when-atom-concatenation
                 tree-list-list-match-concatenation-p-of-cons-concatenation
                 tree-list-terminatedp-of-car-when-tree-list-list-terminatedp
                 nth
                 (:e zp)
                 len))))))
       (check-conc-fn-equiv-thm-event?
        (and
         info.check-conc-fn-equiv-thm
         `((defruled ,info.check-conc-fn-equiv-thm
             (implies (,matchp cst ,rulename-string)
                      (iff (equal (,check-conc-fn cst) ,i)
                           (,conc-matchp
                            (tree-nonleaf->branches cst)
                            ,(pretty-print-concatenation info.conc))))
             :in-theory
             '(,(packn-pos (list check-conc-fn '-tree-equiv-congruence-on-cst)
                           check-conc-fn)
               ,(packn-pos (list matchp '$-tree-equiv-congruence-on-tree)
                           matchp)
               tree-nonleaf->branches$inline-tree-equiv-congruence-on-x
               return-type-of-tree-fix.new-x
               tree-fix-under-tree-equiv)
             :use (:instance lemma (cst (tree-fix cst)))
             :prep-lemmas
             ((defrule lemma
                (implies (and (treep cst)
                              (,matchp cst ,rulename-string))
                         (iff (equal (,check-conc-fn cst) ,i)
                              (,conc-matchp
                               (tree-nonleaf->branches cst)
                               ,(pretty-print-concatenation info.conc))))
                :in-theory '(,check-conc-fn
                             ,conc-equivs-thm
                             (:e rulename))
                :use (:guard-theorem ,check-conc-fn)))))))
       (get-tree-list-list-fn-event?
        (and
         (or alt-singletonp
             check-conc-fn)
         `((define ,info.get-tree-list-list-fn ((cst treep))
             :guard ,(if check-conc-fn
                         `(and (,matchp cst ,rulename-string)
                               (equal (,check-conc-fn cst) ,i))
                       `(,matchp cst ,rulename-string))
             :returns (cstss tree-list-listp
                             :hints
                             (("Goal"
                               :in-theory
                               '(,info.get-tree-list-list-fn
                                 tree-list-listp-of-tree-nonleaf->branches))))
             (tree-nonleaf->branches cst)
             :prepwork ((local (in-theory nil)))
             :guard-hints (("Goal" :in-theory '((:e elementp)
                                                ,nonleaf-thm)))
             ///
             (more-returns
              (cstss
               (,conc-matchp cstss
                             ,(pretty-print-concatenation info.conc))
               :hyp ,(if check-conc-fn
                         `(and (,matchp cst ,rulename-string)
                               (equal (,check-conc-fn cst) ,i))
                       `(,matchp cst ,rulename-string))
               :name ,(packn-pos (list info.get-tree-list-list-fn
                                       '-matchp)
                                 info.get-tree-list-list-fn)
               :hints
               ,(if check-conc-fn
                    `(("Goal"
                       :in-theory
                       '(,info.get-tree-list-list-fn
                         ,info.check-conc-fn-equiv-thm)))
                  `(("Goal"
                     :in-theory
                     '(,info.get-tree-list-list-fn
                       ,alt-matchp
                       ,conc-matchp
                       tree-list-list-match-alternation-p-when-atom-alternation
                       tree-list-list-match-alternation-p-of-cons-alternation)
                     :use ,alt-match-thm)))))
             (fty::deffixequiv ,info.get-tree-list-list-fn
               :hints
               (("Goal"
                 :in-theory
                 '(,info.get-tree-list-list-fn
                   tree-nonleaf->branches$inline-tree-equiv-congruence-on-x
                   tree-fix-under-tree-equiv))))))))
       (conc-singletonp (and (consp conc)
                             (not (consp (cdr conc)))))
       ((unless (or (not conc-singletonp)
                    (= (len info.rep-infos) (len conc))))
        (raise "Internal error: length of ~x0 differs from length of ~x1."
               info.rep-infos conc)
        (mv nil nil nil nil))
       (rep-events
        (and conc-singletonp
             (deftreeops-gen-rep-list-events conc info.rep-infos prefix))))
    (mv matching-thm-event?
        check-conc-fn-equiv-thm-event?
        get-tree-list-list-fn-event?
        rep-events)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-conc-list-events ((alt alternationp)
                                         (infos deftreeops-conc-info-listp)
                                         (conc-equivs-thm acl2::symbolp)
                                         (check-conc-fn acl2::symbolp)
                                         (nonleaf-thm acl2::symbolp)
                                         (alt-match-thm acl2::symbolp)
                                         (alt-singletonp booleanp)
                                         (rulename rulenamep)
                                         (prefix acl2::symbolp))
  :guard (equal (len infos) (len alt))
  :returns (mv (matching-thm-events pseudo-event-form-listp)
               (check-conc-fn-equiv-thm-events pseudo-event-form-listp)
               (get-tree-list-list-fn-events pseudo-event-form-listp)
               (rep-events pseudo-event-form-listp))
  :short "Lift @(tsee deftreeops-gen-conc-events) to lists."
  (deftreeops-gen-conc-list-events-aux
    alt infos 1
    conc-equivs-thm check-conc-fn nonleaf-thm alt-match-thm
    alt-singletonp rulename prefix)

  :prepwork
  ((define deftreeops-gen-conc-list-events-aux
     ((alt alternationp)
      (infos deftreeops-conc-info-listp)
      (i posp)
      (conc-equivs-thm acl2::symbolp)
      (check-conc-fn acl2::symbolp)
      (nonleaf-thm acl2::symbolp)
      (alt-match-thm acl2::symbolp)
      (alt-singletonp booleanp)
      (rulename rulenamep)
      (prefix acl2::symbolp))
     :guard (equal (len infos) (len alt))
     :returns (mv (matching-thm-events pseudo-event-form-listp)
                  (check-conc-fn-equiv-thm-events pseudo-event-form-listp)
                  (get-tree-list-list-fn-events pseudo-event-form-listp)
                  (rep-events pseudo-event-form-listp))
     :parents nil
     (b* (((when (endp alt)) (mv nil nil nil nil))
          ((mv matching-thm-event?
               check-conc-fn-equiv-thm-event?
               get-tree-list-list-fn-event?
               rep-events)
           (deftreeops-gen-conc-events
             (car alt) (car infos) i
             conc-equivs-thm check-conc-fn nonleaf-thm alt-match-thm
             alt-singletonp rulename prefix))
          ((mv more-matching-thm-events
               more-check-conc-fn-equiv-thm-events
               more-get-tree-list-list-fn-events
               more-rep-events)
           (deftreeops-gen-conc-list-events-aux
             (cdr alt) (cdr infos) (1+ i)
             conc-equivs-thm check-conc-fn nonleaf-thm alt-match-thm
             alt-singletonp rulename prefix)))
       (mv (append matching-thm-event?
                   more-matching-thm-events)
           (append check-conc-fn-equiv-thm-event?
                   more-check-conc-fn-equiv-thm-events)
           (append get-tree-list-list-fn-event?
                   more-get-tree-list-list-fn-events)
           (append rep-events
                   more-rep-events))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-rulename-events
  ((rulename rulenamep)
   (alt alternationp)
   (info deftreeops-rulename-infop)
   (prefix acl2::symbolp)
   (rulename-infos deftreeops-rulename-info-alistp
                   "Information about all the rule names."))
  :returns (mv (nonleaf-thm-event pseudo-event-formp)
               (rulename-thm-event pseudo-event-formp)
               (match-thm-event pseudo-event-formp)
               (concs-thm-event pseudo-event-formp)
               (conc-equivs-thm-event? pseudo-event-form-listp)
               (check-conc-fn-event? pseudo-event-form-listp)
               (get-tree-list-list-fn-events pseudo-event-form-listp)
               (conc-matching-thm-events pseudo-event-form-listp)
               (rep-matching-thm-events pseudo-event-form-listp))
  :short "Generate the events for a rule name."
  (b* (((deftreeops-rulename-info info) info)
       (rulename-string (rulename->get rulename))
       (alt-string (pretty-print-alternation alt))
       (matchp (deftreeops-match-pred prefix))
       (alt-matchp (deftreeops-alt-match-pred prefix))
       (conc-matchp (deftreeops-conc-match-pred prefix))
       ((unless (equal (len info.conc-infos) (len alt)))
        (raise "Internal error: ~x0 and ~x1 have different lengths."
               info.conc-infos alt)
        (mv '(_) '(_) '(_) '(_) nil nil nil nil nil))
       (alt-singletonp (and (consp alt)
                            (endp (cdr alt))))
       ((mv conc-matching-thm-events
            check-conc-fn-equiv-thm-events
            get-tree-list-list-fn-events
            rep-matching-thm-events)
        (deftreeops-gen-conc-list-events
          alt
          info.conc-infos
          info.conc-equivs-thm
          info.check-conc-fn
          info.nonleaf-thm
          info.match-thm
          alt-singletonp
          rulename
          prefix))
       (nonleaf-thm-event
        `(defruled ,info.nonleaf-thm
           (implies (,matchp cst ,rulename-string)
                    (equal (tree-kind cst) :nonleaf))
           :in-theory '(,matchp
                        tree-nonleaf-when-match-rulename/group/option
                        (:e element-kind)
                        (:e member-equal))))
       (rulename-thm-event
        `(defruled ,info.rulename-thm
           (implies (,matchp cst ,rulename-string)
                    (equal (tree-nonleaf->rulename? cst)
                           (rulename ,rulename-string)))
           :in-theory '(,matchp
                        tree-rulename-when-match-rulename
                        (:e element-kind)
                        (:e element-rulename->get)
                        (:e rulename))))
       (match-thm-event
        `(defruled ,info.match-thm
           (implies (,matchp cst ,rulename-string)
                    (,alt-matchp
                     (tree-nonleaf->branches cst) ,alt-string))
           :in-theory '(,matchp
                        ,alt-matchp
                        tree-branches-match-alt-when-match-rulename
                        tree-terminatedp
                        (:e element-kind)
                        (:e element-rulename->get)
                        (:e lookup-rulename))
           :use ,info.nonleaf-thm))
       (concs-thm-event
        `(defruled ,info.concs-thm
           (implies (,alt-matchp cstss ,alt-string)
                    (or ,@(deftreeops-gen-rulename-events-aux1
                            alt conc-matchp)))
           :do-not '(preprocess)
           :in-theory
           '(,alt-matchp
             ,conc-matchp
             tree-list-list-match-alternation-p-when-atom-alternation
             tree-list-list-match-alternation-p-of-cons-alternation)))
       (conc-equivs-thm-event?
        (and
         info.conc-equivs-thm
         (b* (((mv conjuncts rules lemma-instances)
               (deftreeops-gen-rulename-events-aux2
                 alt info.conc-infos rulename-infos conc-matchp)))
           `((defruled ,info.conc-equivs-thm
               (implies (,matchp cst ,rulename-string)
                        (and ,@conjuncts))
               :in-theory '((:e rulename)
                            ,info.match-thm
                            ,@rules)
               :use ((:instance ,info.concs-thm
                                (cstss (tree-nonleaf->branches cst)))
                     ,@lemma-instances))))))
       (check-conc-fn-event?
        (and
         info.check-conc-fn
         (b* (((mv cond-arms disjuncts rules)
               (deftreeops-gen-rulename-events-aux3
                 alt info.conc-infos 1 rulename-infos)))
           `((define ,info.check-conc-fn ((cst treep))
               :guard (,matchp cst ,rulename-string)
               :returns (number posp
                                :hints (("Goal" :in-theory '(,info.check-conc-fn
                                                             (:e posp)))))
               (cond ,@cond-arms
                     (t (prog2$ (acl2::impossible) 1)))
               :guard-hints
               (("Goal"
                 :in-theory
                 '(acl2::true-listp-of-nth-when-true-list-listp
                   (:t tree-nonleaf->branches)
                   abnf::true-list-listp-of-tree-nonleaf->branches
                   abnf::treep-of-nth-when-tree-listp
                   abnf::tree-listp-of-nth-when-tree-list-listp
                   abnf::tree-list-listp-of-tree-nonleaf->branches
                   (:e abnf::rulename)
                   (:e nfix)
                   ,info.nonleaf-thm
                   ,info.match-thm
                   ,@rules)
                 :use
                 ((:instance ,info.concs-thm
                             (cstss (tree-nonleaf->branches cst))))))
               ///
               (more-returns
                (number (or ,@disjuncts)
                        :name ,(packn-pos (list info.check-conc-fn
                                                '-possibilities)
                                          info.check-conc-fn)
                        :rule-classes
                        ((:forward-chaining
                          :trigger-terms ((,info.check-conc-fn cst))))
                        :hints (("Goal" :in-theory '(,info.check-conc-fn)))))
               (fty::deffixequiv ,info.check-conc-fn
                 :hints
                 (("Goal"
                   :in-theory
                   '(,info.check-conc-fn
                     tree-nonleaf->branches$inline-tree-equiv-congruence-on-x
                     tree-fix-under-tree-equiv))))
               ,@check-conc-fn-equiv-thm-events))))))
    (mv nonleaf-thm-event
        rulename-thm-event
        match-thm-event
        concs-thm-event
        conc-equivs-thm-event?
        check-conc-fn-event?
        get-tree-list-list-fn-events
        conc-matching-thm-events
        rep-matching-thm-events))

  :prepwork

  ((define deftreeops-gen-rulename-events-aux1 ((alt alternationp)
                                               (conc-matchp acl2::symbolp))
     :returns (disjuncts true-listp)
     :parents nil
     (cond ((endp alt) nil)
           (t (cons `(,conc-matchp cstss
                                   ,(pretty-print-concatenation (car alt)))
                    (deftreeops-gen-rulename-events-aux1
                      (cdr alt) conc-matchp)))))

   (define deftreeops-gen-rulename-events-aux2
     ((alt alternationp)
      (conc-infos deftreeops-conc-info-listp)
      (rulename-infos deftreeops-rulename-info-alistp)
      (conc-matchp acl2::symbolp))
     :guard (equal (len conc-infos) (len alt))
     :returns (mv (conjuncts true-listp)
                  (rules symbol-listp)
                  (lemma-instances true-listp))
     :parents nil
     (b* (((when (endp alt)) (mv nil nil nil))
          (conc (car alt))
          ((deftreeops-conc-info conc-info) (car conc-infos))
          ((unless (and (consp conc)
                        (endp (cdr conc))))
           (raise "Internal error: non-singleton concatenation ~x0." conc)
           (mv nil nil nil))
          (rep (car conc))
          ((unless (equal (repetition->range rep)
                          (make-repeat-range :min 1
                                             :max (nati-finite 1))))
           (raise "Internal error: non-singleton repetition ~x0." rep)
           (mv nil nil nil))
          (elem (repetition->element rep))
          ((unless (element-case elem :rulename))
           (raise "Internal error: element ~x0 is not a rule name." elem)
           (mv nil nil nil))
          (rulename (element-rulename->get elem))
          (rulename-string (rulename->get rulename))
          (conjunct
           `(iff (,conc-matchp (tree-nonleaf->branches cst) ,rulename-string)
                 ,conc-info.discriminant-term))
          ((unless (and (consp conc-info.rep-infos)
                        (endp (cdr conc-info.rep-infos))))
           (raise "Internal error:
                   non-singleton list of repetition information ~x0."
                  conc-info.rep-infos)
           (mv nil nil nil))
          ((deftreeops-rep-info rep-info) (car conc-info.rep-infos))
          (rulename-info (cdr (assoc-equal rulename rulename-infos)))
          ((unless rulename-info)
           (raise "Internal error: no information for rule name ~x0." rulename)
           (mv nil nil nil))
          (rules (list conc-info.matching-thm
                       rep-info.matching-thm))
          (lemma-instance
           `(:instance ,(deftreeops-rulename-info->rulename-thm rulename-info)
                       (cst (nth 0 (nth 0 (tree-nonleaf->branches cst))))))
          ((mv more-conjuncts more-rules more-lemma-instances)
           (deftreeops-gen-rulename-events-aux2
             (cdr alt) (cdr conc-infos) rulename-infos conc-matchp)))
       (mv (cons conjunct more-conjuncts)
           (append rules more-rules)
           (cons lemma-instance more-lemma-instances))))

   (define deftreeops-gen-rulename-events-aux3
     ((alt alternationp)
      (conc-infos deftreeops-conc-info-listp)
      (i posp)
      (rulename-infos deftreeops-rulename-info-alistp))
     :guard (equal (len conc-infos) (len alt))
     :returns (mv (cond-arms true-listp)
                  (disjuncts true-listp)
                  (rules symbol-listp))
     :parents nil
     (b* (((when (endp alt)) (mv nil nil nil))
          (conc (car alt))
          ((deftreeops-conc-info conc-info) (car conc-infos))
          ((unless (and (consp conc)
                        (endp (cdr conc))))
           (raise "Internal error: non-singleton concatenation ~x0." conc)
           (mv nil nil nil))
          (rep (car conc))
          ((unless (equal (repetition->range rep)
                          (make-repeat-range :min 1
                                             :max (nati-finite 1))))
           (raise "Internal error: non-singleton repetition ~x0." rep)
           (mv nil nil nil))
          (elem (repetition->element rep))
          ((unless (element-case elem :rulename))
           (raise "Internal error: element ~x0 is not a rule name." elem)
           (mv nil nil nil))
          (rulename (element-rulename->get elem))
          (cond-arm `(,conc-info.discriminant-term ,i))
          (disjunct `(equal number ,i))
          (rep-infos (deftreeops-conc-info->rep-infos conc-info))
          ((unless (and (consp rep-infos)
                        (endp (cdr rep-infos))))
           (raise "Internal error:
                   non-singleton list of repetition information ~x0."
                  rep-infos)
           (mv nil nil nil))
          ((deftreeops-rep-info rep-info) (car rep-infos))
          (rulename-info (cdr (assoc-equal rulename rulename-infos)))
          ((unless rulename-info)
           (raise "Internal error: no information for rule name ~x0." rulename)
           (mv nil nil nil))
          (rules (list (deftreeops-rulename-info->nonleaf-thm rulename-info)
                       (deftreeops-rulename-info->rulename-thm rulename-info)
                       conc-info.matching-thm
                       rep-info.matching-thm))
          ((mv more-cond-arms
               more-disjuncts
               more-rules)
           (deftreeops-gen-rulename-events-aux3
             (cdr alt) (cdr conc-infos) (1+ i) rulename-infos)))
       (mv (cons cond-arm more-cond-arms)
           (cons disjunct more-disjuncts)
           (append rules more-rules))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-rulename-alist-events
  ((rulename-infos deftreeops-rulename-info-alistp)
   (prefix acl2::symbolp))
  :returns (mv (nonleaf-thm-events pseudo-event-form-listp)
               (rulename-thm-events pseudo-event-form-listp)
               (match-thm-events pseudo-event-form-listp)
               (concs-thm-events pseudo-event-form-listp)
               (conc-equivs-thm-events pseudo-event-form-listp)
               (check-conc-fn-events pseudo-event-form-listp)
               (get-tree-list-list-fn-events pseudo-event-form-listp)
               (conc-matching-thm-events pseudo-event-form-listp)
               (rep-matching-thm-events pseudo-event-form-listp))
  :short "Generate the events for all the rule names in the alist."
  (deftreeops-gen-rulename-alist-events-aux
    rulename-infos prefix rulename-infos)

  :prepwork
  ((define deftreeops-gen-rulename-alist-events-aux
     ((rest-rulename-infos deftreeops-rulename-info-alistp)
      (prefix acl2::symbolp)
      (all-rulename-infos deftreeops-rulename-info-alistp))
     :returns (mv (nonleaf-thm-events pseudo-event-form-listp)
                  (rulename-thm-events pseudo-event-form-listp)
                  (match-thm-events pseudo-event-form-listp)
                  (concs-thm-events pseudo-event-form-listp)
                  (conc-equivs-thm-events pseudo-event-form-listp)
                  (check-conc-fn-events pseudo-event-form-listp)
                  (get-tree-list-list-fn-events pseudo-event-form-listp)
                  (conc-matching-thm-events pseudo-event-form-listp)
                  (rep-matching-thm-events pseudo-event-form-listp))
     :parents nil
     (b* (((when (endp rest-rulename-infos))
           (mv nil nil nil nil nil nil nil nil nil))
          ((cons rulename info) (car rest-rulename-infos))
          (alt (deftreeops-rulename-info->alt info))
          ((mv nonleaf-thm-event
               rulename-thm-event
               match-thm-event
               concs-thm-event
               conc-equivs-thm-event?
               check-conc-fn-event?
               get-tree-list-list-fn-events
               conc-matching-thm-events
               rep-matching-thm-events)
           (deftreeops-gen-rulename-events
             rulename alt info prefix all-rulename-infos))
          ((mv more-nonleaf-thm-events
               more-rulename-thm-events
               more-match-thm-events
               more-concs-thm-events
               more-conc-equivs-thm-events
               more-check-conc-fn-events
               more-get-tree-list-list-fn-events
               more-conc-matching-thm-events
               more-rep-matching-thm-events)
           (deftreeops-gen-rulename-alist-events-aux
             (cdr rest-rulename-infos) prefix all-rulename-infos)))
       (mv (cons nonleaf-thm-event more-nonleaf-thm-events)
           (cons rulename-thm-event more-rulename-thm-events)
           (cons match-thm-event more-match-thm-events)
           (cons concs-thm-event more-concs-thm-events)
           (append conc-equivs-thm-event? more-conc-equivs-thm-events)
           (append check-conc-fn-event? more-check-conc-fn-events)
           (append get-tree-list-list-fn-events
                   more-get-tree-list-list-fn-events)
           (append conc-matching-thm-events more-conc-matching-thm-events)
           (append rep-matching-thm-events more-rep-matching-thm-events))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-all-rulename-infos+events ((rules rulelistp)
                                                  (prefix acl2::symbolp))
  :returns (mv (rulename-infos deftreeops-rulename-info-alistp)
               (rulename-events pseudo-event-form-listp))
  :short "Generate the information and events
          for all the rulenames in a grammar."
  :long
  (xdoc::topstring
   (xdoc::p
    "We generate the events in an order that
     keeps analogous events together
     and avoids forward references."))
  (b* ((infos (deftreeops-gen-rulename-info-alist rules prefix))
       ((mv nonleaf-thm-events
            rulename-thm-events
            match-thm-events
            concs-thm-events
            conc-equivs-thm-events
            check-conc-fn-events
            get-tree-list-list-fn-events
            conc-matching-thm-events
            rep-matching-thm-events)
        (deftreeops-gen-rulename-alist-events infos prefix))
       (events (append nonleaf-thm-events
                       rulename-thm-events
                       match-thm-events
                       concs-thm-events
                       conc-matching-thm-events
                       rep-matching-thm-events
                       conc-equivs-thm-events
                       check-conc-fn-events
                       get-tree-list-list-fn-events)))
    (mv infos events)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-gen-everything ((grammar acl2::symbolp)
                                   (rules rulelistp)
                                   (prefix acl2::symbolp)
                                   (call pseudo-event-formp))
  :returns (event pseudo-event-formp)
  :short "Generate all the events."
  (b* ((matchers (deftreeops-gen-matchers grammar prefix))
       ((mv rulename-infos rulename-events)
        (deftreeops-gen-all-rulename-infos+events rules prefix))
       (table-value (deftreeops-table-value rulename-infos))
       (event `(defsection ,(add-suffix grammar "-TREE-OPERATIONS")
                 :parents (,grammar)
                 :short ,(str::cat
                          "Tree operations specialized to @(tsee "
                          (str::downcase-string (symbol-name grammar))
                          ").")
                 ,@matchers
                 ,@rulename-events
                 ,(deftreeops-table-add call table-value))))
    event))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-process-inputs-and-gen-everything ((args true-listp)
                                                      (call pseudo-event-formp)
                                                      (wrld plist-worldp))
  :returns (mv erp (event pseudo-event-formp))
  :parents (deftreeops-implementation)
  :short "Process the inputs and generate the events."
  (b* (((reterr) '(_))
       ((when (deftreeops-table-lookup call wrld))
        (retok '(value-triple :redundant)))
       ((erp grammar rules prefix) (deftreeops-process-inputs args wrld)))
    (retok (deftreeops-gen-everything grammar rules prefix call))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define deftreeops-fn ((args true-listp)
                       (call pseudo-event-formp)
                       (ctx ctxp)
                       state)
  :returns (mv erp (event pseudo-event-formp) state)
  :parents (deftreeops-implementation)
  :short "Event expansion of @(tsee deftreeops)."
  (b* (((mv erp event)
        (deftreeops-process-inputs-and-gen-everything args call (w state)))
       ((when erp) (er-soft+ ctx t '(_) "~@0" erp)))
    (value event)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection deftreeops-macro-definition
  :parents (deftreeops-implementation)
  :short "Definition of @(tsee deftreeops)."
  (defmacro deftreeops (&whole call &rest args)
    `(make-event (deftreeops-fn ',args ',call 'deftreeops state))))
