; Examples of Compositional Verification of PFCS Gadgets
;
; Copyright (C) 2024 Kestrel Institute (http://www.kestrel.edu)
; Copyright (C) 2024 Aleo Systems Inc. (https://www.aleo.org)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Authors: Alessandro Coglio (www.alessandrocoglio.info)
;          Eric McCarthy (mccarthy@kestrel.edu)
;          Eric Smith (eric.smith@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ZKPAPER")

(include-book "kestrel/utilities/typed-lists/bit-listp" :dir :system)
(include-book "projects/pfcs/convenience-constructors" :dir :system)
(include-book "projects/pfcs/indexed-names" :dir :system)
(include-book "projects/pfcs/lifting" :dir :system)
(include-book "std/testing/must-be-redundant" :dir :system)

(local (include-book "kestrel/prime-fields/prime-fields-rules" :dir :system))
(local (include-book "std/basic/inductions" :dir :system))
(local (include-book "std/lists/len" :dir :system))
(local (include-book "std/typed-lists/string-listp" :dir :system))
(local (include-book "kestrel/utilities/nfix" :dir :system))

(local (include-book "kestrel/built-ins/disable" :dir :system))
(local (acl2::disable-most-builtin-logic-defuns))
(local (acl2::disable-builtin-rewrite-rules-for-defaults))
(set-induction-depth-limit 0)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; This file contains some of the supporting materials for
; the ACL2-2023 Workshop paper "Formal Verification of Zero-Knowledge Circuits"
; by A. Coglio, E. McCarthy, and E. Smith.

; This file contains the formal details, overviewed in Section 6.2,
; of (most of) the example gadgets described in Section 2,
; formalized and verified compositionally in PFCS form.

; This file parallels the file r1cs.lisp,
; which is best read alongside this file.

; This file may be moved to the PFCS library at some point.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; This is the PFCS version of the gadget to assert that a variable is boolean.
; It is constructed by a nullary ACL2 function
; that returns the abstract syntax of the gadget,
; a definitionp (this predicate recognizes PFCS relation definitions).
; This PFCS relation definition uses a specific variable "x",
; which can be instantiated with expressions (e.g. different variables)
; when this PFCS relation is called from another PFCS relation definition.

(define boolean-assert-gadget ()
  :returns (pdef definitionp)
  (pfdef "boolean_assert"
         (list "x")
         (pf= (pf* (pfvar "x")
                   (pf+ (pfconst 1) (pfmon -1 "x")))
              (pfconst 0))))

; This deeply embedded gadget can be automatically lifted
; to a shallowly embedded predicate,
; via the following call of the lifter,
; which operates on the abstract syntax of the PFCS relation definition.

(lift (boolean-assert-gadget))

; The predicate automatically generated by the lift call above is the following.
; It takes one parameter x, as well as the prime p,
; and evaluates the constraints.
; The syntactic additions and multiplications above
; are turned into calls of the actual prime field operations.

(must-be-redundant
 (DEFUN BOOLEAN-ASSERT (X P)
   (AND (EQUAL (MUL X
                    (ADD (MOD 1 P) (MUL (MOD -1 P) X P) P)
                    P)
               (MOD 0 P)))))

; The lifting theorem generated by the lift call above is the following.
; The first hypothesis says that the definition of the gadget is available,
; and has the expected form (defs is rho in the paper).
; The second hypothesis says that x is a field element,
; and the third one that p is a prime.
; The conclusion says that the satisfaction of the gadget by x
; is equivalent to the shallowly embeedded predicate holding on x.
; That is, this relates deep and shallow embeddings of the gadget.

(must-be-redundant
 (DEFTHM DEFINITION-SATP-OF-BOOLEAN-ASSERT-TO-SHALLOW
   (IMPLIES
    (AND
     (EQUAL
      (LOOKUP-DEFINITION "boolean_assert" DEFS)
      '(:DEFINITION
        (PFCS::NAME . "boolean_assert")
        (PFCS::PARA "x")
        (PFCS::BODY
         (:EQUAL (:MUL (:VAR "x")
                  (:ADD (:CONST 1)
                   (:MUL (:CONST -1) (:VAR "x"))))
          (:CONST 0)))))
     (FEP X P)
     (PRIMEP P))
    (EQUAL (DEFINITION-SATP "boolean_assert" DEFS (LIST X) P)
           (BOOLEAN-ASSERT X P)))))

; Given that the gadget has been lifted,
; we can prove correctness for the shallowly embedded predicate,
; which is simpler.
; It is very easy in this case.

(defruled boolean-assert-correctness
  (implies (and (primep p)
                (fep x p))
           (equal (boolean-assert x p)
                  (bitp x)))
  :enable boolean-assert)

; Then we can obtain a correctness proof for the deeply embedded gadget
; by merely combining the correctness theorem for the shallow embedding
; with the lifting theorem.
; This could be automatically generated; note the use of a quoted theory.
; The hypothesese are the same as the lifting theorem,
; but with the gadget definition unexpanded
; (so we enable its executable counterpart here, to bridge the gap).

(defruled boolean-assert-gadget-correctness
  (implies (and (equal (lookup-definition "boolean_assert" defs)
                       (boolean-assert-gadget))
                (primep p)
                (fep x p))
           (equal (definition-satp "boolean_assert" defs (list x) p)
                  (bitp x)))
  :in-theory '((:e boolean-assert-gadget)
               definition-satp-of-boolean-assert-to-shallow
               boolean-assert-correctness))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; This is the PFCS version of the conditional gadget.
; It is similar to the boolean gadget above, but with more variables,
; which are all external, i.e. all parameters of the PFCS relation.
; The constraint is also a little more complicated than the boolean gadget.

(define if-then-else-gadget ()
  :returns (pdef definitionp)
  (pfdef "if_then_else"
         (list "w" "x" "y" "z")
         (pf= (pf* (pfvar "w")
                   (pf+ (pfvar "x")
                        (pfmon -1 "y")))
              (pf+ (pfvar "z")
                   (pfmon -1 "y")))))

; This gadget is also lifted automatically.

(lift (if-then-else-gadget))

; This is the automatically generated shallowly embedded predicate.
; Not different in structure from the one for the boolean,
; except for more variables and a more complex constraint.

(must-be-redundant
 (DEFUN IF-THEN-ELSE (W X Y Z P)
   (AND (EQUAL (MUL W (ADD X (MUL (MOD -1 P) Y P) P) P)
               (ADD Z (MUL (MOD -1 P) Y P) P)))))

; This is the automatically generated lifting theorem.
; Again, similar to the previous gadget, just a bit richer.

(must-be-redundant
 (DEFTHM DEFINITION-SATP-OF-IF-THEN-ELSE-TO-SHALLOW
   (IMPLIES
    (AND
     (EQUAL
      (LOOKUP-DEFINITION "if_then_else" DEFS)
      '(:DEFINITION
        (PFCS::NAME . "if_then_else")
        (PFCS::PARA "w" "x" "y" "z")
        (PFCS::BODY
         (:EQUAL (:MUL (:VAR "w")
                  (:ADD (:VAR "x")
                   (:MUL (:CONST -1) (:VAR "y"))))
          (:ADD (:VAR "z")
           (:MUL (:CONST -1) (:VAR "y")))))))
     (FEP W P)
     (FEP X P)
     (FEP Y P)
     (FEP Z P)
     (PRIMEP P))
    (EQUAL (DEFINITION-SATP "if_then_else" DEFS (LIST W X Y Z) P)
           (IF-THEN-ELSE W X Y Z P)))))

; As with the previous gadget, we (easily) prove correctness,
; first for the shallowly embedded predicate.

(defruled if-then-else-correctness
  (implies (and (primep p)
                (fep x p)
                (fep y p)
                (fep z p)
                (fep w p)
                (bitp w))
           (equal (if-then-else w x y z p)
                  (equal z (if (equal w 1) x y))))
  :enable if-then-else)

; Then we extend correctness to the deep embedding.
; This could be generated automatically.

(defruled if-then-else-gadget-correctness
  (implies (and (equal (lookup-definition "if_then_else" defs)
                       (if-then-else-gadget))
                (primep p)
                (fep w p)
                (fep x p)
                (fep y p)
                (fep z p)
                (bitp w))
           (equal (definition-satp "if_then_else" defs (list w x y z) p)
                  (equal z (if (equal w 1) x y))))
  :in-theory '((:e if-then-else-gadget)
               definition-satp-of-if-then-else-to-shallow
               if-then-else-correctness))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; This is the PFCS version of the equality test gadget,
; which, unlike the gadgets above, has an internal variable, "s".
; This is signaled by the fact that "s" is used in the constraints,
; but it is not a parameter of the PFCS relation.
; This gadget definition also shows the use of a sub-gadget,
; namely the one that constrains w to be boolean.
; Note that the ACL2 function is still unary;
; it does not call boolean-assert-gadget.
; Instead, one of the constraints of the PFCS relation's body
; is a call of the "boolean_assert" gadget;
; note that the variable "x" is instantiated with "w" (as an expression).
; This is a crucial difference with the definition of gadgets in R1CS form:
; the gadget hierarchy is captured directly in the gadgets,
; not in the ACL2 function calls.

(define equality-test-gadget ()
  :returns (pdef definitionp)
  (pfdef "equality_test"
         (list "u" "v" "w")
         (pfcall "boolean_assert" (pfvar "w"))
         (pf= (pf* (pf+ (pfvar "u")
                        (pfmon -1 "v"))
                   (pfvar "s"))
              (pf+ (pfconst 1)
                   (pfmon -1 "w")))
         (pf= (pf* (pf+ (pfvar "u")
                        (pfmon -1 "v"))
                   (pf+ (pfvar "w")))
              (pfconst 0))))

; This gadget is lifted automatically to the shallow embedding.

(lift (equality-test-gadget))

; The predicate automatically generated by the lift call above
; now has an existential quantifier, unlike the previous gadgets.
; The internal variable is existentially quantified;
; the predicate only has parameters for the external variable,
; like the PFCS relation.
; Note also that this predicate calls boolean-assert,
; i.e. the shallowly embedded gadget:
; at the shallow embedding level,
; gadgets call each other in the ACL2 function graph,
; unlike in the deeply embedded versions as pointed out above.

(must-be-redundant
 (DEFUN-SK EQUALITY-TEST (U V W P)
   (EXISTS
    (S)
    (AND (FEP S P)
         (AND (BOOLEAN-ASSERT W P)
              (EQUAL (MUL (ADD U (MUL (MOD -1 P) V P) P) S P)
                     (ADD (MOD 1 P) (MUL (MOD -1 P) W P) P))
              (EQUAL (MUL (ADD U (MUL (MOD -1 P) V P) P) W P)
                     (MOD 0 P)))))))

; The automatically generated lifting theorem has a more complicated proof,
; due to the existential quantification,
; but that is transparent to the caller of the lifter.
; Since the equality test gadget uses the boolean gadget,
; we need the fact that the boolean gadget is available;
; so that is an additional hypothesis in the lifting theorem.
; Note that the conclusion of the lifting theorem does not involve any s;
; it is just about the external variables.

(must-be-redundant
 (DEFTHM DEFINITION-SATP-OF-EQUALITY-TEST-TO-SHALLOW
   (IMPLIES
    (AND
     (EQUAL
      (LOOKUP-DEFINITION "equality_test" DEFS)
      '(:DEFINITION
        (PFCS::NAME . "equality_test")
        (PFCS::PARA "u" "v" "w")
        (PFCS::BODY
         (:RELATION "boolean_assert" ((:VAR "w")))
         (:EQUAL (:MUL (:ADD (:VAR "u")
                        (:MUL (:CONST -1) (:VAR "v")))
                  (:VAR "s"))
          (:ADD (:CONST 1)
           (:MUL (:CONST -1) (:VAR "w"))))
         (:EQUAL (:MUL (:ADD (:VAR "u")
                        (:MUL (:CONST -1) (:VAR "v")))
                  (:VAR "w"))
          (:CONST 0)))))
     (EQUAL
      (LOOKUP-DEFINITION "boolean_assert" DEFS)
      '(:DEFINITION
        (PFCS::NAME . "boolean_assert")
        (PFCS::PARA "x")
        (PFCS::BODY
         (:EQUAL (:MUL (:VAR "x")
                  (:ADD (:CONST 1)
                   (:MUL (:CONST -1) (:VAR "x"))))
          (:CONST 0)))))
     (FEP U P)
     (FEP V P)
     (FEP W P)
     (PRIMEP P))
    (EQUAL (DEFINITION-SATP "equality_test" DEFS (LIST U V W) P)
           (EQUALITY-TEST U V W P)))))

; As above, we first prove the correctness of the shallowly embedded gadget.
; Because of the existential quantification,
; the proof is a little more elaborate.
; We prove soundness and completeness, then put them together.
; For soundness, we just expand the shallowly embedded predicate,
; which involves a witness term for s,
; and then we use the correctness theorem
; for the (shallowly embedded) boolean gadget;
; this is enough to prove soundness.
; For completeness, we use the equality-test-suff theorem,
; as customary when proving defun-sk's with existentials,
; where we need to supply a witness for s:
; if u and v are equal, any s would do, and we just pick 0;
; if u and v differ, we pick 1/(u-v) for s.
; We also use the correctness theorem for the boolean gadget.

; This kind of theorem could be partially automated:
; a lot of it is boilerplate, so there could be an event macro
; that supplies just the interesting parts,
; such as the hints to prove soundness
; and the hints to prove completeness;
; the latter could be further simplified by
; only supplying witness terms for the internal variables,
; from which the :use hints can be automatically generated.

; We need to disable a prime field rule,
; which otherwise sabotages the proof,
; by turning terms into a form for which the satisfaction of the constraints
; is harder to prove
; (which is otherwise easy to prove, see Section 2 of the paper).

(defruled equality-test-correctness
  (implies (and (primep p)
                (fep u p)
                (fep v p)
                (fep w p))
           (equal (equality-test u v w p)
                  (equal w (if (equal u v) 1 0))))
  :use (soundness completeness)
  :prep-lemmas
  ((defrule soundness
     (implies (and (primep p)
                   (fep u p)
                   (fep v p)
                   (fep w p))
              (implies (equality-test u v w p)
                       (equal w (if (equal u v) 1 0))))
     :rule-classes nil
     :enable (equality-test
              boolean-assert-correctness))
   (defrule completeness
     (implies (and (primep p)
                   (fep u p)
                   (fep v p)
                   (fep w p))
              (implies (equal w (if (equal u v) 1 0))
                       (equality-test u v w p)))
     :rule-classes nil
     :use (:instance equality-test-suff
                     (s (if (equal u v)
                            0
                          (inv (sub u v p) p)))
                     (p p))
     :enable boolean-assert-correctness
     :disable pfield::mul-of-add-arg1)))

; As in the other gadgets,
; we compose the correctness theorem with the lifting theorem
; (which could be done automatically),
; to obtain the correctness theorem for the deep embedding.
; Note that, besides the definition of the gadget,
; we need hypotheses about the definitions of the sub-gadgets.

(defruled equality-test-gadget-correctness
  (implies (and (equal (lookup-definition "equality_test" defs)
                       (equality-test-gadget))
                (equal (lookup-definition "boolean_assert" defs)
                       (boolean-assert-gadget))
                (primep p)
                (fep u p)
                (fep v p)
                (fep w p))
           (equal (definition-satp "equality_test" defs (list u v w) p)
                  (equal w (if (equal u v) 1 0))))
  :in-theory '((:e equality-test-gadget)
               (:e boolean-assert-gadget)
               definition-satp-of-equality-test-to-shallow
               equality-test-correctness))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; This is the PFCS version of the gadget that combines
; the conditional gadget and the equality test.
; It consists of two gadget calls.
; The "w" variable is internal to this gadget.
; Note that there is no sign of the "s" variable
; of the equality test sub-gadget here;
; it has been completely handled above.

(define if-equal-then-else-gadget ()
  :returns (pdef definitionp)
  (pfdef "if_equal_then_else"
         (list "u" "v" "x" "y" "z")
         (pfcall "if_then_else" (pfvar "w") (pfvar "x") (pfvar "y") (pfvar "z"))
         (pfcall "equality_test" (pfvar "u") (pfvar "v") (pfvar "w"))))

; Lifting is again automatic.

(lift (if-equal-then-else-gadget))

; The generated shallowly embedded predicate has w existentially quantified,
; and it just calls the other two shallowly embedded gadgets.
; Again, no sign of the variable s of the equality sub-gadget here.

(must-be-redundant
 (DEFUN-SK IF-EQUAL-THEN-ELSE (U V X Y Z P)
   (EXISTS (W)
           (AND (FEP W P)
                (AND (IF-THEN-ELSE W X Y Z P)
                     (EQUALITY-TEST U V W P))))))

; The generated lifting theorem has the usual form,
; with additional hypotheses for the sub-gadgets and their sub-gadgets.
; This might raise a concern that hypotheses about gadget definitions
; will keep growing.
; They will indeed keep growing,
; but not out of proportion
; like in the case of internal variables in the R1CS gadget constructions.
; A gadget definition is like
; a function (or procedure, or method) definition in a traditional program:
; it is normal for properties of a function to depend on
; properties of transitively called functions.
; The transitive closure of the gadgets called by a gadget
; should always remain manageable;
; note that the formulation of the hypotheses below
; is amenable to abbreviation, using suitable macros.

(must-be-redundant
 (DEFTHM DEFINITION-SATP-OF-IF-EQUAL-THEN-ELSE-TO-SHALLOW
   (IMPLIES
    (AND
     (EQUAL
      (LOOKUP-DEFINITION "if_equal_then_else" DEFS)
      '(:DEFINITION
        (PFCS::NAME . "if_equal_then_else")
        (PFCS::PARA "u" "v" "x" "y" "z")
        (PFCS::BODY
         (:RELATION "if_then_else"
          ((:VAR "w")
           (:VAR "x")
           (:VAR "y")
           (:VAR "z")))
         (:RELATION "equality_test"
          ((:VAR "u") (:VAR "v") (:VAR "w"))))))
     (EQUAL
      (LOOKUP-DEFINITION "equality_test" DEFS)
      '(:DEFINITION
        (PFCS::NAME . "equality_test")
        (PFCS::PARA "u" "v" "w")
        (PFCS::BODY
         (:RELATION "boolean_assert" ((:VAR "w")))
         (:EQUAL (:MUL (:ADD (:VAR "u")
                        (:MUL (:CONST -1) (:VAR "v")))
                  (:VAR "s"))
          (:ADD (:CONST 1)
           (:MUL (:CONST -1) (:VAR "w"))))
         (:EQUAL (:MUL (:ADD (:VAR "u")
                        (:MUL (:CONST -1) (:VAR "v")))
                  (:VAR "w"))
          (:CONST 0)))))
     (EQUAL
      (LOOKUP-DEFINITION "boolean_assert" DEFS)
      '(:DEFINITION
        (PFCS::NAME . "boolean_assert")
        (PFCS::PARA "x")
        (PFCS::BODY
         (:EQUAL (:MUL (:VAR "x")
                  (:ADD (:CONST 1)
                   (:MUL (:CONST -1) (:VAR "x"))))
          (:CONST 0)))))
     (EQUAL
      (LOOKUP-DEFINITION "if_then_else" DEFS)
      '(:DEFINITION
        (PFCS::NAME . "if_then_else")
        (PFCS::PARA "w" "x" "y" "z")
        (PFCS::BODY
         (:EQUAL (:MUL (:VAR "w")
                  (:ADD (:VAR "x")
                   (:MUL (:CONST -1) (:VAR "y"))))
          (:ADD (:VAR "z")
           (:MUL (:CONST -1) (:VAR "y")))))))
     (FEP U P)
     (FEP V P)
     (FEP X P)
     (FEP Y P)
     (FEP Z P)
     (PRIMEP P))
    (EQUAL (DEFINITION-SATP "if_equal_then_else" DEFS (LIST U V X Y Z) P)
           (IF-EQUAL-THEN-ELSE U V X Y Z P)))))

; We prove correctness of the shallowly embedded version, as above.
; This is again split into soundness and completeness,
; in order to handle the existentially quantified w.
; the witness is 1 if u and v are equal, 0 if they differ.
; Other than supplying the witness for w,
; soundness and completeness are easily proved by
; opening the predicate for the gadget
; and enabling the correctness theorems of the sub-gadgets.

(defruled if-equal-then-else-correctness
  (implies (and (primep p)
                (fep u p)
                (fep v p)
                (fep x p)
                (fep y p)
                (fep z p))
           (equal (if-equal-then-else u v x y z p)
                  (equal z (if (equal u v) x y))))
  :use (soundness completeness)
  :prep-lemmas
  ((defrule soundness
     (implies (and (primep p)
                   (fep u p)
                   (fep v p)
                   (fep x p)
                   (fep y p)
                   (fep z p))
              (implies (if-equal-then-else u v x y z p)
                       (equal z (if (equal u v) x y))))
     :rule-classes nil
     :enable (if-equal-then-else
              if-then-else-correctness
              equality-test-correctness))
   (defrule completeness
     (implies (and (primep p)
                   (fep u p)
                   (fep v p)
                   (fep x p)
                   (fep y p)
                   (fep z p))
              (implies (equal z (if (equal u v) x y))
                       (if-equal-then-else u v x y z p)))
     :rule-classes nil
     :use (:instance if-equal-then-else-suff
                     (w (if (equal u v) 1 0))
                     (p p))
     :enable (if-then-else-correctness
              equality-test-correctness))))

; As with the other gadgets, we extend correctness to the deep embedding,
; in a way that could be automated.

(defruled if-equal-then-else-gadget-correctness
  (implies (and (equal (lookup-definition "if_equal_then_else" defs)
                       (if-equal-then-else-gadget))
                (equal (lookup-definition "if_then_else" defs)
                       (if-then-else-gadget))
                (equal (lookup-definition "equality_test" defs)
                       (equality-test-gadget))
                (equal (lookup-definition "boolean_assert" defs)
                       (boolean-assert-gadget))
                (primep p)
                (fep u p)
                (fep v p)
                (fep x p)
                (fep y p)
                (fep z p))
           (equal (definition-satp "if_equal_then_else" defs (list u v x y z) p)
                  (equal z (if (equal u v) x y))))
  :in-theory '((:e if-equal-then-else-gadget)
               (:e if-then-else-gadget)
               (:e equality-test-gadget)
               (:e boolean-assert-gadget)
               definition-satp-of-if-equal-then-else-to-shallow
               if-equal-then-else-correctness))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; This is the PFCS version of the gadget that
; consists of the boolean gadget applied to a sequence of variables.
; Since this gadget has a varying number of variables and constraints,
; the boolean-assert-list-gadget function takes a parameter n
; that is the number of both the variables and the gadgets
; (in this case; in other cases,
; variables and constraints may be in different numbers,
; often inter-dependent, so that often just one parameter is needed).
; This is different from all the other functions above,
; which are nullary because they construct fixed gadgets.
; This function constructs a family of gadgets, parameterized over n,
; which is a natural number.

; This function makes use of an auxiliary function
; that recursively constructs the constraints,
; which the main function then puts together into a PFCS relation definition.
; The auxiliary function is a bit more general
; than strictly necessary to build the gadget,
; but that generality actually comes handy in induction proofs.
; The generality is that it takes a generic list of variables as input,
; and generates a boolean constraint for each of them,
; as a call of the boolean gadget defined earlier.
; Again, note that this function does not call boolean-assert-gadget;
; instead, the constraints contain calls of the PFCS relation boolean_assert.
; Given that, the definition of the auxiliary function is straightforward.

; Going back to the main function, there are a few things to explain.
; First, note that the PFCS relation does not have a fixed name,
; but instead a name that depends on n.
; The iname function, from the PFCS library,
; adds an underscore and a numeric index to a given string.
; So the name of this PFCS relation has the form boolean_assert_list_<n>,
; where <n> consists of the digits forming n;
; for instance, if n is 32, the name is boolean_assert_list_32.
; This is important because
; there can be only one PFCS relation with a given name,
; but here we are defining a family of them,
; so each of them must have a unique name;
; note that we may need to use different instances of this gadget
; as sub-gadgets of a super-gadget, with different values of n,
; and thus those have to be distinct PFCS relations.

; Even though the auxiliary function takes an arbitrary list of variables,
; the main function uses a fixed list of variables,
; built via the iname-list function from the PFCS library,
; which returns a list of names obtained by applying iname
; to a base string with increasing indices.
; For instance, (iname-list "x" 32) returns the list
; ("x_0" "x_1" ... "x_30" "x_31"),
; i.e. starting from 0 (inclusive) and ending at n (exclusive).
; This list is used as the parameters of the PFCS relation,
; and also passed to the auxiliary function,
; so it builds the constraints for exactly those variables.

(define boolean-assert-list-gadget ((n natp))
  :returns (pdef definitionp)
  (pfdef (iname "boolean_assert_list" n)
         (iname-list "x" n)
         (boolean-assert-list-gadget-aux (iname-list "x" n)))
  :prepwork
  ((define boolean-assert-list-gadget-aux ((xs string-listp))
     :returns (constrs constraint-listp)
     (cond ((endp xs) nil)
           (t (cons (pfcall "boolean_assert" (pfvar (car xs)))
                    (boolean-assert-list-gadget-aux (cdr xs))))))))

; Our current PFCS lifter only works on the PFCS abstract syntax.
; This is why, in the preceding gadgets,
; we applied it to (the result of) the nullary gadget constructor,
; obtaining the shallowly embedded predicate and the lifting theorem.
; But this gadget constructor is not nullary; it takes n as parameter.
; So we need to lift the gadget manually, which we do below.
; A future version of the PFCS lifter may be able to handle this,
; but note that it would have to look at
; the definition of boolean-assert-list-gadget (and the auxiliary function),
; in order to figure out the shallow embedding;
; even though boolean-assert-list-gadget returns PFCS abstract syntax,
; it is not a fixed one, and the lifter must operate statically.
; While the future lifter may not be able to handle all the possible ways
; in which a function (collection) may construct a gadget,
; there are probably patterns that can be recognized, such as the one above.
; Another idea we have is to extend the PFCS syntax
; with the ability to define parameterized gadgets,
; in which case one could always use nullary gadget constructors,
; and the lifter would know what to expect.

; Back from future work to the current work,
; the shallowly embedded predicate is actually quite simple to write manually.
; It is just a list of calls of the boolean_assert gadget,
; so here we recursively call the corresponding shallowly embedded predicate.

(defund boolean-assert-list (xs p)
  (or (endp xs)
      (and (boolean-assert (car xs) p)
           (boolean-assert-list (cdr xs) p))))

; The lifting theorem must be also created manually for now.
; This is not too difficult, but it needs some theorems about omaps,
; which should be in the omaps library, but currently are not.
; The theorems are the following,
; which should be also amenable to simplifications and generalizations
; (currently they are perhaps a bit too specific).
; These are actually not specific to PFCSes, they are just about omaps.
; No explanations are given here for these theorems,
; because they are not interesting for the current purposes.
; The reader can imagine that they, or suitable generalizations of them,
; are in the omaps library, and just skip to the next comments in this file.

(defruled omap-in-of-supermap-when-submap
  (implies (and (omap::submap sub sup)
                (omap::in key sub))
           (equal (omap::in key sup)
                  (omap::in key sub)))
  :induct t
  :enable omap::submap)

(defruled omap-submap-of-update-right
  (implies (not (omap::in key map))
           (equal (omap::submap map (omap::update key val map2))
                  (omap::submap map map2)))
  :induct t
  :enable omap::submap)

(defruled omap-not-in-when-key-less
  (implies (or (omap::emptyp map)
               (<< key (mv-nth 0 (omap::head map))))
           (not (omap::in key map)))
  :induct t
  :enable (omap::in
           omap::head
           omap::tail
           omap::mapp
           omap::mfix
           omap::emptyp))

(defruled omap-head-not-in-tail
  (implies (not (omap::emptyp map))
           (not (omap::in (mv-nth 0 (omap::head map))
                          (omap::tail map))))
  :enable omap::head-tail-order
  :use (:instance omap-not-in-when-key-less
                  (map (omap::tail map))
                  (key (mv-nth 0 (omap::head map)))))

(defruled omap-submap-of-tail-lemma
  (implies (and (not (omap::emptyp map))
                (omap::submap (omap::tail map)
                              (omap::tail map)))
           (omap::submap (omap::tail map) map))
  :enable omap-head-not-in-tail
  :use (:instance omap-submap-of-update-right
                  (key (mv-nth 0 (omap::head map)))
                  (map (omap::tail map))
                  (val (mv-nth 1 (omap::head map)))
                  (map2 (omap::tail map))))

(defrule omap-submap-reflexive
  (omap::submap map map)
  :induct t
  :enable (omap::submap
           omap-submap-of-tail-lemma))

(defruled omap-submap-of-tail
  (implies (not (omap::emptyp map))
           (omap::submap (omap::tail map) map))
  :enable omap-submap-of-tail-lemma)

(defruled omap-in-of-car-of-from-lists
  (implies (consp keys)
           (equal (omap::in (car keys) (omap::from-lists keys vals))
                  (cons (car keys) (car vals))))
  :enable omap::from-lists)

(defruled omap-in-of-car-of-supermap-of-from-lists
  (implies (and (consp keys)
                (omap::submap (omap::from-lists keys vals) map))
           (equal (omap::in (car keys) map)
                  (cons (car keys) (car vals))))
  :use (:instance omap-in-of-supermap-when-submap
                  (key (car keys))
                  (sub (omap::from-lists keys vals))
                  (sup map))
  :enable omap-in-of-car-of-from-lists)

(defruled omap-in-of-cadr-of-from-lists
  (implies (and (consp (cdr keys))
                (not (member-equal (car keys) (cdr keys))))
           (equal (omap::in (cadr keys) (omap::from-lists keys vals))
                  (cons (cadr keys) (cadr vals))))
  :enable omap::from-lists)

(defruled omap-in-of-cadr-of-supermap-of-from-lists
  (implies (and (consp (cdr keys))
                (not (member-equal (car keys) (cdr keys)))
                (omap::submap (omap::from-lists keys vals) map))
           (equal (omap::in (cadr keys) map)
                  (cons (cadr keys) (cadr vals))))
  :use (:instance omap-in-of-supermap-when-submap
                  (key (cadr keys))
                  (sub (omap::from-lists keys vals))
                  (sup map))
  :enable omap-in-of-cadr-of-from-lists)

(defruled omap-not-in-from-lists-when-not-member
  (implies (not (member-equal key keys))
           (not (omap::in key (omap::from-lists keys vals))))
  :induct t
  :enable omap::from-lists)

(defruled omap-submap-of-from-lists-of-cdr-cdr
  (implies (and (no-duplicatesp-equal keys)
                (consp keys))
           (omap::submap (omap::from-lists (cdr keys) (cdr vals))
                         (omap::from-lists keys vals)))
  :enable (omap::from-lists
           no-duplicatesp-equal
           omap-submap-of-update-right
           omap-not-in-from-lists-when-not-member))

(defruled omap-submap-transitive
  (implies (and (omap::submap map1 map2)
                (omap::submap map2 map3))
           (omap::submap map1 map3))
  :induct (omap::size map1)
  :enable (omap::size
           omap::submap
           omap-in-of-supermap-when-submap))

(defruled omap-submap-of-from-lists-of-cdr-cdr-when-submap-of-from-lists
  (implies (and (no-duplicatesp-equal keys)
                (omap::submap (omap::from-lists keys vals) map)
                (consp keys))
           (omap::submap (omap::from-lists (cdr keys) (cdr vals)) map))
  :use (omap-submap-of-from-lists-of-cdr-cdr
        (:instance omap-submap-transitive
                   (map1 (omap::from-lists (cdr keys) (cdr vals)))
                   (map2 (omap::from-lists keys vals))
                   (map3 map))))

; Given that the gadget constructor calls an auxiliary recursive function,
; it should not be a surprise that the lifting theorem is proved
; by first proving a lifting lemma about the auxiliary function.
; This lifting lemma says that
; the constraints generated by the auxiliary function
; are satisfied iff the shallowly embedded predicate holds.
; This is proved for all choices of variables xs-vars.

; The satisfaction of the constraints is expressed by
; the predicate constraint-list-satp,
; which operates on an assignment
; (similarly to the r1cs-constraints-holdp in the R1CS semantics,
; except that it is an omap in the PFCS semantics instead of an alist).
; A hypothesis says that the assignment includes at least mappings
; from the variables in xs-var to the values (prime field elements) in xs-val;
; the function omap::from-lists from the omaps library
; builds an omap from a list of keys and a list of values of the same length,
; putting keys and values together in order.
; We use this more general assignment and hypothesis,
; instead of simply using (omap::from-lists xs-vars xs-vals),
; in order to have a sufficiently strong induction hypothesis in the proof.

; The other hypotheses are straightforward:
; the definition of the called boolean gadget must be available,
; p is a prime,
; xs-vars are distinct variables (strings),
; xs-vals are field elements (fe-listp lifts fep to lists),
; and asg is a well-formed assignment that includes omap::from-lists.
; The fact that xs-vars have no duplicates is critical,
; otherwise omap::from-lists could not guarantee that
; the value (nth i xs-vals) is assigned to the variable (nth i xs-vars)
; for each i below the common length of xs-vars and xs-vals.

; The right side of the conclusion of this theorem
; is the shallowly embedded predicate applied to the values xs-vals.

; The proof is by induction on both xs-vars and xs-vals,
; and just requires enabling certain functions,
; including some from the PFCS semantics.

(defruled constraint-list-satp-of-boolean-assert-list-gadget-aux-to-shallow
  (implies (and (equal (lookup-definition "boolean_assert" defs)
                       (boolean-assert-gadget))
                (primep p)
                (string-listp xs-vars)
                (no-duplicatesp-equal xs-vars)
                (fe-listp xs-vals p)
                (equal (len xs-vars) (len xs-vals))
                (assignmentp asg)
                (assignment-wfp asg p)
                (omap::submap (omap::from-lists xs-vars xs-vals) asg))
           (equal (constraint-list-satp
                   (boolean-assert-list-gadget-aux xs-vars) defs asg p)
                  (boolean-assert-list xs-vals p)))
  :induct (acl2::cdr-cdr-induct xs-vars xs-vals)
  :enable (boolean-assert-list-gadget-aux
           boolean-assert-list
           pfcs::constraint-list-satp-of-cons
           pfcs::constraint-list-satp-of-nil
           pfcs::constraint-satp-to-definition-satp
           eval-expr-list
           eval-expr
           acl2::not-reserrp-when-natp
           acl2::not-reserrp-when-nat-listp
           definition-satp-of-boolean-assert-to-shallow
           no-duplicatesp-equal
           fe-listp
           omap-in-of-car-of-supermap-of-from-lists
           omap-submap-of-from-lists-of-cdr-cdr-when-submap-of-from-lists))

; Before we can use the lifting lemma to prove the lifting theorem,
; we need a few simple facts about the PFCS relation,
; to avoid opening it during the proof.

; First, the PFCS relation's parameters and body are as expected.

(defrule definition->para-of-boolean-assert-list-gadget
  (equal (definition->para (boolean-assert-list-gadget n))
         (iname-list "x" n))
  :enable boolean-assert-list-gadget)

(defrule definition->body-of-boolean-assert-list-gadget
  (equal (definition->body (boolean-assert-list-gadget n))
         (boolean-assert-list-gadget-aux (iname-list "x" n)))
  :enable boolean-assert-list-gadget)

; We need to know the free variables of the PFCS relation,
; which is this case are none.
; To calculate that, we first need to know all the variables in the constraints,
; which we prove for the auxiliary function to be
; exactly the list of strings passed as argument.
; Then the free variables are those minus the parameters,
; but since the auxiliary function is called exactly with the parameters,
; the set of free variables is the empty one.

(defrule constraint-list-vars-of-boolean-assert-list-gadget-aux
  (equal (constraint-list-vars (boolean-assert-list-gadget-aux xs))
         (set::mergesort (str::string-list-fix xs)))
  :induct t
  :enable (boolean-assert-list-gadget-aux
           constraint-list-vars
           constraint-vars
           expression-list-vars
           expression-vars
           set::mergesort))

(defrule definition-free-vars-of-boolean-assert-list-gadget
  (set::emptyp (definition-free-vars (boolean-assert-list-gadget n)))
  :enable (boolean-assert-list-gadget
           definition-free-vars))

; This is the lifting theorem, which is similar in nature to
; the lifting theorems for the previous gadgets,
; except that this is for a gadget family, for generic n.
; Here n is the length of xs, a generic list of field elements.
; We need the hypothesis that the definition of boolean_assert is available,
; and that the definition of boolean_assert_list_<n> is also available,
; for the same n as the length of xs.

(defruled definition-satp-of-boolean-assert-list-gadget-to-shallow
  (implies (and (equal (lookup-definition
                        (iname "boolean_assert_list" (len xs))
                        defs)
                       (boolean-assert-list-gadget (len xs)))
                (equal (lookup-definition "boolean_assert" defs)
                       (boolean-assert-gadget))
                (primep p)
                (fe-listp xs p))
           (equal (definition-satp
                    (iname "boolean_assert_list" (len xs)) defs xs p)
                  (boolean-assert-list xs p)))
  :enable (constraint-list-satp-of-boolean-assert-list-gadget-aux-to-shallow
           definition-satp
           pfcs::constraint-satp-of-relation-when-nofreevars
           pfcs::constraint-relation-nofreevars-satp
           nfix
           pfcs::eval-expr-list-of-expression-var-list-and-omap-from-lists))

; That concludes the manual lifting of the gadget.

; We proceed with proving the correctness of the shallowly embedded gadget
; (more precisely, gadget family), which is easy.
; Here the specification S is bit-listp.

(defruled boolean-assert-list-correctness
  (implies (and (primep p)
                (fe-listp xs p))
           (equal (boolean-assert-list xs p)
                  (bit-listp xs)))
  :induct t
  :enable (boolean-assert-list
           boolean-assert-correctness
           bit-listp))

; Finally we combine the lifting theorem with the shallow correctness theorem
; to obtain the correctness theorem for the deeply embedded gadget.

(defruled boolean-assert-list-gadget-correctness
  (implies (and (equal (lookup-definition
                        (iname "boolean_assert_list" (len xs))
                        defs)
                       (boolean-assert-list-gadget (len xs)))
                (equal (lookup-definition "boolean_assert" defs)
                       (boolean-assert-gadget))
                (primep p)
                (fe-listp xs p))
           (equal (definition-satp
                    (iname "boolean_assert_list" (len xs)) defs xs p)
                  (bit-listp xs)))
  :enable (boolean-assert-list-correctness
           definition-satp-of-boolean-assert-list-gadget-to-shallow))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; As mentioned in the paper,
; our development and use of PFCSes is still somewhat preliminary.
; At the time of this writing, we do not have a fully formalized and verified
; PFCS gadget (family) for the unsigned n-bit integer addition yet.
; As this involves multiple lists of variables,
; we need to develop a little more machinery,
; in both the PFCS library and the omaps library.
; We are actively working on this,
; so we should be able to add this relatively soon.

; In any case, it should not be hard to imagine how to construct the gadget,
; which is parameterized over the number of bits n;
; it is similar to boolean_assert_list, but a bit richer.
; It also involves a separate function to construct recursively
; a powers-of-two weighted sum PFCS expression, similarly to the R1CS version.
; The lifting theorem is the one that needs more library machinery,
; but again it should not be difficult
; to imagine the lifted predicate and theorem;
; there will also be a lifting theorem
; for the powers-of-two weighted sum expression.
; The correctness of the gadget will be proved similarly to the R1CS version:
; the core argument, about the weighted sums becoming lebits=>nat with mod,
; and the mod being eliminated because their arguments are below p,
; apply here as in the R1CS version.
