/* register allocation
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "treerep.h"
#include "assemble.h"
#include "stacksg.h"
#include "scanner.h"
#include "codegen.h"
#include "regalloc.h"
#include "reswords.h"
#include "compdriver.h"
#include "parser.h"
#include "treetools.h"


int used_bit;
int used_byte;
tree list_of_virtuals;

void gather_virtuals(tree p)
{
    tree np;
    stack_guard;

    if ((p == NULL) || (p->is_gathered)) {
        return;
    }

    for (np = NULL; p != NULL; p = np) {

        if (verbose_ralloc) {
            log((m, "gather virtuals node nr=%04d %s", p->nr, node_name[p->kind]));
        }

        switch (p->kind) {

        case node_test:
        case node_label:
        case node_type:
        case node_ref:
        case node_return:
        case node_precall:
        case node_error:
        case node_w:
        case node_var:
        case node_org:
        case node_const:{
                /* do nothing */
                np = NULL;
                break;
            }

        case node_procedure:{
#ifdef __DEBUG__
                log((m, "%s %d %d %d", p->name, p->is_virtual, p->is_gathered, p->has_transfer));
#endif
                if (p->is_virtual) {
                    if (!p->is_gathered) {
                        if (p->has_transfer) {
                            assert_kind(p->loc, p, node_procedure);
                            p->is_gathered = true;
#ifdef __DEBUG__
                            log((m, "added %s", p->name));
#endif
                            list_of_virtuals = new_chain2(p, list_of_virtuals);
                        } else {
                            tree q = arg(p, 1,
                                         false);
                            p->transfer1 = NULL;
                            p->transfer2 = NULL;
#ifdef __DEBUG__
                            if (q != NULL)
                                q->address = NULL;
                            if (p->result != NULL)
                                p->result->address = NULL;
#endif
                        }
                        gather_virtuals(p->first);
                    }
                } else {
                    gather_virtuals(p->first);
                }

                /* there is no next */
                np = NULL;
                break;
            }

        case node_asm:{
                if (p->opcode == opcode_call) {
                    gather_virtuals(p->first);
                }

                /* there is no next */
                np = NULL;
                break;
            }

        case node_chain:
        case node_assign:
        case node_decl:
        case node_call:
        case node_if:
        case node_op:
        case node_for:
        case node_while:{
                gather_virtuals(p->condition);
                gather_virtuals(p->first);
                gather_virtuals(p->step);
                gather_virtuals(p->end);

                /* next is done via looping */
                np = p->next;
                break;
            }

        default:{
                snark_node(p->loc, p);
                break;
            }
        }
        p->is_gathered = true;
    }
}

tree new_location(tree p, int *free_bit, int *free_byte)
{
    tree q;
    stack_guard;
    assert_kind(NULL, p, node_var);
    assert_pointer(NULL, p->type);

    if (p->type == type_bit) {
        if ((p->address == NULL)
            || (8 * p->address->first->x + p->address->next->x <= *free_bit)
            ) {
            q = new_address_value(*free_bit / 8, *free_bit % 8);
            if (*free_bit > used_bit)
                used_bit = *free_bit;

            if (verbose_ralloc) {
                log((m, "new_location (bit) nr=%04d bit=%02X byte=%02X %s", p->nr, *free_bit,
                     *free_byte, p->name));
            }

            (*free_bit)++;
            return q;
        } else {
            return p->address;
        }
    }

    if (p->type == type_byte) {
        if ((p->address == NULL)
            || (p->address->first->x <= *free_byte)
            ) {
            q = new_address_value(*free_byte, 0);
            if (*free_byte > used_byte)
                used_byte = *free_byte;

            if (verbose_ralloc) {
                log((m, "new_location (byte) nr=%04d bit=%02X byte=%02X %s", p->nr, *free_bit,
                     *free_byte, p->name));
            }

            (*free_byte)++;
            return q;
        } else {
            return p->address;
        }
    }

    snark_node(p->loc, p);
    return NULL;
}

void register_allocate_chain(tree p, int *free_bit, int *free_byte, boolean alloc_procs,        /* unused! */
                             int level)
{

    if ((p->first != NULL)
        && (p->first->kind == node_decl)
        && (p->first->first != NULL)
        && (p->first->first->kind == node_var)
        ) {
        tree q = p->first->first;
        assert_pointer(NULL, q->type);

        if (q->uncle != NULL) {

            /* inherit the address later */
            q->fixed = true;

        } else if (q->indirect) {

            /* don't allocate, should not have an address */
            jal_assert(q->loc, q->address == NULL);

        } else if (q->fixed) {

            if (q->address == NULL) {
                show_subtree(q);
            }

            assert_kind(q->loc, q->address, node_chain);
            /* the master address will be copied
             * lateron in the code generation
             */

        } else {

            q->address = new_location(q, free_bit, free_byte);
            if (verbose_ralloc) {
                log((m, "register_allocate node nr=%04d kind=%d %s %d:%d", p->nr, p->kind,
                     node_name[p->kind], q->address->first->x, q->address->next->x));
            }
#ifdef __DEBUG__
            log((m, "%s %d %s %d:%d", bar(level, '-'), q->nr, q->name, *free_bit, *free_byte));
#endif
        }
    }
}

void register_allocate_precall(tree p, int *free_bit, int *free_byte, boolean alloc_procs,      /* unused! */
                               int level)
{
    tree q;
    if ((p->first != NULL)
        && (p->first->kind == node_precall)
        ) {
#ifdef __DEBUG__
        trace;
        trace_subtree(p->first);
        trace;
#endif
        p = p->first;
        assert_kind(p->loc, p->first, node_procedure);

        /* allocate the result! */
        if (p->first->result != NULL) {
            tree temp = new_chain2(p->first->result, NULL);
            register_allocate_chain(temp, free_bit, free_byte, false, level);
        }

        q = p->first->first;
        if (q->first == NULL) {
            q = q->next;
        }
#ifdef __DEBUG__
        trace;
        trace_subtree(q);
        trace;
#endif
        while ((q != NULL)
               && (q->first != NULL)
               && (q->first->kind == node_decl)
            ) {
#ifdef __DEBUG__
            trace_subtree(q->first);
#endif
            register_allocate_chain(q, free_bit, free_byte, false, level);
#ifdef __DEBUG__
            trace;
            trace_subtree(q);
            trace;
#endif
            q = q->next;
        }
    }
}

void register_allocate(tree p, int free_bit, int free_byte, boolean alloc_procs, int level)
{
    tree np;
    stack_guard;
    for (np = NULL; p != NULL; p = np) {

        if (verbose_ralloc) {
            log((m, "register_allocate node [%03d,%02X,%02X] nr=%04d %s", level, free_byte,
                 free_bit, p->nr, node_name[p->kind]));
        }

        switch (p->kind) {

        case node_test:
        case node_label:
        case node_var:
        case node_type:
        case node_ref:
        case node_return:
        case node_precall:
        case node_w:
        case node_error:
        case node_org:
        case node_const:{
                /* do nothing */
                np = NULL;
                break;
            }

        case node_chain:{

                /* speedup: don't allocate again
                 * if that will not change anything
                 */
                if ((free_bit <= p->free_bit)
                    && (free_byte <= p->free_byte)
                    ) {
                    if (verbose_ralloc) {
                        log((m, "register_allocate chop at nr=%04d", p->nr));
                    }
                    np = NULL;
                } else {

                    p->free_bit = free_bit;
                    p->free_byte = free_byte;

                    register_allocate_precall(p, &free_bit, &free_byte, alloc_procs, level + 1);
                    register_allocate_chain(p, &free_bit, &free_byte, alloc_procs, level);
                    register_allocate(p->first, free_bit, free_byte, alloc_procs, level + 1);

                    /* won't work!!! */
                    /* next is done via looping */
                    /* np = p->next; */
                    register_allocate(p->next, free_bit, free_byte, alloc_procs, level + 1);

                    np = NULL;
                }

                break;
            }

        case node_asm:{
                if (p->opcode != opcode_call) {
                    break;
                }

                /* fallthrough! */
            }

        case node_call:{
                if (p->indirect) {
                    register_allocate(list_of_virtuals, free_bit, free_byte, true, level + 1);
                } else {
                    tree q = p->first->first;
                    if (q->first == NULL) {
                        q = q->next;
                    }
                    while ((q != NULL)
                           && (q->first != NULL)
                           && (q->first->kind == node_decl)
                        ) {
                        q = q->next;
                    }
                    register_allocate(q, free_bit, free_byte, true, level + 1);
                }

                /* next is done via looping */
                np = p->next;
                break;
            }

        case node_assign:
        case node_decl:
        case node_if:
        case node_for:
        case node_op:
        case node_while:{
                register_allocate(p->condition, free_bit, free_byte, alloc_procs, level);
                register_allocate(p->step, free_bit, free_byte, alloc_procs, level);
                register_allocate(p->end, free_bit, free_byte, alloc_procs, level);
                if ((p->first != NULL)
                    && (p->first->kind != node_procedure)
                    ) {
                    register_allocate(p->first, free_bit, free_byte, alloc_procs, level);
                }

                /* next is done via looping */
                np = p->next;
                break;
            }

        case node_procedure:{
                if ((!p->is_virtual)
                    || (alloc_procs)
                    ) {
                    register_allocate(p->first, free_bit, free_byte, alloc_procs, level);
                }

                /* next is done via looping */
                np = p->next;
                break;
            }

        default:{
                snark_node(p->loc, p);
                break;
            }
        }
    }
}

int interrupt_save_w = -1;
int interrupt_save_p = -1;

void ralloc(tree * p)
{
    int free_bit, free_byte, first_free;
    stack_guard;
    first_free = target_first_ram;

    if (interrupt_service_routine != NULL) {
        switch (target_chip) {

        case t_16c84:
        case t_16f84:{
                interrupt_save_w = first_free++;
                interrupt_save_p = first_free;
                first_free += 3;
                break;
            }

        case t_16f88:
        case t_16f877:
        case t_16f876:         /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        case t_16f873:         /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        case t_16f628:
        case t_12f629:
        case t_12f675:
        case t_18f242:
        case t_18f252:
        case t_18f442:
        case t_18f452:{
                interrupt_save_w = target_last_ram--;
                interrupt_save_p = first_free;
                first_free += 3;
                break;
            }

        case t_sx18:
        case t_sx28:{
                snark(NULL);
                break;
            }

        default:{
                log((m, "target chip = %d", target_chip));
                snark(NULL);
                break;
            }
        }
    }

    free_bit = first_free * 8;
    free_byte = first_free;

    /* gather virtual routines */
    list_of_virtuals = NULL;
    gather_virtuals(*p);
    if (verbose_ralloc) {
        log((m, "virtuals"));
        show_subtree(list_of_virtuals);
        log((m, "end of virtuals"));
    }

    /* allocate bits in main */
    used_bit = free_bit - 1;
    used_byte = free_byte - 1;
    register_allocate(*p, free_bit, free_byte, false, 1);

    /* allocate bits in interrupt */
    free_bit = used_bit + 1;
    register_allocate(interrupt_service_routine, free_bit, free_byte, false, 1);

    /* allocate bytes in main */
    free_bit = target_first_ram * 8;
    free_byte = (used_bit + 7) / 8;
    register_allocate(*p, free_bit, free_byte, false, 1);

    /* allocate bytes in interrupt */
    free_byte = used_byte + 1;
    register_allocate(interrupt_service_routine, free_bit, free_byte, false, 1);

#ifdef __DEBUG__
    if (0)
        log((m, "n=%d a=%d", used_byte, target_last_ram + 1));
#endif
}
