package analyzer

import (
	"testing"
)

func Test_isValidErrorTypeName_validTypes(t *testing.T) {
	for _, tt := range []string{
		"AddrError",
		"AmbiguousImportError",
		"BuildError",
		"BuildListError",
		"CertificateInvalidError",
		"CheckEqualError",
		"CheckError",
		"ConstraintError",
		"ConstraintViolationError",
		"CustomError",
		"CycleInRequiresGraphError",
		"DLLError",
		"DNSConfigError",
		"DNSError",
		"DeadlineExceededError",
		"DecodeError",
		"DecodingError",
		"DirectImportFromImplicitDependencyError",
		"DownloadDirPartialError",
		"EmbedError",
		"Error",
		"ERROR",
		"ExecError",
		"ExitError",
		"FileError",
		"FlagNotDefinedError",
		"FormatError",
		"HTTPError",
		"HostnameError",
		"ImportMismatchError",
		"ImportMissingError",
		"ImportMissingSumError",
		"InvalidPathError",
		"InvalidSignatureError",
		"InvalidUTF8Error",
		"InvalidUnmarshalError",
		"InvalidVersionError",
		"LinkError",
		"MarshalerError",
		"MatchError",
		"ModuleError",
		"ModuleRetractedError",
		"MultiplePackageError",
		"MyError",
		"NoGoError",
		"NoMatchingVersionError",
		"NoPatchBaseError",
		"NonFlagError",
		"NumError",
		"OpError",
		"PackageError",
		"PackageNotInModuleError",
		"ParseError",
		"PathError",
		"ProtocolError",
		"PtyError",
		"QueryMatchesMainModuleError",
		"QueryMatchesPackagesInMainModuleError",
		"ReadError",
		"RecordHeaderError",
		"ReferralError",
		"RunError",
		"StructuralError",
		"SyntaxError",
		"SyscallError",
		"SystemRootsError",
		"TagPathError",
		"TypeAssertionError",
		"UnknownAuthorityError",
		"UnknownLineError",
		"UnknownRevisionError",
		"UnknownVerifierError",
		"UnmarshalFieldError",
		"UnmarshalTypeError",
		"UnsupportedTypeError",
		"UnsupportedValueError",
		"UnverifiedNoteError",
		"VCSError",
		"ValueError",
		"WildcardInFirstElementError",
		"WriteError",
		"alwaysError",
		"ambiguousVerifierError",
		"badPathError",
		"boundsError",
		"cgoError",
		"conflictError",
		"deadlineExceededError",
		"entryNotFoundError",
		"err",
		"error",
		"excludedError",
		"formatError",
		"goModDirtyError",
		"gobError",
		"http2GoAwayError",
		"http2StreamError",
		"http2connError",
		"http2goAwayFlowError",
		"http2httpError",
		"http2noCachedConnError",
		"http2resAndError",
		"httpError",
		"importError",
		"invalidImportError",
		"invalidUnmarshalError",
		"issue22091Error",
		"jsonError",
		"labelError",
		"localError",
		"lookupDisabledError",
		"mainPackageError",
		"myError",
		"nestedError",
		"noCommitsError",
		"noWrapError",
		"notExistError",
		"nothingWrittenError",
		"onceError",
		"osError",
		"panicError",
		"pasteIndicatorError",
		"permanentError",
		"queryDisabledError",
		"requestBodyReadError",
		"responseAndError",
		"retractionLoadingError",
		"scanError",
		"statusError",
		"stringError",
		"sumMissingError",
		"summaryError",
		"temporaryError",
		"testError",
		"timeoutError",
		"timeoutTemporaryError",
		"tlsHandshakeTimeoutError",
		"transportReadFromServerError",
		"unexportedCustomError",
		"unexportedError",
		"unsupportedTEError",
		"wantedError",
		"withStackError",
		"withTimeoutError",
		"wrapError",
		"writeError",
		"writerWithReadFromError",
		"zipError",
	} {
		if !isValidErrorTypeName(tt) {
			t.Errorf("%q must be valid error type name", tt)
		}
	}
}

func Test_isValidErrorTypeName_invalidTypes(t *testing.T) {
	for _, tt := range []string{
		"AmbiguousImportErr",
		"BuildErr",
		"CertificateInvalidErr",
		"CheckEqualErrorErr",
		"CheckEqualErrorError",
		"Err",
		"ErrorBuildList",
		"ErrorBuildListError",
		"errWithStack",
		"errZIP",
		"nestedErr",
		"withStackErr",
	} {
		if isValidErrorTypeName(tt) {
			t.Errorf("%q must be invalid error type name", tt)
		}
	}
}

func Test_isValidErrorArrayTypeName(t *testing.T) {
	for _, tt := range []string{
		"ValidationErrors",
		"validationErrors",
		"ERRORS",
		"errors",
	} {
		if !isValidErrorArrayTypeName(tt) {
			t.Errorf("%q must be valid error array type name", tt)
		}
	}

	for _, tt := range []string{
		"ErrorsFromValidation",
		"errorsFromValidation",
	} {
		if isValidErrorArrayTypeName(tt) {
			t.Errorf("%q must be invalid error array type name", tt)
		}
	}
}

func Test_isValidErrorVarName_validVars(t *testing.T) {
	for _, tt := range []string{
		"EOF",
		"ErrAbortHandler",
		"ErrAdvanceTooFar",
		"ErrAlgorithm",
		"ErrBadConn",
		"ErrBadName",
		"ErrBadPattern",
		"ErrBadReadCount",
		"ErrBadStat",
		"ErrBareQuote",
		"ErrBodyNotAllowed",
		"ErrBodyReadAfterClose",
		"ErrBufferFull",
		"ErrChecksum",
		"ErrClosed",
		"ErrClosedPipe",
		"ErrConnClosed",
		"ErrConnDone",
		"ErrContentLength",
		"ErrDecryption",
		"ErrDictionary",
		"ErrDisallowed",
		"ErrEndOfSpan",
		"ErrExecError",
		"ErrExist",
		"ErrFieldCount",
		"ErrFieldTooLong",
		"ErrFileClosing",
		"ErrFinalToken",
		"ErrFlagTerminator",
		"ErrFormat",
		"ErrFrameTooLarge",
		"ErrGONOSUMDB",
		"ErrHandlerTimeout",
		"ErrHeader",
		"ErrHeaderNotPresent",
		"ErrHelp",
		"ErrHijacked",
		"ErrInvalid",
		"ErrInvalidHuffman",
		"ErrInvalidMediaParameter",
		"ErrInvalidMode",
		"ErrInvalidPublicKey",
		"ErrInvalidUnreadByte",
		"ErrInvalidUnreadRune",
		"ErrLength",
		"ErrLineTooLong",
		"ErrMessageTooLarge",
		"ErrMessageTooLong",
		"ErrMissingFile",
		"ErrNegativeAdvance",
		"ErrNegativeCount",
		"ErrNoCookie",
		"ErrNoDeadline",
		"ErrNoLocation",
		"ErrNoModRoot",
		"ErrNoProgress",
		"ErrNoRows",
		"ErrNoSymbols",
		"ErrNoTrustSettings",
		"ErrNotExist",
		"ErrNotFound",
		"ErrNotMangledName",
		"ErrNotPollable",
		"ErrNotStarted",
		"ErrNotSupported",
		"ErrPermission",
		"ErrProcessDone",
		"ErrPushLimitReached",
		"ErrQuote",
		"ErrRange",
		"ErrRecursivePush",
		"ErrRemoveArgument",
		"ErrRequestAborted",
		"ErrSectionDone",
		"ErrSecurity",
		"ErrServerClosed",
		"ErrShortBuffer",
		"ErrShortDst",
		"ErrShortSrc",
		"ErrShortStat",
		"ErrShortWrite",
		"ErrShutdown",
		"ErrSkip",
		"ErrSkipAltProtocol",
		"ErrStringLength",
		"ErrSyntax",
		"ErrTimeout",
		"ErrTooLarge",
		"ErrTooLong",
		"ErrTooManyErrors",
		"ErrTrailingComma",
		"ErrTruncated",
		"ErrTxDone",
		"ErrUnexpectedEOF",
		"ErrUnexpectedType",
		"ErrUnknownPC",
		"ErrUnrecognized",
		"ErrUnsupportedAlgorithm",
		"ErrUseLastResponse",
		"ErrVerification",
		"ErrWriteAfterClose",
		"ErrWriteAfterFlush",
		"ErrWriteConflict",
		"ErrWriteToConnected",
		"ErrWriteTooLong",
		"errBad",
		"errBadCgo",
		"errBadComparison",
		"errBadComparisonType",
		"errBadCount",
		"errBadELF",
		"errBadPixel",
		"errBadTestInputs",
		"errBadType",
		"errBadUint",
		"errBaseLen",
		"errCalcLen",
		"errCallerOwnsConn",
		"errCanceled",
		"errCannotMarshalDNSMessage",
		"errCannotRewind",
		"errCannotUnmarshalDNSMessage",
		"errClientConnClosed",
		"errClientConnGotGoAway",
		"errClientConnUnusable",
		"errClientDisconnected",
		"errClientKeyExchange",
		"errCloseConn",
		"errCloseIdle",
		"errCloseIdleConns",
		"errClosed",
		"errClosedBody",
		"errClosedPipeWrite",
		"errClosedResponseBody",
		"errComplex",
		"errCompressedSRV",
		"errConnBroken",
		"errCorruptArchive",
		"errCorruptObject",
		"errDBClosed",
		"errDepStreamID",
		"errDial",
		"errDirectoryNotFound",
		"errDiscardedBytes",
		"errEarlyCloseWrite",
		"errFail",
		"errFake",
		"errFakeConnSessionDirty",
		"errFakeRoundTrip",
		"errFileTooNew",
		"errGoBuildWithoutBuild",
		"errGoModCase",
		"errHandlerComplete",
		"errHandlerPanicked",
		"errHgArchivalTxt",
		"errIO",
		"errIdleConnTimeout",
		"errIllegalDomain",
		"errInconsistentByteCount",
		"errInternal",
		"errInvalSep",
		"errInvalidAddr",
		"errInvalidDNSResponse",
		"errInvalidDelim",
		"errInvalidInterface",
		"errInvalidInterfaceIndex",
		"errInvalidInterfaceName",
		"errInvalidMessage",
		"errInvalidPtr",
		"errInvalidSigner",
		"errInvalidWord",
		"errInvalidWrite",
		"errKeepAlivesDisabled",
		"errLameReferral",
		"errLeadingInt",
		"errLocation",
		"errLongExtra",
		"errLongName",
		"errMain",
		"errMalformedDomain",
		"errMalformedNote",
		"errMalformedRecord",
		"errMalformedTree",
		"errMessageMismatch",
		"errMessageTooShort",
		"errMissData",
		"errMissing",
		"errMissingAddress",
		"errMissingHost",
		"errMissingParams",
		"errMissingReadDir",
		"errMissingSeek",
		"errMissingU8Prefix",
		"errMixPseudoHeaderTypes",
		"errMultipleGoBuild",
		"errNUL",
		"errNeedMore",
		"errNegativeRead",
		"errNegativeWrite",
		"errNilPanicOrGoexit",
		"errNilPtr",
		"errNilResouceBody",
		"errNoAnswerFromDNSServer",
		"errNoAvailableAddress",
		"errNoAvailableInterface",
		"errNoBody",
		"errNoBuildID",
		"errNoCertificates",
		"errNoChange",
		"errNoComparison",
		"errNoDigits",
		"errNoFallocate",
		"errNoHostname",
		"errNoModules",
		"errNoOverlap",
		"errNoRows",
		"errNoSuchHost",
		"errNoSuchInterface",
		"errNoSuchMulticastInterface",
		"errNoSuitableAddress",
		"errNonCanonicalName",
		"errNotAbsolute",
		"errNotConstraint",
		"errNotDir",
		"errNotEnough",
		"errNotObject",
		"errNotParsed",
		"errNotPermitted",
		"errNotRegular",
		"errOffline",
		"errOffset",
		"errOpen",
		"errOutOfCodes",
		"errPadBytes",
		"errPadLength",
		"errParse",
		"errPathNotClean",
		"errPathNotRelative",
		"errPatternHasSeparator",
		"errPkgIsBuiltin",
		"errPkgIsGorootSrc",
		"errPrefaceTimeout",
		"errPrintedOutput",
		"errProofFailed",
		"errPseudoAfterRegular",
		"errPseudoSyntax",
		"errPublicExponentLarge",
		"errPublicExponentSmall",
		"errPublicModulus",
		"errRange",
		"errRawToken",
		"errReadEmpty",
		"errReadLoopExiting",
		"errReadOnClosedResBody",
		"errRemoteOriginNotFound",
		"errReqBodyTooLong",
		"errRequestCanceled",
		"errRequestCanceledConn",
		"errRequestHeaderListSize",
		"errResTooLong",
		"errReserved",
		"errResourceLen",
		"errResponseHeaderListSize",
		"errRevQuery",
		"errRowsClosed",
		"errSeeker",
		"errSegTooLong",
		"errSegmentSelector",
		"errServerClosedIdle",
		"errServerKeyExchange",
		"errServerMisbehaving",
		"errServerTemporarilyMisbehaving",
		"errShortBuffer",
		"errShortInternal",
		"errShutdown",
		"errSignerAlg",
		"errSignerHash",
		"errSignerID",
		"errStopReqBodyWrite",
		"errStopReqBodyWriteAndCancel",
		"errStreamClosed",
		"errStreamID",
		"errStringTooLong",
		"errSubmoduleDir",
		"errSubmoduleFile",
		"errSymlink",
		"errSyntax",
		"errTagKeySyntax",
		"errTagSpace",
		"errTagSyntax",
		"errTagValueSpace",
		"errTagValueSyntax",
		"errTestScanWrap",
		"errTooBig",
		"errTooLarge",
		"errTooManyAdditionals",
		"errTooManyAnswers",
		"errTooManyAuthorities",
		"errTooManyIdle",
		"errTooManyIdleHost",
		"errTooManyPtr",
		"errTooManyQuestions",
		"errTooMuch",
		"errTrailerEOF",
		"errTruncatedArchive",
		"errUnknownSite",
		"errUnreadByte",
		"errUnrefData",
		"errUnsupportedMessage",
		"errVCS",
		"errValue",
		"errVendored",
		"errVerifierAlg",
		"errVerifierHash",
		"errVerifierID",
		"errVerifyMode",
		"errVersionChange",
		"errWhence",
		"errWriteAtInAppendMode",
		"errWriteHole",
		"errZeroParam",
		"errZeroSegLen",
	} {
		if !isValidErrorVarName(tt) {
			t.Errorf("%q must be valid error var name", tt)
		}
	}
}

func Test_isValidErrorVarName_invalidVars(t *testing.T) {
	for _, tt := range []string{
		"Canceled",
		"ErrExecErr",
		"ErrorExec",
		"IncorrectPasswordError",
		"SkipDir",
		"alwaysError",
		"atoiError",
		"badData",
		"badErr",
		"boolError",
		"brokenConnErr",
		"checkErr",
		"complexError",
		"fakeErr",
		"isInternal",
		"isNotExistErr",
		"myError",
		"outErr",
		"pingError",
		"sentinelErr",
		"someErr",
		"statErr",
		"testError",
		"timeoutProtoErr",
		"viewCloseError",
		"viewError",
	} {
		if isValidErrorVarName(tt) {
			t.Errorf("%q must be invalid error var name", tt)
		}
	}
}

func Test_isErrorType(t *testing.T) {
	t.Run("custom error type", func(t *testing.T) {
		b := isErrorType(
			"AddrError",
			stringSet{"AddrError": {}, "User": {}},
			stringSet{"AddrError": {}},
		)
		if !b {
			t.Fail()
		}
	})

	t.Run("builtin error type", func(t *testing.T) {
		b := isErrorType(
			"error",
			stringSet{"AddrError": {}, "User": {}},
			stringSet{"AddrError": {}},
		)
		if !b {
			t.Fail()
		}
	})

	t.Run("collided error type (without Error method)", func(t *testing.T) {
		b := isErrorType(
			"error",
			stringSet{"AddrError": {}, "User": {}, "error": {}},
			stringSet{"AddrError": {}},
		)
		if b {
			t.Fail()
		}
	})

	t.Run("collided error type (with Error method)", func(t *testing.T) {
		b := isErrorType(
			"error",
			stringSet{"AddrError": {}, "User": {}, "error": {}},
			stringSet{"AddrError": {}, "error": {}},
		)
		if !b {
			t.Fail()
		}
	})
}

func Test_isInitialism(t *testing.T) {
	for _, tt := range []struct {
		in       string
		expected bool
	}{
		{in: "E", expected: true},
		{in: "e", expected: true},
		{in: "EOF", expected: true},
		{in: "eof", expected: true},
		{in: "ERPCMISMATCH", expected: true},
		{in: "EndOfFile", expected: false},
		{in: "withStackError", expected: false},
		{in: "badData", expected: false},
	} {
		t.Run(tt.in, func(t *testing.T) {
			if isInitialism(tt.in) != tt.expected {
				t.Fail()
			}
		})
	}
}
