/*
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include "sgtk-hig.h"
#include "sgtk-util.h"
#include "st-network-preferences-page.h"
#include "st-preferences.h"
#include "st-settings.h"
#include "st-util.h"
#include "st-util-api.h"

/*** type definitions ********************************************************/

struct _STNetworkPreferencesPagePrivate
{
  GtkWidget	*proxy_check;
  GtkWidget	*type_label;
  GtkWidget	*http_radio;
  GtkWidget	*socks5_radio;
  GtkWidget	*server_label;
  GtkWidget	*server_entry;
  GtkWidget	*port_label;
  GtkWidget	*port_spin;
  GtkWidget	*auth_check;
  GtkWidget	*name_label;
  GtkWidget	*name_entry;
  GtkWidget	*password_label;
  GtkWidget	*password_entry;
};
  
/*** function declarations ***************************************************/

static void st_network_preferences_page_class_init (STNetworkPreferencesPageClass *class);
static void st_network_preferences_page_init (STNetworkPreferencesPage *page);

static void st_network_preferences_page_update_sensitivity (STNetworkPreferencesPage *page);

/*** implementation **********************************************************/

GType
st_network_preferences_page_get_type (void)
{
  static GType type = 0;
  
  if (! type)
    {
      static const GTypeInfo info = {
	sizeof(STNetworkPreferencesPageClass),
	NULL,
	NULL,
	(GClassInitFunc) st_network_preferences_page_class_init,
	NULL,
	NULL,
	sizeof(STNetworkPreferencesPage),
	0,
	(GInstanceInitFunc) st_network_preferences_page_init,
      };

      type = g_type_register_static(ST_TYPE_PREFERENCES_PAGE,
				    "STNetworkPreferencesPage",
				    &info,
				    0);
    }

  return type;
}

static void
st_network_preferences_page_class_init (STNetworkPreferencesPageClass *class)
{
  g_type_class_add_private(class, sizeof(STNetworkPreferencesPagePrivate));
}

static void
st_network_preferences_page_init (STNetworkPreferencesPage *page)
{
  STPreferencesPage *ppage = ST_PREFERENCES_PAGE(page);
  GtkWidget *table;
  GtkWidget *type_box;
  GtkWidget *server_box;

  page->priv = G_TYPE_INSTANCE_GET_PRIVATE(page, ST_TYPE_NETWORK_PREFERENCES_PAGE, STNetworkPreferencesPagePrivate);

  st_preferences_page_set_name(ppage, "network");
  st_preferences_page_set_stock_id(ppage, GTK_STOCK_NETWORK);
  st_preferences_page_set_label(ppage, _("Network"));
  st_preferences_page_set_help_link_id(ppage, "preferences-network");

  page->priv->proxy_check = gtk_check_button_new_with_mnemonic(_("_Use a proxy server"));
  st_set_tooltip(page->priv->proxy_check, _("If this option is enabled and the server field is not empty, streamtuner will use that proxy server for data transfers."));
  
  page->priv->type_label = gtk_label_new(_("Type:"));

  type_box = gtk_hbox_new(FALSE, SGTK_HIG_CONTROL_SPACING);
  page->priv->http_radio = gtk_radio_button_new_with_mnemonic(NULL, _("_HTTP"));
  page->priv->socks5_radio = gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(page->priv->http_radio), _("_Socks 5"));
  gtk_box_pack_start(GTK_BOX(type_box), page->priv->http_radio, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(type_box), page->priv->socks5_radio, FALSE, FALSE, 0);

  st_set_tooltip(page->priv->http_radio, _("Select this option if your proxy server is a HTTP proxy."));
  st_set_tooltip(page->priv->socks5_radio, _("Select this option if your proxy server is a Socks 5 proxy."));

  page->priv->server_label = gtk_label_new_with_mnemonic(_("S_erver:"));

  server_box = gtk_hbox_new(FALSE, SGTK_HIG_CONTROL_LABEL_SPACING);
  page->priv->server_entry = gtk_entry_new();
  page->priv->port_label = gtk_label_new_with_mnemonic(_("P_ort:"));
  page->priv->port_spin = gtk_spin_button_new_with_range(0, 65535, 1);
  gtk_box_pack_start(GTK_BOX(server_box), page->priv->server_entry, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(server_box), page->priv->port_label, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(server_box), page->priv->port_spin, FALSE, FALSE, 0);

  gtk_label_set_mnemonic_widget(GTK_LABEL(page->priv->server_label), page->priv->server_entry);
  gtk_label_set_mnemonic_widget(GTK_LABEL(page->priv->port_label), page->priv->port_spin);

  st_set_tooltip(page->priv->server_entry, _("The hostname or IP address of the proxy server"));
  st_set_tooltip(page->priv->port_spin,
		 _("The port number of the proxy server.\n"
		   "\n"
		   "HTTP proxies commonly use port 8080 or 3128.\n"
		   "Socks 5 proxies commonly use port 1080.\n"
		   "\n"
		   "Contact your network administrator for more details."));

  page->priv->auth_check = gtk_check_button_new_with_mnemonic(_("Use proxy _authentication"));
  st_set_tooltip(page->priv->auth_check, _("Enable this option if your proxy server requires user authentication."));

  page->priv->name_label = gtk_label_new_with_mnemonic(_("_Name:"));
  page->priv->name_entry = gtk_entry_new();
  gtk_label_set_mnemonic_widget(GTK_LABEL(page->priv->name_label), page->priv->name_entry);

  st_set_tooltip(page->priv->name_entry,
		 _("Your proxy login name.\n"
		   "\n"
		   "If left blank, you will be prompted for your login name when needed."));

  page->priv->password_label = gtk_label_new_with_mnemonic(_("_Password:"));
  page->priv->password_entry = gtk_entry_new();
  gtk_label_set_mnemonic_widget(GTK_LABEL(page->priv->password_label), page->priv->password_entry);

  gtk_entry_set_visibility(GTK_ENTRY(page->priv->password_entry), FALSE);
  st_set_tooltip(page->priv->password_entry,
		 _("Your proxy password.\n"
		   "\n"
		   "If left blank, you will be prompted for your password when needed."));

  gtk_misc_set_alignment(GTK_MISC(page->priv->type_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(page->priv->server_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(page->priv->name_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(page->priv->password_label), 1, 0.5);

  st_preferences_bind_boolean(GTK_TOGGLE_BUTTON(page->priv->proxy_check),
			      &st_settings.proxy_enabled);
  st_preferences_bind_int_radio(GTK_RADIO_BUTTON(page->priv->http_radio),
				(int *) &st_settings.proxy_type,
				ST_TRANSFER_PROXY_HTTP);
  st_preferences_bind_int_radio(GTK_RADIO_BUTTON(page->priv->socks5_radio),
				(int *) &st_settings.proxy_type,
				ST_TRANSFER_PROXY_SOCKS5);
  st_preferences_bind_string(GTK_ENTRY(page->priv->server_entry),
			     &st_settings.proxy_server);
  st_preferences_bind_int_spin(GTK_SPIN_BUTTON(page->priv->port_spin),
			       &st_settings.proxy_port);
  st_preferences_bind_boolean(GTK_TOGGLE_BUTTON(page->priv->auth_check),
			      &st_settings.proxy_auth_enabled);
  st_preferences_bind_string(GTK_ENTRY(page->priv->name_entry),
			     &st_settings.proxy_auth_name);
  st_preferences_bind_string(GTK_ENTRY(page->priv->password_entry),
			     &st_settings.proxy_auth_password);

  g_signal_connect_swapped(page->priv->proxy_check, "toggled", G_CALLBACK(st_network_preferences_page_update_sensitivity), page);
  g_signal_connect_swapped(page->priv->auth_check, "toggled", G_CALLBACK(st_network_preferences_page_update_sensitivity), page);

  sgtk_entry_set_next_widget(GTK_ENTRY(page->priv->server_entry), page->priv->port_spin);
  sgtk_entry_set_next_widget(GTK_ENTRY(page->priv->port_spin), page->priv->name_entry);
  sgtk_entry_set_next_widget(GTK_ENTRY(page->priv->name_entry), page->priv->password_entry);
  sgtk_entry_set_next_widget(GTK_ENTRY(page->priv->password_entry), page->priv->server_entry);

  table = gtk_table_new(6, 2, FALSE);

  gtk_table_set_row_spacings(GTK_TABLE(table), SGTK_HIG_CONTROL_SPACING);
  gtk_table_set_col_spacings(GTK_TABLE(table), SGTK_HIG_CONTROL_LABEL_SPACING);

  gtk_table_attach(GTK_TABLE(table), page->priv->proxy_check, 0, 2, 0, 1,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), page->priv->type_label, 0, 1, 1, 2,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), type_box, 1, 2, 1, 2,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), page->priv->server_label, 0, 1, 2, 3,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), server_box, 1, 2, 2, 3,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), page->priv->auth_check, 0, 2, 3, 4,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), page->priv->name_label, 0, 1, 4, 5,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), page->priv->name_entry, 1, 2, 4, 5,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), page->priv->password_label, 0, 1, 5, 6,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), page->priv->password_entry, 1, 2, 5, 6,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);

  gtk_widget_show_all(table);
  gtk_box_pack_start(GTK_BOX(page), table, FALSE, FALSE, 0);

  st_network_preferences_page_update_sensitivity(page);
}

static void
st_network_preferences_page_update_sensitivity (STNetworkPreferencesPage *page)
{
  gboolean proxy, auth;

  g_return_if_fail(ST_IS_NETWORK_PREFERENCES_PAGE(page));

  proxy = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(page->priv->proxy_check));
  auth = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(page->priv->auth_check));

  gtk_widget_set_sensitive(page->priv->type_label, proxy); 
  gtk_widget_set_sensitive(page->priv->http_radio, proxy); 
  gtk_widget_set_sensitive(page->priv->socks5_radio, proxy); 
  gtk_widget_set_sensitive(page->priv->server_label, proxy); 
  gtk_widget_set_sensitive(page->priv->server_entry, proxy);
  gtk_widget_set_sensitive(page->priv->port_label, proxy); 
  gtk_widget_set_sensitive(page->priv->port_spin, proxy);
  gtk_widget_set_sensitive(page->priv->auth_check, proxy);
  gtk_widget_set_sensitive(page->priv->name_label, proxy && auth);
  gtk_widget_set_sensitive(page->priv->name_entry, proxy && auth);
  gtk_widget_set_sensitive(page->priv->password_label, proxy && auth);
  gtk_widget_set_sensitive(page->priv->password_entry, proxy && auth);
}

STPreferencesPage *
st_network_preferences_page_new (void)
{
  return g_object_new(ST_TYPE_NETWORK_PREFERENCES_PAGE, NULL);
}
