import type * as ts from "./tsserverlibrary.shim"
import {
  areTypesMutuallyAssignableTsServer,
  getElementTypeTsServer,
  getSymbolTypeTsServer,
  getTypePropertiesTsServer
} from "./get-element-type-ts-server"
import {
  AreTypesMutuallyAssignableArguments,
  GetElementTypeArguments,
  GetSymbolTypeArguments,
  GetTypePropertiesArguments,
  Range
} from "./protocol"
import {areTypesMutuallyAssignable, getElementType, getSymbolType, getTypeProperties, ReverseMapper} from "./ide-get-element-type"

type tsServerHandler<T> = (ts: typeof import("./tsserverlibrary.shim"),
                           projectService: ts.server.ProjectService,
                           requestArgument: T) => ts.server.HandlerResponse | undefined

type lspHandler<T> = (ts: typeof import("./tsserverlibrary.shim"),
                      requestArgument: T,
                      context: LspSupport) => Promise<ts.server.HandlerResponse | undefined>

interface IdeCommandArgumentsTypes {
  "ideGetElementType": GetElementTypeArguments
  "ideGetSymbolType": GetSymbolTypeArguments
  "ideGetTypeProperties": GetTypePropertiesArguments
  "ideAreTypesMutuallyAssignable": AreTypesMutuallyAssignableArguments
  "ideCloseSafely": ts.server.protocol.FileRequestArgs
  "ideEnsureFileAndProjectOpenedCommand": ts.server.protocol.FileLocationRequestArgs,
  "ideTestCancellation": {}
}

const customHandlers: {
  [K in keyof IdeCommandArgumentsTypes]: [tsServerHandler<IdeCommandArgumentsTypes[K]>, lspHandler<IdeCommandArgumentsTypes[K]> | undefined]
} = {
  "ideGetElementType": [getElementTypeTsServer, getElementTypeLsp],
  "ideGetSymbolType": [getSymbolTypeTsServer, getSymbolTypeLsp],
  "ideGetTypeProperties": [getTypePropertiesTsServer, getTypePropertiesLsp],
  "ideAreTypesMutuallyAssignable": [areTypesMutuallyAssignableTsServer, areTypesMutuallyAssignableLsp],
  "ideCloseSafely": [closeSafelyTsServer, undefined /* not supported on LSP */],
  "ideEnsureFileAndProjectOpenedCommand": [ensureFileAndProjectOpenedTsServer, undefined /* not supported on LSP */],
  "ideTestCancellation": [testCancellationTsServer, testCancellationLsp], // Special command to test cancellation support on the server
}

/** This method is used to register handlers for TS 5+ */
export function registerProtocolHandlers(
  session: ts.server.Session,
  ts: typeof import("./tsserverlibrary.shim"),
  projectService: ts.server.ProjectService,
) {
  for (let command in customHandlers) {
    session.addProtocolHandler(command, (request: ts.server.protocol.Request) => {
      try {
        return customHandlers[command as keyof IdeCommandArgumentsTypes][0](ts, projectService, request.arguments) || emptyDoneResponse()
      }
      catch (e) {
        if ((e as Error).isOperationCancelledError) {
          return cancelledResponse()
        }
        else if ((e as Error).isFileOutsideOfImportGraphError) {
          return fileOutsideOfImportGraphResponse()
        }
        else {
          throw e
        }
      }
    });
  }
}

/** This method is used by the old session provider logic for TS <5 **/
export function initCommandNamesForSessionProvider(TypeScriptCommandNames: any) {
  for (let command in customHandlers) {
    TypeScriptCommandNames[command] = command
  }
}

/** This method is used by the old session provider logic for TS <5  **/
export function tryHandleTsServerCommand(ts_impl: typeof import("./tsserverlibrary.shim"),
                                         projectService: ts.server.ProjectService,
                                         request: ts.server.protocol.Request): ts.server.HandlerResponse | undefined {
  try {
    return customHandlers[request.command as keyof IdeCommandArgumentsTypes]?.[0]?.(ts_impl, projectService, request.arguments)
  }
  catch (e) {
    return processError(e)
  }
}


export interface LspSupport {
  cancellationToken: ts.CancellationToken,
  process<T>(fileUri: string, range: Range | undefined, processor: (context: LspProcessingContext) => T): Promise<T | undefined>
}

export interface LspProcessingContext {
  cancellationToken: ts.CancellationToken,
  languageService: ts.LanguageService,
  program: ts.Program,
  sourceFile: ts.SourceFile,
  range: Range | undefined,
  reverseMapper: ReverseMapper
}

export async function tryHandleCustomTsServerCommandLsp(ts: typeof import("./tsserverlibrary.shim"),
                                                        commandName: any,
                                                        requestArguments: any,
                                                        context: LspSupport): Promise<ts.server.HandlerResponse | undefined> {
  return customHandlers[commandName as keyof IdeCommandArgumentsTypes]?.[1]?.(ts, requestArguments, context)
}

function closeSafelyTsServer(_ts: typeof import("./tsserverlibrary.shim"),
                             projectService: ts.server.ProjectService,
                             requestArguments: ts.server.protocol.FileRequestArgs) {
  projectService.ideProjectService.closeClientFileSafely(requestArguments.file)
  return notRequiredResponse()
}

function ensureFileAndProjectOpenedTsServer(_ts: typeof import("./tsserverlibrary.shim"),
                                            projectService: ts.server.ProjectService,
                                            requestArguments: ts.server.protocol.FileLocationRequestArgs): ts.server.HandlerResponse {
  // Ensure that there is a project opened for the file
  projectService.ideProjectService.getProjectAndSourceFile(requestArguments.file, requestArguments.projectFileName)
  return emptyDoneResponse()
}

let lastIdeProjectId = 0

function getElementTypeLsp(ts: typeof import("./tsserverlibrary.shim"), requestArgument: GetElementTypeArguments,
                           lspSupport: LspSupport) {
  return lspSupport.process(requestArgument.file, requestArgument.range, context => {
    const ideProjectId = context.languageService.ideProjectId ?? (context.languageService.ideProjectId = lastIdeProjectId++)
    const range = context.range
    if (!range) return undefined
    // TODO consider using languageService.webStormGetElementType
    return getElementType(ts, ideProjectId, context.program, context.sourceFile, range, requestArgument.forceReturnType,
                          context.cancellationToken, context.reverseMapper)
  })
}

function getSymbolTypeLsp(ts: typeof import("./tsserverlibrary.shim"), requestArgument: GetSymbolTypeArguments,
                          lspSupport: LspSupport) {
  return lspSupport.process(requestArgument.originalRequestUri, undefined, context => {
    if (context.languageService.ideProjectId !== requestArgument.ideProjectId
      || context.program.getTypeChecker().webStormCacheInfo?.ideTypeCheckerId !== requestArgument.ideTypeCheckerId) {
      return undefined;
    }
    return getSymbolType(ts, context.program, requestArgument.symbolId, context.cancellationToken, context.reverseMapper)
  })
}

function getTypePropertiesLsp(ts: typeof import("./tsserverlibrary.shim"), requestArgument: GetTypePropertiesArguments,
                              lspSupport: LspSupport) {
  return lspSupport.process(requestArgument.originalRequestUri, undefined, context => {
    if (context.languageService.ideProjectId !== requestArgument.ideProjectId
      || context.program.getTypeChecker().webStormCacheInfo?.ideTypeCheckerId !== requestArgument.ideTypeCheckerId) {
      return undefined;
    }
    return getTypeProperties(ts, context.program, requestArgument.typeId, context.cancellationToken, context.reverseMapper)
  })
}

function areTypesMutuallyAssignableLsp(ts: typeof import("./tsserverlibrary.shim"), requestArgument: AreTypesMutuallyAssignableArguments,
                                       lspSupport: LspSupport) {
  return lspSupport.process(requestArgument.originalRequestUri, undefined, context => {
    if (context.languageService.ideProjectId !== requestArgument.ideProjectId
      || context.program.getTypeChecker().webStormCacheInfo?.ideTypeCheckerId !== requestArgument.ideTypeCheckerId) {
      return undefined;
    }
    return areTypesMutuallyAssignable(ts, context.program, requestArgument.type1Id, requestArgument.type2Id, context.cancellationToken)
  })
}

function testCancellationTsServer(_ts: typeof import("./tsserverlibrary.shim"),
                                  projectService: ts.server.ProjectService,
                                  _requestArguments: {}): ts.server.HandlerResponse | undefined {
  const start = new Date().getTime()
  while (!projectService.cancellationToken.isCancellationRequested()) {
    // not possible to use promise here
    if (new Date().getTime() - start > 10_000) {
      throw new Error("Test cancellation timeout - waited 10s")
    }
  }
  return cancelledResponse()
}

async function testCancellationLsp(_ts: typeof import("./tsserverlibrary.shim"),
                                   _requestArgument: {},
                                   lspSupport: LspSupport): Promise<ts.server.HandlerResponse | undefined> {
  const start = new Date().getTime()
  while (!lspSupport.cancellationToken.isCancellationRequested()) {
    if (new Date().getTime() - start > 10_000) {
      throw new Error("Test cancellation timeout - waited 10s")
    }
  }
  return cancelledResponse()
}

function notRequiredResponse() {
  return {
    responseRequired: false
  }
}

function emptyDoneResponse(): { response: any; responseRequired: true } {
  return {
    responseRequired: true,
    response: null
  }
}

function cancelledResponse() {
  return {
    responseRequired: true,
    response: {
      cancelled: true
    }
  }
}

function fileOutsideOfImportGraphResponse() {
  return {
    responseRequired: true,
    response: {
      error: "file-outside-of-import-graph"
    }
  }
}