/* vim:set ts=2 sw=2 sts=2 et: */
/**
 * \author     Marcus Holland-Moritz (github@mhxnet.de)
 * \copyright  Copyright (c) Marcus Holland-Moritz
 *
 * This file is part of dwarfs.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the “Software”), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * SPDX-License-Identifier: MIT
 */

#pragma once

#include <condition_variable>
#include <cstdint>
#include <mutex>

namespace dwarfs {

class counting_semaphore {
 public:
  void post(int64_t n) {
    {
      std::lock_guard lock(mx_);
      count_ += n;
    }
    cond_.notify_all();
  }

  int64_t wait(int64_t n) {
    std::unique_lock lock(mx_);
    cond_.wait(lock, [&] { return count_ >= n; });
    count_ -= n;
    return n;
  }

 private:
  std::mutex mx_;
  std::condition_variable cond_;
  int64_t count_{0};
};

class scoped_lease {
 public:
  scoped_lease(counting_semaphore& sem, int64_t n)
      : sem_{&sem}
      , n_{sem_->wait(n)} {}

  ~scoped_lease() { release(); }

  void release() {
    if (n_ > 0) {
      sem_->post(n_);
      n_ = 0;
    }
  }

  int64_t size() const noexcept { return n_; }

  void shrink(int64_t n) {
    if (n <= n_) {
      sem_->post(n_ - n);
      n_ = n;
    }
  }

  scoped_lease(scoped_lease const&) = delete;
  scoped_lease& operator=(scoped_lease const&) = delete;

  scoped_lease(scoped_lease&& other) noexcept = delete;
  scoped_lease& operator=(scoped_lease&& other) noexcept = delete;

 private:
  counting_semaphore* sem_{nullptr};
  int64_t n_{0};
};

} // namespace dwarfs
