// -*- indent-tabs-mode: nil -*-

#include <cppunit/extensions/HelperMacros.h>

#include <string>
#include <fstream>
#include <iostream>
#include <sys/stat.h>
#include <dirent.h>
#include <cerrno>
#include <sys/wait.h>
#include <spawn.h>
#include <signal.h>
#include <unistd.h>

#include "../canlxx.h"
#include "../fileutil.h"
#include "../opensslgenericutil.h"

class CredentialsTest
  : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(CredentialsTest);
  CPPUNIT_TEST(TestCredentialsRequest);
  CPPUNIT_TEST(TestCredentials);
  CPPUNIT_TEST_SUITE_END();

public:
  CredentialsTest() : opensslcnf("openssl.cnf"), certfile("cert.pem"),
    keyfile("key.pem"), crlfile("crl.pem"), cadir("testCA"),
    trusted_cadir("trusted_certificates"), eec_subject("CN=Test,OU=ARC,OU=EMI,O=EU"),
    ocsp_responder_uri("http://localhost:8888") {}
  void setUp();
  void tearDown();
  void TestCredentialsRequest();
  void TestCredentials();

private:
  void createOpenSSLconf();
  void launch_ocsp_responder();
  void kill_ocsp_responder();
  std::string opensslcnf;
  std::string certfile;
  std::string keyfile;
  std::string cadir;
  std::string crlfile;
  std::string trusted_cadir;
  std::string eec_subject;
  std::string ocsp_responder_uri;
  pid_t pid;
};

void CredentialsTest::createOpenSSLconf() {
  //The configuration file does not include a typical complete information, 
  //only the info that is needed in this test
  std::ofstream f(opensslcnf.c_str(), std::ifstream::trunc);
  f << "[ ca ]" << std::endl;
  f << "default_ca  = CA_default" << std::endl;
  f << "[ CA_default ]" << std::endl;
  f << "dir             = ./" << cadir << std::endl;
  f << "certs           = $dir/certs" << std::endl;
  f << "crl_dir         = $dir/crl" << std::endl;
  f << "database        = $dir/index.txt" << std::endl;
  f << "new_certs_dir   = $dir/newcerts" << std::endl;
  f << "certificate     = $dir/cacert.pem" << std::endl;
  f << "serial          = $dir/serial" << std::endl;
  f << "crlnumber       = $dir/crlnumber" << std::endl;
  f << "crl             = $dir/crl.pem" << std::endl;
  f << "private_key     = $dir/cakey.pem" << std::endl;
  //f << "private_key     = $dir/private/cakey.pem" << std::endl;
  //f << "RANDFILE        = $dir/private/.rand" << std::endl;
  f << "x509_extensions = usr_cert" << std::endl;
  f << "default_days    = 365" << std::endl;
  f << "default_crl_days= 30" << std::endl;
  f << "default_md      = sha512" << std::endl;
  f << "[ req ]" << std::endl;
  f << "default_bits    = 1024" << std::endl;
  f << "default_keyfile = privkey.pem" << std::endl;
  f << "distinguished_name  = req_distinguished_name" << std::endl;
  f << "attributes          = req_attributes" << std::endl;
  f << "x509_extensions = v3_ca" << std::endl;
  f << "req_extensions = v3_req" << std::endl;
  f << "[ req_distinguished_name ]" << std::endl;
  f << "O = EU" << std::endl;
  f << "OU = EMI" << std::endl;
  f << "CN = CA" << std::endl;
  f << "[ req_attributes ]" << std::endl;
  f << "challengePassword  = A challenge password" << std::endl;
  f << "[ usr_cert ]" << std::endl;
  f << "basicConstraints=CA:FALSE" << std::endl;
  f << "subjectKeyIdentifier=hash" << std::endl;
  f << "authorityKeyIdentifier=keyid,issuer" << std::endl;
  f << "authorityInfoAccess=OCSP;URI:http://localhost:8888" << std::endl;
  f << "[ v3_req ]" << std::endl;
  f << "basicConstraints = CA:FALSE" << std::endl;
  f << "keyUsage = nonRepudiation, digitalSignature, keyEncipherment" << std::endl;
  f << "[ v3_ca ]" << std::endl;
  f << "basicConstraints=CA:true" << std::endl;
  f << "subjectKeyIdentifier=hash" << std::endl;
  f << "authorityKeyIdentifier=keyid:always,issuer:always" << std::endl;
  f << "[ proxy_cert_ext ]" << std::endl;
  f << "basicConstraints=CA:FALSE" << std::endl;
  f << "subjectKeyIdentifier=hash" << std::endl;
  f << "authorityKeyIdentifier=keyid,issuer:always" << std::endl;
  f << "proxyCertInfo=critical,language:id-ppl-anyLanguage,pathlen:3,policy:foo" << std::endl;
  f.close();

}

void CredentialsTest::TestCredentialsRequest() {
  AuthN::Context ctx(AuthN::Context::EmptyContext);

  // Make a credential request
  AuthN::ProxyCredentialsRequest credreq(ctx);
  int keybits = 1024;
  credreq.MakeKeys(keybits);

  // Set subject
  std::string subject = "CN=Test,OU=EMI,O=Grid";
  credreq.SetSubjectName(subject);

  // Set the proxy cert info extension into credential request

  AuthN::Credentials::Extension policy;
  policy.value = "my test proxy policy";
  credreq.SetPolicy(policy);

  /*
  time_t now;
  time_t till = now + 43200;
  credreq.SetValidFrom(now);
  credreq.SetValidTill(till);
  */

  // Make the request
  CPPUNIT_ASSERT_EQUAL(credreq.MakeRequest(), AuthN::Status(0));
  CPPUNIT_ASSERT((bool)credreq);

  // Add an keyusage extension, the request will be
  // re-signed after extension added

  AuthN::Credentials::Extension ext;
/*
  ext.critical = true;
  ext.oid = "X509v3 Key Usage";
  ext.value = "Digital Signature, Key Encipherment";
  AuthN::Status stat = credreq.AddExtension(ext);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
*/
/*
  ext.critical = false;
  ext.oid = "X509v3 Basic Constraints";
  ext.value = "CA:FALSE";
  stat = credreq.AddExtension(ext);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
*/

  std::string attr_name = "X509v3 Basic Constraints";
  std::string ext_attr_value = "CA:TRUE, pathlen:2";
  credreq.AddAttributes(attr_name, ext_attr_value);

  // Get private and public key
  bool enc = false;
  std::string privk_str, pubk_str;
  credreq.GetPrivateKey(privk_str);
  credreq.GetPublicKey(pubk_str);

  // Get the X509 request in string
  std::string csr_str;
  credreq.GetRequest(csr_str);

  std::cout<<"X509 Request: "<<csr_str<<std::endl;

  // Get extension
  std::string ku_name = "X509v3 Basic Constraints";
  AuthN::Credentials::Extension ku_ext;
  AuthN::Status status = credreq.GetExtension(ku_name, ku_ext);
  CPPUNIT_ASSERT((bool)status);
  CPPUNIT_ASSERT(!(ku_ext.value.empty()));

  // Copy to another CredentialRequest object
  AuthN::CredentialsRequest* copy_credreq;
  copy_credreq = &credreq.Copy();
  CPPUNIT_ASSERT((bool)(*copy_credreq));

  std::string privk_str_copy, pubk_str_copy, csr_str_copy;
  copy_credreq->GetPrivateKey(privk_str_copy);
  CPPUNIT_ASSERT(!privk_str.compare(privk_str_copy));

  copy_credreq->GetPublicKey(pubk_str_copy);
  CPPUNIT_ASSERT(!pubk_str.compare(pubk_str_copy));

  copy_credreq->GetRequest(csr_str_copy);
  CPPUNIT_ASSERT(!csr_str.compare(csr_str_copy));

  std::string subject_copy;
  subject_copy = copy_credreq->GetSubjectName();
  CPPUNIT_ASSERT(!subject.compare(subject_copy));

  AuthN::Credentials::Extension ku_ext_copy;
  status = copy_credreq->GetExtension(ku_name, ku_ext_copy);
  CPPUNIT_ASSERT(!(ku_ext_copy.value.empty()));
  CPPUNIT_ASSERT(!ku_ext.value.compare(ku_ext_copy.value));

  delete copy_credreq;

  // Assign to another CredentialRequest object,
  AuthN::CredentialsRequest assigned_credreq(ctx);
  assigned_credreq.AssignRequest(csr_str, privk_str);
  CPPUNIT_ASSERT((bool)(assigned_credreq));

  std::string privk_str_assigned, pubk_str_assigned, csr_str_assigned;
  assigned_credreq.GetPrivateKey(privk_str_assigned);
  CPPUNIT_ASSERT(!privk_str.compare(privk_str_assigned));

  assigned_credreq.GetPublicKey(pubk_str_assigned);
  CPPUNIT_ASSERT(!pubk_str.compare(pubk_str_assigned));

  assigned_credreq.GetRequest(csr_str_assigned);
  CPPUNIT_ASSERT(!csr_str.compare(csr_str_assigned));

  std::string subject_assigned;
  subject_assigned = assigned_credreq.GetSubjectName();
  CPPUNIT_ASSERT(!subject.compare(subject_assigned));

  AuthN::Credentials::Extension ku_ext_assigned;
  status = assigned_credreq.GetExtension(ku_name, ku_ext_assigned);
  CPPUNIT_ASSERT(!(ku_ext_assigned.value.empty()));
  CPPUNIT_ASSERT(!ku_ext.value.compare(ku_ext_assigned.value));

  std::string assigned_attr_name = "X509v3 Basic Constraints";
  std::list<std::string> assigned_attrs;
  assigned_credreq.GetAttributes(assigned_attr_name, assigned_attrs);

  CPPUNIT_ASSERT(!assigned_attrs.front().compare(ext_attr_value));
  
}

void CredentialsTest::TestCredentials() {
  AuthN::Context ctx(AuthN::Context::EmptyContext);

  //----------------------------------
  // Sign a CA certificate
  AuthN::CACredentialsRequest cacredreq(ctx);
  int keybits = 1024;
  cacredreq.MakeKeys(keybits);
  CPPUNIT_ASSERT_EQUAL(cacredreq.MakeRequest(), AuthN::Status(0));

  AuthN::Credentials cred(ctx);
  AuthN::Credentials out(ctx);
  cacredreq.SetValidFrom(time(NULL));
  cacredreq.SetValidTill(time(NULL) + 3600*24*365);
  cacredreq.SetSubjectName("/O=EU/OU=EMI/CN=CA");
  AuthN::Status stat = cred.Sign(cacredreq, out, opensslcnf);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
  CPPUNIT_ASSERT_EQUAL(cred.GetIssuerName(), cred.GetSubjectName());

  //Output the CA certificate into CA directory 
  //TODO: adapt for API without direct access to OpenSSL objects
/*
  X509* cacert = NULL;
  cacert = cred.GetCertificate();
  unsigned long ca_hash = X509_NAME_hash(cacert->cert_info->subject);
  char buf[16];
  std::string ca_name_str;
  snprintf(buf, sizeof(buf), "%0lx", ca_hash);
  ca_name_str = buf;
  std::cout<<ca_name_str<<std::endl;
*/
  std::string ca_cert_str;
  cred.GetCertificate(ca_cert_str);
  std::string ca_name_hash = AuthN::OpenSSL::get_hash_from_x509str(ca_cert_str);

  std::string ca_file_loc = trusted_cadir + "/" + ca_name_hash + ".0";
  std::string namespaces_loc = trusted_cadir + "/" + ca_name_hash + ".namespaces";
  std::string globus_signing_loc = trusted_cadir + "/" + ca_name_hash + ".signing_policy";
  std::ofstream ca_f(ca_file_loc.c_str(), std::ifstream::trunc);
  cred.GetCertificate(ca_f); ca_f.close();

  std::string ca_cert = cadir + "/cacert.pem";
  std::string ca_key = cadir + "/cakey.pem";
  std::ofstream ca_cert_stream(ca_cert.c_str(), std::ifstream::trunc);
  cred.GetCertificate(ca_cert_stream); ca_cert_stream.close();
  std::ofstream ca_key_stream(ca_key.c_str(), std::ifstream::trunc);
  cred.GetPrivateKey(ca_key_stream); ca_key_stream.close();

  // Get extensions of the CA certificate
/*
  AuthN::Credentials::Extension extension;
  for(int i=0; i<8; i++) {
    cred.GetExtension(i, extension);
    std::cout<<extension.oid<<"  "<<extension.value<<std::endl;
  }
*/

  //------------------------------
  //Sign an EEC certificate
  AuthN::CredentialsRequest eecreq(ctx);
  eecreq.MakeKeys(keybits);
  CPPUNIT_ASSERT_EQUAL(eecreq.MakeRequest(), AuthN::Status(0));

  AuthN::Credentials eec(ctx);
  eecreq.SetValidFrom(time(NULL));
  eecreq.SetValidTill(time(NULL) + 3600*24*30);
  eecreq.SetSubjectName(eec_subject);
  stat = cred.Sign(eecreq, eec, opensslcnf);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
  CPPUNIT_ASSERT_EQUAL(eec_subject, eec.GetSubjectName());
  CPPUNIT_ASSERT_EQUAL(eec_subject, eec.GetIdentityName());
  CPPUNIT_ASSERT_EQUAL(eec.GetIssuerName(), cred.GetSubjectName());
 
  std::string repo_cert = cadir + "/certs" + "/cert.pem";
  std::string repo_key = cadir + "/certs" + "/key.pem";
  std::ofstream repo_cert_stream(repo_cert.c_str(), std::ifstream::trunc);
  eec.GetCertificate(repo_cert_stream); repo_cert_stream.close();
  std::ofstream repo_key_stream(repo_key.c_str(), std::ifstream::trunc);
  // As EEC/proxy cert is not supposed to be self-signed, 
  // the related private key should be retrieved from the request side
  eecreq.GetPrivateKey(repo_key_stream); repo_key_stream.close();
  std::string eec_keystr;
  eecreq.GetPrivateKey(eec_keystr);

  //---------------------------
  //Sign a proxy certificate
  AuthN::ProxyCredentialsRequest proxyreq1(ctx);
  proxyreq1.MakeKeys(keybits);
  CPPUNIT_ASSERT_EQUAL(proxyreq1.MakeRequest(), AuthN::Status(0));
  AuthN::Credentials::Extension policy1;
  policy1.value = "my test proxy policy level 1";
  proxyreq1.SetPolicy(policy1);
  eec.Assign("", "", eec_keystr); // Assign the private key into eec object
  AuthN::Credentials proxy1(ctx);
  proxyreq1.SetValidFrom(time(NULL));
  proxyreq1.SetValidTill(time(NULL) + 3600*12);
 
  stat = eec.Sign(proxyreq1, proxy1, opensslcnf);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
  CPPUNIT_ASSERT_EQUAL(eec_subject, proxy1.GetIdentityName());
  CPPUNIT_ASSERT_EQUAL(proxy1.GetIssuerName(), eec.GetSubjectName());

  std::string proxy1_keystr;
  proxyreq1.GetPrivateKey(proxy1_keystr);

  //-----------------------------------
  //Sign a second level proxy certificate
  AuthN::ProxyCredentialsRequest proxyreq2(ctx);
  proxyreq2.MakeKeys(keybits);
  CPPUNIT_ASSERT_EQUAL(proxyreq2.MakeRequest(), AuthN::Status(0));
  AuthN::Credentials::Extension policy2;
  policy2.value = "my test proxy policy level 2";
  proxyreq2.SetPolicy(policy2);

  AuthN::Credentials proxy2(ctx);
  proxyreq2.SetValidFrom(time(NULL));
  proxyreq2.SetValidTill(time(NULL) + 3600*12);
  proxy1.Assign("", "", proxy1_keystr); // Assign the private key into eec object
  stat = proxy1.Sign(proxyreq2, proxy2, opensslcnf);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
  CPPUNIT_ASSERT_EQUAL(eec_subject, proxy2.GetIdentityName());
  CPPUNIT_ASSERT_EQUAL(proxy2.GetIssuerName(), proxy1.GetSubjectName());

  //--------------------------------
  //Check the credential assignment
  //EEC credential
  AuthN::Credentials assign_to_eec(ctx);
  std::string eec_cert_str;
  std::string eec_key_str;
  std::string eec_chain_str;
  eec.GetCertificate(eec_cert_str);
  eec.GetPrivateKey(eec_key_str);
  eec.GetChain(eec_chain_str);
  stat = assign_to_eec.Assign(eec_cert_str, eec_chain_str, eec_key_str);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
  std::string assign_to_eec_cert_str;
  std::string assign_to_eec_key_str; 
  std::string assign_to_eec_chain_str; 
  assign_to_eec.GetCertificate(assign_to_eec_cert_str);
  assign_to_eec.GetPrivateKey(assign_to_eec_key_str);
  assign_to_eec.GetChain(assign_to_eec_chain_str);
  CPPUNIT_ASSERT_EQUAL(eec_cert_str, assign_to_eec_cert_str);
  CPPUNIT_ASSERT_EQUAL(eec_key_str, assign_to_eec_key_str);
  CPPUNIT_ASSERT_EQUAL(eec_chain_str, assign_to_eec_chain_str);

  //Proxy credential --- level 1
  AuthN::Credentials assign_to_proxy1(ctx);
  std::string proxy1_cert_str;
  std::string proxy1_key_str;
  std::string proxy1_chain_str;
  proxy1.GetCertificate(proxy1_cert_str);
  proxy1.GetPrivateKey(proxy1_key_str);
  proxy1.GetChain(proxy1_chain_str);
  stat = assign_to_proxy1.Assign(proxy1_cert_str, proxy1_chain_str, proxy1_key_str);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
  std::string assign_to_proxy1_cert_str;
  std::string assign_to_proxy1_key_str;
  std::string assign_to_proxy1_chain_str;
  assign_to_proxy1.GetCertificate(assign_to_proxy1_cert_str);
  assign_to_proxy1.GetPrivateKey(assign_to_proxy1_key_str);
  assign_to_proxy1.GetChain(assign_to_proxy1_chain_str);
  CPPUNIT_ASSERT_EQUAL(proxy1_cert_str, assign_to_proxy1_cert_str);
  CPPUNIT_ASSERT_EQUAL(proxy1_key_str, assign_to_proxy1_key_str);
  CPPUNIT_ASSERT_EQUAL(proxy1_chain_str, assign_to_proxy1_chain_str);

  //Proxy credential --- level 2
  AuthN::Credentials assign_to_proxy2(ctx);
  std::string proxy2_cert_str;
  std::string proxy2_key_str;
  std::string proxy2_chain_str;
  proxy2.GetCertificate(proxy2_cert_str);
  proxy2.GetPrivateKey(proxy2_key_str);
  proxy2.GetChain(proxy2_chain_str);
  stat = assign_to_proxy2.Assign(proxy2_cert_str, proxy2_chain_str, proxy2_key_str);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));
  std::string assign_to_proxy2_cert_str;
  std::string assign_to_proxy2_key_str;
  std::string assign_to_proxy2_chain_str;
  assign_to_proxy2.GetCertificate(assign_to_proxy2_cert_str);
  assign_to_proxy2.GetPrivateKey(assign_to_proxy2_key_str);
  assign_to_proxy2.GetChain(assign_to_proxy2_chain_str);
  CPPUNIT_ASSERT_EQUAL(proxy2_cert_str, assign_to_proxy2_cert_str);
  CPPUNIT_ASSERT_EQUAL(proxy2_key_str, assign_to_proxy2_key_str);
  CPPUNIT_ASSERT_EQUAL(proxy2_chain_str, assign_to_proxy2_chain_str);

  //Check the Copy method
  AuthN::Credentials& eec_copy = eec.Copy();
  std::string eec_copy_cert_str;
  std::string eec_copy_key_str;
  std::string eec_copy_chain_str;
  eec_copy.GetCertificate(eec_copy_cert_str);
  eec_copy.GetPrivateKey(eec_copy_key_str);
  eec_copy.GetChain(eec_copy_chain_str);
  CPPUNIT_ASSERT_EQUAL(eec_cert_str, eec_copy_cert_str);
  CPPUNIT_ASSERT_EQUAL(eec_key_str, eec_copy_key_str);
  CPPUNIT_ASSERT_EQUAL(eec_chain_str, eec_copy_chain_str);  

  //-----------------------
  //Certificate validation checking
  ctx.SetCAPath(trusted_cadir);
  AuthN::Validator validator(ctx);

  // 1. Validation checking against namespaces policy
  //Set the namespaces policy
  std::ofstream n_f(namespaces_loc.c_str(), std::ifstream::trunc);
  n_f << "TO Issuer \"/O=EU/OU=EMI/CN=CA\" \\"<< std::endl << "PERMIT Subject \"/O=EU/OU=EMI/.*\""; n_f.close();
  //std::ofstream g_f(globus_signing_loc.c_str(), std::ifstream::trunc);
  //g_f << "access_id_CA  X509   \'/O=EU/OU=EMI/CN=CA\'"<<std::endl
  //    << "pos_rights    globus CA:sign" << std::endl
  //    << "cond_subjects globus \'\"/O=EU/OU=EMI/CN=CA\" \"/O=EU/OU=EMI/*\"\'"<<std::endl; g_f.close();

  //eec namespaces validation
  validator.SetMode(AuthN::Validator::ValidationNamespaceCheck);
  eec.SetValidator(validator);
  stat = eec.Validate();
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));

  // 2. Validation checking against CRL
  //Revoke the eec certificate, and set the crl
  int res;
  std::string cmd;
  cmd = "openssl ca -config " + opensslcnf + " -revoke " + repo_cert;
  res = system(cmd.c_str());
  CPPUNIT_ASSERT_EQUAL(res, 0);
  std::string ca_crl_file = cadir + "/" + "crl.pem";
  cmd = "openssl ca -config " + opensslcnf + " -gencrl -out " + ca_crl_file;
  res = system(cmd.c_str());
  CPPUNIT_ASSERT_EQUAL(res, 0);

  //Copy the crl content into crl file under trusted ca directory
  std::ifstream in(ca_crl_file.c_str(), std::ios::in);
  CPPUNIT_ASSERT_EQUAL(!in, false);
  std::string s; std::getline<char>(in, s, 0); in.close();  
  std::string crl_file = trusted_cadir + "/" + ca_name_hash + ".r0"; //".crl";
  std::ofstream c_f(crl_file.c_str(), std::ifstream::trunc);
  c_f << s; c_f.close();

  //eec CRL validation
  validator.SetMode(AuthN::Validator::ValidationCRLMandatory);
  eec.SetValidator(validator);
  stat = eec.Validate();
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(-1)); //It is expected not to be valid since the crl has been there

  //proxy CRL validation
  proxy1.SetValidator(validator);
  stat = proxy1.Validate();
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(-1));

  //proxy CRL validation
  proxy2.SetValidator(validator);
  stat = proxy2.Validate();
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(-1));

  // 3. Validation checking against OCSP responder
  //Launch the local ocsp responder daemon
  launch_ocsp_responder();

  AuthN::Context ocsp_ctx(AuthN::Context::EmptyContext);
  ocsp_ctx.SetCAPath(trusted_cadir);
  //Set the credential for validator, which will be
  //used to sign the OCSP request
  ocsp_ctx.SetCredentials(repo_cert, repo_key);

  AuthN::Validator ocsp_validator(ocsp_ctx);

  //Set the validation mode
  ocsp_validator.SetMode(AuthN::Validator::ValidationOCSPIfPresent);

  //eec OCSP validation
  eec.SetValidator(ocsp_validator);
  stat = eec.Validate();
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));

  //proxy OCSP validation
  proxy1.SetValidator(ocsp_validator);
  stat = proxy1.Validate();
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));

  //proxy OCSP validation
  proxy2.SetValidator(ocsp_validator);
  stat = proxy2.Validate();
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));

  kill_ocsp_responder();
}

void CredentialsTest::setUp() {
  std::string certfile, keyfile;
  AuthN::Utils::File::makedir(cadir.c_str());
  AuthN::Utils::File::makedir(trusted_cadir.c_str());
  std::string certs = cadir + "/certs";
  AuthN::Utils::File::makedir(certs.c_str());
  std::string crl = cadir + "/crl";
  AuthN::Utils::File::makedir(crl.c_str());
  std::string serial = cadir + "/serial";
  std::ofstream s_f(serial.c_str(), std::ifstream::trunc);
  s_f << "00"; s_f.close();
  std::string index = cadir + "/index.txt";
  std::ofstream i_f(index.c_str(), std::ifstream::trunc);
  i_f << ""; i_f.close();
  std::string crlnumber = cadir + "/crlnumber";
  std::ofstream c_f(crlnumber.c_str(), std::ifstream::trunc);
  c_f << "00"; c_f.close();

  createOpenSSLconf();

}

void CredentialsTest::tearDown() {
  //remove(opensslcnf.c_str());
  //AuthN::File::removedir(cadir.c_str());
  //AuthN::File::removedir(trusted_cadir.c_str());
}

void CredentialsTest::launch_ocsp_responder() {
  //pid_t pid;
  int status;
  bool res;
  char *argv[16];
  std::string cafile = cadir + "/cacert.pem";
  std::string cakey = cadir + "/cakey.pem";
  std::string index_file = cadir + "/ocspid.txt";

  argv[0] = (char*)"/usr/bin/openssl";
  argv[1] = (char*)"ocsp";
  argv[2] = (char*)"-port";
  argv[3] = (char*)"8888";
  argv[4] = (char*)"-rsigner";
  argv[5] = (char*)(cafile.c_str()); //"./test/testCA/cacert.pem";
  argv[6] = (char*)"-rkey";
  argv[7] = (char*)(cakey.c_str()); //"./test/testCA/cakey.pem";
  argv[8] = (char*)"-ndays";
  argv[9] = (char*)"10";
  argv[10] = (char*)"-CA";
  argv[11] = (char*)(cafile.c_str()); //"./test/testCA/cacert.pem";
  argv[12] = (char*)"-resp_key_id";
  argv[13] = (char*)"-index";
  argv[14] = (char*)(index_file.c_str()); //"./ocspid.txt";
  argv[15] = NULL;

  std::ofstream f(index_file.c_str(), std::ifstream::trunc);
  f.close();

  if(posix_spawn(&pid, argv[0], NULL, NULL, argv, NULL) != 0) {
    std::cout<<"posix_spawn failed"<<std::endl;
    exit (1);
  }
  else sleep(2); //sleep a little time to wait the openssl ocsp server being launched

}

void CredentialsTest::kill_ocsp_responder() {
  int sig = SIGTERM;
  if(kill(pid, sig)!=0) { std::cout<<"failed to kill child"<<std::endl; exit(1); }
}

CPPUNIT_TEST_SUITE_REGISTRATION(CredentialsTest);
