/*	$NetBSD: wm8731_zaudio.c,v 1.2 2018/06/16 21:22:13 thorpej Exp $	*/

/*-
 * Copyright (c) 2013 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by TOYOKURA Atsushi.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * TODO:
 *	- powerhooks (currently only works until first suspend)
 */

#include "opt_cputypes.h"
#include "opt_zaudio.h"

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: wm8731_zaudio.c,v 1.2 2018/06/16 21:22:13 thorpej Exp $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/callout.h>
#include <sys/device.h>
#include <sys/kmem.h>
#include <sys/kernel.h>
#include <sys/audioio.h>
#include <sys/mutex.h>
#include <sys/intr.h>
#include <sys/bus.h>

#include <dev/audio_if.h>
#include <dev/mulaw.h>
#include <dev/auconv.h>

#include <dev/i2c/i2cvar.h>

#include <arm/xscale/pxa2x0reg.h>
#include <arm/xscale/pxa2x0var.h>
#include <arm/xscale/pxa2x0_i2c.h>
#include <arm/xscale/pxa2x0_i2s.h>
#include <arm/xscale/pxa2x0_dmac.h>
#include <arm/xscale/pxa2x0_gpio.h>

#include <zaurus/zaurus/zaurus_var.h>
#include <zaurus/dev/zaudiovar.h>
#include <zaurus/dev/wm8731reg.h>
#include <zaurus/dev/wm8731var.h>
#include <zaurus/dev/scoopvar.h>

#define WM8731_ADDRESS  0x1B

/* GPIO pins */
#define GPIO_HP_IN_C860	4

#define WM8731_OP_SPKR	0
#define WM8731_OP_MIC	1
#define WM8731_OP_NUM	2

static int	wm8731_finalize(device_t);
static bool	wm8731_suspend(device_t, const pmf_qual_t *);
static bool	wm8731_resume(device_t, const pmf_qual_t *);
static void	wm8731_volume_up(device_t);
static void	wm8731_volume_down(device_t);
static void	wm8731_volume_toggle(device_t);

static struct audio_device wm8731_device = {
	"WM8731",
	"1.0",
	"wm"
};

static const struct audio_format wm8731_formats[] = {
	{
		.driver_data	= NULL,
		.mode		= AUMODE_PLAY | AUMODE_RECORD,
		.encoding	= AUDIO_ENCODING_SLINEAR_LE,
		.validbits	= 16,
		.precision	= 16,
		.channels	= 2,
		.channel_mask	= AUFMT_STEREO,
		.frequency_type	= 0,
		.frequency	= { 4000, 48000 }
	}
};
static const int wm8731_nformats = (int)__arraycount(wm8731_formats);

static void wm8731_init(struct zaudio_softc *);
static int wm8731_jack_intr(void *);
static void wm8731_jack(void *);
static void wm8731_standby(struct zaudio_softc *);
static void wm8731_update_volume(struct zaudio_softc *, int);
static void wm8731_update_mutes(struct zaudio_softc *, int);
static void wm8731_play_setup(struct zaudio_softc *);
/*static*/ void wm8731_record_setup(struct zaudio_softc *);
static int wm8731_query_encoding(void *, struct audio_encoding *);
static int wm8731_set_params(void *, int, int, audio_params_t *,
    audio_params_t *, stream_filter_list_t *, stream_filter_list_t *);
static int wm8731_start_output(void *, void *, int, void (*)(void *), void *);
static int wm8731_start_input(void *, void *, int, void (*)(void *), void *);
static int wm8731_halt_output(void *);
static int wm8731_halt_input(void *);
static int wm8731_getdev(void *, struct audio_device *);
static int wm8731_set_port(void *, struct mixer_ctrl *);
static int wm8731_get_port(void *, struct mixer_ctrl *);
static int wm8731_query_devinfo(void *, struct mixer_devinfo *);

static struct audio_hw_if wm8731_hw_if = {
	.open			= zaudio_open,
	.close			= zaudio_close,
	.drain			= NULL,
	.query_encoding		= wm8731_query_encoding,
	.set_params		= wm8731_set_params,
	.round_blocksize	= zaudio_round_blocksize,
	.commit_settings	= NULL,
	.init_output		= NULL,
	.init_input		= NULL,
	.start_output		= wm8731_start_output,
	.start_input		= wm8731_start_input,
	.halt_output		= wm8731_halt_output,
	.halt_input		= wm8731_halt_input,
	.speaker_ctl		= NULL,
	.getdev			= wm8731_getdev,
	.setfd			= NULL,
	.set_port		= wm8731_set_port,
	.get_port		= wm8731_get_port,
	.query_devinfo		= wm8731_query_devinfo,
	.allocm			= zaudio_allocm,
	.freem			= zaudio_freem,
	.round_buffersize	= zaudio_round_buffersize,
	.mappage		= zaudio_mappage,
	.get_props		= zaudio_get_props,
	.trigger_output		= NULL,
	.trigger_input		= NULL,
	.dev_ioctl		= NULL,
	.get_locks		= zaudio_get_locks,
};

static const uint16_t playback_regs[][2] = {
	/* Power Down Control */
	{ WM8731_PD_REG, WM8731_CLKOUTPD | WM8731_OSCPD | WM8731_OUTPD
	    | WM8731_ADCPD | WM8731_MICPD | WM8731_LINEINPD },

	/* Digital Audio Path Control */
	{ WM8731_DAP_REG, 0 },

	/* Analogue Audio Path Control */
	{ WM8731_AAP_REG, WM8731_DACSEL | WM8731_MUTEMIC },

	/* Activating DSP and DAI */
	{ WM8731_AC_REG, WM8731_ACTIVE },
	
	/* End of list */
	{ 0xffff, 0xffff }
};

static const uint16_t record_regs[][2] = {
	/* Power Down Control */
	{ WM8731_PD_REG, WM8731_CLKOUTPD | WM8731_OSCPD | WM8731_DACPD
	    | WM8731_LINEINPD },

	/* Digital Audio Path Control */
	{ WM8731_DAP_REG, 0 },

	/* Analogue Audio Path Control */
	{ WM8731_AAP_REG, WM8731_INSEL | WM8731_MICBOOST },

	/* Activating DSP and DAI */
	{ WM8731_AC_REG, WM8731_ACTIVE },

	/* End of list */
	{ 0xffff, 0xffff }
};

static __inline int
wm8731_write(struct zaudio_softc *sc, int reg, int val)
{
	uint16_t tmp;
	uint8_t cmd;
	uint8_t data;

	tmp = (reg << 9) | (val & 0x1ff);
	cmd = tmp >> 8;
	data = tmp;
	return iic_exec(sc->sc_i2c, I2C_OP_WRITE_WITH_STOP, WM8731_ADDRESS,
	    &cmd, 1, &data, 1, 0);
}

int
wm8731_match(device_t parent, cfdata_t cf, struct i2c_attach_args *ia)
{
	int match_result;

	if (ZAURUS_ISC1000 || ZAURUS_ISC3000)
		return 0;

	if (iic_use_direct_match(ia, cf, NULL, &match_result))
		return match_result;

	/* indirect config - check typical address */
	if (ia->ia_addr == WM8731_ADDRESS)
		return I2C_MATCH_ADDRESS_ONLY;

	return 0;
}

void
wm8731_attach(device_t parent, device_t self, struct i2c_attach_args *ia)
{
	struct zaudio_softc *sc = device_private(self);
	int error;

	aprint_normal(": I2S, WM8731 Audio\n");
	aprint_naive("\n");

	/* Check for an I2C response from the wm8731 */
	iic_acquire_bus(sc->sc_i2c, 0);
	error = wm8731_write(sc, WM8731_RESET_REG, 0);
	iic_release_bus(sc->sc_i2c, 0);
	if (error) {
		aprint_error_dev(self, "codec failed to respond\n");
		goto fail_i2c;
	}
	delay(100);

	/* Allocate memory for volume & mute operations */
	sc->sc_volume = kmem_zalloc(sizeof(*sc->sc_volume) * WM8731_OP_NUM,
	    KM_SLEEP);
	sc->sc_unmute = kmem_zalloc(sizeof(*sc->sc_unmute) * WM8731_OP_NUM,
	    KM_SLEEP);
	sc->sc_unmute_toggle = kmem_zalloc(
	    sizeof(*sc->sc_unmute_toggle) * WM8731_OP_NUM, KM_SLEEP);

	/* Speaker On by default. */
	sc->sc_volume[WM8731_OP_SPKR].left = 180;
	sc->sc_volume[WM8731_OP_SPKR].right = 180;
	sc->sc_jack = FALSE;
	UNMUTE(sc, WM8731_OP_SPKR, 1);
	sc->sc_volume[WM8731_OP_MIC].left = 180;
	UNMUTE(sc, WM8731_OP_MIC, 0);

	/* Configure headphone jack state change handling. */
	callout_setfunc(&sc->sc_to, wm8731_jack, sc);
	pxa2x0_gpio_set_function(GPIO_HP_IN_C860, GPIO_IN);
	(void) pxa2x0_gpio_intr_establish(GPIO_HP_IN_C860, IST_EDGE_BOTH,
	    IPL_BIO, wm8731_jack_intr, sc);

	/* wm8731_init() implicitly depends on ioexp or scoop */
	config_finalize_register(self, wm8731_finalize);

	audio_attach_mi(&wm8731_hw_if, sc, self);

	if (!pmf_device_register(self, wm8731_suspend, wm8731_resume))
		aprint_error_dev(self, "couldn't establish power handler\n");
	if (!pmf_event_register(self, PMFE_AUDIO_VOLUME_UP,
	    wm8731_volume_up, true))
		aprint_error_dev(self, "couldn't register event handler\n");
	if (!pmf_event_register(self, PMFE_AUDIO_VOLUME_DOWN,
	    wm8731_volume_down, true))
		aprint_error_dev(self, "couldn't register event handler\n");
	if (!pmf_event_register(self, PMFE_AUDIO_VOLUME_TOGGLE,
	    wm8731_volume_toggle, true))
		aprint_error_dev(self, "couldn't register event handler\n");

	return;

fail_i2c:
	pxa2x0_i2s_detach_sub(&sc->sc_i2s);
}

static int
wm8731_finalize(device_t dv)
{
	struct zaudio_softc *sc = device_private(dv);

	wm8731_init(sc);
	return 0;
}

static bool
wm8731_suspend(device_t dv, const pmf_qual_t *qual)
{
	struct zaudio_softc *sc = device_private(dv);

	callout_stop(&sc->sc_to);
	wm8731_standby(sc);

	return true;
}

static bool
wm8731_resume(device_t dv, const pmf_qual_t *qual)
{
	struct zaudio_softc *sc = device_private(dv);

	pxa2x0_i2s_init(&sc->sc_i2s);
	wm8731_init(sc);

	return true;
}

static __inline uint8_t
vol_sadd(int vol, int stride)
{

	vol += stride;
	if (vol > 255)
		return 255;
	return (uint8_t)vol;
}

#ifndef	ZAUDIO_VOLUME_STRIDE
#define	ZAUDIO_VOLUME_STRIDE	8
#endif

static void
wm8731_volume_up(device_t dv)
{
	struct zaudio_softc *sc = device_private(dv);
	int s;

	s = splbio();
	iic_acquire_bus(sc->sc_i2c, 0);

	sc->sc_volume[WM8731_OP_SPKR].left =
	    vol_sadd(sc->sc_volume[WM8731_OP_SPKR].left, ZAUDIO_VOLUME_STRIDE);
	sc->sc_volume[WM8731_OP_SPKR].right =
	    vol_sadd(sc->sc_volume[WM8731_OP_SPKR].right, ZAUDIO_VOLUME_STRIDE);

	wm8731_update_volume(sc, WM8731_OP_SPKR);

	iic_release_bus(sc->sc_i2c, 0);
	splx(s);
}

static __inline uint8_t
vol_ssub(int vol, int stride)
{

	vol -= stride;
	if (vol < 0)
		return 0;
	return (uint8_t)vol;
}

static void
wm8731_volume_down(device_t dv)
{
	struct zaudio_softc *sc = device_private(dv);
	int s;

	s = splbio();
	iic_acquire_bus(sc->sc_i2c, 0);

	sc->sc_volume[WM8731_OP_SPKR].left =
	    vol_ssub(sc->sc_volume[WM8731_OP_SPKR].left, ZAUDIO_VOLUME_STRIDE);
	sc->sc_volume[WM8731_OP_SPKR].right =
	    vol_ssub(sc->sc_volume[WM8731_OP_SPKR].right, ZAUDIO_VOLUME_STRIDE);

	wm8731_update_volume(sc, WM8731_OP_SPKR);

	iic_release_bus(sc->sc_i2c, 0);
	splx(s);
}

static void
wm8731_volume_toggle(device_t dv)
{
	struct zaudio_softc *sc = device_private(dv);
	int s;

	s = splbio();
	iic_acquire_bus(sc->sc_i2c, 0);

	if (!sc->sc_unmute[WM8731_OP_SPKR]) {
		sc->sc_unmute[WM8731_OP_SPKR] =
		    sc->sc_unmute_toggle[WM8731_OP_SPKR];
	} else {
		sc->sc_unmute[WM8731_OP_SPKR] = 0;
	}
	wm8731_update_mutes(sc, 1);

	iic_release_bus(sc->sc_i2c, 0);
	splx(s);
}

static void
wm8731_init(struct zaudio_softc *sc)
{

	iic_acquire_bus(sc->sc_i2c, 0);

	/* Reset the codec */
	wm8731_write(sc, WM8731_RESET_REG, 0);
	delay(100);

	/* Switch to standby power only */
	wm8731_write(sc, WM8731_PD_REG, WM8731_CLKOUTPD | WM8731_OSCPD |
	      WM8731_OUTPD | WM8731_DACPD | WM8731_ADCPD | WM8731_MICPD |
	      WM8731_LINEINPD);

	/* Configure digital interface for I2S */
	wm8731_write(sc, WM8731_DAI_REG, WM8731_SET_IWL(2) | WM8731_SET_FORMAT(2));

	/* Initialise volume levels */
	wm8731_update_volume(sc, WM8731_OP_SPKR);
	wm8731_update_volume(sc, WM8731_OP_MIC);

	scoop_set_headphone(0);
	scoop_set_speaker(0);
	scoop_set_mic_bias(0);

	iic_release_bus(sc->sc_i2c, 0);

	/* Assume that the jack state has changed. */ 
	wm8731_jack(sc);
}

static int
wm8731_jack_intr(void *v)
{
	struct zaudio_softc *sc = v;

	if (!callout_active(&sc->sc_to))
		wm8731_jack(sc);
	
	return 1;
}

static void
wm8731_jack(void *v)
{
	struct zaudio_softc *sc = v;

	switch (sc->sc_state) {
	case ZAUDIO_JACK_STATE_OUT:
		if (pxa2x0_gpio_get_bit(GPIO_HP_IN_C860)) {
			sc->sc_state = ZAUDIO_JACK_STATE_INS;
			sc->sc_icount = 0;
		}
		break;

	case ZAUDIO_JACK_STATE_INS:
		if (sc->sc_icount++ > 2) {
			if (pxa2x0_gpio_get_bit(GPIO_HP_IN_C860)) {
				sc->sc_state = ZAUDIO_JACK_STATE_IN;
				sc->sc_jack = TRUE;
				UNMUTE(sc, WM8731_OP_MIC, 1);
				goto update_mutes;
			} else 
				sc->sc_state = ZAUDIO_JACK_STATE_OUT;
		}
		break;

	case ZAUDIO_JACK_STATE_IN:
		if (!pxa2x0_gpio_get_bit(GPIO_HP_IN_C860)) {
			sc->sc_state = ZAUDIO_JACK_STATE_REM;
			sc->sc_icount = 0;
		}
		break;

	case ZAUDIO_JACK_STATE_REM: 
		if (sc->sc_icount++ > 2) {
			if (!pxa2x0_gpio_get_bit(GPIO_HP_IN_C860)) {
				sc->sc_state = ZAUDIO_JACK_STATE_OUT;
				sc->sc_jack = FALSE;
				UNMUTE(sc, WM8731_OP_MIC, 0);
				goto update_mutes;
			} else
				sc->sc_state = ZAUDIO_JACK_STATE_IN;
		}
		break;
	}
	
	callout_schedule(&sc->sc_to, hz/4);

	return;

update_mutes:
	callout_stop(&sc->sc_to);

	if (sc->sc_playing || sc->sc_recording) {
		iic_acquire_bus(sc->sc_i2c, 0);
		if (sc->sc_playing)
			wm8731_update_mutes(sc, 1);
		if (sc->sc_recording)
			wm8731_update_mutes(sc, 2);
		iic_release_bus(sc->sc_i2c, 0);
	}
}

static void
wm8731_standby(struct zaudio_softc *sc)
{

	iic_acquire_bus(sc->sc_i2c, 0);

	/* Switch to standby power only */
	wm8731_write(sc, WM8731_PD_REG, WM8731_CLKOUTPD | WM8731_OSCPD |
	      WM8731_OUTPD | WM8731_DACPD | WM8731_ADCPD | WM8731_MICPD |
	      WM8731_LINEINPD);

	scoop_set_headphone(0);
	scoop_set_speaker(0);
	scoop_set_mic_bias(0);

	/* Activating DSP and DAI */
	wm8731_write(sc, WM8731_AC_REG, 0);

	iic_release_bus(sc->sc_i2c, 0);
}

static void
wm8731_update_volume(struct zaudio_softc *sc, int output)
{
	struct zaudio_volume *volume;

	switch (output) {
	case WM8731_OP_SPKR:
		volume = &sc->sc_volume[WM8731_OP_SPKR];
		wm8731_write(sc, WM8731_LHP_REG,
		       WM8731_SET_LHPVOL(volume->left >> 1));
		wm8731_write(sc, WM8731_RHP_REG,
		       WM8731_SET_RHPVOL(volume->right >> 1));
		break;

	case WM8731_OP_MIC:
		volume = &sc->sc_volume[WM8731_OP_MIC];
		wm8731_write(sc, WM8731_LIN_REG, WM8731_LRINBOTH |
		    WM8731_SET_LINVOL(volume->left >> 3));
		break;
	}
}

static void
wm8731_update_mutes(struct zaudio_softc *sc, int mask)
{
	uint16_t val = WM8731_CLKOUTPD | WM8731_OSCPD | WM8731_LINEINPD;

	/* playback */
	if (mask & 1) {
		val |= WM8731_ADCPD | WM8731_MICPD;
		if (!sc->sc_unmute[WM8731_OP_SPKR]) {
			val |= WM8731_OUTPD | WM8731_DACPD;
		}
		wm8731_write(sc, WM8731_PD_REG, val);
		scoop_set_headphone(sc->sc_unmute[WM8731_OP_SPKR] & sc->sc_jack);
		scoop_set_speaker(sc->sc_unmute[WM8731_OP_SPKR] & !sc->sc_jack);
	}

	/* record */
	if (mask & 2) {
		val = WM8731_OUTPD | WM8731_DACPD;
		if (!sc->sc_unmute[WM8731_OP_MIC]) {
			val |= WM8731_ADCPD | WM8731_MICPD;
		}
		wm8731_write(sc, WM8731_PD_REG, val);
		scoop_set_mic_bias(sc->sc_unmute[WM8731_OP_MIC]);
	}
}

static void
wm8731_play_setup(struct zaudio_softc *sc)
{
	int i;

	iic_acquire_bus(sc->sc_i2c, 0);

	/* Program the codec with playback settings */
	for (i = 0; playback_regs[i][0] != 0xffff; i++) {
		wm8731_write(sc, playback_regs[i][0], playback_regs[i][1]);
	}
	wm8731_update_mutes(sc, 1);

	iic_release_bus(sc->sc_i2c, 0);
}

/*static*/ void
wm8731_record_setup(struct zaudio_softc *sc)
{
	int i;

	iic_acquire_bus(sc->sc_i2c, 0);

	/* Program the codec with playback settings */
	for (i = 0; record_regs[i][0] != 0xffff; i++) {
		wm8731_write(sc, record_regs[i][0], record_regs[i][1]);
	}
	wm8731_update_mutes(sc, 2);

	iic_release_bus(sc->sc_i2c, 0);
}

static int
wm8731_query_encoding(void *hdl, struct audio_encoding *aep)
{

	switch (aep->index) {
	case 0:
		strlcpy(aep->name, AudioEulinear, sizeof(aep->name));
		aep->encoding = AUDIO_ENCODING_ULINEAR;
		aep->precision = 8;
		aep->flags = AUDIO_ENCODINGFLAG_EMULATED;
		break;

	case 1:
		strlcpy(aep->name, AudioEmulaw, sizeof(aep->name));
		aep->encoding = AUDIO_ENCODING_ULAW;
		aep->precision = 8;
		aep->flags = AUDIO_ENCODINGFLAG_EMULATED;
		break;

	case 2:
		strlcpy(aep->name, AudioEalaw, sizeof(aep->name));
		aep->encoding = AUDIO_ENCODING_ALAW;
		aep->precision = 8;
		aep->flags = AUDIO_ENCODINGFLAG_EMULATED;
		break;

	case 3:
		strlcpy(aep->name, AudioEslinear, sizeof(aep->name));
		aep->encoding = AUDIO_ENCODING_SLINEAR;
		aep->precision = 8;
		aep->flags = AUDIO_ENCODINGFLAG_EMULATED;
		break;

	case 4:
		strlcpy(aep->name, AudioEslinear_le, sizeof(aep->name));
		aep->encoding = AUDIO_ENCODING_SLINEAR_LE;
		aep->precision = 16;
		aep->flags = 0;
		break;

	case 5:
		strlcpy(aep->name, AudioEulinear_le, sizeof(aep->name));
		aep->encoding = AUDIO_ENCODING_ULINEAR_LE;
		aep->precision = 16;
		aep->flags = AUDIO_ENCODINGFLAG_EMULATED;
		break;

	case 6:
		strlcpy(aep->name, AudioEslinear_be, sizeof(aep->name));
		aep->encoding = AUDIO_ENCODING_SLINEAR_BE;
		aep->precision = 16;
		aep->flags = AUDIO_ENCODINGFLAG_EMULATED;
		break;

	case 7:
		strlcpy(aep->name, AudioEulinear_be, sizeof(aep->name));
		aep->encoding = AUDIO_ENCODING_ULINEAR_BE;
		aep->precision = 16;
		aep->flags = AUDIO_ENCODINGFLAG_EMULATED;
		break;

	default:
		return EINVAL;
	}

	return 0;
}

static int
wm8731_set_params(void *hdl, int setmode, int usemode, audio_params_t *play,
    audio_params_t *rec, stream_filter_list_t *pfil, stream_filter_list_t *rfil)
{
	struct zaudio_softc *sc = hdl;
	struct audio_params *p;
	stream_filter_list_t *fil;
	int mode, i;

	if (play->sample_rate != rec->sample_rate &&
	    usemode == (AUMODE_PLAY | AUMODE_RECORD)) {
		if (setmode == AUMODE_PLAY) {
			rec->sample_rate = play->sample_rate;
			setmode |= AUMODE_RECORD;
		} else if (setmode == AUMODE_RECORD) {
			play->sample_rate = rec->sample_rate;
			setmode |= AUMODE_PLAY;
		} else
			return EINVAL;
	}

	for (mode = AUMODE_RECORD; mode != -1;
	     mode = (mode == AUMODE_RECORD) ? AUMODE_PLAY : -1) {
		if ((setmode & mode) == 0)
			continue;

		p = (mode == AUMODE_PLAY) ? play : rec;

		if (p->sample_rate < 4000 || p->sample_rate > 48000 ||
		    (p->precision != 8 && p->precision != 16) ||
		    (p->channels != 1 && p->channels != 2))
			return EINVAL;

		fil = (mode == AUMODE_PLAY) ? pfil : rfil;
		i = auconv_set_converter(wm8731_formats, wm8731_nformats,
					 mode, p, true, fil);
		if (i < 0)
			return EINVAL;
	}

	if (setmode == AUMODE_RECORD)
		pxa2x0_i2s_setspeed(&sc->sc_i2s, &rec->sample_rate);
	else
		pxa2x0_i2s_setspeed(&sc->sc_i2s, &play->sample_rate);

	return 0;
}

static int
wm8731_halt_output(void *hdl)
{
	struct zaudio_softc *sc = hdl;
	int rv;

	rv = pxa2x0_i2s_halt_output(&sc->sc_i2s);
	if (!sc->sc_recording)
		wm8731_standby(sc);
	sc->sc_playing = 0;

	return rv;
}

static int
wm8731_halt_input(void *hdl)
{
	struct zaudio_softc *sc = hdl;
	int rv;

	rv = pxa2x0_i2s_halt_input(&sc->sc_i2s);
	if (!sc->sc_playing)
		wm8731_standby(sc);
	sc->sc_recording = 0;

	return rv;
}

static int
wm8731_getdev(void *hdl, struct audio_device *ret)
{

	*ret = wm8731_device;
	return 0;
}

#define WM8731_SPKR_LVL		0
#define WM8731_SPKR_MUTE	1
#define WM8731_MIC_LVL		2
#define WM8731_MIC_MUTE		3
#define WM8731_RECORD_SOURCE	4
#define WM8731_OUTPUT_CLASS	5
#define WM8731_INPUT_CLASS	6
#define WM8731_RECORD_CLASS	7

static int
wm8731_set_port(void *hdl, struct mixer_ctrl *mc)
{
	struct zaudio_softc *sc = hdl;
	int error = EINVAL;
	int s;

	s = splbio();
	iic_acquire_bus(sc->sc_i2c, 0);

	switch (mc->dev) {
	case WM8731_SPKR_LVL:
		if (mc->type != AUDIO_MIXER_VALUE)
			break;
		if (mc->un.value.num_channels == 1) {
			sc->sc_volume[WM8731_OP_SPKR].left =
			    mc->un.value.level[AUDIO_MIXER_LEVEL_MONO];
			sc->sc_volume[WM8731_OP_SPKR].right =
			    mc->un.value.level[AUDIO_MIXER_LEVEL_MONO];
		} else if (mc->un.value.num_channels == 2) {
			sc->sc_volume[WM8731_OP_SPKR].left =
			    mc->un.value.level[AUDIO_MIXER_LEVEL_LEFT];
			sc->sc_volume[WM8731_OP_SPKR].right =
			    mc->un.value.level[AUDIO_MIXER_LEVEL_RIGHT];
		}
		else
			break;
		wm8731_update_volume(sc, WM8731_OP_SPKR);
		error = 0;
		break;

	case WM8731_SPKR_MUTE:
		if (mc->type != AUDIO_MIXER_ENUM)
			break;
		UNMUTE(sc, WM8731_OP_SPKR, mc->un.ord ? 1 : 0);
		wm8731_update_mutes(sc, 1);
		error = 0;
		break;

	case WM8731_MIC_LVL:
		if (mc->type != AUDIO_MIXER_VALUE)
			break;
		if (mc->un.value.num_channels == 1)
			sc->sc_volume[WM8731_OP_MIC].left =
			    mc->un.value.level[AUDIO_MIXER_LEVEL_MONO];
		else
			break;
		wm8731_update_volume(sc, WM8731_OP_MIC);
		error = 0;
		break;

	case WM8731_MIC_MUTE:
		if (mc->type != AUDIO_MIXER_ENUM)
			break;
		UNMUTE(sc, WM8731_OP_MIC, mc->un.ord ? 1 : 0);
		wm8731_update_mutes(sc, 2);
		error = 0;
		break;

	case WM8731_RECORD_SOURCE:
		if (mc->type != AUDIO_MIXER_ENUM)
			break;
		if (mc->un.ord != 0)
			break;
		/* MIC only */
		error = 0;
		break;
	}

	iic_release_bus(sc->sc_i2c, 0);
	splx(s);

	return error;
}

static int
wm8731_get_port(void *hdl, struct mixer_ctrl *mc)
{
	struct zaudio_softc *sc = hdl;
	int error = EINVAL;

	switch (mc->dev) {
	case WM8731_SPKR_LVL:
		if (mc->type != AUDIO_MIXER_VALUE)
			break;
		if (mc->un.value.num_channels == 1)
			mc->un.value.level[AUDIO_MIXER_LEVEL_MONO] =
			    sc->sc_volume[WM8731_OP_SPKR].left;
		else if (mc->un.value.num_channels == 2) {
			mc->un.value.level[AUDIO_MIXER_LEVEL_LEFT] =
			    sc->sc_volume[WM8731_OP_SPKR].left;
			mc->un.value.level[AUDIO_MIXER_LEVEL_RIGHT] =
			    sc->sc_volume[WM8731_OP_SPKR].right;
		}
		else
			break;
		error = 0;
		break;

	case WM8731_SPKR_MUTE:
		if (mc->type != AUDIO_MIXER_ENUM)
			break;
		mc->un.ord = sc->sc_unmute[WM8731_OP_SPKR] ? 1 : 0;
		error = 0;
		break;

	case WM8731_MIC_LVL:
		if (mc->type != AUDIO_MIXER_VALUE)
			break;
		if (mc->un.value.num_channels == 1)
			mc->un.value.level[AUDIO_MIXER_LEVEL_MONO] =
			    sc->sc_volume[WM8731_OP_MIC].left;
		else
			break;
		error = 0;
		break;

	case WM8731_MIC_MUTE:
		if (mc->type != AUDIO_MIXER_ENUM)
			break;
		mc->un.ord = sc->sc_unmute[WM8731_OP_MIC] ? 1 : 0;
		error = 0;
		break;

	case WM8731_RECORD_SOURCE:
		if (mc->type != AUDIO_MIXER_ENUM)
			break;
		mc->un.ord = 0; /* MIC only */
		error = 0;
		break;
	}

	return error;
}

/*ARGSUSED*/
static int
wm8731_query_devinfo(void *hdl, struct mixer_devinfo *di)
{

	switch (di->index) {
	case WM8731_SPKR_LVL:
		di->type = AUDIO_MIXER_VALUE;
		di->mixer_class = WM8731_OUTPUT_CLASS;
		di->prev = AUDIO_MIXER_LAST;
		di->next = WM8731_SPKR_MUTE;
		strlcpy(di->label.name, AudioNspeaker,
		    sizeof(di->label.name));
		di->un.v.num_channels = 1;
		strlcpy(di->un.v.units.name, AudioNvolume,
		    sizeof(di->un.v.units.name));
		break;

	case WM8731_SPKR_MUTE:
		di->type = AUDIO_MIXER_ENUM;
		di->mixer_class = WM8731_OUTPUT_CLASS;
		di->prev = WM8731_SPKR_LVL;
		di->next = AUDIO_MIXER_LAST;
mute:
		strlcpy(di->label.name, AudioNmute, sizeof(di->label.name));
		di->un.e.num_mem = 2;
		strlcpy(di->un.e.member[0].label.name, AudioNon,
		    sizeof(di->un.e.member[0].label.name));
		di->un.e.member[0].ord = 0;
		strlcpy(di->un.e.member[1].label.name, AudioNoff,
		    sizeof(di->un.e.member[1].label.name));
		di->un.e.member[1].ord = 1;
		break;

	case WM8731_MIC_LVL:
		di->type = AUDIO_MIXER_VALUE;
		di->mixer_class = WM8731_INPUT_CLASS;
		di->prev = AUDIO_MIXER_LAST;
		di->next = WM8731_MIC_MUTE;
		strlcpy(di->label.name, AudioNmicrophone,
		    sizeof(di->label.name));
		strlcpy(di->un.v.units.name, AudioNvolume,
		    sizeof(di->un.v.units.name));
		di->un.v.num_channels = 1;
		break;

	case WM8731_MIC_MUTE:
		di->type = AUDIO_MIXER_ENUM;
		di->mixer_class = WM8731_INPUT_CLASS;
		di->prev = WM8731_MIC_LVL;
		di->next = AUDIO_MIXER_LAST;
		goto mute;

	case WM8731_RECORD_SOURCE:
		di->type = AUDIO_MIXER_ENUM;
		di->mixer_class = WM8731_RECORD_CLASS;
		di->prev = AUDIO_MIXER_LAST;
		di->next = AUDIO_MIXER_LAST;
		strlcpy(di->label.name, AudioNsource, sizeof(di->label.name));
		di->un.e.num_mem = 1;
		strlcpy(di->un.e.member[0].label.name, AudioNmicrophone,
		    sizeof(di->un.e.member[0].label.name));
		di->un.e.member[0].ord = 0;
		break;

	case WM8731_OUTPUT_CLASS:
		di->type = AUDIO_MIXER_CLASS;
		di->mixer_class = WM8731_OUTPUT_CLASS;
		di->prev = AUDIO_MIXER_LAST;
		di->next = AUDIO_MIXER_LAST;
		strlcpy(di->label.name, AudioCoutputs, sizeof(di->label.name));
		break;

	case WM8731_INPUT_CLASS:
		di->type = AUDIO_MIXER_CLASS;
		di->mixer_class = WM8731_INPUT_CLASS;
		di->prev = AUDIO_MIXER_LAST;
		di->next = AUDIO_MIXER_LAST;
		strlcpy(di->label.name, AudioCinputs, sizeof(di->label.name));
		break;

	case WM8731_RECORD_CLASS:
		di->type = AUDIO_MIXER_CLASS;
		di->mixer_class = WM8731_RECORD_CLASS;
		di->prev = AUDIO_MIXER_LAST;
		di->next = AUDIO_MIXER_LAST;
		strlcpy(di->label.name, AudioCinputs, sizeof(di->label.name));
		break;

	default:
		return ENXIO;
	}

	return 0;
}

static int
wm8731_start_output(void *hdl, void *block, int bsize, void (*intr)(void *),
    void *intrarg)
{
	struct zaudio_softc *sc = hdl;
	int rv;

	/* Power up codec if we are not already playing. */
	if (!sc->sc_playing) {
		sc->sc_playing = 1;
		wm8731_play_setup(sc);
	}

	/* Start DMA via I2S */
	rv = pxa2x0_i2s_start_output(&sc->sc_i2s, block, bsize, intr, intrarg);
	if (rv) {
		if (!sc->sc_recording)
			wm8731_standby(sc);
		sc->sc_playing = 0;
	}

	return rv;
}

static int
wm8731_start_input(void *hdl, void *block, int bsize, void (*intr)(void *),
    void *intrarg)
{
	struct zaudio_softc *sc = hdl;
	int rv;

	/* Power up codec if we are not already recording. */
	if (!sc->sc_recording) {
		sc->sc_recording = 1;
		wm8731_record_setup(sc);
	}

	/* Start DMA via I2S */
	rv = pxa2x0_i2s_start_input(&sc->sc_i2s, block, bsize, intr, intrarg);
	if (rv) {
		if (!sc->sc_playing)
			wm8731_standby(sc);
		sc->sc_recording = 0;
	}
	return rv;
}
