/*
 * Copyright © 2007 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *    Eric Anholt <eric@anholt.net>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <pciaccess.h>
#include <err.h>
#include <unistd.h>

#include "reg_dumper.h"
#include "../i810_reg.h"

struct idle_flags {
    uint32_t instdone_flag;
    char *name;
    int ignore;
    unsigned int count;
};

struct idle_flags i915_idle_flags[] = {
    {IDCT_DONE, "IDCT", 1},
    {IQ_DONE, "IQ", 1},
    {PR_DONE, "PR", 1},
    {VLD_DONE, "VLD", 1},
    {IP_DONE, "IP", 1},
    {FBC_DONE, "FBC"},
    {BINNER_DONE, "BINNER"},
    {SF_DONE, "SF"},
    {SE_DONE, "SE"},
    {WM_DONE, "WM"},
    {IZ_DONE, "IZ"},
    {PERSPECTIVE_INTERP_DONE, "perspective interpolation"},
    {DISPATCHER_DONE, "dispatcher"},
    {PROJECTION_DONE, "projection and LOD"},
    {DEPENDENT_ADDRESS_DONE, "dependent address calc"},
    {TEXTURE_FETCH_DONE, "texture fetch"},
    {TEXTURE_DECOMPRESS_DONE, "texture decompress"},
    {SAMPLER_CACHE_DONE, "sampler cache"},
    {FILTER_DONE, "filter"},
    {BYPASS_FIFO_DONE, "bypass FIFO"},
    {PS_DONE, "PS"},
    {CC_DONE, "CC"},
    {MAP_FILTER_DONE, "map filter"},
    {MAP_L2_IDLE, "map L2", 1},
    {0x80000038, "reserved bits", 1},
    {0, "total"},
    {0, "other"},
};

struct idle_flags i965_idle_flags[] = {
    {I965_SF_DONE, "SF"},
    {I965_SE_DONE, "SE"},
    {I965_WM_DONE, "WM"},
    {I965_TEXTURE_FETCH_DONE, "texture fetch"},
    {I965_SAMPLER_CACHE_DONE, "sampler cache"},
    {I965_FILTER_DONE, "filter"},
    {I965_PS_DONE, "PS"},
    {I965_CC_DONE, "CC"},
    {I965_MAP_FILTER_DONE, "map filter"},
    {I965_MAP_L2_IDLE, "map L2"},
    {I965_CP_DONE, "CP"},
    {0, "total"},
    {0, "other"},
};

/* Fills in the "other" and "total" fields' idle flags */
static void
setup_other_flags(I830Ptr pI830,
		  struct idle_flags *idle_flags, int idle_flag_count)
{
    uint32_t other_idle_flags, total_idle_flags = 0;
    int i;

    if (IS_I965G(pI830))
	other_idle_flags = ~(I965_RING_0_ENABLE);
    else
	other_idle_flags = ~(RING_0_ENABLE | RING_1_ENABLE | RING_2_ENABLE);

    for (i = 0; i < idle_flag_count - 2; i++) {
	other_idle_flags &= ~idle_flags[i].instdone_flag;
	if (!idle_flags[i].ignore)
	    total_idle_flags |= idle_flags[i].instdone_flag;
    }
    idle_flags[idle_flag_count - 2].instdone_flag = total_idle_flags;
    idle_flags[idle_flag_count - 1].instdone_flag = other_idle_flags;
}

int main(int argc, char **argv)
{
    struct pci_device *dev;
    I830Rec i830;
    I830Ptr pI830 = &i830;
    ScrnInfoRec scrn;
    int err, mmio_bar;
    void *mmio;
    struct idle_flags *idle_flags;
    int idle_flag_count;

    err = pci_system_init();
    if (err != 0) {
	fprintf(stderr, "Couldn't initialize PCI system: %s\n", strerror(err));
	exit(1);
    }

    /* Grab the graphics card */
    dev = pci_device_find_by_slot(0, 0, 2, 0);
    if (dev == NULL)
	errx(1, "Couldn't find graphics card");

    err = pci_device_probe(dev);
    if (err != 0) {
	fprintf(stderr, "Couldn't probe graphics card: %s\n", strerror(err));
	exit(1);
    }

    if (dev->vendor_id != 0x8086)
	errx(1, "Graphics card is non-intel");

    i830.PciInfo = dev;

    mmio_bar = IS_I9XX((&i830)) ? 0 : 1;

    err = pci_device_map_range (dev,
				dev->regions[mmio_bar].base_addr,
				dev->regions[mmio_bar].size, 
				PCI_DEV_MAP_FLAG_WRITABLE,
				&mmio);

    if (err != 0) {
	fprintf(stderr, "Couldn't map MMIO region: %s\n", strerror(err));
	exit(1);
    }
    i830.mmio = mmio;

    scrn.scrnIndex = 0;
    scrn.pI830 = &i830;

    if (IS_I965G(pI830)) {
	idle_flags = i965_idle_flags;
	idle_flag_count = ARRAY_SIZE(i965_idle_flags);
    } else {
	idle_flags = i915_idle_flags;
	idle_flag_count = ARRAY_SIZE(i915_idle_flags);
    }

    setup_other_flags(pI830, idle_flags, idle_flag_count);

    for (;;) {
	int i, j;

	for (i = 0; i < 100; i++) {
	    uint32_t instdone;

	    if (IS_I965G(pI830))
		instdone = INREG(INST_DONE_I965);
	    else
		instdone = INREG(INST_DONE);

	    for (j = 0; j < idle_flag_count; j++) {
		if ((instdone & idle_flags[j].instdone_flag) !=
		    idle_flags[j].instdone_flag)
		    idle_flags[j].count++;
	    }

	    usleep (10000);
	}

	for (j = 0; j < idle_flag_count; j++) {
	    if (!idle_flags[j].ignore)
		printf("%25s: %3d\n", idle_flags[j].name, idle_flags[j].count);
	    else
		printf("%25s: %3d (unreliable)\n",
		       idle_flags[j].name, idle_flags[j].count);
	    idle_flags[j].count = 0;
	}
	printf("\n");
    }

    return 0;
}

void xf86DrvMsg(int scrnIndex, int severity, const char *format, ...)
{
    va_list va;

    switch (severity) {
    case X_INFO:
	printf("(II): ");
	break;
    case X_WARNING:
	printf("(WW): ");
	break;
    case X_ERROR:
	printf("(EE): ");
	break;
    }

    va_start(va, format);
    vprintf(format, va);
    va_end(va);
}
