/****************************************************************
Copyright (C) Lucent Technologies 1997
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name Lucent Technologies or any of
its entities not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.

LUCENT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
IN NO EVENT SHALL LUCENT OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.
****************************************************************/

#ifndef LIBAWK_H_
#define LIBAWK_H_	20070501

#include <sys/types.h>

#include <inttypes.h>
#include <assert.h>
#include <setjmp.h>

typedef double	awkfloat_t;

#define	RECSIZE	(8 * 1024)	/* sets limit on records, fields, etc., etc. */

/* awkcell_t:  all information about a variable or constant */

typedef struct awkcell_t {
	uint8_t		 ctype;		/* OCELL, OBOOL, OJUMP, etc. */
	uint8_t		 csub;		/* CCON, CTEMP, CFLD, etc. */
	char		*nval;		/* name, for variables only */
	char		*sval;		/* string value */
	awkfloat_t	 fval;		/* value as number */
	int		 tval;		/* type info: STR|NUM|ARR|FCN|FLD|CON|DONTFREE */
	struct awkcell_t *cnext;	/* ptr to next if chained */
} awkcell_t;

typedef struct awkarray_t {		/* symbol table array */
	int		nelem;		/* elements in table right now */
	int		size;		/* size of tab */
	awkcell_t	**tab;		/* hash table pointers */
} awkarray_t;

#define	NSYMTAB	50	/* initial size of a symbol table */


/* awkcell_t.tval values: */
#define	NUM	01	/* number value is valid */
#define	STR	02	/* string value is valid */
#define DONTFREE 04	/* string space is not freeable */
#define	CON	010	/* this is a constant */
#define	ARR	020	/* this is an array */
#define	FCN	040	/* this is a function name */
#define FLD	0100	/* this is a field $1, $2, ... */
#define	REC	0200	/* this is $0 */


/* function types */
#define	FLENGTH	1
#define	FSQRT	2
#define	FEXP	3
#define	FLOG	4
#define	FINT	5
#define	FSYSTEM	6
#define	FRAND	7
#define	FSRAND	8
#define	FSIN	9
#define	FCOS	10
#define	FATAN	11
#define	FTOUPPER 12
#define	FTOLOWER 13
#define	FFLUSH	14
#define FSYSTIME	15
#define FSTRFTIME	16

/* awknode_t:  parse tree is made of nodes, with awkcell_t's at bottom */
typedef struct awknode_t {
	int		  ntype;
	struct awknode_t *nnext;
	int		  lineno;
	int		  nobj;
	struct awknode_t *narg[1];	/* variable: actual size set by calling malloc */
} awknode_t;

#define	NIL	((awknode_t *) NULL)

/* ctypes */
#define OCELL	1
#define OBOOL	2
#define OJUMP	3

/* awkcell_t subtypes: csub */
#define	CFREE	7
#define CCOPY	6
#define CCON	5
#define CTEMP	4
#define CNAME	3 
#define CVAR	2
#define CFLD	1
#define	CUNK	0

/* bool subtypes */
#define BTRUE	11
#define BFALSE	12

/* jump subtypes */
#define JEXIT	21
#define JNEXT	22
#define	JBREAK	23
#define	JCONT	24
#define	JRET	25
#define	JNEXTFILE	26

/* node types */
#define NVALUE	1
#define NSTAT	2
#define NEXPR	3



#define NOTLEGAL(n)	(n <= FIRSTTOKEN || n >= LASTTOKEN || awklib_proctab[n-FIRSTTOKEN] == awklib_nullproc)
#define ISVALUE(n)	((n)->ntype == NVALUE)
#define ISEXPR(n)	((n)->ntype == NEXPR)
#define ISJUMP(n)	((n)->ctype == OJUMP)
#define ISEXIT(n)	((n)->csub == JEXIT)
#define	ISBREAK(n)	((n)->csub == JBREAK)
#define	ISCONT(n)	((n)->csub == JCONT)
#define	ISNEXT(n)	((n)->csub == JNEXT || (n)->csub == JNEXTFILE)
#define	ISRET(n)	((n)->csub == JRET)
#define ISREC(n)	((n)->tval & REC)
#define ISFLD(n)	((n)->tval & FLD)
#define ISSTR(n)	((n)->tval & STR)
#define ISNUM(n)	((n)->tval & NUM)
#define ISARR(n)	((n)->tval & ARR)
#define ISFCN(n)	((n)->tval & FCN)
#define ISTRUE(n)	((n)->csub == BTRUE)
#define ISTEMP(n)	((n)->csub == CTEMP)
#define	ISARGUMENT(n)	((n)->nobj == ARG)
/* #define freeable(p)	(!((p)->tval & DONTFREE)) */
#define FREEABLE(p)	( ((p)->tval & (STR|DONTFREE)) == STR )

/* structures used by regular expression matching machinery, mostly b.c: */

#define NCHARS	(256+3)		/* 256 handles 8-bit chars; 128 does 7-bit */
				/* watch out in match(), etc. */
typedef struct awkrrow_t {
	long	ltype;	/* long avoids pointer warnings on 64-bit */
	union {
		int		 i;
		awknode_t	*np;
		uint8_t		*up;
	} lval;		/* because Al stores a pointer in it! */
	int	*lfollow;
} awkrrow_t;

typedef struct fa_t {
	unsigned int	**gototab;
	uint8_t		 *out;
	uint8_t		 *restr;
	int		**posns;
	int		  state_count;
	int		  anchor;
	int		  use;
	int		  initstat;
	int		  curstat;
	int		  accept;
	struct awkrrow_t  re[1];	/* variable: actual size set by calling malloc */
} fa_t;

typedef struct awkfile_t {
	FILE		*fp;
	const char	*fname;
	int		 mode;	/* '|', 'a', 'w' => LE/LT, GT */
} awkfile_t;

typedef struct awkframe_t {	/* stack frame for awk function calls */
	int		  nargs;	/* number of arguments in this call */
	awkcell_t	 *fcncell;	/* pointer to awkcell_t for function */
	awkcell_t	**args;		/* pointer to array of arguments after execute */
	awkcell_t	 *retval;	/* return value */
} awkframe_t;

#define	NARGS	50	/* max args in a call */

#define PA2NUM	50	/* max number of pat,pat patterns allowed */

#define MAXFLD	2

#define	NFA	128	/* cache this many dynamic fa_t's */

enum {
	AWKLIB_RUNNING,
	AWKLIB_COMPILING,
	AWKLIB_ERROR_PRINTING
};

typedef struct awk_t {
	int		  stage;		/* from the enum above */
	int		  safe;		/* 0 => unsafe, 1 => safe */
	char		**FS;		/* input file separators */
	char		**RS;		/* record separator */
	char		**OFS;		/* output field separator */
	char		**ORS;		/* output record separator */
	char		**OFMT;		/* output format for numbers */
	awkfloat_t	 *NR;		/* number of current record */
	awkfloat_t	 *FNR;	/* number of current record in current file */
	awkfloat_t	 *NF;		/* number of fields in current record */
	int		  recsize;	/* size of current record, orig RECSIZE */
	char		**FILENAME;	/* current filename argument */
	char		**SUBSEP;	/* subscript separator for a[i,j,k]; default \034 */
	awkfloat_t	 *RSTART;	/* start of re matched with ~; origin 1 (!) */
	awkfloat_t	 *RLENGTH;	/* length of same */
	uint8_t		 *record;	/* points to $0 */
	int		  lineno;		/* line number in awk program */
	int		  errorflag;	/* 1 if error has occurred */
	int		  donefld;	/* 1 if record broken into fields */
	int		  donerec;	/* 1 if record is valid (no fld has changed */
	int		  dbg;
	uint8_t		 *patbeg;	/* beginning of pattern matched */
	int		  patlen;		/* length of pattern matched.  set in b.c */
	awkarray_t	*symtab;
	awkcell_t	*nrloc;		/* NR */
	awkcell_t	*fnrloc;	/* FNR */
	awkcell_t	*nfloc;		/* NF */
	int		 shiftc;	/* shift count for current line */
	awkcell_t	*rstartloc;	/* RSTART */
	awkcell_t	*rlengthloc;	/* RLENGTH */
	awknode_t	*winner;
	awknode_t	*nullnode;
	int		pairstack[PA2NUM];/* state of each pat,pat */
	int		paircnt;	/* number of them in use */
	char		*lexprog;	/* points to program argument if it exists */
	int		nfields;	/* last allocated slot for $i */
	awkfloat_t *ARGC;		/* number of arguments from command line */
	awkarray_t	*ARGVtab;	/* symbol table containing ARGV[...] */
	awkcell_t	*fsloc;		/* FS */
	char		**CONVFMT;	/* format for conversions in awklib_getsval */
	awkarray_t	*ENVtab;	/* symbol table containing ENVIRON[...] */
	awkcell_t	*symtabloc;	/* SYMTAB */
	awkcell_t	**fldtab;	/* pointers to awkcell_t */
	awkcell_t	*literal0;
	char		*cmdname;	/* gets argv[0] for error messages */
	char		*curfname;	/* current function name */
	int		infunc;		/* = 1 if in arglist or body of func */
	int		bracecnt;
	int		brackcnt;
	int		parencnt;
	awknode_t	*curnode;	/* the node being executed, for debugging */
	awknode_t	*arglist;	/* list of args for current function */
	char		ebuf[300];
	char		*ep;
	char		yysbuf[100];	/* pushback buffer */
	char		*yysptr;
	char		**scripts;	/* program filenames from -f's */
	int		scriptc;	/* number of filenames */
	int		scriptsize;	/* size of scripts array */
	int		curscript;	/* current filename */
	FILE		*yyin;		/* lex input file */
	int		bracechecked;	/* if we've already checked braces */
	awkfile_t	*files;
	unsigned	 nfiles;
	awkframe_t	 *frame;	/* base of stack frames; dynamically allocated */
	int		nframe;		/* number of frames allocated */
	awkframe_t	 *fp;		/* frame pointer. bottom level unused */
	int		init_getrec;
	FILE		*infile;
	char		*file;
	char		*fields;
	int		fieldssize;
	/* regexp vars */
	uint8_t		*prestr;	/* current position in current re */
	uint8_t		*lastre;	/* origin of last re */
	uint8_t		*rlxstr;
	int		maxsetvec;
	int		*setvec;
	int		*tmpset;
	int		lastfld;	/* last used field */
	int		rtok;		/* next token in current re */
	int		rlxval;
	int		setcnt;
	int		poscnt;
	fa_t		*fatab[NFA];
	int		nfatab;		/* entries in fatab */
	int		sc;		/* 1 => return a } right now */
	int		reg;		/* 1 => return a REGEXPR now */
	int		lexbufsize; 	/* BUG: setting this small causes core dump! */
	char		*lexbuf;
	int		stringbufsz;
	uint8_t		*stringbuf;
	uint8_t		*regexprbuf;
	int	 	regexprbufsz;
	int		make_dfa_now;
	uint8_t		*cclenterbuf;
	int		cclenterbufsz;
	uint8_t		*relexbuf;
	int		relexbufsz;
	/* misc */
	int		argno;		/* current input argument number */
	/* FS stuff */
	char		static_inputFS[16];
	size_t		len_inputFS;
	char		*inputFS;
	/* visited various functions */
	int		syntaxc;	/* number of times in SYNTAX() */
	int		eprintc;	/* number of times in eprint() */
	/* control info */
	jmp_buf		 env;
	awkcell_t	*tmps;		/* free temporary cells for execution */
	/* lex/parse */
	/* prototype regexps */
} awk_t;

/* external prototypes */
int	awklib_parse(void *);
int	awklib_lex(void *, void *);
void	awklib_syminit(awk_t *);
void	awklib_arginit(awk_t *, int, char **);
void	awklib_envinit(awk_t *, char **);
void	awklib_recinit(awk_t *, unsigned int);
void	awklib_error(const char *);
void	awklib_bracecheck(awk_t *);
void	awklib_run(awk_t *, awknode_t *);
void	awklib_setdefaults(awk_t *);
int	awklib_setvar(awk_t *, const char *, int, void *);
int	awklib_getvarint(awk_t *, const char *);
void	*awklib_getvarptr(awk_t *, const char *);

#endif /* LIBAWK_H_ */
