/****************************************************************
Copyright (C) Lucent Technologies 1997
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name Lucent Technologies or any of
its entities not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.

LUCENT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
IN NO EVENT SHALL LUCENT OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.
****************************************************************/

#define DEBUG
#include <stdio.h>
#include <ctype.h>
#include <dlfcn.h>
#include <wchar.h>
#include <wctype.h>
#include <limits.h>
#include <math.h>
#include <regex.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "awklib.h"
#include "awkgram.h"
#include "proto.h"

#ifndef MIN
#define	MIN(a,b)	(((a) < (b)) ? (a) : (b))
#endif

#ifndef ABS
#define ABS(x) 		(((x) >= 0) ? (x) : -(x))
#endif

#define TEMPFREE(awkp, x)	do {					\
	if (ISTEMP(x)) {						\
		tfree(awkp, x);						\
	} else {							\
	}								\
} while (/* CONSTCOND */0)

/*
#undef tempfree

void tempfree(awkcell_t *p) {
	if (p->ctype == OCELL && (p->csub < CUNK || p->csub > CFREE)) {
		WARNING("bad csub %d in awkcell_t %d %s",
			p->csub, p->ctype, p->sval);
	}
	if (ISTEMP(p))
		tfree(p);
}
*/

/* do we really need these? */
/* #ifdef _NFILE */
/* #ifndef FOPEN_MAX */
/* #define FOPEN_MAX _NFILE */
/* #endif */
/* #endif */
/*  */
/* #ifndef	FOPEN_MAX */
/* #define	FOPEN_MAX	40 */	/* max number of open files */
/* #endif */
/*  */
/* #ifndef RAND_MAX */
/* #define RAND_MAX	32767 */	/* all that ansi guarantees */
/* #endif */



static awkcell_t	truecell	={ OBOOL, BTRUE, 0, 0, 1.0, NUM, NULL };
awkcell_t	*True	= &truecell;
static awkcell_t	falsecell	={ OBOOL, BFALSE, 0, 0, 0.0, NUM, NULL };
awkcell_t	*False	= &falsecell;
static awkcell_t	breakcell	={ OJUMP, JBREAK, 0, 0, 0.0, NUM, NULL };
awkcell_t	*jbreak	= &breakcell;
static awkcell_t	contcell	={ OJUMP, JCONT, 0, 0, 0.0, NUM, NULL };
awkcell_t	*jcont	= &contcell;
static awkcell_t	nextcell	={ OJUMP, JNEXT, 0, 0, 0.0, NUM, NULL };
awkcell_t	*jnext	= &nextcell;
static awkcell_t	nextfilecell	={ OJUMP, JNEXTFILE, 0, 0, 0.0, NUM, NULL };
awkcell_t	*jnextfile	= &nextfilecell;
static awkcell_t	exitcell	={ OJUMP, JEXIT, 0, 0, 0.0, NUM, NULL };
awkcell_t	*jexit	= &exitcell;
static awkcell_t	retcell		={ OJUMP, JRET, 0, 0, 0.0, NUM, NULL };
awkcell_t	*jret	= &retcell;
static awkcell_t	tempcell	={ OCELL, CTEMP, 0, "", 0.0, NUM|STR|DONTFREE, NULL };

/* free a tempcell */
static void
tfree(awk_t *awkp, awkcell_t *a)
{
	if (FREEABLE(a)) {
		DPRINTF(awkp, ("freeing %s %s %o\n", NN(a->nval), NN(a->sval), a->tval) );
		XFREE(a->sval);
	}
	if (a == awkp->tmps)
		FATAL(awkp, "tempcell list is curdled");
	a->cnext = awkp->tmps;
	awkp->tmps = a;
}

/* buffer memory management */
int awklib_adjbuf(awk_t *awkp, uint8_t **pbuf, int *psiz, int minlen, int quantum, uint8_t **pbptr,
	const char *whatrtn)
/* pbuf:    address of pointer to buffer being managed
 * psiz:    address of buffer size variable
 * minlen:  minimum length of buffer needed
 * quantum: buffer size quantum
 * pbptr:   address of movable pointer into buffer, or 0 if none
 * whatrtn: name of the calling routine if failure should cause fatal error
 *
 * return   0 for realloc failure, !=0 for success
 */
{
	if (minlen > *psiz) {
		char *tbuf;
		int rminlen = quantum ? minlen % quantum : 0;
		int boff = pbptr ? *pbptr - *pbuf : 0;
		/* round up to next multiple of quantum */
		if (rminlen)
			minlen += quantum - rminlen;
		tbuf = realloc(*pbuf, minlen);
		DPRINTF(awkp, ("awklib_adjbuf %s: %d %d (pbuf=%p, tbuf=%p)\n", whatrtn, *psiz, minlen, *pbuf, tbuf) );
		if (tbuf == NULL) {
			if (whatrtn)
				FATAL(awkp, "out of memory in %s", whatrtn);
			return 0;
		}
		*pbuf = (uint8_t *)tbuf;
		*psiz = minlen;
		if (pbptr)
			*pbptr = (uint8_t *)tbuf + boff;
	}
	return 1;
}

/* in case stdin, etc., are not constants */
static void
stdinit(awk_t *awkp)
{
	awkp->nfiles = FOPEN_MAX;
	awkp->files = calloc(awkp->nfiles, sizeof(awkfile_t));
	if (awkp->files == NULL)
		FATAL(awkp, "can't allocate file memory for %zu files", awkp->nfiles);
	awkp->files[0].fp = stdin;
	awkp->files[0].fname = "/dev/stdin";
	awkp->files[0].mode = LT;
	awkp->files[1].fp = stdout;
	awkp->files[1].fname = "/dev/stdout";
	awkp->files[1].mode = GT;
	awkp->files[2].fp = stderr;
	awkp->files[2].fname = "/dev/stderr";
	awkp->files[2].mode = GT;
}

static void
closeall(awk_t *awkp)
{
	size_t i;
	int stat;

	for (i = 0; i < awkp->nfiles; i++) {
		if (awkp->files[i].fp) {
			if (ferror(awkp->files[i].fp))
				WARNING(awkp,  "i/o error occurred on %s", awkp->files[i].fname );
			if (awkp->files[i].mode == '|' || awkp->files[i].mode == LE)
				stat = pclose(awkp->files[i].fp);
			else
				stat = fclose(awkp->files[i].fp);
			if (stat == EOF)
				WARNING(awkp,  "i/o error occurred while closing %s", awkp->files[i].fname );
		}
	}
}

static FILE *
openfile(awk_t *awkp, int a, const char *us)
{
	const char *s = us;
	size_t i;
	int m;
	FILE *fp = 0;

	if (*s == '\0')
		FATAL(awkp, "null file name in print or getline");
	for (i = 0; i < awkp->nfiles; i++)
		if (awkp->files[i].fname && strcmp(s, awkp->files[i].fname) == 0) {
			if (a == awkp->files[i].mode || (a==APPEND && awkp->files[i].mode==GT))
				return awkp->files[i].fp;
			if (a == FFLUSH)
				return awkp->files[i].fp;
		}
	if (a == FFLUSH)	/* didn't find it, so don't create it! */
		return NULL;

	for (i = 0; i < awkp->nfiles; i++)
		if (awkp->files[i].fp == NULL)
			break;
	if (i >= awkp->nfiles) {
		awkfile_t *nf;
		size_t nnf = awkp->nfiles + FOPEN_MAX;

		nf = realloc(awkp->files, nnf * sizeof(*nf));
		if (nf == NULL)
			FATAL(awkp, "cannot grow files for %s and %zu files", s, nnf);
		(void)memset(&nf[awkp->nfiles], 0, FOPEN_MAX * sizeof(*nf));
		awkp->nfiles = nnf;
		awkp->files = nf;
	}
	fflush(stdout);	/* force a semblance of order */
	m = a;
	if (a == GT) {
		fp = fopen(s, "w");
	} else if (a == APPEND) {
		fp = fopen(s, "a");
		m = GT;	/* so can mix > and >> */
	} else if (a == '|') {	/* output pipe */
		fp = popen(s, "w");
	} else if (a == LE) {	/* input pipe */
		fp = popen(s, "r");
	} else if (a == LT) {	/* getline <file */
		fp = strcmp(s, "-") == 0 ? stdin : fopen(s, "r");	/* "-" is stdin */
	} else	/* can't happen */
		FATAL(awkp, "illegal redirection %d", a);
	if (fp != NULL) {
		awkp->files[i].fname = awklib_tostring(awkp, s);
		awkp->files[i].fp = fp;
		awkp->files[i].mode = m;
	}
	return fp;
}

static const char *
filename(awk_t *awkp, FILE *fp)
{
	size_t i;

	for (i = 0; i < awkp->nfiles; i++)
		if (fp == awkp->files[i].fp)
			return awkp->files[i].fname;
	return "???";
}

static void
backsub(uint8_t **pb_ptr, const uint8_t **sptr_ptr)/* handle \\& variations */
{						/* sptr[0] == '\\' */
	uint8_t *pb = *pb_ptr;
	const uint8_t *sptr = *sptr_ptr;

	if (sptr[1] == '\\') {
		if (sptr[2] == '\\' && sptr[3] == '&') { /* \\\& -> \& */
			*pb++ = '\\';
			*pb++ = '&';
			sptr += 4;
		} else if (sptr[2] == '&') {	/* \\& -> \ + matched */
			*pb++ = '\\';
			sptr += 2;
		} else {			/* \\x -> \\x */
			*pb++ = *sptr++;
			*pb++ = *sptr++;
		}
	} else if (sptr[1] == '&') {	/* literal & */
		sptr++;
		*pb++ = *sptr++;
	} else				/* literal \ */
		*pb++ = *sptr++;

	*pb_ptr = pb;
	*sptr_ptr = sptr;
}

/* execution of parse tree starts here */
void
awklib_run(awk_t *awkp, awknode_t *a)
{
	stdinit(awkp);
	awklib_execute(awkp, a);
	closeall(awkp);
}

/* execute a node of the parse tree */
awkcell_t *
awklib_execute(awk_t *awkp, awknode_t *u)
{
	awkcell_t *(*proc)(awk_t *, awknode_t **, int);
	awkcell_t *x;
	awknode_t *a;

	if (u == NULL)
		return(True);
	for (a = u; ; a = a->nnext) {
		awkp->curnode = a;
		if (ISVALUE(a)) {
			x = (awkcell_t *) (a->narg[0]);
			if (ISFLD(x) && !awkp->donefld)
				awklib_fldbld(awkp);
			else if (ISREC(x) && !awkp->donerec)
				awklib_recbld(awkp);
			return(x);
		}
		if (NOTLEGAL(a->nobj))	/* probably a awkcell_t* but too risky to print */
			FATAL(awkp, "illegal statement");
		proc = awklib_proctab[a->nobj-FIRSTTOKEN];
		x = (*proc)(awkp, a->narg, a->nobj);
		if (ISFLD(x) && !awkp->donefld) {
			awklib_fldbld(awkp);
		} else if (ISREC(x) && !awkp->donerec) {
			awklib_recbld(awkp);
		}
		if (ISEXPR(a)) {
			return(x);
		}
		if (ISJUMP(x)) {
			return(x);
		}
		if (a->nnext == NULL) {
			return(x);
		}
		TEMPFREE(awkp, x);
	}
}


/* execute an awk program */
awkcell_t *
awklib_program(awk_t *awkp, awknode_t **a, int n)
{				/* a[0] = BEGIN, a[1] = body, a[2] = END */
	awkcell_t *x;

	if (setjmp(awkp->env) != 0)
		goto ex;
	if (a[0]) {		/* BEGIN */
		x = awklib_execute(awkp, a[0]);
		if (ISEXIT(x))
			return(True);
		if (ISJUMP(x))
			FATAL(awkp, "illegal break, continue, next or nextfile from BEGIN");
		TEMPFREE(awkp, x);
	}
	if (a[1] || a[2])
		while (awklib_getrec(awkp, &awkp->record, &awkp->recsize, 1) > 0) {
			x = awklib_execute(awkp, a[1]);
			if (ISEXIT(x))
				break;
			TEMPFREE(awkp, x);
		}
  ex:
	if (setjmp(awkp->env) != 0)	/* handles exit within END */
		goto ex1;
	if (a[2]) {		/* END */
		x = awklib_execute(awkp, a[2]);
		if (ISBREAK(x) || ISNEXT(x) || ISCONT(x))
			FATAL(awkp, "illegal break, continue, next or nextfile from END");
		TEMPFREE(awkp, x);
	}
  ex1:
	return(True);
}

awkcell_t *
awklib_call(awk_t *awkp, awknode_t **a, int n)	/* function call.  very kludgy and fragile */
{
	static const awkcell_t newcopycell = { OCELL, CCOPY, 0, "", 0.0, NUM|STR|DONTFREE, NULL };
	int i, ncall, ndef;
	int freed = 0; /* handles potential double freeing when fcn & param share a tempcell */
	awknode_t *x;
	awkcell_t *args[NARGS], *oargs[NARGS];	/* BUG: fixed size arrays */
	awkcell_t *y, *z, *fcn;
	char *s;

	fcn = awklib_execute(awkp, a[0]);	/* the function itself */
	s = fcn->nval;
	if (!ISFCN(fcn))
		FATAL(awkp, "calling undefined function %s", s);
	if (awkp->frame == NULL) {
		awkp->fp = awkp->frame = calloc(awkp->nframe += 100, sizeof(awkframe_t));
		if (awkp->frame == NULL)
			FATAL(awkp, "out of space for stack frames calling %s", s);
	}
	for (ncall = 0, x = a[1]; x != NULL; x = x->nnext)	/* args in call */
		ncall++;
	ndef = (int) fcn->fval;			/* args in awklib_defn */
	DPRINTF(awkp, ("calling %s, %d args (%d in defn), fp=%d\n", s, ncall, ndef, (int) (awkp->fp-awkp->frame)) );
	if (ncall > ndef)
		WARNING(awkp, "function %s called with %d args, uses only %d",
			s, ncall, ndef);
	if (ncall + ndef > NARGS)
		FATAL(awkp, "function %s has %d arguments, limit %d", s, ncall+ndef, NARGS);
	for (i = 0, x = a[1]; x != NULL; i++, x = x->nnext) {	/* get call args */
		DPRINTF(awkp, ("evaluate args[%d], fp=%d:\n", i, (int) (awkp->fp-awkp->frame)) );
		y = awklib_execute(awkp, x);
		oargs[i] = y;
		DPRINTF(awkp, ("args[%d]: %s %f <%s>, t=%o\n",
			   i, NN(y->nval), y->fval, ISARR(y) ? "(array)" : NN(y->sval), y->tval) );
		if (ISFCN(y))
			FATAL(awkp, "can't use function %s as argument in %s", y->nval, s);
		if (ISARR(y))
			args[i] = y;	/* arrays by ref */
		else
			args[i] = awklib_copycell(awkp, y);
		TEMPFREE(awkp, y);
	}
	for ( ; i < ndef; i++) {	/* add null args for ones not provided */
		args[i] = awklib_gettemp(awkp);
		*args[i] = newcopycell;
	}
	awkp->fp++;	/* now ok to up frame */
	if (awkp->fp >= awkp->frame + awkp->nframe) {
		int dfp = awkp->fp - awkp->frame;	/* old index */
		awkp->frame = realloc(awkp->frame, (awkp->nframe += 100) * sizeof(awkframe_t));
		if (awkp->frame == NULL)
			FATAL(awkp, "out of space for stack frames in %s", s);
		awkp->fp = awkp->frame + dfp;
	}
	awkp->fp->fcncell = fcn;
	awkp->fp->args = args;
	awkp->fp->nargs = ndef;	/* number defined with (excess are locals) */
	awkp->fp->retval = awklib_gettemp(awkp);

	DPRINTF(awkp, ("start exec of %s, fp=%d\n", s, (int) (awkp->fp-awkp->frame)) );
	y = awklib_execute(awkp, (awknode_t *)(fcn->sval));	/* execute body */
	DPRINTF(awkp, ("finished exec of %s, fp=%d\n", s, (int) (awkp->fp-awkp->frame)) );

	for (i = 0; i < ndef; i++) {
		awkcell_t *t = awkp->fp->args[i];
		if (ISARR(t)) {
			if (t->csub == CCOPY) {
				if (i >= ncall) {
					awklib_freesymtab(awkp, t);
					t->csub = CTEMP;
					TEMPFREE(awkp, t);
				} else {
					oargs[i]->tval = t->tval;
					oargs[i]->tval &= ~(STR|NUM|DONTFREE);
					oargs[i]->sval = t->sval;
					TEMPFREE(awkp, t);
				}
			}
		} else if (t != y) {	/* kludge to prevent freeing twice */
			t->csub = CTEMP;
			TEMPFREE(awkp, t);
		} else if (t == y && t->csub == CCOPY) {
			t->csub = CTEMP;
			TEMPFREE(awkp, t);
			freed = 1;
		}
	}
	TEMPFREE(awkp, fcn);
	if (ISEXIT(y) || ISNEXT(y))
		return y;
	if (freed == 0) {
		TEMPFREE(awkp, y);	/* don't free twice! */
	}
	z = awkp->fp->retval;			/* return value */
	DPRINTF(awkp, ("%s returns %g |%s| %o\n", s, awklib_getfval(awkp, z), awklib_getsval(awkp, z), z->tval) );
	awkp->fp--;
	return(z);
}

/* make a copy of a cell in a temp */
awkcell_t *
awklib_copycell(awk_t *awkp, awkcell_t *x)
{
	awkcell_t *y;

	/* copy is not constant or field */

	y = awklib_gettemp(awkp);
	y->tval = x->tval & ~(CON|FLD|REC);
	y->csub = CCOPY;	/* prevents freeing until call is over */
	y->nval = x->nval;	/* BUG? */
	if (ISSTR(x) /* || x->ctype == OCELL */) {
		y->sval = awklib_tostring(awkp, x->sval);
		y->tval &= ~DONTFREE;
	} else
		y->tval |= DONTFREE;
	y->fval = x->fval;
	return y;
}

/* nth argument of a function */
awkcell_t *
awklib_arg(awk_t *awkp, awknode_t **a, int n)
{

	n = awklib_ptoi(a[0]);	/* argument number, counting from 0 */
	DPRINTF(awkp, ("arg(%d), fp->nargs=%d\n", n, awkp->fp->nargs) );
	if (n+1 > awkp->fp->nargs)
		FATAL(awkp, "argument #%d of function %s was not supplied",
			n+1, awkp->fp->fcncell->nval);
	return awkp->fp->args[n];
}

/* break, continue, next, nextfile, return */
awkcell_t *
awklib_jump(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *y;

	switch (n) {
	case EXIT:
		if (a[0] != NULL) {
			y = awklib_execute(awkp, a[0]);
			awkp->errorflag = (int) awklib_getfval(awkp, y);
			TEMPFREE(awkp, y);
		}
		longjmp(awkp->env, 1);
	case RETURN:
		if (a[0] != NULL) {
			y = awklib_execute(awkp, a[0]);
			if ((y->tval & (STR|NUM)) == (STR|NUM)) {
				awklib_setsval(awkp, awkp->fp->retval, awklib_getsval(awkp, y));
				awkp->fp->retval->fval = awklib_getfval(awkp, y);
				awkp->fp->retval->tval |= NUM;
			}
			else if (y->tval & STR)
				awklib_setsval(awkp, awkp->fp->retval, awklib_getsval(awkp, y));
			else if (y->tval & NUM)
				awklib_setfval(awkp, awkp->fp->retval, awklib_getfval(awkp, y));
			else		/* can't happen */
				FATAL(awkp, "bad type variable %d", y->tval);
			TEMPFREE(awkp, y);
		}
		return(jret);
	case NEXT:
		return(jnext);
	case NEXTFILE:
		awklib_nextfile(awkp);
		return(jnextfile);
	case BREAK:
		return(jbreak);
	case CONTINUE:
		return(jcont);
	default:	/* can't happen */
		FATAL(awkp, "illegal jump type %d", n);
	}
	return 0;	/* not reached */
}

/* get next line from specific input */
/* a[0] is variable, a[1] is operator, a[2] is filename */
awkcell_t *
awklib_getline(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *r, *x;
	FILE *fp;
	uint8_t *buf;
	int bufsize = awkp->recsize;
	int mode;

	if ((buf = malloc(bufsize)) == NULL)
		FATAL(awkp, "out of memory in getline");

	fflush(stdout);	/* in case someone is waiting for a prompt */
	r = awklib_gettemp(awkp);
	if (a[1] != NULL) {		/* getline < file */
		x = awklib_execute(awkp, a[2]);		/* filename */
		mode = awklib_ptoi(a[1]);
		if (mode == '|')		/* input pipe */
			mode = LE;	/* arbitrary flag */
		fp = openfile(awkp, mode, awklib_getsval(awkp, x));
		TEMPFREE(awkp, x);
		if (fp == NULL)
			n = -1;
		else
			n = awklib_readrec(awkp, &buf, &bufsize, fp);
		if (n <= 0) {
			;
		} else if (a[0] != NULL) {	/* getline var <file */
			x = awklib_execute(awkp, a[0]);
			awklib_setsval(awkp, x, (char *)buf);
			TEMPFREE(awkp, x);
		} else {			/* getline <file */
			awklib_setsval(awkp, awkp->fldtab[0], (char *)buf);
			if (awklib_is_number(awkp->fldtab[0]->sval)) {
				awkp->fldtab[0]->fval = atof(awkp->fldtab[0]->sval);
				awkp->fldtab[0]->tval |= NUM;
			}
		}
	} else {			/* bare getline; use current input */
		if (a[0] == NULL)	/* getline */
			n = awklib_getrec(awkp, &awkp->record, &awkp->recsize, 1);
		else {			/* getline var */
			n = awklib_getrec(awkp, &buf, &bufsize, 0);
			x = awklib_execute(awkp, a[0]);
			awklib_setsval(awkp, x, (char *)buf);
			TEMPFREE(awkp, x);
		}
	}
	awklib_setfval(awkp, r, (awkfloat_t) n);
	free(buf);
	return r;
}

/* get NF */
awkcell_t *
awklib_getnf(awk_t *awkp, awknode_t **a, int n)
{
	if (awkp->donefld == 0)
		awklib_fldbld(awkp);
	return (awkcell_t *) a[0];
}

/* a[0] is symtab, a[1] is list of subscripts */
awkcell_t *
awklib_array(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x, *y, *z;
	char *s;
	awknode_t *np;
	uint8_t *buf;
	int bufsz = awkp->recsize;
	int nsub = strlen(*awkp->SUBSEP);

	if ((buf = malloc(bufsz)) == NULL)
		FATAL(awkp, "out of memory in array");

	x = awklib_execute(awkp, a[0]);	/* awkcell_t* for symbol table */
	buf[0] = 0;
	for (np = a[1]; np; np = np->nnext) {
		y = awklib_execute(awkp, np);	/* subscript */
		s = awklib_getsval(awkp, y);
		if (!awklib_adjbuf(awkp, &buf, &bufsz, strlen((char *)buf)+strlen(s)+nsub+1, awkp->recsize, 0, "array"))
			FATAL(awkp, "out of memory for %s[%s...]", x->nval, buf);
		strlcat((char *)buf, s, bufsz);
		if (np->nnext)
			strlcat((char *)buf, *awkp->SUBSEP, bufsz);
		TEMPFREE(awkp, y);
	}
	if (!ISARR(x)) {
		DPRINTF(awkp, ("making %s into an array\n", NN(x->nval)) );
		if (FREEABLE(x))
			XFREE(x->sval);
		x->tval &= ~(STR|NUM|DONTFREE);
		x->tval |= ARR;
		x->sval = (char *) awklib_makesymtab(awkp, NSYMTAB);
	}
	z = setawkvar(awkp, (char *)buf, "", 0.0, STR|NUM, (awkarray_t *) x->sval);
	z->ctype = OCELL;
	z->csub = CVAR;
	TEMPFREE(awkp, x);
	free(buf);
	return(z);
}

awkcell_t *
awklib_delete(awk_t *awkp, awknode_t **a, int n)	/* a[0] is symtab, a[1] is list of subscripts */
{
	awkcell_t *x, *y;
	awknode_t *np;
	uint8_t *s;
	int nsub = strlen(*awkp->SUBSEP);

	x = awklib_execute(awkp, a[0]);	/* awkcell_t* for symbol table */
	if (!ISARR(x))
		return True;
	if (a[1] == 0) {	/* delete the elements, not the table */
		awklib_freesymtab(awkp, x);
		x->tval &= ~STR;
		x->tval |= ARR;
		x->sval = (char *) awklib_makesymtab(awkp, NSYMTAB);
	} else {
		int bufsz = awkp->recsize;
		uint8_t *buf;
		if ((buf = malloc(bufsz)) == NULL)
			FATAL(awkp, "out of memory in adelete");
		buf[0] = 0;
		for (np = a[1]; np; np = np->nnext) {
			y = awklib_execute(awkp, np);	/* subscript */
			s = (uint8_t *)awklib_getsval(awkp, y);
			if (!awklib_adjbuf(awkp, &buf, &bufsz, strlen((char *)buf)+strlen((char *)s)+nsub+1, awkp->recsize, 0, "awklib_delete"))
				FATAL(awkp, "out of memory deleting %s[%s...]", x->nval, buf);
			strlcat((char *)buf, (char *)s, bufsz);
			if (np->nnext)
				strlcat((char *)buf, *awkp->SUBSEP, bufsz);
			TEMPFREE(awkp, y);
		}
		awklib_freeelem(x, (char *)buf);
		free(buf);
	}
	TEMPFREE(awkp, x);
	return True;
}

/* a[0] is index (list), a[1] is symtab */
awkcell_t *
awklib_intest(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x, *ap, *k;
	awknode_t *p;
	uint8_t *buf;
	char *s;
	int bufsz = awkp->recsize;
	int nsub = strlen(*awkp->SUBSEP);

	ap = awklib_execute(awkp, a[1]);	/* array name */
	if (!ISARR(ap)) {
		DPRINTF(awkp, ("making %s into an array\n", ap->nval) );
		if (FREEABLE(ap))
			XFREE(ap->sval);
		ap->tval &= ~(STR|NUM|DONTFREE);
		ap->tval |= ARR;
		ap->sval = (char *) awklib_makesymtab(awkp, NSYMTAB);
	}
	if ((buf = malloc(bufsz)) == NULL) {
		FATAL(awkp, "out of memory in intest");
	}
	buf[0] = 0;
	for (p = a[0]; p; p = p->nnext) {
		x = awklib_execute(awkp, p);	/* expr */
		s = awklib_getsval(awkp, x);
		if (!awklib_adjbuf(awkp, &buf, &bufsz, strlen((char *)buf)+strlen(s)+nsub+1, awkp->recsize, 0, "intest"))
			FATAL(awkp, "out of memory deleting %s[%s...]", x->nval, buf);
		strcat((char *)buf, s);
		TEMPFREE(awkp, x);
		if (p->nnext)
			strcat((char *)buf, *awkp->SUBSEP);
	}
	k = awklib_lookup((char *)buf, (awkarray_t *) ap->sval);
	TEMPFREE(awkp, ap);
	free(buf);
	if (k == NULL)
		return(False);
	else
		return(True);
}


/* ~ and match() */
awkcell_t *
awklib_matchop(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x, *y;
	uint8_t *s;
	char *t;
	int i;
	fa_t *pfa;
	int (*mf)(awk_t *, fa_t *, const char *) = awklib_match, mode = 0;

	if (n == MATCHFCN) {
		mf = awklib_pmatch;
		mode = 1;
	}
	x = awklib_execute(awkp, a[1]);	/* a[1] = target text */
	s = (uint8_t *)awklib_getsval(awkp, x);
	if (a[0] == 0)		/* a[1] == 0: already-compiled reg expr */
		i = (*mf)(awkp, (fa_t *) a[2], (char *)s);
	else {
		y = awklib_execute(awkp, a[2]);	/* a[2] = regular expr */
		t = awklib_getsval(awkp, y);
		pfa = awklib_makedfa(awkp, t, mode);
		i = (*mf)(awkp, pfa, (char *)s);
		TEMPFREE(awkp, y);
	}
	TEMPFREE(awkp, x);
	if (n == MATCHFCN) {
		int start = awkp->patbeg - s + 1;
		if (awkp->patlen < 0)
			start = 0;
		awklib_setfval(awkp, awkp->rstartloc, (awkfloat_t) start);
		awklib_setfval(awkp, awkp->rlengthloc, (awkfloat_t) awkp->patlen);
		x = awklib_gettemp(awkp);
		x->tval = NUM;
		x->fval = start;
		return x;
	} else if ((n == MATCH && i == 1) || (n == NOTMATCH && i == 0))
		return(True);
	else
		return(False);
}


/* a[0] || a[1], a[0] && a[1], !a[0] */
awkcell_t *
awklib_boolop(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x, *y;
	int i;

	x = awklib_execute(awkp, a[0]);
	i = ISTRUE(x);
	TEMPFREE(awkp, x);
	switch (n) {
	case BOR:
		if (i) return(True);
		y = awklib_execute(awkp, a[1]);
		i = ISTRUE(y);
		TEMPFREE(awkp, y);
		if (i) return(True);
		else return(False);
	case AND:
		if ( !i ) return(False);
		y = awklib_execute(awkp, a[1]);
		i = ISTRUE(y);
		TEMPFREE(awkp, y);
		if (i) return(True);
		else return(False);
	case NOT:
		if (i) return(False);
		else return(True);
	default:	/* can't happen */
		FATAL(awkp, "unknown boolean operator %d", n);
	}
	return 0;	/*NOTREACHED*/
}

/* a[0 < a[1], etc. */
awkcell_t *
awklib_relop(awk_t *awkp, awknode_t **a, int n)
{
	int i;
	awkcell_t *x, *y;
	awkfloat_t j;

	x = awklib_execute(awkp, a[0]);
	y = awklib_execute(awkp, a[1]);
	if (x->tval&NUM && y->tval&NUM) {
		j = x->fval - y->fval;
		i = j<0? -1: (j>0? 1: 0);
	} else {
		i = strcmp(awklib_getsval(awkp, x), awklib_getsval(awkp, y));
	}
	TEMPFREE(awkp, x);
	TEMPFREE(awkp, y);
	switch (n) {
	case LT:	if (i<0) return(True);
			else return(False);
	case LE:	if (i<=0) return(True);
			else return(False);
	case NE:	if (i!=0) return(True);
			else return(False);
	case EQ:	if (i == 0) return(True);
			else return(False);
	case GE:	if (i>=0) return(True);
			else return(False);
	case GT:	if (i>0) return(True);
			else return(False);
	default:	/* can't happen */
		FATAL(awkp, "unknown relational operator %d", n);
	}
	return 0;	/*NOTREACHED*/
}

/* get a tempcell */
awkcell_t *
awklib_gettemp(awk_t *awkp)
{
	int i;
	awkcell_t *x;

	if (!awkp->tmps) {
		awkp->tmps = calloc(100, sizeof(awkcell_t));
		if (!awkp->tmps)
			FATAL(awkp, "out of space for temporaries");
		for(i = 1; i < 100; i++)
			awkp->tmps[i-1].cnext = &awkp->tmps[i];
		awkp->tmps[i-1].cnext = 0;
	}
	x = awkp->tmps;
	awkp->tmps = x->cnext;
	*x = tempcell;
	return(x);
}

/* $( a[0] ) */
awkcell_t *
awklib_indirect(awk_t *awkp, awknode_t **a, int n)
{
	awkfloat_t val;
	awkcell_t *x;
	int m;
	char *s;

	x = awklib_execute(awkp, a[0]);
	val = awklib_getfval(awkp, x);	/* freebsd: defend against super large field numbers */
	if ((awkfloat_t)INT_MAX < val)
		FATAL(awkp, "trying to access out of range field %s", x->nval);
	m = (int) val;
	if (m == 0 && !awklib_is_number(s = awklib_getsval(awkp, x)))	/* suspicion! */
		FATAL(awkp, "illegal field $(%s), name \"%s\"", s, x->nval);
		/* BUG: can x->nval ever be null??? */
	TEMPFREE(awkp, x);
	x = awklib_fieldadr(awkp, m);
	x->ctype = OCELL;	/* BUG?  why are these needed? */
	x->csub = CFLD;
	return(x);
}

/* substr(a[0], a[1], a[2]) */
awkcell_t *
awklib_substr(awk_t *awkp, awknode_t **a, int nnn)
{
	int k, m, n;
	char *s;
	int temp;
	awkcell_t *x, *y, *z = 0;

	x = awklib_execute(awkp, a[0]);
	y = awklib_execute(awkp, a[1]);
	if (a[2] != 0)
		z = awklib_execute(awkp, a[2]);
	s = awklib_getsval(awkp, x);
	k = strlen(s) + 1;
	if (k <= 1) {
		TEMPFREE(awkp, x);
		TEMPFREE(awkp, y);
		if (a[2] != 0) {
			TEMPFREE(awkp, z);
		}
		x = awklib_gettemp(awkp);
		awklib_setsval(awkp, x, "");
		return(x);
	}
	m = (int) awklib_getfval(awkp, y);
	if (m <= 0)
		m = 1;
	else if (m > k)
		m = k;
	TEMPFREE(awkp, y);
	if (a[2] != 0) {
		n = (int) awklib_getfval(awkp, z);
		TEMPFREE(awkp, z);
	} else
		n = k - 1;
	if (n < 0)
		n = 0;
	else if (n > k - m)
		n = k - m;
	DPRINTF(awkp, ("substr: m=%d, n=%d, s=%s\n", m, n, s) );
	y = awklib_gettemp(awkp);
	temp = s[n+m-1];	/* with thanks to John Linderman */
	s[n+m-1] = '\0';
	awklib_setsval(awkp, y, s + m - 1);
	s[n+m-1] = temp;
	TEMPFREE(awkp, x);
	return(y);
}

awkcell_t *
awklib_index(awk_t *awkp, awknode_t **a, int nnn)		/* index(a[0], a[1]) */
{
	awkcell_t *x, *y, *z;
	char *s1, *s2, *p1, *p2, *q;
	awkfloat_t v = 0.0;

	x = awklib_execute(awkp, a[0]);
	s1 = awklib_getsval(awkp, x);
	y = awklib_execute(awkp, a[1]);
	s2 = awklib_getsval(awkp, y);

	z = awklib_gettemp(awkp);
	for (p1 = s1; *p1 != '\0'; p1++) {
		for (q=p1, p2=s2; *p2 != '\0' && *q == *p2; q++, p2++)
			;
		if (*p2 == '\0') {
			v = (awkfloat_t) (p1 - s1 + 1);	/* origin 1 */
			break;
		}
	}
	TEMPFREE(awkp, x);
	TEMPFREE(awkp, y);
	awklib_setfval(awkp, z, v);
	return(z);
}

#define	MAXNUMSIZE	50

/* printf-like conversions */
static int
format(awk_t *awkp, char **pbuf, int *pbufsize, const char *s, awknode_t *a)
{
	uint8_t *fmt, *p, *t;
	const char *os;
	awkcell_t *x;
	int flag = 0, n;
	int fmtwd; /* format width */
	int fmtsz = awkp->recsize;
	uint8_t *buf = (uint8_t *)*pbuf;
	int bufsize = *pbufsize;
#define FMTSZ(a)   (fmtsz - ((a) - fmt))
#define BUFSZ(a)   (bufsize - ((a) - buf))

	os = s;
	p = buf;
	if ((fmt = malloc(fmtsz)) == NULL)
		FATAL(awkp, "out of memory in format()");
	while (*s) {
		awklib_adjbuf(awkp, &buf, &bufsize, MAXNUMSIZE+1+p-buf, awkp->recsize, &p, "format1");
		if (*s != '%') {
			*p++ = *s++;
			continue;
		}
		if (*(s+1) == '%') {
			*p++ = '%';
			s += 2;
			continue;
		}
		/* have to be real careful in case this is a huge number, eg, %100000d */
		fmtwd = atoi(s+1);
		if (fmtwd < 0)
			fmtwd = -fmtwd;
		awklib_adjbuf(awkp, &buf, &bufsize, fmtwd+1+p-buf, awkp->recsize, &p, "format2");
		for (t = fmt; (*t++ = *s) != '\0'; s++) {
			if (!awklib_adjbuf(awkp, &fmt, &fmtsz, MAXNUMSIZE+1+t-fmt, awkp->recsize, &t, "format3"))
				FATAL(awkp, "format item %.30s... ran format() out of memory", os);
			if (*s == 'l' || *s == 'h' || *s == 'L')
				goto weird;
			if (isalpha((uint8_t)*s))
				break;	/* the ansi panoply */
			if (*s == '*') {
				if (a == NULL)
					FATAL(awkp, "not enough args in printf("
					    "\"%.30s\")", os);
				x = awklib_execute(awkp, a);
				a = a->nnext;
				snprintf((char *)t - 1, FMTSZ(t - 1), 
				    "%d", fmtwd=(int) awklib_getfval(awkp, x));
				if (fmtwd < 0)
					fmtwd = -fmtwd;
				awklib_adjbuf(awkp, &buf, &bufsize, fmtwd+1+p-buf, awkp->recsize, &p, "format");
				t = fmt + strlen((char *)fmt);
				TEMPFREE(awkp, x);
			}
		}
		*t = '\0';
		if (fmtwd < 0)
			fmtwd = -fmtwd;
		awklib_adjbuf(awkp, &buf, &bufsize, fmtwd+1+p-buf, awkp->recsize, &p, "format4");

		switch (*s) {
		case 'f': case 'e': case 'g': case 'E': case 'G':
			flag = 'f';
			break;
		case 'd': case 'i':
			flag = 'd';
			if(*(s-1) == 'l') break;
			*(t-1) = 'l';
			*t = 'd';
			*++t = '\0';
			break;
		case 'o': case 'x': case 'X': case 'u':
			flag = *(s-1) == 'l' ? 'd' : 'u';
			break;
		case 's':
			flag = 's';
			break;
		case 'c':
			flag = 'c';
			break;
		default:
weird:
			WARNING(awkp, "weird printf conversion %s", fmt);
			flag = '?';
			break;
		}
		if (a == NULL)
			FATAL(awkp, "not enough args in printf(%s)", os);
		x = awklib_execute(awkp, a);
		a = a->nnext;
		n = MAXNUMSIZE;
		if (fmtwd > n)
			n = fmtwd;
		awklib_adjbuf(awkp, &buf, &bufsize, 1+n+p-buf, awkp->recsize, &p, "format5");
		switch (flag) {
		case '?':
			snprintf((char *)p, BUFSZ(p), "%s", fmt);	/* unknown, so dump it too */
			t = (uint8_t *)awklib_getsval(awkp, x);
			n = strlen((char *)t);
			if (fmtwd > n)
				n = fmtwd;
			awklib_adjbuf(awkp, &buf, &bufsize, 1+strlen((char *)p)+n+p-buf, awkp->recsize, &p, "format6");
			p += strlen((char *)p);
			snprintf((char *)p, BUFSZ(p), "%s", t);
			break;
		case 'f':
			snprintf((char *)p, BUFSZ(p), (const char *)fmt, awklib_getfval(awkp, x));
			break;
		case 'd':
			snprintf((char *)p, BUFSZ(p), (const char *)fmt, (long) awklib_getfval(awkp, x));
			break;
		case 'u':
			snprintf((char *)p, BUFSZ(p), (const char *)fmt, (int) awklib_getfval(awkp, x));
			break;
		case 's':
			t = (uint8_t *)awklib_getsval(awkp, x);
			n = strlen((char *)t);
			if (fmtwd > n)
				n = fmtwd;
			if (!awklib_adjbuf(awkp, &buf, &bufsize, 1+n+p-buf, awkp->recsize, &p, "format7"))
				FATAL(awkp, "huge string/format (%d chars) in printf %.30s... ran format() out of memory", n, t);
			snprintf((char *)p, BUFSZ(p), (char *)fmt, t);
			break;
		case 'c':
			if (ISNUM(x)) {
				if (awklib_getfval(awkp, x))
					snprintf((char *)p, BUFSZ(p), (char *)fmt, (int) awklib_getfval(awkp, x));
				else {
					*p++ = '\0'; /* explicit null byte */
					*p = '\0';   /* next output will start here */
				}
			} else
				snprintf((char *)p, BUFSZ(p), (char *)fmt, awklib_getsval(awkp, x)[0]);
			break;
		default:
			FATAL(awkp, "can't happen: bad conversion %c in format()", flag);
		}
		TEMPFREE(awkp, x);
		p += strlen((char *)p);
		s++;
	}
	*p = '\0';
	free(fmt);
	for ( ; a; a = a->nnext)		/* evaluate any remaining args */
		awklib_execute(awkp, a);
	*pbuf = (char *)buf;
	*pbufsize = bufsize;
	return p - buf;
}

/* sprintf(a[0]) */
awkcell_t *
awklib_sprintf(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;
	awknode_t *y;
	char *buf;
	int bufsz=3*awkp->recsize;

	if ((buf = malloc(bufsz)) == NULL)
		FATAL(awkp, "out of memory in awklib_sprintf");
	y = a[0]->nnext;
	x = awklib_execute(awkp, a[0]);
	if (format(awkp, &buf, &bufsz, awklib_getsval(awkp, x), y) == -1)
		FATAL(awkp, "sprintf string %.30s... too long.  can't happen.", buf);
	TEMPFREE(awkp, x);
	x = awklib_gettemp(awkp);
	x->sval = buf;
	x->tval = STR;
	return(x);
}

/* printf */
awkcell_t *
awklib_printf(awk_t *awkp, awknode_t **a, int n)
{	/* a[0] is list of args, starting with format string */
	/* a[1] is redirection operator, a[2] is redirection file */
	FILE *fp;
	awkcell_t *x;
	awknode_t *y;
	char *buf;
	int len;
	int bufsz=3*awkp->recsize;

	if ((buf = malloc(bufsz)) == NULL)
		FATAL(awkp, "out of memory in awklib_printf");
	y = a[0]->nnext;
	x = awklib_execute(awkp, a[0]);
	if ((len = format(awkp, &buf, &bufsz, awklib_getsval(awkp, x), y)) == -1)
		FATAL(awkp, "printf string %.30s... too long.  can't happen.", buf);
	TEMPFREE(awkp, x);
	if (a[1] == NULL) {
		/* fputs(buf, stdout); */
		fwrite(buf, len, 1, stdout);
		if (ferror(stdout))
			FATAL(awkp, "write error on stdout");
	} else {
		fp = awklib_redirect(awkp, awklib_ptoi(a[1]), a[2]);
		/* fputs(buf, fp); */
		fwrite(buf, len, 1, fp);
		fflush(fp);
		if (ferror(fp))
			FATAL(awkp, "write error on %s", filename(awkp, fp));
	}
	free(buf);
	return(True);
}

/* a[0] + a[1], etc.  also -a[0] */
awkcell_t *
awklib_arith(awk_t *awkp, awknode_t **a, int n)
{
	awkfloat_t i, j = 0;
	double v;
	awkcell_t *x, *y, *z;

	x = awklib_execute(awkp, a[0]);
	i = awklib_getfval(awkp, x);
	TEMPFREE(awkp, x);
	if (n != UMINUS) {
		y = awklib_execute(awkp, a[1]);
		j = awklib_getfval(awkp, y);
		TEMPFREE(awkp, y);
	}
	z = awklib_gettemp(awkp);
	switch (n) {
	case ADD:
		i += j;
		break;
	case MINUS:
		i -= j;
		break;
	case MULT:
		i *= j;
		break;
	case DIVIDE:
		if (j == 0)
			FATAL(awkp, "division by zero");
		i /= j;
		break;
	case MOD:
		if (j == 0)
			FATAL(awkp, "division by zero in mod");
		modf(i/j, &v);
		i = i - j * v;
		break;
	case UMINUS:
		i = -i;
		break;
	case POWER:
		if (j >= 0 && modf(j, &v) == 0.0)	/* pos integer exponent */
			i = awklib_ipow(i, (int) j);
		else
			i = errcheck(awkp, pow(i, j), "pow");
		break;
	default:	/* can't happen */
		FATAL(awkp, "illegal arithmetic operator %d", n);
	}
	awklib_setfval(awkp, z, i);
	return(z);
}

/* x**n.  ought to be done by pow, but isn't always */
double
awklib_ipow(double x, int n)
{
	double v;

	if (n <= 0)
		return 1;
	v = awklib_ipow(x, n/2);
	if (n % 2 == 0)
		return v * v;
	else
		return x * v * v;
}

/* a[0]++, etc. */
awkcell_t *
awklib_incrdecr(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x, *z;
	int k;
	awkfloat_t xf;

	x = awklib_execute(awkp, a[0]);
	xf = awklib_getfval(awkp, x);
	k = (n == PREINCR || n == POSTINCR) ? 1 : -1;
	if (n == PREINCR || n == PREDECR) {
		awklib_setfval(awkp, x, xf + k);
		return(x);
	}
	z = awklib_gettemp(awkp);
	awklib_setfval(awkp, z, xf);
	awklib_setfval(awkp, x, xf + k);
	TEMPFREE(awkp, x);
	return(z);
}

/* a[0] = a[1], a[0] += a[1], etc. */
awkcell_t *
awklib_assign(awk_t *awkp, awknode_t **a, int n)
{
	/* this is subtle; don't muck with it. */
	awkcell_t *x, *y;
	awkfloat_t xf, yf;
	double v;

	y = awklib_execute(awkp, a[1]);
	x = awklib_execute(awkp, a[0]);
	if (n == ASSIGN) {	/* ordinary assignment */
		if (x == y && !(x->tval & (FLD|REC)))	/* self-assignment: */
			;		/* leave alone unless it's a field */
		else if ((y->tval & (STR|NUM)) == (STR|NUM)) {
			awklib_setsval(awkp, x, awklib_getsval(awkp, y));
			x->fval = awklib_getfval(awkp, y);
			x->tval |= NUM;
		}
		else if (ISSTR(y))
			awklib_setsval(awkp, x, awklib_getsval(awkp, y));
		else if (ISNUM(y))
			awklib_setfval(awkp, x, awklib_getfval(awkp, y));
		else
			awklib_funnyvar(awkp, y, "read value of");
		TEMPFREE(awkp, y);
		return(x);
	}
	xf = awklib_getfval(awkp, x);
	yf = awklib_getfval(awkp, y);
	switch (n) {
	case ADDEQ:
		xf += yf;
		break;
	case SUBEQ:
		xf -= yf;
		break;
	case MULTEQ:
		xf *= yf;
		break;
	case DIVEQ:
		if (yf == 0)
			FATAL(awkp, "division by zero in /=");
		xf /= yf;
		break;
	case MODEQ:
		if (yf == 0)
			FATAL(awkp, "division by zero in %%=");
		modf(xf/yf, &v);
		xf = xf - yf * v;
		break;
	case POWEQ:
		if (yf >= 0 && modf(yf, &v) == 0.0)	/* pos integer exponent */
			xf = awklib_ipow(xf, (int) yf);
		else
			xf = errcheck(awkp, pow(xf, yf), "pow");
		break;
	default:
		FATAL(awkp, "illegal assignment operator %d", n);
		break;
	}
	TEMPFREE(awkp, y);
	awklib_setfval(awkp, x, xf);
	return(x);
}

/* a[0] cat a[1] */
awkcell_t *
awklib_cat(awk_t *awkp, awknode_t **a, int q)
{
	awkcell_t *x, *y, *z;
	int n1, n2;
	char *s;

	x = awklib_execute(awkp, a[0]);
	y = awklib_execute(awkp, a[1]);
	awklib_getsval(awkp, x);
	awklib_getsval(awkp, y);
	n1 = strlen(x->sval);
	n2 = strlen(y->sval);
	s = malloc(n1 + n2 + 1);
	if (s == NULL)
		FATAL(awkp, "out of space concatenating %.15s... and %.15s...",
			x->sval, y->sval);
	strcpy(s, x->sval);
	strcpy(s+n1, y->sval);
	TEMPFREE(awkp, y);
	z = awklib_gettemp(awkp);
	z->sval = s;
	z->tval = STR;
	TEMPFREE(awkp, x);
	return(z);
}

/* a[0] { a[1] } */
awkcell_t *
awklib_pastat(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;

	if (a[0] == 0)
		x = awklib_execute(awkp, a[1]);
	else {
		x = awklib_execute(awkp, a[0]);
		if (ISTRUE(x)) {
			TEMPFREE(awkp, x);
			x = awklib_execute(awkp, a[1]);
		}
	}
	return x;
}

/* a[0], a[1] { a[2] } */
awkcell_t *
awklib_dopa2(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;
	int pair;

	pair = awklib_ptoi(a[3]);
	if (awkp->pairstack[pair] == 0) {
		x = awklib_execute(awkp, a[0]);
		if (ISTRUE(x))
			awkp->pairstack[pair] = 1;
		TEMPFREE(awkp, x);
	}
	if (awkp->pairstack[pair] == 1) {
		x = awklib_execute(awkp, a[1]);
		if (ISTRUE(x))
			awkp->pairstack[pair] = 0;
		TEMPFREE(awkp, x);
		x = awklib_execute(awkp, a[2]);
		return(x);
	}
	return(False);
}

/* split(a[0], a[1], a[2]); a[3] is type */
awkcell_t *
awklib_split(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t *x = 0, *y, *ap;
	char *s;
	int sep;
	char *t, temp, num[50], *fs = 0;
	int n, tempstat, arg3type;

	y = awklib_execute(awkp, a[0]);	/* source string */
	s = awklib_getsval(awkp, y);
	arg3type = awklib_ptoi(a[3]);
	if (a[2] == 0)		/* fs string */
		fs = *awkp->FS;
	else if (arg3type == STRING) {	/* split(str,arr,"string") */
		x = awklib_execute(awkp, a[2]);
		fs = awklib_getsval(awkp, x);
	} else if (arg3type == REGEXPR)
		fs = "(regexpr)";	/* split(str,arr,/regexpr/) */
	else
		FATAL(awkp, "illegal type of split");
	sep = *fs;
	ap = awklib_execute(awkp, a[1]);	/* array name */
	awklib_freesymtab(awkp, ap);
	DPRINTF(awkp, ("split: s=|%s|, a=%s, sep=|%s|\n", s, NN(ap->nval), fs) );
	ap->tval &= ~STR;
	ap->tval |= ARR;
	ap->sval = (char *) awklib_makesymtab(awkp, NSYMTAB);

	n = 0;
	if (*s != '\0' && (strlen(fs) > 1 || arg3type == REGEXPR)) {	/* reg expr */
		fa_t *pfa;
		if (arg3type == REGEXPR) {	/* it's ready already */
			pfa = (fa_t *) a[2];
		} else {
			pfa = awklib_makedfa(awkp, fs, 1);
		}
		if (awklib_nematch(awkp, pfa,s)) {
			tempstat = pfa->initstat;
			pfa->initstat = 2;
			do {
				n++;
				snprintf(num, sizeof(num), "%d", n);
				temp = *awkp->patbeg;
				*awkp->patbeg = '\0';
				if (awklib_is_number(s))
					setawkvar(awkp, num, s, atof(s), STR|NUM, (awkarray_t *) ap->sval);
				else
					setawkvar(awkp, num, s, 0.0, STR, (awkarray_t *) ap->sval);
				*awkp->patbeg = temp;
				s = (char *)awkp->patbeg + awkp->patlen;
				if (*(awkp->patbeg+awkp->patlen-1) == 0 || *s == 0) {
					n++;
					snprintf(num, sizeof(num), "%d", n);
					setawkvar(awkp, num, "", 0.0, STR, (awkarray_t *) ap->sval);
					pfa->initstat = tempstat;
					goto spdone;
				}
			} while (awklib_nematch(awkp, pfa,s));
			pfa->initstat = tempstat; 	/* bwk: has to be here to reset */
							/* cf gsub and refldbld */
		}
		n++;
		snprintf(num, sizeof(num), "%d", n);
		if (awklib_is_number(s))
			setawkvar(awkp, num, s, atof(s), STR|NUM, (awkarray_t *) ap->sval);
		else
			setawkvar(awkp, num, s, 0.0, STR, (awkarray_t *) ap->sval);
spdone:
		pfa = NULL;
	} else if (sep == ' ') {
		for (n = 0; ; ) {
			while (*s == ' ' || *s == '\t' || *s == '\n')
				s++;
			if (*s == 0)
				break;
			n++;
			t = s;
			do
				s++;
			while (*s!=' ' && *s!='\t' && *s!='\n' && *s!='\0');
			temp = *s;
			*s = '\0';
			snprintf(num, sizeof(num), "%d", n);
			if (awklib_is_number(t))
				setawkvar(awkp, num, t, atof(t), STR|NUM, (awkarray_t *) ap->sval);
			else
				setawkvar(awkp, num, t, 0.0, STR, (awkarray_t *) ap->sval);
			*s = temp;
			if (*s != 0)
				s++;
		}
	} else if (sep == 0) {	/* new: split(s, a, "") => 1 char/elem */
		for (n = 0; *s != 0; s++) {
			char buf[2];
			n++;
			snprintf(num, sizeof(num), "%d", n);
			buf[0] = *s;
			buf[1] = 0;
			if (isdigit((uint8_t)buf[0]))
				setawkvar(awkp, num, buf, atof(buf), STR|NUM, (awkarray_t *) ap->sval);
			else
				setawkvar(awkp, num, buf, 0.0, STR, (awkarray_t *) ap->sval);
		}
	} else if (*s != 0) {
		for (;;) {
			n++;
			t = s;
			while (*s != sep && *s != '\n' && *s != '\0')
				s++;
			temp = *s;
			*s = '\0';
			snprintf(num, sizeof(num), "%d", n);
			if (awklib_is_number(t))
				setawkvar(awkp, num, t, atof(t), STR|NUM, (awkarray_t *) ap->sval);
			else
				setawkvar(awkp, num, t, 0.0, STR, (awkarray_t *) ap->sval);
			*s = temp;
			if (*s++ == 0)
				break;
		}
	}
	TEMPFREE(awkp, ap);
	TEMPFREE(awkp, y);
	if (a[2] != 0 && arg3type == STRING) {
		TEMPFREE(awkp, x);
	}
	x = awklib_gettemp(awkp);
	x->tval = NUM;
	x->fval = n;
	return(x);
}

/* a[0] ? a[1] : a[2] */
awkcell_t *
awklib_condexpr(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;

	x = awklib_execute(awkp, a[0]);
	if (ISTRUE(x)) {
		TEMPFREE(awkp, x);
		x = awklib_execute(awkp, a[1]);
	} else {
		TEMPFREE(awkp, x);
		x = awklib_execute(awkp, a[2]);
	}
	return(x);
}

/* if (a[0]) a[1]; else a[2] */
awkcell_t *
awklib_ifstat(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;

	x = awklib_execute(awkp, a[0]);
	if (ISTRUE(x)) {
		TEMPFREE(awkp, x);
		x = awklib_execute(awkp, a[1]);
	} else if (a[2] != 0) {
		TEMPFREE(awkp, x);
		x = awklib_execute(awkp, a[2]);
	}
	return(x);
}

/* while (a[0]) a[1] */
awkcell_t *
awklib_whilestat(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;

	for (;;) {
		x = awklib_execute(awkp, a[0]);
		if (!ISTRUE(x))
			return(x);
		TEMPFREE(awkp, x);
		x = awklib_execute(awkp, a[1]);
		if (ISBREAK(x)) {
			x = True;
			return(x);
		}
		if (ISNEXT(x) || ISEXIT(x) || ISRET(x))
			return(x);
		TEMPFREE(awkp, x);
	}
}

/* do a[0]; while(a[1]) */
awkcell_t *
awklib_dostat(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;

	for (;;) {
		x = awklib_execute(awkp, a[0]);
		if (ISBREAK(x))
			return True;
		if (ISNEXT(x) || ISEXIT(x) || ISRET(x))
			return(x);
		TEMPFREE(awkp, x);
		x = awklib_execute(awkp, a[1]);
		if (!ISTRUE(x))
			return(x);
		TEMPFREE(awkp, x);
	}
}

/* for (a[0]; a[1]; a[2]) a[3] */
awkcell_t *
awklib_forstat(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;

	x = awklib_execute(awkp, a[0]);
	TEMPFREE(awkp, x);
	for (;;) {
		if (a[1]!=0) {
			x = awklib_execute(awkp, a[1]);
			if (!ISTRUE(x)) return(x);
			else TEMPFREE(awkp, x);
		}
		x = awklib_execute(awkp, a[3]);
		if (ISBREAK(x))		/* turn off break */
			return True;
		if (ISNEXT(x) || ISEXIT(x) || ISRET(x))
			return(x);
		TEMPFREE(awkp, x);
		x = awklib_execute(awkp, a[2]);
		TEMPFREE(awkp, x);
	}
}

/* for (a[0] in a[1]) a[2] */
awkcell_t *
awklib_instat(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x, *vp, *arrayp, *cp, *ncp;
	awkarray_t *tp;
	int i;

	vp = awklib_execute(awkp, a[0]);
	arrayp = awklib_execute(awkp, a[1]);
	if (!ISARR(arrayp)) {
		return True;
	}
	tp = (awkarray_t *) arrayp->sval;
	TEMPFREE(awkp, arrayp);
	for (i = 0; i < tp->size; i++) {	/* this routine knows too much */
		for (cp = tp->tab[i]; cp != NULL; cp = ncp) {
			awklib_setsval(awkp, vp, cp->nval);
			ncp = cp->cnext;
			x = awklib_execute(awkp, a[2]);
			if (ISBREAK(x)) {
				TEMPFREE(awkp, vp);
				return True;
			}
			if (ISNEXT(x) || ISEXIT(x) || ISRET(x)) {
				TEMPFREE(awkp, vp);
				return(x);
			}
			TEMPFREE(awkp, x);
		}
	}
	return True;
}

static char *
nawk_toXXX(awk_t *awkp, const char *s, int (*fun_c)(int), wint_t (*fun_wc)(wint_t))
{
	char *buf      = NULL;
	char *pbuf     = NULL;
	const char *ps = NULL;
	size_t n       = 0;
	mbstate_t mbs, mbs2;
	wchar_t wc;
	size_t sz = MB_CUR_MAX;

	if (sz == 1) {
		buf = awklib_tostring(awkp, s);

		for (pbuf = buf; *pbuf; pbuf++)
			*pbuf = fun_c((uint8_t)*pbuf);

		return buf;
	} else {
		/* upper/lower character may be shorter/longer */
		buf = awklib_tostringN(awkp, s, strlen(s) * sz + 1);

		memset(&mbs,  0, sizeof(mbs));
		memset(&mbs2, 0, sizeof(mbs2));

		ps   = s;
		pbuf = buf;
		while (n = mbrtowc(&wc, ps, sz, &mbs),
		       n > 0 && n != (size_t)-1 && n != (size_t)-2)
		{
			ps += n;

			n = wcrtomb(pbuf, fun_wc(wc), &mbs2);
			if (n == (size_t)-1)
				FATAL(awkp, "illegal wide character %s", s);

			pbuf += n;
		}

		*pbuf = 0;

		if (n)
			FATAL(awkp, "illegal byte sequence %s", s);

		return buf;
	}
}

static char *
nawk_toupper(awk_t *awkp, const char *s)
{
	return nawk_toXXX(awkp, s, toupper, towupper);
}

static char *
nawk_tolower(awk_t *awkp, const char *s)
{
	return nawk_toXXX(awkp, s, tolower, towlower);
}

static void
flush_all(awk_t *awkp)
{
	size_t i;

	for (i = 0; i < awkp->nfiles; i++)
		if (awkp->files[i].fp)
			fflush(awkp->files[i].fp);
}

/* builtin functions. a[0] is type, a[1] is arg list */
awkcell_t *
awklib_bltin(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x, *y;
	awkfloat_t u;
	int t, sz;
	char *buf, *fmt;
	awknode_t *nextarg;
	FILE *fp;
	time_t tv;
	struct tm *tm;

	t = awklib_ptoi(a[0]);
	x = awklib_execute(awkp, a[1]);
	nextarg = a[1]->nnext;
	switch (t) {
	case FLENGTH:
		if (ISARR(x))
			u = ((awkarray_t *) x->sval)->nelem;	/* GROT.  should be function*/
		else
			u = strlen(awklib_getsval(awkp, x));
		break;
	case FLOG:
		u = errcheck(awkp, log(awklib_getfval(awkp, x)), "log"); break;
	case FINT:
		modf(awklib_getfval(awkp, x), &u); break;
	case FEXP:
		u = errcheck(awkp, exp(awklib_getfval(awkp, x)), "exp"); break;
	case FSQRT:
		u = errcheck(awkp, sqrt(awklib_getfval(awkp, x)), "sqrt"); break;
	case FSIN:
		u = sin(awklib_getfval(awkp, x)); break;
	case FCOS:
		u = cos(awklib_getfval(awkp, x)); break;
	case FATAN:
		if (nextarg == 0) {
			WARNING(awkp, "atan2 requires two arguments; returning 1.0");
			u = 1.0;
		} else {
			y = awklib_execute(awkp, a[1]->nnext);
			u = atan2(awklib_getfval(awkp, x), awklib_getfval(awkp, y));
			TEMPFREE(awkp, y);
			nextarg = nextarg->nnext;
		}
		break;
	case FSYSTEM:
		fflush(stdout);		/* in case something is buffered already */
		u = (awkfloat_t) system(awklib_getsval(awkp, x)) / 256;   /* 256 is unix-dep */
		break;
	case FRAND:
		/* in principle, rand() returns something in 0..RAND_MAX */
		u = (awkfloat_t) (rand() % RAND_MAX) / RAND_MAX;
		break;
	case FSRAND:
		if (ISREC(x))	/* no argument provided */
			u = time((time_t *)0);
		else
			u = awklib_getfval(awkp, x);
		srand((unsigned int) u);
		break;
	case FTOUPPER:
	case FTOLOWER:
		if (t == FTOUPPER)
			buf = nawk_toupper(awkp, awklib_getsval(awkp, x));
		else
			buf = nawk_tolower(awkp, awklib_getsval(awkp, x));
		TEMPFREE(awkp, x);
		x = awklib_gettemp(awkp);
		awklib_setsval(awkp, x, buf);
		free(buf);
		return x;
	case FFLUSH:
		if (ISREC(x) || strlen(awklib_getsval(awkp, x)) == 0) {
			flush_all(awkp);	/* fflush() or fflush("") -> all */
			u = 0;
		} else if ((fp = openfile(awkp, FFLUSH, awklib_getsval(awkp, x))) == NULL)
			u = EOF;
		else
			u = fflush(fp);
		break;
	case FSYSTIME:
		u = time((time_t *) 0); break;
	case FSTRFTIME:
		/* strftime([format [,timestamp]]) */
		if (nextarg) {
			y = awklib_execute(awkp, nextarg), nextarg = nextarg->nnext;
			tv = (time_t) awklib_getfval(awkp, y);
			TEMPFREE(awkp, y);
		} else
			tv = time((time_t *) 0);
		tm = localtime(&tv);

		if (ISREC(x)) {
			/* format argument not provided, use default */
			fmt = awklib_tostring(awkp, "%a %b %d %H:%M:%S %Z %Y");
		} else
			fmt = awklib_tostring(awkp, awklib_getsval(awkp, x));

		sz = 32, buf = NULL;
		do {
			if ((buf = realloc(buf, (sz *= 2))) == NULL)
				FATAL(awkp, "out of memory in strftime");
		} while(strftime(buf, sz, fmt, tm) == 0);

		y = awklib_gettemp(awkp);
		awklib_setsval(awkp, y, buf);
		free(fmt);
		free(buf);

		return y;
	default:	/* can't happen */
		FATAL(awkp, "illegal function type %d", t);
		break;
	}
	TEMPFREE(awkp, x);
	x = awklib_gettemp(awkp);
	awklib_setfval(awkp, x, u);
	if (nextarg != 0) {
		WARNING(awkp, "warning: function has too many arguments");
		for ( ; nextarg; nextarg = nextarg->nnext)
			awklib_execute(awkp, nextarg);
	}
	return(x);
}

/* print a[0] */
awkcell_t *
awklib_print(awk_t *awkp, awknode_t **a, int n)
{
	awknode_t *x;
	awkcell_t *y;
	FILE *fp;

	if (a[1] == 0)	/* a[1] is redirection operator, a[2] is file */
		fp = stdout;
	else
		fp = awklib_redirect(awkp, awklib_ptoi(a[1]), a[2]);
	for (x = a[0]; x != NULL; x = x->nnext) {
		y = awklib_execute(awkp, x);
		fputs(awklib_getpssval(awkp, y), fp);
		TEMPFREE(awkp, y);
		if (x->nnext == NULL)
			fputs(*awkp->ORS, fp);
		else
			fputs(*awkp->OFS, fp);
	}
	if (a[1] != 0)
		fflush(fp);
	if (ferror(fp))
		FATAL(awkp, "write error on %s", filename(awkp, fp));
	return(True);
}

awkcell_t *
awklib_nullproc(awk_t *awkp, awknode_t **a, int n)
{
	awkp = awkp;
	n = n;
	a = a;
	return 0;
}


/* set up all i/o redirections */
FILE *
awklib_redirect(awk_t *awkp, int a, awknode_t *b)
{
	FILE *fp;
	awkcell_t *x;
	char *fname;

	x = awklib_execute(awkp, b);
	fname = awklib_getsval(awkp, x);
	fp = openfile(awkp, a, fname);
	if (fp == NULL)
		FATAL(awkp, "can't open file %s", fname);
	TEMPFREE(awkp, x);
	return fp;
}

awkcell_t *
awklib_closefile(awk_t *awkp, awknode_t **a, int n)
{
	awkcell_t *x;
	size_t i;
	int stat;

	n = n;
	x = awklib_execute(awkp, a[0]);
	awklib_getsval(awkp, x);
	stat = -1;
	for (i = 0; i < awkp->nfiles; i++) {
		if (awkp->files[i].fname && strcmp(x->sval, awkp->files[i].fname) == 0) {
			if (ferror(awkp->files[i].fp))
				WARNING(awkp,  "i/o error occurred on %s", awkp->files[i].fname );
			if (awkp->files[i].mode == '|' || awkp->files[i].mode == LE)
				stat = pclose(awkp->files[i].fp);
			else
				stat = fclose(awkp->files[i].fp);
			if (stat == EOF)
				WARNING(awkp,  "i/o error occurred closing %s", awkp->files[i].fname );
			if (i > 2)	/* don't do /dev/std... */
				XFREE(awkp->files[i].fname);
			awkp->files[i].fname = NULL;	/* watch out for ref thru this */
			awkp->files[i].fp = NULL;
		}
	}
	TEMPFREE(awkp, x);
	x = awklib_gettemp(awkp);
	awklib_setfval(awkp, x, (awkfloat_t) stat);
	return(x);
}

/* substitute command */
awkcell_t *
awklib_sub(awk_t *awkp, awknode_t **a, int nnn)
{
	const uint8_t *sptr;
	uint8_t *q;
	awkcell_t *x, *y, *result;
	uint8_t *t, *buf, *pb;
	fa_t *pfa;
	int bufsz = awkp->recsize;

	if ((buf = malloc(bufsz)) == NULL)
		FATAL(awkp, "out of memory in awklib_sub");
	x = awklib_execute(awkp, a[3]);	/* target string */
	t = (uint8_t *)awklib_getsval(awkp, x);
	if (a[0] == 0)		/* 0 => a[1] is already-compiled regexpr */
		pfa = (fa_t *) a[1];	/* regular expression */
	else {
		y = awklib_execute(awkp, a[1]);
		pfa = awklib_makedfa(awkp, awklib_getsval(awkp, y), 1);
		TEMPFREE(awkp, y);
	}
	y = awklib_execute(awkp, a[2]);	/* replacement string */
	result = False;
	if (awklib_pmatch(awkp, pfa, (char *)t)) {
		sptr = t;
		awklib_adjbuf(awkp, &buf, &bufsz, 1+awkp->patbeg-sptr, awkp->recsize, 0, "sub");
		pb = buf;
		while (sptr < awkp->patbeg)
			*pb++ = *sptr++;
		sptr = (uint8_t *)awklib_getsval(awkp, y);
		while (*sptr != 0) {
			awklib_adjbuf(awkp, &buf, &bufsz, 5+pb-buf, awkp->recsize, &pb, "sub");
			if (*sptr == '\\') {
				backsub(&pb, &sptr);
			} else if (*sptr == '&') {
				sptr++;
				awklib_adjbuf(awkp, &buf, &bufsz, 1+awkp->patlen+pb-buf, awkp->recsize, &pb, "sub");
				for (q = awkp->patbeg; q < awkp->patbeg+awkp->patlen; )
					*pb++ = *q++;
			} else
				*pb++ = *sptr++;
		}
		*pb = '\0';
		if (pb > buf + bufsz)
			FATAL(awkp, "sub result1 %.30s too big; can't happen", buf);
		sptr = awkp->patbeg + awkp->patlen;
		if ((awkp->patlen == 0 && *awkp->patbeg) || (awkp->patlen && *(sptr-1))) {
			awklib_adjbuf(awkp, &buf, &bufsz, 1+strlen((char *)sptr)+pb-buf, 0, &pb, "sub");
			while ((*pb++ = *sptr++) != 0)
				;
		}
		if (pb > buf + bufsz)
			FATAL(awkp, "sub result2 %.30s too big; can't happen", buf);
		awklib_setsval(awkp, x, (char *)buf);	/* BUG: should be able to avoid copy */
		result = True;;
	}
	TEMPFREE(awkp, x);
	TEMPFREE(awkp, y);
	free(buf);
	return result;
}

/* global substitute */
awkcell_t *
awklib_gsub(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t *x, *y;
	const char *rptr;
	const uint8_t *sptr;
	uint8_t *t, *q;
	uint8_t *pb, *buf;
	fa_t *pfa;
	int mflag, tempstat, num;
	int bufsz = awkp->recsize;

	if ((buf = malloc(bufsz)) == NULL)
		FATAL(awkp, "out of memory in awklib_gsub");
	mflag = 0;	/* if mflag == 0, can replace empty string */
	num = 0;
	x = awklib_execute(awkp, a[3]);	/* target string */
	t = (uint8_t *)awklib_getsval(awkp, x);
	if (a[0] == 0)		/* 0 => a[1] is already-compiled regexpr */
		pfa = (fa_t *) a[1];	/* regular expression */
	else {
		y = awklib_execute(awkp, a[1]);
		pfa = awklib_makedfa(awkp, awklib_getsval(awkp, y), 1);
		TEMPFREE(awkp, y);
	}
	y = awklib_execute(awkp, a[2]);	/* replacement string */
	if (awklib_pmatch(awkp, pfa, (char *)t)) {
		tempstat = pfa->initstat;
		pfa->initstat = 2;
		pb = buf;
		rptr = awklib_getsval(awkp, y);
		do {
			if (awkp->patlen == 0 && *awkp->patbeg != 0) {	/* matched empty string */
				if (mflag == 0) {	/* can replace empty */
					num++;
					sptr = (uint8_t *)rptr;
					while (*sptr != 0) {
						awklib_adjbuf(awkp, &buf, &bufsz, 5+pb-buf, awkp->recsize, &pb, "gsub");
						if (*sptr == '\\') {
							backsub(&pb, &sptr);
						} else if (*sptr == '&') {
							sptr++;
							awklib_adjbuf(awkp, &buf, &bufsz, 1+awkp->patlen+pb-buf, awkp->recsize, &pb, "gsub");
							for (q = awkp->patbeg; q < awkp->patbeg+awkp->patlen; )
								*pb++ = *q++;
						} else
							*pb++ = *sptr++;
					}
				}
				if (*t == 0)	/* at end */
					goto done;
				awklib_adjbuf(awkp, &buf, &bufsz, 2+pb-buf, awkp->recsize, &pb, "gsub");
				*pb++ = *t++;
				if (pb > buf + bufsz)	/* BUG: not sure of this test */
					FATAL(awkp, "gsub result0 %.30s too big; can't happen", buf);
				mflag = 0;
			}
			else {	/* matched nonempty string */
				num++;
				sptr = t;
				awklib_adjbuf(awkp, &buf, &bufsz, 1+(awkp->patbeg-sptr)+pb-buf, awkp->recsize, &pb, "gsub");
				while (sptr < awkp->patbeg)
					*pb++ = *sptr++;
				sptr = (uint8_t *)rptr;
				while (*sptr != 0) {
					awklib_adjbuf(awkp, &buf, &bufsz, 5+pb-buf, awkp->recsize, &pb, "gsub");
					if (*sptr == '\\') {
						backsub(&pb, &sptr);
					} else if (*sptr == '&') {
						sptr++;
						awklib_adjbuf(awkp, &buf, &bufsz, 1+awkp->patlen+pb-buf, awkp->recsize, &pb, "gsub");
						for (q = awkp->patbeg; q < awkp->patbeg+awkp->patlen; )
							*pb++ = *q++;
					} else
						*pb++ = *sptr++;
				}
				t = awkp->patbeg + awkp->patlen;
				if (awkp->patlen == 0 || *t == 0 || *(t-1) == 0)
					goto done;
				if (pb > buf + bufsz)
					FATAL(awkp, "gsub result1 %.30s too big; can't happen", buf);
				mflag = 1;
			}
		} while (awklib_pmatch(awkp, pfa, (char *)t));
		sptr = t;
		awklib_adjbuf(awkp, &buf, &bufsz, 1+strlen((char *)sptr)+pb-buf, 0, &pb, "gsub");
		while ((*pb++ = *sptr++) != 0)
			;
	done:	if (pb > buf + bufsz)
			FATAL(awkp, "gsub result2 %.30s too big; can't happen", buf);
		*pb = '\0';
		awklib_setsval(awkp, x, (char *)buf);	/* BUG: should be able to avoid copy + free */
		pfa->initstat = tempstat;
	}
	TEMPFREE(awkp, x);
	TEMPFREE(awkp, y);
	x = awklib_gettemp(awkp);
	x->tval = NUM;
	x->fval = num;
	free(buf);
	return(x);
}

/* global selective substitute */
/* XXX incomplete - doesn't support backreferences \0 ... \9 */
awkcell_t *
awklib_gensub(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t *x, *y, *res, *h;
	char *rptr;
	const uint8_t *sptr;
	uint8_t *q, *pb, *t, *buf;
	fa_t *pfa;
	int mflag, tempstat, num, whichm;
	int bufsz = awkp->recsize;

	if ((buf = malloc(bufsz)) == NULL)
		FATAL(awkp, "out of memory in awklib_gensub");
	mflag = 0;	/* if mflag == 0, can replace empty string */
	num = 0;
	x = awklib_execute(awkp, a[4]);	/* source string */
	t = (uint8_t *)awklib_getsval(awkp, x);
	res = awklib_copycell(awkp, x);	/* target string - initially copy of source */
	if (a[0] == 0)		/* 0 => a[1] is already-compiled regexpr */
		pfa = (fa_t *) a[1];	/* regular expression */
	else {
		y = awklib_execute(awkp, a[1]);
		pfa = awklib_makedfa(awkp, awklib_getsval(awkp, y), 1);
		TEMPFREE(awkp, y);
	}
	y = awklib_execute(awkp, a[2]);	/* replacement string */
	h = awklib_execute(awkp, a[3]);	/* which matches should be replaced */
	sptr = (uint8_t *)awklib_getsval(awkp, h);
	if (sptr[0] == 'g' || sptr[0] == 'G')
		whichm = -1;
	else {
		/*
		 * The specified number is index of replacement, starting
		 * from 1. GNU awk treats index lower than 0 same as
		 * 1, we do same for compatibility.
		 */
		whichm = (int) awklib_getfval(awkp, h) - 1;
		if (whichm < 0)
			whichm = 0;
	}
	TEMPFREE(awkp, h);

	if (awklib_pmatch(awkp, pfa, (char *)t)) {
		char *sl;

		tempstat = pfa->initstat;
		pfa->initstat = 2;
		pb = buf;
		rptr = awklib_getsval(awkp, y);
		/*
		 * XXX if there are any backreferences in subst string,
		 * complain now.
		 */
		for(sl=rptr; (sl = strchr(sl, '\\')) && sl[1]; sl++) {
			if (strchr("0123456789", sl[1])) {
				FATAL(awkp, "gensub doesn't support backreferences (subst \"%s\")", rptr);
			}
		}
		
		do {
			if (whichm >= 0 && whichm != num) {
				num++;
				awklib_adjbuf(awkp, &buf, &bufsz, (pb - buf) + (awkp->patbeg - t) + awkp->patlen, awkp->recsize, &pb, "gensub");

				/* copy the part of string up to and including
				 * match to output buffer */
				while (t < awkp->patbeg + awkp->patlen)
					*pb++ = *t++;
				continue;
			}

			if (awkp->patlen == 0 && *awkp->patbeg != 0) {	/* matched empty string */
				if (mflag == 0) {	/* can replace empty */
					num++;
					sptr = (uint8_t *)rptr;
					while (*sptr != 0) {
						awklib_adjbuf(awkp, &buf, &bufsz, 5+pb-buf, awkp->recsize, &pb, "gensub");
						if (*sptr == '\\') {
							backsub(&pb, &sptr);
						} else if (*sptr == '&') {
							sptr++;
							awklib_adjbuf(awkp, &buf, &bufsz, 1+awkp->patlen+pb-buf, awkp->recsize, &pb, "gensub");
							for (q = awkp->patbeg; q < awkp->patbeg+awkp->patlen; )
								*pb++ = *q++;
						} else
							*pb++ = *sptr++;
					}
				}
				if (*t == 0)	/* at end */
					goto done;
				awklib_adjbuf(awkp, &buf, &bufsz, 2+pb-buf, awkp->recsize, &pb, "gensub");
				*pb++ = *t++;
				if (pb > buf + bufsz)	/* BUG: not sure of this test */
					FATAL(awkp, "gensub result0 %.30s too big; can't happen", buf);
				mflag = 0;
			}
			else {	/* matched nonempty string */
				num++;
				sptr = t;
				awklib_adjbuf(awkp, &buf, &bufsz, 1+(awkp->patbeg-sptr)+pb-buf, awkp->recsize, &pb, "gensub");
				while (sptr < awkp->patbeg)
					*pb++ = *sptr++;
				sptr = (uint8_t *)rptr;
				while (*sptr != 0) {
					awklib_adjbuf(awkp, &buf, &bufsz, 5+pb-buf, awkp->recsize, &pb, "gensub");
					if (*sptr == '\\') {
						backsub(&pb, &sptr);
					} else if (*sptr == '&') {
						sptr++;
						awklib_adjbuf(awkp, &buf, &bufsz, 1+awkp->patlen+pb-buf, awkp->recsize, &pb, "gensub");
						for (q = awkp->patbeg; q < awkp->patbeg+awkp->patlen; )
							*pb++ = *q++;
					} else
						*pb++ = *sptr++;
				}
				t = awkp->patbeg + awkp->patlen;
				if (awkp->patlen == 0 || *t == 0 || *(t-1) == 0)
					goto done;
				if (pb > buf + bufsz)
					FATAL(awkp, "gensub result1 %.30s too big; can't happen", buf);
				mflag = 1;
			}
		} while (awklib_pmatch(awkp, pfa, (char *)t));
		sptr = t;
		awklib_adjbuf(awkp, &buf, &bufsz, 1+strlen((char *)sptr)+pb-buf, 0, &pb, "gensub");
		while ((*pb++ = *sptr++) != 0)
			;
	done:	if (pb > buf + bufsz)
			FATAL(awkp, "gensub result2 %.30s too big; can't happen", buf);
		*pb = '\0';
		awklib_setsval(awkp, res, (char *)buf);
		pfa->initstat = tempstat;
	}
	TEMPFREE(awkp, x);
	TEMPFREE(awkp, y);
	free(buf);
	return(res);
}

/* set default values for everything */
void
awklib_setdefaults(awk_t *awkp)
{
	awkp->safe = 0;		/* 1 => "safe" mode */
	awkp->recsize = RECSIZE;
	awkp->lineno = 1;
	awkp->nfields = MAXFLD;	/* last allocated slot for $i */

	awkp->ep = awkp->ebuf;
	awkp->yysptr = awkp->yysbuf;

	awkp->file = "";
	awkp->fieldssize = RECSIZE;
	awkp->argno = 1;	/* current input argument number */

	awkp->lexbufsize = 5; /* BUG: setting this small causes core dump! */
	awkp->stringbufsz = 500;
	awkp->regexprbufsz = 500;
	awkp->make_dfa_now = 1;
	awkp->cclenterbufsz = 100;
	awkp->relexbufsz = 100;

	awkp->static_inputFS[0] = ' ';
	awkp->static_inputFS[1] = 0x0;
	awkp->len_inputFS = sizeof(awkp->static_inputFS) - 1;
	awkp->inputFS = awkp->static_inputFS;
}

/*
 * Call a C function that returns nothing
 */
static void 
voidcall(void *f, long *a, double *d)
{
	void            (*func) () = f;

	(*func) (
		 a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9],
	a[10], a[11], a[12], a[13], a[14], a[15], a[16], a[17], a[18], a[19],
	a[20], a[21], a[22], a[23], a[24], a[25], a[26], a[27], a[28], a[29],
	a[30], a[31], a[32], a[33], a[34], a[35], a[36], a[37], a[38], a[39],
		 d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7], d[8], d[9],
	d[10], d[11], d[12], d[13], d[14], d[15], d[16], d[17], d[18], d[19]
		);
}

/*
 * Call a C function that returns an int
 */
static int 
intcall(void *f, long *a, double *d)
{
	int             (*func) () = f;
	int             res = 0;

	res = (*func) (
		 a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9],
	a[10], a[11], a[12], a[13], a[14], a[15], a[16], a[17], a[18], a[19],
	a[20], a[21], a[22], a[23], a[24], a[25], a[26], a[27], a[28], a[29],
	a[30], a[31], a[32], a[33], a[34], a[35], a[36], a[37], a[38], a[39],
		 d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7], d[8], d[9],
	d[10], d[11], d[12], d[13], d[14], d[15], d[16], d[17], d[18], d[19]
		);
	return res;
}

/*
 * Call a C function that returns a pointer
 */
static void    *
ptrcall(void *f, long *a, double *d)
{
	void           *(*func) () = f;
	void           *res = NULL;

	res = (*func) (
		 a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9],
	a[10], a[11], a[12], a[13], a[14], a[15], a[16], a[17], a[18], a[19],
	a[20], a[21], a[22], a[23], a[24], a[25], a[26], a[27], a[28], a[29],
	a[30], a[31], a[32], a[33], a[34], a[35], a[36], a[37], a[38], a[39],
		 d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7], d[8], d[9],
	d[10], d[11], d[12], d[13], d[14], d[15], d[16], d[17], d[18], d[19]
		);
	return res;
}

/*
 * Call a C function that returns a double
 */
static double 
floatcall(void *f, long *a, double *d)
{
	double          (*func) () = f;
	double          res = 0.0;

	res = (*func) (
		 a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9],
	a[10], a[11], a[12], a[13], a[14], a[15], a[16], a[17], a[18], a[19],
	a[20], a[21], a[22], a[23], a[24], a[25], a[26], a[27], a[28], a[29],
	a[30], a[31], a[32], a[33], a[34], a[35], a[36], a[37], a[38], a[39],
		 d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7], d[8], d[9],
	d[10], d[11], d[12], d[13], d[14], d[15], d[16], d[17], d[18], d[19]
		);
	return res;
}

/* set up the C arg stack using values that were passed to us, and prototype arg types */
static int
makeargs(awk_t *awkp, awknode_t *a, awkarray_t *types, char *funcname, long *largs, double *dargs)
{
	awkcell_t	 *typecell;
	awkcell_t	 *argcell;
	awknode_t	 *node;
	uint32_t	  lsize;
	uint32_t	  dsize;
	uint32_t	  lpos;
	uint32_t	  dpos;
	char		**sput;
	char		 *argtype;
	char		 *argument;
	char		  buf[1024];
	int		  i;

	lsize = dsize = lpos = dpos = 0;
	for (i = 0, node = a ; node != NULL ; i++, node = node->nnext) {
		if (lsize > 40 * sizeof(int) || dsize > 20) {
			return 0;
		}
		(void) snprintf(buf, sizeof(buf), "%s:%d", funcname, i);
		typecell = awklib_lookup(buf, types);
		argtype = awklib_getsval(awkp, typecell);
		if (strcmp(argtype, "void") == 0) {
			lsize += sizeof(int);
			largs[lpos++] = 0;
		} else if (strcmp(argtype, "bool") == 0) {
			lsize += sizeof(int);
			argcell = awklib_execute(awkp, node);
			argument = awklib_getsval(awkp, argcell);
			largs[lpos++] = (*argument != '0');
		} else if (strcmp(argtype, "int") == 0 ||
			   strcmp(argtype, "storage") == 0) {
			lsize += sizeof(int);
			argcell = awklib_execute(awkp, node);
			argument = awklib_getsval(awkp, argcell);
			largs[lpos++] = atol(argument);
		} else if (strcmp(argtype, "float") == 0 ||
		           strcmp(argtype, "double") == 0) {
			lsize += sizeof(double);
			dsize += 1;
		} else if (strcmp(argtype, "void*") == 0 ||
			   strcmp(argtype, "pointer") == 0 ||
			   strcmp(argtype, "ptr") == 0) {
			lsize += sizeof(void *);
			argcell = awklib_execute(awkp, node);
			/* this gets freed in unmakeargs() */
			argument = strdup(awklib_getsval(awkp, argcell));
			sput = (char **) (largs + lpos);
			*sput = argument;
			lpos += sizeof(char *) / sizeof(long);
		} else {
			return 0;
		}
	}
	return 1;
}

/* set up the C arg stack using values that were passed to us, and prototype arg types */
static int
unmakeargs(awk_t *awkp, awknode_t *a, awkarray_t *types, char *funcname, long *largs, double *dargs)
{
	awkcell_t	 *typecell;
	awknode_t	 *node;
	uint32_t	  lsize;
	uint32_t	  dsize;
	uint32_t	  lpos;
	uint32_t	  dpos;
	char		**sput;
	char		 *argtype;
	char		  buf[1024];
	int		  i;

	lsize = dsize = lpos = dpos = 0;
	for (i = 0, node = a ; node != NULL ; i++, node = node->nnext) {
		if (lsize > 40 * sizeof(int) || dsize > 20) {
			return 0;
		}
		(void) snprintf(buf, sizeof(buf), "%s:%d", funcname, i);
		typecell = awklib_lookup(buf, types);
		argtype = awklib_getsval(awkp, typecell);
		if (strcmp(argtype, "void") == 0) {
			lsize += sizeof(int);
			lpos += 1;
		} else if (strcmp(argtype, "bool") == 0) {
			lsize += sizeof(int);
			lpos += 1;
		} else if (strcmp(argtype, "int") == 0 ||
			   strcmp(argtype, "storage") == 0) {
			lsize += sizeof(int);
			lpos += 1;
		} else if (strcmp(argtype, "float") == 0 ||
		           strcmp(argtype, "double") == 0) {
			lsize += sizeof(double);
			dsize += 1;
		} else if (strcmp(argtype, "void*") == 0 ||
			   strcmp(argtype, "pointer") == 0 ||
			   strcmp(argtype, "ptr") == 0) {
			lsize += sizeof(void *);
			sput = (char **) (largs + lpos);
			(void) free(*sput);
			lpos += sizeof(char *) / sizeof(long);
		} else {
			return 0;
		}
	}
	return 1;
}

/* parse a C prototype */
static int
parse_prototype(awk_t *awkp, const char *proto, awkcell_t *arr, char *emerging_funcname, int esize)
{
	regex_t		 ret_regex;
	regex_t		 arg_regex;
	regmatch_t	 matchv[10];
	char		 funcname[1024];
	char		 varname[1024];
	char		 value[1024];
	char		*symtab;
	int		 protolen;
	int		 cc;
	int		 i;

	/* XXX - agc make this use pfa_t */
	(void) regcomp(&ret_regex, "([^ \t]*)[ \t]*([^ \t(]*)[ \t]*\\(", REG_EXTENDED);
	symtab = arr->sval;
	regexec(&ret_regex, proto, 10, matchv, 0);
	/* use a temp funcname for convenience */
	(void) snprintf(funcname, sizeof(funcname), "%.*s",
		(int)(matchv[2].rm_eo - matchv[2].rm_so),
		&proto[(int)matchv[2].rm_so]);
	if (emerging_funcname != NULL) {
		(void) snprintf(emerging_funcname, esize, "%s", funcname);
	}
	/* func:return is return type */
	(void) snprintf(varname, sizeof(varname), "%s:return", funcname);
	(void) snprintf(value, sizeof(value), "%.*s", (int)(matchv[1].rm_eo - matchv[1].rm_so), proto);
	setawkvar(awkp, varname, value, 0.0, STR|NUM, (awkarray_t *) arr->sval);
	regfree(&ret_regex);
	/* func:0, func:1 etc are the arg types */
	regcomp(&arg_regex, "^([ \t]*)([^ \t,)]*)[ \t]*([,)])", REG_EXTENDED);
	protolen = strlen(proto);
	for (cc = (int)matchv[0].rm_eo, i = 0 ; cc < protolen ; i++, cc += (int)(matchv[3].rm_eo) + 1) {
		if (regexec(&arg_regex, &proto[cc], 10, matchv, 0) != 0) {
			break;
		}
		(void) snprintf(varname, sizeof(varname), "%s:%d", funcname, i);
		(void) snprintf(value, sizeof(value), "%.*s", (int)(matchv[2].rm_eo - matchv[2].rm_so), &proto[cc]);
		if (i == 0 && strcmp(value, "void") == 0) {
			break;
		}
		setawkvar(awkp, varname, value, 0.0, STR|NUM, (awkarray_t *) arr->sval);
		if (proto[cc + (int)(matchv[3].rm_eo)] == ')') {
			break;
		}
	}
	/* func:argc is the arg count */
	(void) snprintf(value, sizeof(value), "%d", i);
	(void) snprintf(varname, sizeof(varname), "%s:argc", funcname);
	setawkvar(awkp, varname, value, 0.0, STR|NUM, (awkarray_t *) arr->sval);
	return 1;
}

/* save 'n' chars of 's' in malloc'd memory */
/* bound the allocation with 'size' chars */
static char *
strnsave(const char *s, int n, int maxstr)
{
	char	*cp;

	if (n < 0) {
		n = strlen(s);
	}
	if (n > maxstr) {
		n = maxstr;
	}
	if ((cp = calloc(1, n + 1)) != NULL) {
		(void) memcpy(cp, s, n);
		cp[n] = 0x0;
	}
	return cp;
}

/* we use this for the numeric representation of the handle */
typedef union un_t {
	void		*vp;
	uint32_t	 i32;
	uint64_t	 i64;
} un_t;

#if !defined(SO_SUFFIX)
#  ifdef __APPLE__
#    define SO_SUFFIX	"dylib"
#  else
#    define SO_SUFFIX	"so"
#  endif
#endif

#ifndef MAXSTRSIZE
#define MAXSTRSIZE	(2048 * 1024)
#endif

#ifndef MAXFIELDS
#define MAXFIELDS	512
#endif

/* dlopen(a[0], a[1]); */
awkcell_t *
awklib_dlopen(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t	*pathcell;
	awkcell_t	*arr;
	awkcell_t	*res;
	char		 fullpath[1024];
	char		 handle[32];
	char		*path;
	un_t		 un;

	pathcell = awklib_execute(awkp, a[0]);	/* path string */
	path = awklib_getsval(awkp, pathcell);
	arr = awklib_execute(awkp, a[1]);	/* array name */
	awklib_freesymtab(awkp, arr);
	arr->tval &= ~STR;
	arr->tval |= ARR;
	arr->sval = (char *) awklib_makesymtab(awkp, NSYMTAB);
	/* fill in handle and path elements */
	if ((un.vp = dlopen(path, RTLD_LAZY)) == NULL) {
		(void) snprintf(fullpath, sizeof(fullpath), "%s.%s", path, SO_SUFFIX);
		path = fullpath;
	}
	if ((un.vp = dlopen(path, RTLD_LAZY)) == NULL) {
		(void) fprintf(stderr, "Unable to dlopen \"%s\"\n", path);
		res = awklib_gettemp(awkp);
		res->tval = NUM;
		res->fval = 0;
		return res;
	}
	if (/* CONSTCOND */sizeof(void *) == 8) {
		(void) snprintf(handle, sizeof(handle), "%" PRIu64, un.i64);
	} else {
		(void) snprintf(handle, sizeof(handle), "%d", un.i32);
	}
	setawkvar(awkp, "handle", handle, 0.0, STR|NUM, (awkarray_t *) arr->sval);
	setawkvar(awkp, "path", path, 0.0, STR, (awkarray_t *) arr->sval);
	TEMPFREE(awkp, arr);
	TEMPFREE(awkp, pathcell);
	res = awklib_gettemp(awkp);
	res->tval = NUM;
	res->fval = 2;
	return res;
}

/* dlproto(a[0], a[1]); */
awkcell_t *
awklib_dlproto(awk_t *awkp, awknode_t **a, int nnn)
{
	regex_t		 ret_regex;
	regex_t		 arg_regex;
	awkcell_t	*arr;
	awkcell_t	*res;
	awkcell_t	*protocell;
	char		*symtab;
	char		*proto;

	arr = awklib_execute(awkp, a[0]);	/* array name */
	if (!(arr->tval & ARR)) {
		res = awklib_gettemp(awkp);
		res->tval = NUM;
		res->fval = 0;
		return res;
	}
	/* XXX - agc make this use pfa_t */
	(void) regcomp(&ret_regex, "([^ \t]*)[ \t]*([^ \t(]*)[ \t]*\\(", REG_EXTENDED);
	symtab = arr->sval;
	protocell = awklib_execute(awkp, a[1]);	/* C prototype */
	proto = awklib_getsval(awkp, protocell);
	parse_prototype(awkp, proto, arr, NULL, 0);
	TEMPFREE(awkp, arr);
	TEMPFREE(awkp, protocell);
	res = awklib_gettemp(awkp);
	res->tval = NUM;
	res->fval = 1;
	regfree(&arg_regex);
	return res;
}

/* call function */
awkcell_t *
awklib_dlcall(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t	*arr;
	awkcell_t	*handlecell;
	awkcell_t	*funccell;
	awkcell_t	*retcell;
	awkcell_t	*res;
	void		(*vfunc)(void);
	int		(*ifunc)(void);
	double		(*dfunc)(void);
	void		*(*pfunc)(void);
	char		 efunc[1024];
	char		 buf[1024];
	char		*funcname;
	char		*handle;
	char		*rettype;
	long		 largs[40];
	double		 dargs[40];
	un_t		 vpret;
	un_t		 un;

	arr = awklib_execute(awkp, a[0]); /* array */
	if ((handlecell = awklib_lookup("handle", (awkarray_t *)arr->sval)) == NULL) {
		res = awklib_gettemp(awkp);
		res->tval = NUM;
		res->fval = -1;
		return res;
	}
	handle = awklib_getsval(awkp, handlecell);
	funccell = awklib_execute(awkp, a[1]);	/* function name string */
	funcname = awklib_getsval(awkp, funccell);
	if (strchr(funcname, '(') != NULL) {
		/* we have a function prototype */
		parse_prototype(awkp, funcname, arr, efunc, sizeof(efunc));
		funcname = efunc;
	}
	if (/* CONSTCOND */sizeof(void *) == 8) {
		un.i64 = strtoll(handle, NULL, 10);
	} else {
		un.i32 = atoi(handle);
	}
	res = awklib_gettemp(awkp);
	res->tval = NUM;
	(void) snprintf(buf, sizeof(buf), "%s:return", funcname);
	retcell = awklib_lookup(buf, (awkarray_t *)arr->sval);
	rettype = awklib_getsval(awkp, retcell);
	if (strcmp(rettype, "void") == 0) {
		if ((vfunc = dlsym(un.vp, funcname)) != NULL) {
			makeargs(awkp, a[1]->nnext, (awkarray_t *)arr->sval, funcname, largs, dargs);
			voidcall(vfunc, largs, dargs);
			res->fval = 0;
			unmakeargs(awkp, a[1]->nnext, (awkarray_t *)arr->sval, funcname, largs, dargs);
			return res;
		}
	} else if (strcmp(rettype, "void*") == 0 || strcmp(rettype, "ptr") == 0) {
		if ((pfunc = dlsym(un.vp, funcname)) != NULL) {
			makeargs(awkp, a[1]->nnext, (awkarray_t *)arr->sval, funcname, largs, dargs);
			vpret.vp = ptrcall(pfunc, largs, dargs);
			if (/* CONSTCOND */sizeof(void *) == 8) {
				res->tval = vpret.i64;
			} else {
				res->tval = vpret.i32;
			}
			unmakeargs(awkp, a[1]->nnext, (awkarray_t *)arr->sval, funcname, largs, dargs);
			return res;
		}
	} else if (strcmp(rettype, "int") == 0) {
		if ((ifunc = dlsym(un.vp, funcname)) != NULL) {
			makeargs(awkp, a[1]->nnext, (awkarray_t *)arr->sval, funcname, largs, dargs);
			res->fval = intcall(ifunc, largs, dargs);
			unmakeargs(awkp, a[1]->nnext, (awkarray_t *)arr->sval, funcname, largs, dargs);
			return res;
		}
	} else if (strcmp(rettype, "float") == 0) {
		if ((dfunc = dlsym(un.vp, funcname)) != NULL) {
			makeargs(awkp, a[1]->nnext, (awkarray_t *)arr->sval, funcname, largs, dargs);
			res->fval = floatcall(dfunc, largs, dargs);
			unmakeargs(awkp, a[1]->nnext, (awkarray_t *)arr->sval, funcname, largs, dargs);
			return res;
		}
	}
	(void) fprintf(stderr, "Symbol \"%s\" not found\n", funcname);
	res->fval = -1;
	return res;
}

/* dlclose(lib) */
awkcell_t *
awklib_dlclose(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t	*handlecell;
	awkcell_t	*res;
	awkcell_t	*arr;
	char		*handle;
	un_t		 un;
	int		 ret;

	arr = awklib_execute(awkp, a[0]); /* array */
	if ((handlecell = awklib_lookup("handle", (awkarray_t *)arr->sval)) == NULL) {
		res = awklib_gettemp(awkp);
		res->tval = NUM;
		res->fval = -1;
		return res;
	}
	handle = awklib_getsval(awkp, handlecell);
	if (/* CONSTCOND */sizeof(void *) == 8) {
		un.i64 = strtoll(handle, NULL, 10);
	} else {
		un.i32 = atoi(handle);
	}
	ret = dlclose(un.vp);
	res = awklib_gettemp(awkp);
	res->tval = NUM;
	res->fval = ret;
	return res;
}

/* dlalloc(size) */
awkcell_t *
awklib_dlalloc(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t	*sizecell;
	awkcell_t	*res;
	uint32_t	 size;
	un_t		 un;

	sizecell = awklib_execute(awkp, a[0]); /* size */
	size = (uint32_t) awklib_getfval(awkp, sizecell);
	if ((un.vp = calloc(1, size)) == NULL) {
		(void) fprintf(stderr, "Unable to calloc \"%d\" bytes\n", size);
		res = awklib_gettemp(awkp);
		res->tval = NUM;
		res->fval = 0;
		return res;
	}
	TEMPFREE(awkp, sizecell);
	res = awklib_gettemp(awkp);
	res->tval = NUM;
	res->fval = (uint32_t) un.i32;
	return res;
}

/* dlfree(storage) */
awkcell_t *
awklib_dlfree(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t	*handlecell;
	awkcell_t	*res;
	un_t		 un;

	handlecell = awklib_execute(awkp, a[0]); /* handle */
	un.i32 = (uint32_t) awklib_getfval(awkp, handlecell);
	(void) free(un.vp);
	TEMPFREE(awkp, handlecell);
	res = awklib_gettemp(awkp);
	res->tval = NUM;
	res->fval = 1;
	return res;
}

/* dlfield(storage, offset, length, type) */
awkcell_t *
awklib_dlfield(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t	*handlecell;
	awkcell_t	*typecell;
	awkcell_t	*offcell;
	awkcell_t	*res;
	int64_t		 i64;
	int32_t		 i32;
	char		*storage;
	char		*comma;
	char		*type;
	un_t		 un;
	int		 size;
	int		 off;

	handlecell = awklib_execute(awkp, a[0]); /* storage */
	un.i32 = (uint32_t) awklib_getfval(awkp, handlecell);
	storage = (char *) un.vp;
	offcell = awklib_execute(awkp, a[1]); /* offset */
	off = (uint32_t) awklib_getfval(awkp, offcell);
	typecell = awklib_execute(awkp, a[2]); /* type */
	type = awklib_getsval(awkp, typecell);
	res = awklib_gettemp(awkp);
	if (strncmp(type, "storage", 7) == 0 ||
	    strncmp(type, "ptr", 3) == 0 ||
	    strncmp(type, "int32", 5) == 0) {
		(void) memcpy(&i32, &storage[off], sizeof(i32));
		if ((comma = strchr(type, ',')) != NULL) {
			if (strcmp(comma + 1, "htonl") == 0) {
				i32 = htonl(i32);
			} else if (strcmp(comma + 1, "ntohl") == 0) {
				i32 = ntohl(i32);
			}
		}
		res->tval = NUM;
		res->fval = i32;
	} else if (strncmp(type, "int16", 5) == 0) {
		i32 = 0;
		(void) memcpy(&i32, &storage[off], sizeof(i32));
		if ((comma = strchr(type, ',')) != NULL) {
			if (strcmp(comma + 1, "htons") == 0) {
				i32 = htons(i32 & 0xffff);
			} else if (strcmp(comma + 1, "ntohs") == 0) {
				i32 = ntohs(i32 & 0xffff);
			}
		}
		res->tval = NUM;
		res->fval = (i32 & 0xffff);
	} else if (strncmp(type, "int64", 5) == 0) {
		(void) memcpy(&i64, &storage[off], 8);
		res->tval = NUM;
		res->fval = i64;
	} else if (strncmp(type, "int8", 4) == 0 ||
	           strncmp(type, "bool", 4) == 0) {
		res->tval = NUM;
		res->fval = storage[off];
	} else if (strncmp(type, "string", 6) == 0) {
		size = MAXSTRSIZE;
		if ((comma = strchr(type, ',')) != NULL) {
			size = atol(comma + 1);
		}
		res->tval = STR;
		res->sval = strnsave(&storage[off], -1, MIN(MAXSTRSIZE, size));
	} else {
		res->tval = NUM;
		res->fval = -1;
	}
	TEMPFREE(awkp, handlecell);
	TEMPFREE(awkp, typecell);
	TEMPFREE(awkp, offcell);
	return res;
}

/* shift(N) */
awkcell_t *
awklib_shift(awk_t *awkp, awknode_t **a, int nnn)
{
	awkcell_t	*countcell;
	awkcell_t	*res;
	int		 count;

	countcell = awklib_execute(awkp, a[0]); /* shift count */
	count = (int) awklib_getfval(awkp, countcell);
	if (ABS(count) > MAXFIELDS || ABS(awkp->shiftc + count) > MAXFIELDS) {
		/* be reasonable */
		WARNING(awkp,  "requested shift count too large - ignoring request", count);
	} else {
		awkp->shiftc += count;
	}
	TEMPFREE(awkp, countcell);
	res = awklib_gettemp(awkp);
	res->tval = NUM;
	res->fval = awkp->shiftc;
	return res;
}
