/*	$NetBSD: parser1.c,v 1.6 2003/08/07 11:17:28 agc Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#ifndef lint
__RCSID("$NetBSD: parser1.c,v 1.6 2003/08/07 11:17:28 agc Exp $");
#endif /* not lint */

#include <string.h>
#include <stdlib.h>

#include "ww.h"
#include "defs.h"
#include "var.h"
#include "var.h"
#include "lcmd.h"
#include "context.h"
#include "token.h"

#define p_erred()	(cx.x_erred)
#define p_clearerr()	(cx.x_erred = cx.x_synerred = 0)
#define p_abort()	(cx.x_abort)

static char *
str_ncpy(char *s, int n)
{
	char	*p;
	int	 len = strlen(s);

	if (n > len) {
		n = len;
	}
	if ((p = malloc((unsigned)(n + 1))) != NULL) {
		(void) memcpy(p, s, (unsigned)n);
		p[n] = 0x0;
	}
	return p;
}

static char *
str_cat(char *s1, char *s2)
{
	unsigned	 s1len;
	unsigned	 s2len;
	char		*p;

	s1len = strlen(s1);
	s2len = strlen(s2);
	if ((p = malloc(strlen(s1) + strlen(s2) + 1)) == NULL) {
		return NULL;
	}
	(void) memcpy(p, s1, s1len);
	(void) memcpy(&p[s1len], s2, s2len);
	p[s1len + s2len] = 0x0;
	return p;
}

/* forward declarations */
static int	p_assign(char *, value_t *, int);
static int	p_convstr(value_t *v);
static void	p_error(const char *msg, ...);
static int	p_expr(value_t *, char);
static int	p_expr0(value_t *, char);
static int	p_expr1(value_t *, char);
static int	p_expr11(value_t *, char);
static int	p_expr12(value_t *, char);
static int	p_expr2(value_t *, char);
static int	p_expr3_10(int, value_t *, char);
static int	p_expression(char);
static int	p_function(char *, value_t *, int);
static int	p_if(char);
static int	p_statement(char);
static void	p_statementlist(char);
static void	p_synerror(void);


void
p_start(void)
{
	char flag = 1;

	(void) s_gettok();
	for (;;) {
		p_statementlist(flag);
		if (token == T_EOF || p_abort())
			break;
		flag = 0;
		p_synerror();
		while (token != T_EOL && token != T_EOF) {
			if (token == T_STR)
				free(token_str);
			(void) s_gettok();
		}
		if (token == T_EOL)
			(void) s_gettok();
		p_clearerr();
	}
}

void
p_statementlist(char flag)
{
	for (; p_statement(flag) >= 0; p_clearerr())
		;
}

int
p_statement(char flag)
{
	switch (token) {
	case T_EOL:
		(void) s_gettok();
		return 0;
	case T_IF:
		return p_if(flag);
	default:
		return p_expression(flag);
	}
}

int
p_if(char flag)
{
	value_t t;
	char true = 0;

top:
	(void) s_gettok();

	if (p_expr(&t, flag) < 0) {
		p_synerror();
		return -1;
	}
	switch (t.v_type) {
	case V_NUM:
		true = !true && t.v_num != 0;
		break;
	case V_STR:
		p_error("if: Numeric value required.");
		free(t.v_str);
		/* FALLTHROUGH */
	case V_ERR:
		flag = 0;
		break;
	}

	if (token != T_THEN) {
		p_synerror();
		return -1;
	}

	(void) s_gettok();
	p_statementlist(flag && true);
	if (p_erred())
		return -1;

	if (token == T_ELSIF)
		goto top;

	if (token == T_ELSE) {
		(void) s_gettok();
		p_statementlist(flag && !true);
		if (p_erred())
			return -1;
	}

	if (token == T_ENDIF) {
		(void) s_gettok();
		return 0;
	}

	p_synerror();
	return -1;
}

int
p_expression(char flag)
{
	value_t t;
	char *cmd;

	switch (token) {
	case T_NUM:
		t.v_type = V_NUM;
		t.v_num = token_num;
		(void) s_gettok();
		break;
	case T_STR:
		t.v_type = V_STR;
		t.v_str = token_str;
		(void) s_gettok();
		break;
	default:
		if (p_expr(&t, flag) < 0)
			return -1;
		if (token == T_EOF) {
			VAL_FREE(t);
			return 0;
		}
	}
	if (token != T_ASSIGN && p_convstr(&t) < 0)
		return -1;
	cmd = t.v_type == V_STR ? t.v_str : 0;
	if ((*(token == T_ASSIGN ? p_assign : p_function))(cmd, &t, flag) < 0) {
		if (cmd)
			free(cmd);
		return -1;
	}
	if (cmd)
		free(cmd);
	VAL_FREE(t);
	if (token == T_EOL)
		(void) s_gettok();
	else if (token != T_EOF) {
		p_synerror();
		return -1;
	}
	return 0;
}

static int
p_convstr(value_t *v)
{
	if (v->v_type != V_NUM)
		return 0;
	(void) asprintf(&v->v_str, "%d", v->v_num);
	if (v->v_str == NULL) {
		p_memerror();
		v->v_type = V_ERR;
		return -1;
	}
	v->v_type = V_STR;
	return 0;
}

void
p_synerror(void)
{
	if (!cx.x_synerred) {
		cx.x_synerred = cx.x_erred = 1;
		error("Syntax error.");
	}
}

static void
p_error(const char *msg, ...)
{
	va_list ap;

	va_start(ap, msg);
	if (!cx.x_erred) {
		cx.x_erred = 1;
		verror(msg, ap);
	}
	va_end(ap);
}

void
p_memerror(void)
{
	cx.x_erred = cx.x_abort = 1;
	error("Out of memory.");
}

/*
 * name == NULL means we don't have a function name but
 * want to parse the arguments anyway.  flag == 0 in this case.
 */
int
p_function(char *name, value_t *v, int flag)
{
	value_t t;
	struct lcmd_tab *c = NULL;
	variable_t *a = NULL;
	struct lcmd_arg *ap;			/* this arg */
	struct lcmd_arg *lp = NULL;		/* list arg */
	int i;
	value_t av[LCMD_NARG + 1];
	value_t *vp;
	winvars_t	*winvars;

	if (name != NULL) {
		if ((c = lcmd_lookup(name)) != NULL) {
			name = __UNCONST(c->lc_name);
		} else if ((a = alias_lookup(name)) != NULL) {
			name = a->r_name;
		} else {
			p_error("%s: No such command or alias.", name);
			flag = 0;
		}
	}
	for (vp = av; vp < &av[LCMD_NARG + 1]; vp++)
		vp->v_type = V_ERR;

	if (token == T_LP)
		(void) s_gettok();
	i = 0;
	for (;;) {
		ap = 0;
		vp = 0;
		if (token == T_COMMA)		/* null argument */
			t.v_type = V_ERR;
		else {
			if (p_expr0(&t, flag) < 0)
				break;
			if (t.v_type == V_ERR)
				flag = 0;
		}
		if (token != T_ASSIGN) {
			if (i >= LCMD_NARG ||
			    (c != NULL && (ap = lp) == NULL &&
			    (ap = c->lc_arg + i)->arg_name == NULL)) {
				p_error("%s: Too many arguments.", name);
				flag = 0;
			} else
				vp = &av[i++];
		} else {
			char *tmp;
			if (p_convstr(&t) < 0)
				goto abort;
			tmp = t.v_type == V_STR ? t.v_str : 0;
			(void) s_gettok();
			if (p_expr(&t, flag) < 0) {
				if (tmp)
					free(tmp);
				p_synerror();
				goto abort;
			}
			if (t.v_type == V_ERR)
				flag = 0;
			if (tmp) {
				if (c == 0) {
					/* an aliase */
					p_error("%s: Bad alias syntax.", name);
					flag = 0;
				} else {
					for (ap = c->lc_arg, vp = av;
					     ap != 0 && ap->arg_name != 0 &&
					     (*ap->arg_name == '\0' ||
					      strncmp(tmp, ap->arg_name,
							ap->arg_minlen) != 0);
					     ap++, vp++)
						;
					if (ap == 0 || ap->arg_name == 0) {
						p_error("%s: Unknown argument \"%s\".",
							name, tmp);
						flag = 0;
						ap = 0;
						vp = 0;
					}
				}
				free(tmp);
			}
		}
		if (ap != 0) {
			if (ap->arg_flags & ARG_LIST) {
				i = vp - av + 1;
				lp = ap;
			}
			if (vp && vp->v_type != V_ERR) {
				if (*ap->arg_name)
					p_error("%s: Argument %d (%s) duplicated.",
						name, vp - av + 1,
						ap->arg_name);
				else
					p_error("%s: Argument %d duplicated.",
						name, vp - av + 1);
				flag = 0;
				vp = 0;
			} else if (t.v_type == V_ERR) {
				/* do nothing */
			} else if (((ap->arg_flags&ARG_TYPE) == ARG_NUM &&
				    t.v_type != V_NUM) ||
				   ((ap->arg_flags&ARG_TYPE) == ARG_STR &&
				   t.v_type != V_STR)) {
				if (*ap->arg_name)
					p_error("%s: Argument %d (%s) type mismatch.",
						name, vp - av + 1,
						ap->arg_name);
				else
					p_error("%s: Argument %d type mismatch.",
						name, vp - av + 1);
				flag = 0;
				vp = 0;
			}
		}
		if (vp != 0)
			*vp = t;
		else
			VAL_FREE(t);
		if (token == T_COMMA)
			(void) s_gettok();
	}

	if (p_erred())
		flag = 0;
	if (token == T_RP)
		(void) s_gettok();
	else if (token != T_EOL && token != T_EOF)
		flag = 0;		/* look for legal follow set */
	v->v_type = V_ERR;
	if (flag) {
		winvars = get_winvars();
		if (c != NULL) {
			(*c->lc_func)(winvars, v, av);
		} else {
			if (a->r_val.v_type & A_INUSE)
				p_error("%s: Recursive alias.", a->r_name);
			else {
				a->r_val.v_type |= A_INUSE;
				if (dolongcmd(a->r_val.v_str, av, i) < 0)
					p_memerror();
				a->r_val.v_type &= ~A_INUSE;
			}
		}
	}
	if (p_abort()) {
		VAL_FREE(*v);
		v->v_type = V_ERR;
		goto abort;
	}
	for (vp = av; vp < &av[LCMD_NARG]; vp++)
		VAL_FREE(*vp);
	return 0;
abort:
	for (vp = av; vp < &av[LCMD_NARG]; vp++)
		VAL_FREE(*vp);
	return -1;
}

static int
p_assign(char *name, value_t *v, int flag)
{
	(void) s_gettok();

	if (p_expr(v, flag) < 0) {
		p_synerror();
		return -1;
	}
	switch (v->v_type) {
	case V_STR:
	case V_NUM:
		if (name && flag && var_set(name, v) == 0) {
			p_memerror();
			VAL_FREE(*v);
			return -1;
		}
		break;
	}
	return 0;
}

/*
 * =
 * ? :
 * ||
 * &&
 * |
 * ^
 * &
 * == !=
 * <= >=
 * << >>
 * + -
 * * / %
 * unary - + ~ !
 */
static int
p_expr(value_t *v, char flag)
{
	value_t t;
	int ret;

	if (p_expr0(&t, flag) < 0)
		return -1;

	if (token != T_ASSIGN) {
		*v = t;
		return 0;
	}
	switch (t.v_type) {
	case V_NUM:
		p_error("%d: Not a variable.", t.v_num);
		/* FALLTHROUGH */
	case V_ERR:
		t.v_str = 0;
		break;
	}
	ret = p_assign(t.v_str, v, flag);
	if (t.v_str != 0)
		free(t.v_str);
	return ret;
}

/*
 * ? :
 */
static int
p_expr0(value_t *v, char flag)
{
	value_t t;
	char true = 0;

	if (p_expr1(v, flag) < 0)
		return -1;
	if (token != T_QUEST)
		return 0;
	switch (v->v_type) {
	case V_NUM:
		true = v->v_num != 0;
		break;
	case V_STR:
		p_error("?: Numeric left operand required.");
		free(v->v_str);
		v->v_type = V_ERR;
		/* FALLTHROUGH */
	case V_ERR:
		flag = 0;
		break;
	}
	(void) s_gettok();
	v->v_type = V_ERR;
	if ((flag && true ? p_expr1(v, 1) : p_expr1(&t, 0)) < 0)
		return -1;
	if (token != T_COLON) {
		VAL_FREE(*v);
		p_synerror();
		return -1;
	}
	(void) s_gettok();
	return flag && !true ? p_expr1(v, 1) : p_expr1(&t, 0);
}

/*
 * ||
 */
static int
p_expr1(value_t *v, char flag)
{
	char true = 0;

	if (p_expr2(v, flag) < 0)
		return -1;
	if (token != T_OROR)
		return 0;
	for (;;) {
		switch (v->v_type) {
		case V_NUM:
			v->v_num = true = true || v->v_num != 0;
			break;
		case V_STR:
			p_error("||: Numeric operands required.");
			free(v->v_str);
			v->v_type = V_ERR;
			/* FALLTHROUGH */
		case V_ERR:
			flag = 0;
			break;
		}
		if (token != T_OROR)
			return 0;
		(void) s_gettok();
		if (p_expr2(v, flag && !true) < 0)
			return -1;
	}
}

/*
 * &&
 */
int
p_expr2(value_t *v, char flag)
{
	char true = 1;

	if (p_expr3_10(3, v, flag) < 0)
		return -1;
	if (token != T_ANDAND)
		return 0;
	for (;;) {
		switch (v->v_type) {
		case V_NUM:
			v->v_num = true = true && v->v_num != 0;
			break;
		case V_STR:
			p_error("&&: Numeric operands required.");
			free(v->v_str);
			v->v_type = V_ERR;
			/* FALLTHROUGH */
		case V_ERR:
			flag = 0;
			break;
		}
		if (token != T_ANDAND)
			return 0;
		(void) s_gettok();
		if (p_expr3_10(3, v, flag && true) < 0)
			return -1;
	}
	/*NOTREACHED*/
}

/*
 * |		3
 * ^		4
 * &		5
 * == !=	6
 * < <= > >=	7
 * << >>	8
 * + -		9
 * * / %	10
 */
int
p_expr3_10(int level, value_t *v, char flag)
{
	value_t l, r;
	int op;
	const char *opname = NULL;

	if ((level == 10 ? p_expr11(v, flag)
	     : p_expr3_10(level + 1, v, flag)) < 0)
		return -1;
	for (;;) {
		switch (level) {
		case 3:
			if (token != T_OR)
				return 0;
			opname = "|";
			break;
		case 4:
			if (token != T_XOR)
				return 0;
			opname = "^";
			break;
		case 5:
			if (token != T_AND)
				return 0;
			opname = "&";
			break;
		case 6:
			if (token == T_EQ)
				opname = "==";
			else if (token == T_NE)
				opname = "!=";
			else
				return 0;
			break;
		case 7:
			switch (token) {
			case T_LT:
				opname = "<";
				break;
			case T_LE:
				opname = "<=";
				break;
			case T_GT:
				opname = ">";
				break;
			case T_GE:
				opname = ">=";
				break;
			default:
				return 0;
			}
			break;
		case 8:
			if (token == T_LS)
				opname = "<<";
			else if (token == T_RS)
				opname = ">>";
			else
				return 0;
			break;
		case 9:
			if (token == T_PLUS)
				opname = "+";
			else if (token == T_MINUS)
				opname = "-";
			else
				return 0;
			break;
		case 10:
			switch (token) {
			case T_MUL:
				opname = "*";
				break;
			case T_DIV:
				opname = "/";
				break;
			case T_MOD:
				opname = "%";
				break;
			default:
				return 0;
			}
			break;
		}
		l = *v;
		if (l.v_type == V_ERR)
			flag = 0;

		op = token;
		(void) s_gettok();
		if ((level == 10 ? p_expr11(&r, flag)
		     : p_expr3_10(level + 1, &r, flag)) < 0) {
			p_synerror();
			VAL_FREE(l);
			return -1;
		}

		if (r.v_type == V_ERR)
			flag = 0;
		else switch (op) {
		case T_EQ:
		case T_NE:
		case T_LT:
		case T_LE:
		case T_GT:
		case T_GE:
		case T_PLUS:
			if (l.v_type == V_STR) {
				if (r.v_type == V_NUM)
					if (p_convstr(&r) < 0)
						flag = 0;
			} else
				if (r.v_type == V_STR)
					if (p_convstr(&l) < 0)
						flag = 0;
			break;
		case T_LS:
		case T_RS:
			if (r.v_type == V_STR) {
				char *p = r.v_str;
				r.v_type = V_NUM;
				r.v_num = strlen(p);
				free(p);
			}
			break;
		case T_OR:
		case T_XOR:
		case T_AND:
		case T_MINUS:
		case T_MUL:
		case T_DIV:
		case T_MOD:
		default:
			if (l.v_type == V_STR || r.v_type == V_STR) {
				p_error("%s: Numeric operands required.",
					opname);
				flag = 0;
			}
		}
		if (!flag) {
			VAL_FREE(l);
			VAL_FREE(r);
			v->v_type = V_ERR;
			if (p_abort())
				return -1;
			continue;
		}

		v->v_type = V_NUM;
		switch (op) {
		case T_EQ:
		case T_NE:
		case T_LT:
		case T_LE:
		case T_GT:
		case T_GE:
			if (l.v_type == V_STR) {
				int tmp = strcmp(l.v_str, r.v_str);
				free(l.v_str);
				free(r.v_str);
				l.v_type = V_NUM;
				l.v_num = tmp;
				r.v_type = V_NUM;
				r.v_num = 0;
			}
			break;
		}
		switch (op) {
		case T_OR:
			v->v_num = l.v_num | r.v_num;
			break;
		case T_XOR:
			v->v_num = l.v_num ^ r.v_num;
			break;
		case T_AND:
			v->v_num = l.v_num & r.v_num;
			break;
		case T_EQ:
			v->v_num = l.v_num == r.v_num;
			break;
		case T_NE:
			v->v_num = l.v_num != r.v_num;
			break;
		case T_LT:
			v->v_num = l.v_num < r.v_num;
			break;
		case T_LE:
			v->v_num = l.v_num <= r.v_num;
			break;
		case T_GT:
			v->v_num = l.v_num > r.v_num;
			break;
		case T_GE:
			v->v_num = l.v_num >= r.v_num;
			break;
		case T_LS:
			if (l.v_type == V_STR) {
				int i;
				if ((i = strlen(l.v_str)) > r.v_num)
					i = r.v_num;
				v->v_str = str_ncpy(l.v_str, i);
				v->v_type = V_STR;
			} else
				v->v_num = l.v_num << r.v_num;
			break;
		case T_RS:
			if (l.v_type == V_STR) {
				int i;
				if ((i = strlen(l.v_str)) > r.v_num)
					i -= r.v_num;
				else
					i = 0;
				v->v_str = strdup(l.v_str + i);
				v->v_type = V_STR;
			} else {
				/* LINTED */
				v->v_num = l.v_num >> r.v_num;
			}
			break;
		case T_PLUS:
			if (l.v_type == V_STR) {
				v->v_str = str_cat(l.v_str, r.v_str);
				v->v_type = V_STR;
			} else
				v->v_num = l.v_num + r.v_num;
			break;
		case T_MINUS:
			v->v_num = l.v_num - r.v_num;
			break;
		case T_MUL:
			v->v_num = l.v_num * r.v_num;
			break;
		case T_DIV:
			v->v_num = l.v_num / r.v_num;
			break;
		case T_MOD:
			v->v_num = l.v_num % r.v_num;
			break;
		}
		VAL_FREE(l);
		VAL_FREE(r);
	}
	/*NOTREACHED*/
}

/*
 * unary $ $? + - ! ~
 */
int
p_expr11(value_t *v, char flag)
{
	int op;
	const char *opname;

	switch (token) {
	case T_DOLLAR:
		opname = "$";
		break;
	case T_DQ:
		opname = "$?";
		break;
	case T_PLUS:
		opname = "unary +";
		break;
	case T_MINUS:
		opname = "unary -";
		break;
	case T_NOT:
		opname = "!";
		break;
	case T_COMP:
		opname = "~";
		break;
	default:
		return p_expr12(v, flag);
	}
	op = token;
	(void) s_gettok();
	if (p_expr11(v, flag) < 0)
		return -1;
	switch (v->v_type) {
	case V_NUM:
		break;
	case V_STR:
		switch (op) {
		case T_MINUS:
		case T_NOT:
		case T_COMP:
			p_error("%s: Numeric operand required.", opname);
			free(v->v_str);
			v->v_type = V_ERR;
			return 0;
		}
		break;
	case V_ERR:
		return 0;
	}
	switch (op) {
	case T_DOLLAR:
	case T_DQ:
		if (v->v_type == V_NUM) {
			int tmp = cx.x_type == X_BUF && cx.x_arg != 0 &&
				v->v_num > 0 && v->v_num <= cx.x_narg;
			if (op == T_DQ)
				v->v_num = tmp;
			else if (tmp)
				*v = cx.x_arg[v->v_num - 1];
			else {
				p_error("%d: No such argument.", v->v_num);
				v->v_type = V_ERR;
			}
		} else {
			char *name = v->v_str;
			variable_t *r = var_lookup(name);
			if (op == T_DQ) {
				v->v_type = V_NUM;
				v->v_num = r != 0;
			} else if (r != 0)
				*v = r->r_val;
			else {
				p_error("%s: Undefined variable.", name);
				v->v_type = V_ERR;
			}
			free(name);
		}
		if (v->v_type == V_STR && (v->v_str = strdup(v->v_str)) == 0) {
			p_memerror();
			return -1;
		}
		break;
	case T_MINUS:
		v->v_num = - v->v_num;
		break;
	case T_NOT:
		v->v_num = ! v->v_num;
		break;
	case T_COMP:
		v->v_num = ~ v->v_num;
		break;
	}
	return 0;
}

/*
 * string, number, ( expr )
 * Plus function calls.
 *
 * Always return v_type == V_ERR when flag == 0.
 */
int
p_expr12(value_t *v, char flag)
{
	v->v_type = V_ERR;
	switch (token) {
	case T_NUM:
		if (flag) {
			v->v_type = V_NUM;
			v->v_num = token_num;
		}
		(void) s_gettok();
		break;
	case T_STR:
		if (flag) {
			v->v_type = V_STR;
			v->v_str = token_str;
		} else
			free(token_str);
		(void) s_gettok();
		break;
	case T_LP:
		(void) s_gettok();
		if (p_expr(v, flag) < 0) {
			p_synerror();
			return -1;
		}
		if (token != T_RP) {
			p_synerror();
			VAL_FREE(*v);
			return -1;
		}
		(void) s_gettok();
		break;
	default:
		return -1;
	}
	while (token == T_LP) {
		char *cmd;

		if (p_convstr(v) < 0)
			return -1;
		cmd = v->v_type == V_STR ? v->v_str : 0;
		if (p_function(cmd, v, flag) < 0) {
			if (cmd)
				free(cmd);
			return -1;
		}
		if (cmd)
			free(cmd);
	}
	return 0;
}
