/*	$NetBSD: wwbox.c,v 1.6 2003/08/07 11:17:36 agc Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#ifndef lint
__RCSID("$NetBSD: wwbox.c,v 1.6 2003/08/07 11:17:36 agc Exp $");
#endif /* not lint */

#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "ww.h"
#include "tt.h"
#include "char.h"


void
wwbox(winvars_t *winvars, ww_t *w, int r, int c, int nr, int nc)
{
	int r1, c1;
	int i;

	r1 = r + nr - 1;
	c1 = c + nc - 1;
	wwframec(winvars, w, r, c, WWF_D|WWF_R);
	for (i = c + 1; i < c1; i++)
		wwframec(winvars, w, r, i, WWF_L|WWF_R);
	wwframec(winvars, w, r, i, WWF_L|WWF_D);
	for (i = r + 1; i < r1; i++)
		wwframec(winvars, w, i, c1, WWF_U|WWF_D);
	wwframec(winvars, w, i, c1, WWF_U|WWF_L);
	for (i = c1 - 1; i > c; i--)
		wwframec(winvars, w, r1, i, WWF_R|WWF_L);
	wwframec(winvars, w, r1, i, WWF_R|WWF_U);
	for (i = r1 - 1; i > r; i--)
		wwframec(winvars, w, i, c, WWF_D|WWF_U);
}

static char cmap[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";

void
wwdumpwin(ww_t *w)
{
	int i, j;

	tt.tt_nmodes = 0;
	(*tt.tt_clear)();
	for (i = w->ww_i.t; i < w->ww_i.b; i++) {
		(*tt.tt_move)(i, w->ww_i.l);
		for (j = w->ww_i.l; j < w->ww_i.r; j++)
			(*tt.tt_putc)(w->ww_win[i][j] & WWM_GLS ? 'G' : ' ');
	}
}

void
wwdumpnvis(ww_t *w)
{
	int i;
	char buf[20];

	tt.tt_nmodes = 0;
	(*tt.tt_clear)();
	for (i = w->ww_i.t; i < w->ww_i.b; i++) {
		(*tt.tt_move)(i, w->ww_i.l);
		(void) snprintf(buf, sizeof(buf), "%d", w->ww_nvis[i]);
		(*tt.tt_write)(buf, (int)strlen(buf));
	}
}

void
wwdumpsmap(winvars_t *winvars)
{
	int i, j;

	tt.tt_nmodes = 0;
	(*tt.tt_clear)();
	for (i = 0; i < winvars->wwnrow; i++) {
		(*tt.tt_move)(i, 0);
		for (j = 0; j < winvars->wwncol; j++)
			(*tt.tt_putc)(cmap[winvars->wwsmap[i][j]]);
	}
}

void
wwdumpns(winvars_t *winvars)
{
	int i, j;

	(*tt.tt_clear)();
	for (i = 0; i < winvars->wwnrow; i++) {
		(*tt.tt_move)(i, 0);
		for (j = 0; j < winvars->wwncol; j++) {
			tt.tt_nmodes = getmodepart(&winvars->wwns[i][j]) & tt.tt_availmodes;
			(*tt.tt_putc)(getcharpart(&winvars->wwns[i][j]));
		}
	}
}

void
wwdumpos(winvars_t *winvars)
{
	int i, j;

	(*tt.tt_clear)();
	for (i = 0; i < winvars->wwnrow; i++) {
		(*tt.tt_move)(i, 0);
		for (j = 0; j < winvars->wwncol; j++) {
			tt.tt_nmodes = getmodepart(&winvars->wwos[i][j]) & tt.tt_availmodes;
			(*tt.tt_putc)(getcharpart(&winvars->wwns[i][j]));
		}
	}
}

#define FRAMEOK(wp, w, r, c) (w1 = (wp)->wwindex[(wp)->wwsmap[r][c]], \
	w1->ww_fmap || w1->ww_order > (w)->ww_order)

void
wwframe(winvars_t *winvars, ww_t *w, ww_t *wframe)
{
	int r, c;
	char a1, a2, a3;
	char b1, b2, b3;
	int code;
	ww_t *w1;

	if (w->ww_w.t > 0) {
		r = w->ww_w.t - 1;
		c = w->ww_i.l - 1;
		a1 = 0;
		a2 = 0;
		b1 = 0;
		b2 = c < 0 || FRAMEOK(winvars, w, r, c);

		for (; c < w->ww_i.r; c++) {
			if (c + 1 >= winvars->wwncol) {
				a3 = 1;
				b3 = 1;
			} else {
				a3 = w->ww_index == winvars->wwsmap[r + 1][c + 1];
				b3 = FRAMEOK(winvars, w, r, c + 1);
			}
			if (b2) {
				code = 0;
				if ((a1 || a2) && b1)
					code |= WWF_L;
				if ((a2 || a3) && b3)
					code |= WWF_R;
				if (code)
					wwframec(winvars, wframe, r, c, code|WWF_TOP);
			}
			a1 = a2;
			a2 = a3;
			b1 = b2;
			b2 = b3;
		}
		if ((a1 || a2) && b1 && b2)
			wwframec(winvars, wframe, r, c, (char)(WWF_L|WWF_TOP));
	}

	if (w->ww_w.b < winvars->wwnrow) {
		r = w->ww_w.b;
		c = w->ww_i.l - 1;
		a1 = 0;
		a2 = 0;
		b1 = 0;
		b2 = c < 0 || FRAMEOK(winvars, w, r, c);

		for (; c < w->ww_i.r; c++) {
			if (c + 1 >= winvars->wwncol) {
				a3 = 1;
				b3 = 1;
			} else {
				a3 = w->ww_index == winvars->wwsmap[r - 1][c + 1];
				b3 = FRAMEOK(winvars, w, r, c + 1);
			}
			if (b2) {
				code = 0;
				if ((a1 || a2) && b1)
					code |= WWF_L;
				if ((a2 || a3) && b3)
					code |= WWF_R;
				if (code)
					wwframec(winvars, wframe, r, c, code);
			}
			a1 = a2;
			a2 = a3;
			b1 = b2;
			b2 = b3;
		}
		if ((a1 || a2) && b1 && b2)
			wwframec(winvars, wframe, r, c, WWF_L);
	}

	if (w->ww_w.l > 0) {
		r = w->ww_i.t - 1;
		c = w->ww_w.l - 1;
		a1 = 0;
		a2 = 0;
		b1 = 0;
		b2 = r < 0 || FRAMEOK(winvars, w, r, c);

		for (; r < w->ww_i.b; r++) {
			if (r + 1 >= winvars->wwnrow) {
				a3 = 1;
				b3 = 1;
			} else {
				a3 = w->ww_index == winvars->wwsmap[r + 1][c + 1];
				b3 = FRAMEOK(winvars, w, r + 1, c);
			}
			if (b2) {
				code = 0;
				if ((a1 || a2) && b1)
					code |= WWF_U;
				if ((a2 || a3) && b3)
					code |= WWF_D;
				if (code)
					wwframec(winvars, wframe, r, c, code);
			}
			a1 = a2;
			a2 = a3;
			b1 = b2;
			b2 = b3;
		}
		if ((a1 || a2) && b1 && b2)
			wwframec(winvars, wframe, r, c, WWF_U);
	}

	if (w->ww_w.r < winvars->wwncol) {
		r = w->ww_i.t - 1;
		c = w->ww_w.r;
		a1 = 0;
		a2 = 0;
		b1 = 0;
		b2 = r < 0 || FRAMEOK(winvars, w, r, c);

		for (; r < w->ww_i.b; r++) {
			if (r + 1 >= winvars->wwnrow) {
				a3 = 1;
				b3 = 1;
			} else {
				a3 = w->ww_index == winvars->wwsmap[r + 1][c - 1];
				b3 = FRAMEOK(winvars, w, r + 1, c);
			}
			if (b2) {
				code = 0;
				if ((a1 || a2) && b1)
					code |= WWF_U;
				if ((a2 || a3) && b3)
					code |= WWF_D;
				if (code)
					wwframec(winvars, wframe, r, c, code);
			}
			a1 = a2;
			a2 = a3;
			b1 = b2;
			b2 = b3;
		}
		if ((a1 || a2) && b1 && b2)
			wwframec(winvars, wframe, r, c, WWF_U);
	}
}

void
wwframec(winvars_t *winvars, ww_t *f, int r, int c, char code)
{
	char oldcode;
	unsigned char *smap;

	if (r < f->ww_i.t || r >= f->ww_i.b || c < f->ww_i.l || c >= f->ww_i.r)
		return;

	smap = &winvars->wwsmap[r][c];

	{
		ww_t *w;

		w = winvars->wwindex[*smap];
		if (w->ww_order > f->ww_order) {
			if (w != &winvars->wwnobody && w->ww_win[r][c] == 0)
				w->ww_nvis[r]--;
			*smap = f->ww_index;
		}
	}

	if (f->ww_fmap != 0) {
		char *fmap;

		fmap = &f->ww_fmap[r][c];
		oldcode = *fmap;
		*fmap |= code;
		if (code & WWF_TOP)
			*fmap &= ~WWF_LABEL;
		code = *fmap;
	} else
		oldcode = 0;
	{
		char *win = &f->ww_win[r][c];

		if (*win == WWM_GLS && *smap == f->ww_index)
			f->ww_nvis[r]++;
		*win &= ~WWM_GLS;
	}
	if (oldcode != code && (code & WWF_LABEL) == 0) {
#if 1
		short frame;

		frame = tt.tt_frame[code & WWF_MASK];
		setwhole(&f->ww_buf[r][c], frame);
#else
		char	ch;
		char	mode;

		mode = tt.tt_frame[code & WWF_MASK];
		ch = tt.tt_frame[code & WWF_MASK];
		setwhole(&f->ww_buf[r][c], frame);
#endif
		if (winvars->wwsmap[r][c] == f->ww_index) {
			winvars->wwtouched[r] |= WWU_TOUCHED;
#if 0
			setwhole(&winvars->wwns[r][c], frame);
#else
			setwhole(&winvars->wwns[r][c], frame);
#endif
		}
	}
}

/*
 * Sufficient but not necessary test for total visibility.
 */
int
wwvisible(winvars_t *winvars, ww_t *w)
{
	int i;
	int nvis = 0;

	for (i = w->ww_i.t; i < w->ww_i.b; i++)
		nvis += w->ww_nvis[i];
	if (ISSET(w->ww_wflags, WWW_HASCURSOR)
	    && w->ww_cur.r >= w->ww_i.t && w->ww_cur.r < w->ww_i.b
	    && w->ww_cur.c >= w->ww_i.l && w->ww_cur.c < w->ww_i.r
	    && winvars->wwsmap[w->ww_cur.r][w->ww_cur.c] == w->ww_index)
		nvis++;
	return nvis == w->ww_i.nr * w->ww_i.nc;
}

void
wwbell(void)
{
	/* LINTED */
	TTPUTC(CONTROL('g'));
}

/*
 * Tty input interrupt handler.
 * (1) Read input into buffer (wwib*).
 * (2) Set the interrupt flag if anything is read.
 * Currently, the last is used to get out of the blocking
 * select() in wwiomux().
 * To avoid race conditions, we only modify wwibq in here, except
 * when the buffer is empty; and everywhere else, we only change wwibp.
 * It should be completely safe.
 */
void
wwrint(winvars_t *winvars)
{
	int n;

	winvars->wwnread++;
	n = read(0, winvars->wwibq, (unsigned)(winvars->wwibe - winvars->wwibq));
	if (n > 0) {
		if (tt.tt_rint) {
			n = (*tt.tt_rint)(winvars->wwibq, n);
		}
		if (n > 0) {
			winvars->wwibq += n;
			winvars->wwnreadc += n;
			/*
			 * Hasten or delay the next checkpoint,
			 * as the case may be.
			 */
			if (tt.tt_checkpoint && !winvars->wwdocheckpoint) {
				(void) alarm(1);
			}
			WWSETINTR(winvars);
		}
	} else if (n == 0) {
		winvars->wwnreadz++;
	} else {
		winvars->wwnreade++;
	}
}

void
wwsuspend(winvars_t *winvars)
{
	sig_t oldsig;

	oldsig = signal(SIGTSTP, SIG_IGN);
	wwend(0);
	(void) signal(SIGTSTP, SIG_DFL);
	(void) kill(0, SIGTSTP);
	(void) signal(SIGTSTP, SIG_IGN);
	wwstart(winvars);
	(void) signal(SIGTSTP, oldsig);
}
