/******************************************************************************
 * Rage 128 Chapter 6 sample code                                               *
 *                                                                            *
 * fixpat.c - This program uses the pattern registers to draw rectangles      *
 * on the screen.  An 8x8 monochrome pattern,                                 *
 *                                                                            *
 * Copyright (c) 1999 ATI Technologies Inc. All rights reserved.              *
 ******************************************************************************/

#include <stdio.h>
#include <i86.h>
#include <conio.h>
#include <stdlib.h>
#include "..\util\regdef.h"
#include "..\util\defines.h"
#include "..\util\main.h"

/******************************************************************************
 * Main Program to demonstrate pattern rectangle blts                         *
 *  Function: Three rectangles are drawn on screen using the pattern          *
 *            registers.  A monochrome 8x8 is used to draw a white and        *
 *            magenta patterned rectangle.                                    *
 *    Inputs: Arguments for mode spatial and colour resolution                *
 *   Outputs: NONE                                                            *
 ******************************************************************************/

void main (int argc, char *argv[])
{
    WORD width;                          // Width of drawing area.
    WORD height;                         // Height of drawing area.
    DWORD bppvalue;

    printf ("Rage 128 Chapter 4 sample code\n"
            "\n"
            "fixpat.c\n"
            "Pattern registers are used as a source to draw rectangles\n"
            "Three rectangles are drawn on screen using the pattern registers.  A\n"
            "monochrome 8x8 is used to draw a white and magenta patterned rectangle.\n"
            "\n"
            "Spatial resolution (640, 800, 1024, 1280, 1600) and Colour Depth\n"
            "(8, 15, 16, 24, 32) should be passed as arguments.\n"
            "Default setting is 640x480 spatial resolution and 8bpp pixel depth.\n");

    R128_StartUp (argc, argv);

    // clear screen.
    R128_ClearScreen (WHITE);

    // Set up values.
    width = R128_AdapterInfo.xres;
    height = R128_AdapterInfo.yres;

    // A rectangle is drawn, filled by a colour expanded 8x8 monochrome
    // fixed pattern via the brush data registers.
    // mono source       = mono 8x8 fixed pattern,
    // foreground source = foreground colour,
    // background source = background colour

    R128_WaitForFifo (3);

    // set DP_DATATYPE to satisfy the above conditions.
    bppvalue = R128_GetBPPValue (R128_AdapterInfo.bpp);
    regw (DP_DATATYPE, BRUSH_8x8MONO | SRC_DSTCOLOR | bppvalue);

    // This is an 8x8 monochrome pattern of a box with a border.
    regw (BRUSH_DATA0, 0x818181FF);
    regw (BRUSH_DATA1, 0xFF818181);

    // foreground colour is used when pattern bit is "1"
    // background colour is used when pattern bit is "0"
    R128_WaitForFifo (4);
    regw (DP_BRUSH_FRGD_CLR, R128_GetColourCode (WHITE));
    regw (DP_BRUSH_BKGD_CLR, R128_GetColourCode (LIGHTMAGENTA));

    // Draw a filled pattern rectangle.
    regw (DST_Y_X, 0);
    regw (DST_HEIGHT_WIDTH, height << 16 | width/3);

    // pause so we can see the result
    getch ();

    // Let's draw the second (middle) pattern.
    // The same pattern is used, but we use expand the pattern to the
    // foreground, and leave the background alone (0x1 - DP_BRUSH_DATATYPE)
    R128_WaitForFifo (5);

    // set DP_DATATYPE to satisfy the above conditions.
    regw (DP_DATATYPE, SRC_DSTCOLOR | BRUSH_8x8MONO_LBKGD | bppvalue);

    // Set the foreground colour.  The background will still be white
    regw (DP_BRUSH_FRGD_CLR, R128_GetColourCode (LIGHTBLUE));
    regw (DP_BRUSH_BKGD_CLR, R128_GetColourCode (BLACK));

    // Draw a filled pattern rectangle.
    regw (DST_Y_X, width/3 | 0 << 16);
    regw (DST_HEIGHT_WIDTH, (height << 16) | (width/3));

    // pause again to see the results
    getch ();

    // Let's draw the last (right hand third) pattern.
    // We will use a 32x1 mono pattern, expanding it to the foreground,
    // and leave the background alone.  (0x7 - DP_BRUSH_DATATYPE)
    R128_WaitForFifo (6);

    // set DP_DATATYPE to satisfy the above conditions.
    regw (DP_DATATYPE, SRC_DSTCOLOR | BRUSH_32x1MONO | bppvalue);
    regw (DP_BRUSH_FRGD_CLR, R128_GetColourCode (LIGHTRED));
    regw (DP_BRUSH_BKGD_CLR, R128_GetColourCode (YELLOW));
    regw (BRUSH_DATA0, 0x55FFAA00);

    // Draw a filled pattern rectangle.
    regw (DST_Y_X, (2 * (width/3)) | 0 << 16);
    regw (DST_HEIGHT_WIDTH, height << 16 | (width/3));

    // Pause to see the result
    getch ();

    // Batch command to restore old mode.
    R128_ShutDown ();

    exit (0);                           // No errors.
} // main

