/****************************************************************************
 * RAGE128 Chapter 6 Sample Code                                            *
 *                                                                          *
 * tritxlst.c - This program demonstrates how to draw a texture mapped      *
 * 3D triangle list using CCE command packets.                              *
 *                                                                          *
 * Copyright (c) 1999 ATI Technologies Inc.  All rights reserved.           *
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <i86.h>
#include <conio.h>
#include "..\..\util\regdef.h"
#include "..\..\util\defines.h"
#include "..\..\util\main.h"
#include "..\..\util\cce.h"
#include "..\util\cntx3d.h"
#include "..\util\prim.h"
#include "..\util\texture.h"

#define NUM_TRIS 30
#define TEX_FILE "..\\image\\brick.bmp"

char gErrMsg [64] = {0};

extern DWORD BPPOverride; // Override default 8bpp mode in Chap 3 lib.


/****************************************************************************
 * main                                                                     *
 *  Function: Main loop.                                                    *
 *    Inputs: none                                                          *
 *   Outputs: none                                                          *
 ****************************************************************************/

void main (int argc, char *argv[])
{
	int i, exitcode = 0;
    PRIMDATA trilist;
    float xstart, xend, ystart, triwidth, triheight;
    TEXINFO texinfo;

    printf ("Rage 128 Sample Code, Chapter 6.\n"
        "TRITXLST.EXE\n\n"
        "This program demonstrates how to draw a texture mapped triangle list using\n"
        "Rage 128 CCE command packets.\n\n"
        "Mode resolution (eg 640 480) and colour depth (15, 16, 32)\n"
        "should be passed as command line arguments.\n"
        "Default resolution and colour depth is 640x480, 32bpp.\n\n"
        "Press any key to continue.\n\n");

    getch ();

    // Inspect arguments for invalid color depth.

    for (i = 1; i < argc; i++)
    {
        if ((strcmp (argv[i], "8") == 0) || (strcmp (argv[i], "24") == 0))
		{
			printf ("\nThis program does not operate at 8 or 24 bpp.");
			printf ("\nProgram terminated.");
			exit (1);
		}
	}

    // Override default 8bpp mode in Chap 3 lib.

    BPPOverride = 32;

    // First, run StartUp function to set up the application.

    R128_StartUp (argc, argv);

    // Clear the screen

    R128_ClearScreen (DARKBLUE);

    // Initialize the CCE microengine.

    if (R128_CCEInit (CCE_MODE_192BM) != CCE_SUCCESS)
    {
        R128_ShutDown ();
        printf ("R128_CCEInit failed!!\n");
        exit (1);
    } // if

	// Set the default 3D context.

    R128_Init3DContext ();

    // Create a triangle list.

    trilist.verts = (LPTLVERTEX2) malloc (NUM_TRIS * 3 * sizeof (TLVERTEX2));
    if (trilist.verts == NULL)
    {
        sprintf (gErrMsg, "trilist malloc failed!!");
        exitcode = 1;
		goto exit_app;
    }
    trilist.numverts = NUM_TRIS * 3;

    // Compute triangle position and size paramters.

    xstart = R128_AdapterInfo.xres/20.0f;   
	xend = R128_AdapterInfo.xres - xstart;
    triwidth = (R128_AdapterInfo.xres - (2.0f * xstart))/((float)NUM_TRIS/3.0f);
    triheight = triwidth;
    ystart = (R128_AdapterInfo.yres - (3.0f * triheight))/2.0f;

    // Initialize the vertices.

    for (i=0; i < trilist.numverts; i+=3)
    {
        // Triangle vertex 0.

        trilist.verts[i].x = xstart;
        trilist.verts[i].y = ystart;
        trilist.verts[i].z = 0.5f;
        trilist.verts[i].rhw = 1.0f;
        trilist.verts[i].diffuse = 0x000000ff;
        trilist.verts[i].specular = 0x00000000;
        trilist.verts[i].s2 = trilist.verts[i].s1 = 0.0f;
        trilist.verts[i].t2 = trilist.verts[i].t1 = 1.0f;

        // Triangle vertex 1.

        trilist.verts[i+1].x = trilist.verts[i].x + triwidth;
        trilist.verts[i+1].y = ystart;
        trilist.verts[i+1].z = 0.5f;
        trilist.verts[i+1].rhw = 1.0f;
        trilist.verts[i+1].diffuse = 0x0000ff00;
        trilist.verts[i+1].specular = 0x00000000;
        trilist.verts[i+1].s2 = trilist.verts[i+1].s1 = 1.0f;
        trilist.verts[i+1].t2 = trilist.verts[i+1].t1 = 1.0f;

        // Triangle vertex 2.

        trilist.verts[i+2].x = trilist.verts[i].x;
        trilist.verts[i+2].y = ystart + triheight;
        trilist.verts[i+2].z = 0.5f;
        trilist.verts[i+2].rhw = 1.0f;
        trilist.verts[i+2].diffuse = 0x00ff0000;
        trilist.verts[i+2].specular = 0x00000000;
        trilist.verts[i+2].s2 = trilist.verts[i+2].s1 = 0.0f;
        trilist.verts[i+2].t2 = trilist.verts[i+2].t1 = 0.0f;

		// Increment xstart.

        xstart+=triwidth;

		// Reset xstart and ystart if at the end of the row.

		if (xstart >= xend)
		{
			xstart = R128_AdapterInfo.xres/20.0f;
			ystart += triheight;
		}
    }

	// Load the texture map.

	memset (&texinfo, 0, sizeof (texinfo));
    ATTEMPT (R128_LoadTextureBMP_ARGB8888 (TEX_FILE, &texinfo), 
        "LoadTextureBMP failed!!")

	// Set the texture.

    ATTEMPT (R128_SetTexture (0, &texinfo), 
        "SetTextureState failed!!")

	// Enable texture mapping.

    ATTEMPT (R128_SetTextureState (0, TEXSTATE_ENABLE, TRUE), 
        "enable texture failed!!")

	// Set the minification filtering mode.

    ATTEMPT (R128_SetTextureState (0, TEXSTATE_MINBLEND, 
		MINBLEND_BILINEAR), 
        "set min blend failed!!")
        
	// Set the magnification filtering mode.

    ATTEMPT (R128_SetTextureState (0, TEXSTATE_MAGBLEND, 
		MAGBLEND_BILINEAR), 
        "set mag blend failed!!")

    ATTEMPT (R128_SetTextureState (0, TEXSTATE_COMB_FNC, 
        TEXCOMBFNC_MODULATE), 
        "set comb fnc failed!!")

    // Draw the triangle list.

    ATTEMPT (R128_DrawTriangleList (&trilist),
        "DrawTriList failed!!")

	// Wait for key press to exit.

    getch ();

exit_app:

    // Free the vertex buffer.

	if (trilist.verts)
		free (trilist.verts);

	// Free the texture.

	if (texinfo.Offset[0])
		R128_FreeTexture (&texinfo);

    // Shut down the microengine.

	// Close 3D context, shut down micro-engine and restore original mode.

	R128_Close3DContext ();
    R128_CCEEnd (CCE_END_WAIT);
    R128_ShutDown ();

	// print error message if any.

	if (exitcode == 1)
		printf ("%s\n", gErrMsg);

	exit (exitcode);
} // main


