/****************************************************************************
 * Rage 128 Chapter 7 sample code                                           *
 *                                                                          *
 * ov1calch.c - This module calculates the horizontal scaling factors       *
 *              for the overlay/scaler.                                     *
 *                                                                          *
 * Copyright (c) 1999 ATI Technologies Inc. All rights reserved.            *
 ****************************************************************************/
#include "overlay.h"

void Calc_H_INC_STEP_BY (
    int fieldvalue_OV0_SURFACE_FORMAT,
    double H_scale_ratio,
    int DisallowFourTapVertFiltering,
    int DisallowFourTapUVVertFiltering,
    int AllowPreDownScale,
    DWORD *val_OV0_P1_H_INC,
    DWORD *val_OV0_P1_H_STEP_BY,
    DWORD *val_OV0_P23_H_INC,
    DWORD *val_OV0_P23_H_STEP_BY,
    int *P1GroupSize,
    int *P1StepSize,
    int *P23StepSize,
    DWORD *val_OV0_P1_PREDWNSC_RATIO,
    DWORD *val_OV0_P23_PREDWNSC_RATIO
)
{

    double ClocksNeededFor16Pixels;

    switch (fieldvalue_OV0_SURFACE_FORMAT)
    {
        case 3:
        case 4: /*16BPP (ARGB1555 and RGB565) */
            /* All colour components are fetched in pairs */
            *P1GroupSize = 2;
            /* We don't support four tap in this mode because G's are split between two bytes. In theory we could support it if */
            /* we saved part of the G when fetching the R, and then filter the G, followed by the B in the following cycles. */
            if (H_scale_ratio>=.5)
            {
                /* We are actually generating two pixels (but 3 colour components) per tick. Thus we don't have to skip */
                /* until we reach .5. P1 and P23 are the same. */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 1;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 1;
                *P1StepSize = 1;
                *P23StepSize = 1;
            }
            else if (H_scale_ratio>=.25)
            {
                /* Step by two */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 2;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 2;
                *P1StepSize = 2;
                *P23StepSize = 2;
            }
            else if (H_scale_ratio>=.125)
            {
                /* Step by four */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 3;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 3;
                *P1StepSize = 4;
                *P23StepSize = 4;
            }
            else if (H_scale_ratio>=.0625)
            {
                /* Step by eight */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 4;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 4;
                *P1StepSize = 8;
                *P23StepSize = 8;
            }
            else if (H_scale_ratio>=0.03125)
            {
                /* Step by sixteen */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 5;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 5;
                *P1StepSize = 16;
                *P23StepSize = 16;
            }
            else
            {
                H_scale_ratio=0.03125;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 5;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 5;
                *P1StepSize = 16;
                *P23StepSize = 16;
            }
            break;
        case 6: /*32BPP RGB */
            if (H_scale_ratio>=1.5 && !DisallowFourTapVertFiltering)
            {
                /* All colour components are fetched in pairs */
                *P1GroupSize = 2;
                /* With four tap filtering, we can generate two colour components every clock, or two pixels every three */
                /* clocks. This means that we will have four tap filtering when scaling 1.5 or more. */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 0;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 0;
                *P1StepSize = 1;
                *P23StepSize = 1;
            }
            else if (H_scale_ratio>=0.75)
            {
                /* Four G colour components are fetched at once */
                *P1GroupSize = 4;
                /* R and B colour components are fetched in pairs */
                /* With two tap filtering, we can generate four colour components every clock. */
                /* This means that we will have two tap filtering when scaling 1.0 or more. */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 1;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 1;
                *P1StepSize = 1;
                *P23StepSize = 1;
            }
            else if (H_scale_ratio>=0.375)
            {
                /* Step by two. */
                /* Four G colour components are fetched at once */
                *P1GroupSize = 4;
                /* R and B colour components are fetched in pairs */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 2;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 2;
                *P1StepSize = 2;
                *P23StepSize = 2;
            }
            else if (H_scale_ratio>=0.25)
            {
                /* Step by two. */
                /* Four G colour components are fetched at once */
                *P1GroupSize = 4;
                /* R and B colour components are fetched in pairs */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 2;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 3;
                *P1StepSize = 2;
                *P23StepSize = 4;
            }
            else if (H_scale_ratio>=0.1875)
            {
                /* Step by four */
                /* Four G colour components are fetched at once */
                *P1GroupSize = 4;
                /* R and B colour components are fetched in pairs */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 3;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 3;
                *P1StepSize = 4;
                *P23StepSize = 4;
            }
            else if (H_scale_ratio>=0.125)
            {
                /* Step by four */
                /* Four G colour components are fetched at once */
                *P1GroupSize = 4;
                /* R and B colour components are fetched in pairs */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 3;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 4;
                *P1StepSize = 4;
                *P23StepSize = 8;
            }
            else if (H_scale_ratio>=0.09375)
            {
                /* Step by eight */
                /* Four G colour components are fetched at once */
                *P1GroupSize = 4;
                /* R and B colour components are fetched in pairs */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 4;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 4;
                *P1StepSize = 8;
                *P23StepSize = 8;
            }
            else if (H_scale_ratio>=0.0625)
            {
                /* Step by eight */
                /* Four G colour components are fetched at once */
                *P1GroupSize = 4;
                /* R and B colour components are fetched in pairs */
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 5;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 5;
                *P1StepSize = 16;
                *P23StepSize = 16;
            }
            else
            {
                H_scale_ratio=0.0625;
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 5;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 5;
                *P1StepSize = 16;
                *P23StepSize = 16;
            }
            break;
        case 9:
            /*ToDo_Active: In mode 9 there is a possibility that HScale ratio may be set to an illegal value, so we have extra conditions in the if statement. For consistancy, these conditions be added to the other modes as well. */
            /* four tap on both (unless Y is too wide) */
            if ((H_scale_ratio>=(ClocksNeededFor16Pixels=8+2+2) / 16.0) &&
               ((WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5)<=0x3000) &&
               ((WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5)<=0x2000) &&
               !DisallowFourTapVertFiltering && !DisallowFourTapUVVertFiltering)
            {   /*0.75 */
                /* Colour components are fetched in pairs */
                *P1GroupSize = 2;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 0;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 0;
                *P1StepSize = 1;
                *P23StepSize = 1;
            }
            /* two tap on Y (because it is too big for four tap), four tap on UV */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=4+2+2) / 16.0) &&
                    ((WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5)<=0x2000) &&
                    DisallowFourTapVertFiltering && !DisallowFourTapUVVertFiltering)
            {   /*0.75 */
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 1;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 0;
                *P1StepSize = 1;
                *P23StepSize = 1;
            }
            /* We scale the Y with the four tap filters, but UV's are generated
               with dual two tap configuration. */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=8+1+1) / 16.0) &&
                    ((WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5)<=0x2000) &&
                    !DisallowFourTapVertFiltering)
            {   /*0.625 */
                *P1GroupSize = 2;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 0;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 1;
                *P1StepSize = 1;
                *P23StepSize = 1;
            }
            /* We scale the Y, U, and V with the two tap filters */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=4+1+1) / 16.0) &&
                    ((WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5)<=0x2000))
            {   /*0.375 */
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 1;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 1;
                *P1StepSize = 1;
                *P23StepSize = 1;
            }
            /* We scale step the U and V by two to allow more bandwidth for fetching Y's,
               thus we won't drop Y's yet. */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=4+.5+.5) / 16.0) &&
                    ((WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4*2)) * (1<<0xc) + 0.5)<=0x2000))
            {   /*>=0.3125 and >.333333~ */
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 1;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4*2)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 2;
                *P1StepSize = 1;
                *P23StepSize = 2;
            }
            /* We step the Y, U, and V by two. */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=2+.5+.5) / 16.0)  &&
                    ((WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4*2)) * (1<<0xc) + 0.5)<=0x2000))
            {
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 2;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4*2)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 2;
                *P1StepSize = 2;
                *P23StepSize = 2;
            }
            /* We step the Y by two and the U and V by four. */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=2+.25+.25) / 16.0) &&
                    ((WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4*4)) * (1<<0xc) + 0.5)<=0x2000))
            {
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 2;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 3;
                *P1StepSize = 2;
                *P23StepSize = 4;
            }
            /* We step the Y, U, and V by four. */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=1+.25+.25) / 16.0) &&
                    ((WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4*4)) * (1<<0xc) + 0.5)<=0x2000))
            {
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 3;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4*4)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 3;
                *P1StepSize = 4;
                *P23StepSize = 4;
            }
            /* We would like to step the Y by four and the U and V by eight, but we can't mix step by 3 and step by 4 for packed modes */

            /* We step the Y, U, and V by eight. */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=.5+.125+.125) / 16.0) &&
                    ((WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4*8)) * (1<<0xc) + 0.5)<=0x2000))
            {
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 4;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4*8)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 4;
                *P1StepSize = 8;
                *P23StepSize = 8;
            }
            /* We step the Y by eight and the U and V by sixteen. */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=.5+.0625+.0625) / 16.0) &&
            ((WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5)<=0x3000) &&
            ((WORD)((1/(H_scale_ratio*4*16)) * (1<<0xc) + 0.5)<=0x2000))
            {
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 4;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4*16)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 5;
                *P1StepSize = 8;
                *P23StepSize = 16;
            }
            /* We step the Y, U, and V by sixteen. */
            else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=.25+.0625+.0625) / 16.0) &&
                    ((WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5)<=0x3000) &&
                    ((WORD)((1/(H_scale_ratio*4*16)) * (1<<0xc) + 0.5)<=0x2000))
            {
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 5;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4*16)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 5;
                *P1StepSize = 16;
                *P23StepSize = 16;
            }
            else
            {
                H_scale_ratio=(ClocksNeededFor16Pixels=.25+.0625+.0625) / 16;
                *P1GroupSize = 4;
                *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                *val_OV0_P1_H_STEP_BY = 5;
                *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*4*16)) * (1<<0xc) + 0.5);
                *val_OV0_P23_H_STEP_BY = 5;
                *P1StepSize = 16;
                *P23StepSize = 16;
            }
            break;
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:    /* YUV12, VYUY422, YUYV422, YOverPkCRCB12, YWovenWithPkCRCB12 */
                /* We scale the Y, U, and V with the four tap filters */
                /* four tap on both (unless Y is too wide) */
                if ((H_scale_ratio>=(ClocksNeededFor16Pixels=8+4+4) / 16.0) &&
                    !DisallowFourTapVertFiltering && !DisallowFourTapUVVertFiltering)
                {       /*0.75 */
                    *P1GroupSize = 2;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 0;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 0;
                    *P1StepSize = 1;
                    *P23StepSize = 1;
                }
                /* two tap on Y (because it is too big for four tap), four tap on UV */
                else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=4+4+4) / 16.0) &&
                        DisallowFourTapVertFiltering && !DisallowFourTapUVVertFiltering)
                {   /*0.75 */
                    *P1GroupSize = 4;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 1;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 0;
                    *P1StepSize = 1;
                    *P23StepSize = 1;
                }
                /* We scale the Y with the four tap filters, but UV's are generated
                   with dual two tap configuration. */
                else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=8+2+2) / 16.0) &&
                          !DisallowFourTapVertFiltering)
                {   /*0.625 */
                    *P1GroupSize = 2;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 0;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 1;
                    *P1StepSize = 1;
                    *P23StepSize = 1;
                }
                /* We scale the Y, U, and V with the two tap filters */
                else if (H_scale_ratio>=(ClocksNeededFor16Pixels=4+2+2) / 16.0)
                {   /*0.375 */
                    *P1GroupSize = 4;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 1;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 1;
                    *P1StepSize = 1;
                    *P23StepSize = 1;
                }
                /* We scale step the U and V by two to allow more bandwidth for
                   fetching Y's, thus we won't drop Y's yet. */
                else if (H_scale_ratio>=(ClocksNeededFor16Pixels=4+1+1) / 16.0)
                {   /*0.312 */
                    *P1GroupSize = 4;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 1;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*2)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 2;
                    *P1StepSize = 1;
                    *P23StepSize = 2;
                }
                /* We step the Y, U, and V by two. */
                else if (H_scale_ratio>=(ClocksNeededFor16Pixels=2+1+1) / 16.0)
                {
                    *P1GroupSize = 4;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 2;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*2)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 2;
                    *P1StepSize = 2;
                    *P23StepSize = 2;
                }
                /* We step the Y by two and the U and V by four. */
                else if (H_scale_ratio>=(ClocksNeededFor16Pixels=2+.5+.5) / 16.0)
                {
                    *P1GroupSize = 4;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*2)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 2;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*4)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 3;
                    *P1StepSize = 2;
                    *P23StepSize = 4;
                }
                /* We step the Y, U, and V by four. */
                else if (H_scale_ratio>=(ClocksNeededFor16Pixels=1+.5+.5) / 16.0)
                {
                    *P1GroupSize = 4;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 3;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*4)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 3;
                    *P1StepSize = 4;
                    *P23StepSize = 4;
                }
                /* We step the Y by four and the U and V by eight. */
                else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=1+.25+.25) / 16.0) &&
                         (fieldvalue_OV0_SURFACE_FORMAT==10))
                {
                    *P1GroupSize = 4;
                    /* Can't mix step by 3 and step by 4 for packed modes */
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*4)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 3;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*8)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 4;
                    *P1StepSize = 4;
                    *P23StepSize = 8;
                }
                /* We step the Y, U, and V by eight. */
                else if (H_scale_ratio>=(ClocksNeededFor16Pixels=.5+.25+.25) / 16.0)
                {
                    *P1GroupSize = 4;
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 4;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*8)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 4;
                    *P1StepSize = 8;
                    *P23StepSize = 8;
                }
                /* We step the Y by eight and the U and V by sixteen. */
                else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=.5+.125+.125) / 16.0) && (fieldvalue_OV0_SURFACE_FORMAT==10))
                {
                    *P1GroupSize = 4;
                    /* Step by 5 not supported for packed modes */
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 4;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*16)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 5;
                    *P1StepSize = 8;
                    *P23StepSize = 16;
                }
                /* We step the Y, U, and V by sixteen. */
                else if ((H_scale_ratio>=(ClocksNeededFor16Pixels=.25+.125+.125) / 16.0) &&
                         (fieldvalue_OV0_SURFACE_FORMAT==10))
                {
                    *P1GroupSize = 4;
                    /* Step by 5 not supported for packed modes */
                    *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                    *val_OV0_P1_H_STEP_BY = 5;
                    *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*16)) * (1<<0xc) + 0.5);
                    *val_OV0_P23_H_STEP_BY = 5;
                    *P1StepSize = 16;
                    *P23StepSize = 16;
                }
                else
                {
                    if (fieldvalue_OV0_SURFACE_FORMAT==10)
                    {
                        H_scale_ratio=(ClocksNeededFor16Pixels=.25+.125+.125) / 16;
                        *P1GroupSize = 4;
                        *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*16)) * (1<<0xc) + 0.5);
                        *val_OV0_P1_H_STEP_BY = 5;
                        *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*16)) * (1<<0xc) + 0.5);
                        *val_OV0_P23_H_STEP_BY = 5;
                        *P1StepSize = 16;
                        *P23StepSize = 16;
                    }
                    else
                    {
                        H_scale_ratio=(ClocksNeededFor16Pixels=.5+.25+.25) / 16;
                        *P1GroupSize = 4;
                        *val_OV0_P1_H_INC = (WORD)((1/(H_scale_ratio*8)) * (1<<0xc) + 0.5);
                        *val_OV0_P1_H_STEP_BY = 4;
                        *val_OV0_P23_H_INC = (WORD)((1/(H_scale_ratio*2*8)) * (1<<0xc) + 0.5);
                        *val_OV0_P23_H_STEP_BY = 4;
                        *P1StepSize = 8;
                        *P23StepSize = 8;
                    }
                }
                break;
        default:    break;

    }

}

