/****************************************************************************
 * OVERLAY.H                                                                *
 *                                                                          *
 * Defines and constants for the Rage 128 overlay/scaler.                   *
 *                                                                          *
 * Copyright (c) 1999 ATI Technologies Inc.  All rights reserved            *
 ****************************************************************************/
/* define OVERLAY_H for other files that include this file */
#ifndef OVERLAY_H
#define OVERLAY_H

#include <stdio.h>
#include <stdlib.h>
#include "main.h"
#include "regdef.h"
#include "defines.h"

/* -------------------------------------------------------------------------- */
/* OVERLAY general constants */
/* -------------------------------------------------------------------------- */

/* OV0_REG_LOAD_CNTL register constants */
#define OV0_LOCK                          0x1L        /* R128 New */

/* OV0_SCALE_CNTL register constants */
#define OV0_DOUBLE_BUFFER_REGS            0x1000000L  /* R128 New */
#define OV0_OVERLAY_EN                     0x40000000L /* R128 Eqv */
#define OV0_SOFT_RESET                    0x80000000L

#define SCALER_SOURCE_15BPP     3
#define SCALER_SOURCE_16BPP     4
#define SCALER_SOURCE_32BPP     6
#define SCALER_SOURCE_YUV9      9
#define SCALER_SOURCE_YUV12     10
#define SCALER_SOURCE_VYUY422   11
#define SCALER_SOURCE_YVYU422   12

/*--------------------------------------------------------------------------
                            Structure definitions
 --------------------------------------------------------------------------*/

typedef struct OverlayDescriptor {

    DWORD UPDATE_OVERLAY_LOCK;      /* Direct Draw (the UpdateOverlay function for example), will set this bit to prevent */
                                                        /* the display call back function from modifying the overlay at the same time. */
    DWORD DISPLAY_CALLBACK_LOCK;    /* The display callback function, will set this bit to prevent Direct Draw from */
                                                        /* modifying the overlay at the same time. */

    /* Bandwidth and other display mode info */
    /* ************************************* */
    /* These values that must be provided to determine available bandwidth (possibly obtained via CWDDE which queries the BIOS) */
    DWORD HTOTAL_InCharactersMinusOne;
    DWORD HDISP_InCharactersMinusOne;
    DWORD PrimaryPixelDepthInBytes;
    DWORD DisplayFIFOSizeInBytes;   /* On the hairy edge, we may have to have control of this to get saler support in every mode display possible. */
    DWORD CRTCInterlaceEnable;      /* Describes the CRT's display mode (1 is interlaced) */
    DWORD PixelClock_VCLK_PeriodInPicoSec;
    DWORD MemoryClock_XCLK_PeriodInPicoSec;
    DWORD ECP_DIV_ValueFromPLLRegs;         /* the PLL register field "ECP_DIV" which the bios should set. */
                                /* If 0 then ECP = VCLK */
                                /* If 1 then ECP = VCLK/2 and the scaler will replicate pixels horizontally. (occurs when pixel clock is from 125 to 250 MHz) */
                                /* If 2 then ECP = VCLK/4 and the scaler will replicate pixels horizontally. (occurs when pixel clock exceeds 250 MHz - which currently it doesn't) */
                                /* Important Note: When ECP = VCLK/2 then in some modes the maximum horizontal downscale ratio is half that when ECP = VCLK!! */
    DWORD MEM_CHECKBOARD;               /* Need to read this after BIOS sets up the chip. It is used to determine how tiled surfaces are stored in the frame buffer. */
    DWORD MissPenalty;
    DWORD CyclesPerOctword;
    DWORD MissAllignmentPenalty;
    DWORD OpenPenalty;                  /* Memory access timing information that we should perhaps obtain from the BIOS rather than hardcode??? */
    DWORD WriteClosePenalty;
    DWORD ReadClosePenalty;
    DWORD BankChangePenalty;
    DWORD RefreshCycleTime;

    DWORD DISP_LINES_ARE_64BYTE_ALLIGNED;
    DWORD CAN_GUARANTEE_NO_SUBPICTURE;
    /* Only ATI's MPEG player will use the SUBPICTURE, thus this should typically */
    /* be set to one. However, the BIOS or drivers should prevent the Subpicture */
    /* from being turned on if the overlay was initialized. */

    DWORD CAN_GUARANTEE_NO_VIDEO_CAPTURE;
    DWORD CAN_GUARANTEE_ONLY_ONE_VIDEO_CAPTURE;
    DWORD CAN_GUARANTEE_THIS_IS_RAGE128_REV_A;

    /* Bandwidth Calculation and Result Interpretation */
    /* *********************************************** */
    DWORD BandwidthDirective;
        /* 1 = BEST_EFFORT (Assume bandwidth info is good and do the best scaling possible with available bandwidth) */
        /* 2 = DETERMINE_HBLANK_NEEDED (Given the scaling ratios, MAX_LN_IN_PER_LN_OUT, and other info provided, work out the minimum CRTCHTotalInPixels needed) */
    DWORD NeededHTotal;
        /* If you set BandwidthDirective to DETERMINE_HBLANK_NEEDED then the scaler determines how large an HBlank is needed and returns "NeededHTotal". */
        /* It sets up the scaler assuming that the HTotal will be adjusted to meet it's needs. The DETERMINE_HBLANK_NEEDED directive should be used for testing only. */
    /* Basic Scaling Settings */
    /* *********************************************** */
    DWORD OVERLAY_ENABLE;           /* Turns on the scaler. (Will also power up the scaler and subpicture clock.) */
    DWORD DOUBLE_BUFFER_REGS;       /* Should always set to one except for test. */

    DWORD SURFACE_FORMAT;

    DWORD SrcHeight;
    DWORD SrcWidth;

    /* Out of bounds areas (negative values or values that exceed SrcHeight and SrcWidth) will be */
    /* replaced with black by hardware if possible. This can be used for DVD letterboxing for example. */

    /* Below are the physical window parameters. If values are derived from DDHAL_UPDATEOVERLAYDATA->rSrc->?????, then */
    /* they must be converted to double precision values (for SrcRight and SrcBottom add one before casting), then */
    /* the effects of a virtual desktop must be factored in, and then */
    /* the PHYSICAL_Src parameters can be programmed. If OverlayDescriptor->USE_DBL_PRECISION_SRC_WIN is set then the values */
    /* must be derived from DDHAL_UPDATEOVERLAYDATA->lpDDSrcSurface->dwReserved1->DBL_PRECISION_Src????, then the effects of */
    /* a virtual desktop must be factored in, and then the PHYSICAL_Src parameters can be programmed. */
    double PHYSICAL_SrcLeft;
    double PHYSICAL_SrcRight;
    double PHYSICAL_SrcTop;
    double PHYSICAL_SrcBottom;

    /* Indicates the surface location on the screen for both video and subpicture */
    DWORD PHYSICAL_DestLeft;
    DWORD PHYSICAL_DestRight;
    DWORD PHYSICAL_DestTop;
    DWORD PHYSICAL_DestBottom;

    /* ****************************** */
    /* If an application wishes to use a non-Microsoft standard to define a planer surface, it can set the USE_DETAILED_SURFACE_DESCRIPTION */
    /* flag. Otherwise, SetupOverlay will derive the detailed parameters from the "Simple..." parameters and the SURFACE_FORMAT based on */
    /* Microsoft's standard. The detailed parameters will be overwritten. */

    DWORD SurfaceIsTiled;

    DWORD SimpleBaseOffsetOfSurface;
    DWORD SimpleSurfaceTiled;
    DWORD SimpleSurfacePitch;

    DWORD USE_DETAILED_SURFACE_DESCRIPTION;
    /* These parameters represent the address of the top left corner of the surface in memory. The SetupOverlay function will use these values */
    /* and the top left corner of the source window to calculate to the Octword address that is programmed into hardware. */
    /* Note that {0, 1, 2} or {3, 4, 5} are used for {Y, U, V} planes of planer modes. */
    DWORD BaseOffsetOfSurface[6];           /* In Bytes (If the surface is tiled, must point to the top left tile. This tile should not be in a checkerboarded row of tiles) */
    DWORD SurfaceSpecificXOffset[6];        /* In Bytes (If the surface is not tiled, set to zero) */
    DWORD SurfaceSpecificYOffset[6];        /* In Lines (If the surface is not tiled, set to zero) */
    DWORD PitchSelectOfSurface[6];
    DWORD TiledFlagOfSurface[6];

    DWORD SurfacePitch[2];

    /* The register fields that relate to SurfaceBaseAddresses are not double buffered as this was deemed too expensive */
    /* in hardware. Normally, these registers are simply updated immediately, and there are no artifacts. However, in */
    /* certain situations, such as zooming, there can be artifacts if the SurfaceBaseAddresses registers are not updated */
    /* synchronously with other double buffered registers (e.g. scale ratio registers). This is not a problem if the video is */
    /* provided by a software application. The application can avoid updating the SurfaceBaseAddress that is currently in use. */
    /* To make the update take effect cleanly, the a software application must submit a field/frame in the same */
    /* "UPDATE_OVERLAY" call that changes the zoom window. In this case an UPDATE_OVERLAY call will replace the simpler "FLIP" */
    /* call. Because a software application is providing the video, it knows which SurfaceBaseAddress is in use. */
    /* Note: If the video is provided by video capture, it is hard to be sure which buffer is active. Even if a register could */
    /* be read to find out, the information can become out of date before actions based in that info are complete. If a */
    /* seamless zoom for video capture is needed, then the application will have to manage the timing via interrupts or polling. */

    /* ************************ */
    /* Scaling Quality Controls */
    /* ************************ */
    DWORD SharpnessEffect;
        /* A nominal value of ???? will result in the most accurate image resampling according to sampling theory. However, image quality is */
        /* subjectively evaluated, and sometimes the degree of sharpness depends on the type of video content. This parameter provides the */
        /* user with the freedom to further sharpen or dull the image. */


    /* ************************ */
    /* Extended Features */
    /* ************************ */

    /* Double precision window for MPEG pan and scan */
    /* ************************ */

    /* To properly support MPEG's pan and scan feature, you must set the USE_DBL_PRECISION_SRC_WIN flag, and use the */
    /* DBL_PRECISION_Src... parameters instead of the UNSIGNED_INT_... parameters. */
    DWORD USE_DBL_PRECISION_SRC_WIN;
    double DBL_PRECISION_SrcLeft;
    double DBL_PRECISION_SrcRight;
    double DBL_PRECISION_SrcTop;
    double DBL_PRECISION_SrcBottom;

    /* Scaling Quality Overrides */
    /* ************************ */

        /* The scaler setup routine will automatically determine the best settings for a given display mode, and set of scaling parameters. */
        /* However, for test of demo purposes, you can force the scaler to make other choices. */
    DWORD P1_MAX_LN_IN_PER_LN_OUT;
    DWORD P23_MAX_LN_IN_PER_LN_OUT;     /* Used only for U and V planes in planer modes */
        /* Describes how many lines you are willing to allow the scaler to fetch per line out. */
        /* More lines means better downscaling, but also more bandwidth! If you set the BandwidthDirective */
        /* to BEST_EFFORT, then the scaler will use as many lines as possible, but it will not exceed P??_MAX_LN_IN_PER_LN_OUT. */
        /* If you can provide correct bandwidth info, then set these values to the max unless you want to reduce the scaling quality. */
    DWORD LOG_V_DECIMATION_FACT;
    DWORD TWO_TAP_LINEAR_HORZ_FILT;
    DWORD TWO_TAP_LINEAR_VERT_FILT;
    DWORD HORZ_PICK_NEAREST;            /* Use 0 unless source keying is needed */
    DWORD VERT_PICK_NEAREST;            /* Use 0 unless source keying is needed */
    DWORD HC_COEF_ON_HORZ_Y;
    DWORD HC_COEF_ON_HORZ_UV;
    DWORD HC_COEF_ON_VERT_Y;
    DWORD HC_COEF_ON_VERT_UV;
    DWORD EMULATE_RAGE_PRO;
        /* Will set P??_MAX_LN_IN_PER_LN_OUT to 1, */
        /* force two tap vertical filtering, */
        /* and drop pixels more aggressively when downscaling horizontally. */
    DWORD EMULATE_RIVA128;
        /* Will set P??_MAX_LN_IN_PER_LN_OUT to 1, */
        /* force TWO_TAP_LINEAR_HORZ_FILT and VERT_PICK_NEAREST filtering, */
        /* and drop pixels very aggressively when downscaling horizontally. */

    DWORD AllowPreDownScale;

    /* Deinterlacing Control */
    /* ********************* */
        /* Note: The selection of bob or weave affects the vertical scaling done by hardware, recommend that deinterlacing and scaling */
        /* setup NOT be separate routines. */
    DWORD AutoFlipVideoProvider;        /* VidPort0=0, VidPort1=1, Software=2 */
    DWORD IGNORE_REPEAT_FIELD;

    /* Note: The following three registers may be hard coded in a future cost reduce revision of the Rage128 */
    DWORD DeinterlacePattern;                   /* Up to 10 two bit values (max 20 bit value) */
                                                        /* Hardware remembers where the last three fields are. */
                                                        /* Each two bit value indicates what should be done with them. */
                                                        /* Each time a new field is delivered, the next two bit value is used. */
    DWORD DeinterlacePatternLenMinusOne;   /* Number from 0 to 9 */
    DWORD DeinterlacePatternPointer;       /* Number from 0 to 9 */

    DWORD ShiftFieldDown;           /* 0 doesn't shift either field */
                                                    /* 1 to position even field below odd. */
                                                    /* 2 to position odd field below even. */
                                                    /* 3 shifts both fields down (not useful) */
    DWORD FIRST_LINE_EVEN;      /* When weaving, indicates that the first source line (pointed to by a base address register) */
                                                    /* Comes from an even field. */

    /* Submit Field */
    /* **************************************** */
        /* Software applications indicate to hardware that there is a new field for display using these parameters. */
        /* The plan is to have a separate call for this */
    DWORD There_Is_A_New_Field;      /* Tells the routine that a new field of source has been provided by software. */
    DWORD SurfaceBaseAddressNum;                /* Tells which "SurfaceBaseAddress" points to the new field of source. */
                                        /* If Planer, selects Y base. U is Y+1, V is Y+2, Y must be 0 or 4. */
    DWORD FieldIsOdd;               /* Indicates that the new field of source is an odd field */
    DWORD RepeatField;
        /* Indicates a repeated field in an NTSC 3:2 Pull Down sequence of fields. */
        /* More generically, it indicates the beginning of a repeating pattern of even/odd fields so that the weaver can be aligned. */
    /* **************************************** */

    /* Colour and Effect Controls */
    /* ************************** */
    DWORD GAMMA_SEL;
    DWORD Y2R_TEMP;                         /* Always use ? */
    DWORD INT_EMU;                          /* Interlace emulation */
    DWORD BRIGHTNESS;
    DWORD SATURATION_U;
    DWORD SATURATION_V;
    DWORD PIX_EXPAND;
        /* Enables dynamic range expansion of RGB1555 and RGB565. (e.g. makes packed 1f,3f,1f value into ff,ff,ff instead of f8,fc,f8). Always use 1. */
    DWORD SIGNED_UV1;                   /* Changes interpretation of UV values in source image. PC's use 0, Apple's use 1 */
    DWORD COMCORE_SHIFT_UP_ONE;
        /* Always use 0. Do not expose to any API. This feature is obsolete and we want to remove it in future chips. */
        /* Notify Phil Swan if anyone still needs this feature. */
    DWORD DIS_LIMIT;                        /* Disables colour range limiters for test */
    DWORD SCALE_Y2R_DIS;                    /* Disables colour conversion for test */

    /* Keying Controls */
    /* ************************** */
    DWORD VIDEO_KEY_CLR;
    DWORD VIDEO_KEY_MSK;
    DWORD VIDEO_KEY_FN;

    DWORD GRAPHICS_KEY_CLR;
    DWORD GRAPHICS_KEY_MSK;
    DWORD GRAPHICS_KEY_FN;

   DWORD COMPARE_MIX;

} OVERLAY_DESCRIPTOR;

typedef struct OverlayRegFields {

    DWORD val_OVERLAY_X_START;
    DWORD val_OVERLAY_X_END;
    DWORD val_OVERLAY_Y_START;
    DWORD val_OVERLAY_Y_END;
    DWORD val_LOCK;
    DWORD val_STALL_GUI_UNTIL_FLIP;
    DWORD val_H_LOAD_CMP;

    /* Added signals for predownscaling */
    DWORD val_OV0_P1_PREDWNSC_RATIO;
    DWORD val_OV0_P23_PREDWNSC_RATIO;

    DWORD val_OV0_PIX_EXPAND;
    DWORD val_OV0_Y2R_TEMP;
    DWORD val_OV0_HORZ_PICK_NEAREST;
    DWORD val_OV0_VERT_PICK_NEAREST;
    DWORD val_OV0_SIGNED_UV;
    DWORD val_OV0_GAMMA_SEL;
    DWORD val_OV0_COMCORE_SHIFT_UP_ONE;
    DWORD val_OV0_SURFACE_FORMAT;
    DWORD val_OV0_SMART_SWITCH;
    DWORD val_OV0_BURST_PER_PLANE;
    DWORD val_OV0_DOUBLE_BUFFER_REGS;
    DWORD val_OV0_DIS_LIMIT;
    DWORD val_OV0_PROGMBL_LOAD_START;
    DWORD val_OV0_INT_EMU;
    DWORD val_OV0_SOFT_RESET;
    DWORD val_OV0_V_INC;
    DWORD val_OV0_P1_MAX_LN_IN_PER_LN_OUT;
    DWORD val_OV0_P1_V_ACCUM_INIT;
    DWORD val_OV0_P23_MAX_LN_IN_PER_LN_OUT;
    DWORD val_OV0_P23_V_ACCUM_INIT;
    DWORD val_OV0_P1_BLNK_LN_AT_TOP_M1;
    DWORD val_OV0_P1_ACTIVE_LINES_M1;
    DWORD val_OV0_P23_BLNK_LN_AT_TOP_M1;
    DWORD val_OV0_P23_ACTIVE_LINES_M1;
    DWORD val_OV0_VID_BUF0_PITCH_SEL;
    DWORD val_OV0_VID_BUF0_TILED_ADRS;
    DWORD val_OV0_VID_BUF0_BASE_ADRS;
    DWORD val_OV0_BUF0_1ST_LINE_LSBS;
    DWORD val_OV0_VID_BUF1_PITCH_SEL;
    DWORD val_OV0_VID_BUF1_TILED_ADRS;
    DWORD val_OV0_VID_BUF1_BASE_ADRS;
    DWORD val_OV0_BUF1_1ST_LINE_LSBS;
    DWORD val_OV0_VID_BUF2_PITCH_SEL;
    DWORD val_OV0_VID_BUF2_TILED_ADRS;
    DWORD val_OV0_VID_BUF2_BASE_ADRS;
    DWORD val_OV0_BUF2_1ST_LINE_LSBS;
    DWORD val_OV0_VID_BUF3_PITCH_SEL;
    DWORD val_OV0_VID_BUF3_TILED_ADRS;
    DWORD val_OV0_VID_BUF3_BASE_ADRS;
    DWORD val_OV0_BUF3_1ST_LINE_LSBS;
    DWORD val_OV0_VID_BUF4_PITCH_SEL;
    DWORD val_OV0_VID_BUF4_TILED_ADRS;
    DWORD val_OV0_VID_BUF4_BASE_ADRS;
    DWORD val_OV0_BUF4_1ST_LINE_LSBS;
    DWORD val_OV0_VID_BUF5_PITCH_SEL;
    DWORD val_OV0_VID_BUF5_TILED_ADRS;
    DWORD val_OV0_VID_BUF5_BASE_ADRS;
    DWORD val_OV0_BUF5_1ST_LINE_LSBS;
    DWORD val_OV0_PITCH0_VALUE;
    DWORD val_OV0_PITCH0_SKIP_LINES;
    DWORD val_OV0_PITCH0_IN_TILES_LSBS;
    DWORD val_OV0_PITCH1_VALUE;
    DWORD val_OV0_PITCH1_SKIP_LINES;
    DWORD val_OV0_PITCH1_IN_TILES_LSBS;
    DWORD val_OV0_SOFT_BUF_NUM;
    DWORD val_OV0_SOFT_REPEAT_FIELD;
    DWORD val_OV0_SOFT_BUF_ODD;
    DWORD val_OV0_IGNORE_REPEAT_FIELD;
    DWORD val_OV0_SOFT_EOF_TOGGLE;
    DWORD val_OV0_VID_PORT_SELECT;
    DWORD val_OV0_P1_FIRST_LINE_EVEN;
    DWORD val_OV0_SHIFT_EVEN_DOWN;
    DWORD val_OV0_SHIFT_ODD_DOWN;
    DWORD val_OV0_DEINT_PAT;
    DWORD val_OV0_DEINT_PAT_LEN_M1;
    DWORD val_OV0_P1_H_INC;
    DWORD val_OV0_P23_H_INC;
    DWORD val_OV0_P1_H_STEP_BY;
    DWORD val_OV0_P23_H_STEP_BY;
    DWORD val_OV0_P1_H_ACCUM_INIT;
    DWORD val_OV0_PRESHIFT_P1_TO;
    DWORD val_OV0_P23_H_ACCUM_INIT;
    DWORD val_OV0_PRESHIFT_P23_TO;
    DWORD val_OV0_P1_X_END;
    DWORD val_OV0_P1_X_START;
    DWORD val_OV0_P2_X_END;
    DWORD val_OV0_P2_X_START;
    DWORD val_OV0_P3_X_END;
    DWORD val_OV0_P3_X_START;
    DWORD val_OV0_HC_COEF_ON_HORZ_Y;
    DWORD val_OV0_HC_COEF_ON_HORZ_UV;
    DWORD val_OV0_HC_COEF_ON_VERT_Y;
    DWORD val_OV0_HC_COEF_ON_VERT_UV;
    DWORD val_OV0_COEF_0_0TH_TAP;
    DWORD val_OV0_COEF_0_1ST_TAP;
    DWORD val_OV0_COEF_0_2ND_TAP;
    DWORD val_OV0_COEF_0_3RD_TAP;
    DWORD val_OV0_COEF_1_0TH_TAP;
    DWORD val_OV0_COEF_1_1ST_TAP;
    DWORD val_OV0_COEF_1_2ND_TAP;
    DWORD val_OV0_COEF_1_3RD_TAP;
    DWORD val_OV0_COEF_2_0TH_TAP;
    DWORD val_OV0_COEF_2_1ST_TAP;
    DWORD val_OV0_COEF_2_2ND_TAP;
    DWORD val_OV0_COEF_2_3RD_TAP;
    DWORD val_OV0_COEF_3_0TH_TAP;
    DWORD val_OV0_COEF_3_1ST_TAP;
    DWORD val_OV0_COEF_3_2ND_TAP;
    DWORD val_OV0_COEF_3_3RD_TAP;
    DWORD val_OV0_COEF_4_0TH_TAP;
    DWORD val_OV0_COEF_4_1ST_TAP;
    DWORD val_OV0_COEF_4_2ND_TAP;
    DWORD val_OV0_COEF_4_3RD_TAP;
    DWORD val_OV0_BRIGHTNESS;
    DWORD val_OV0_SATURATION_U;
    DWORD val_OV0_SATURATION_V;
    DWORD val_OV0_VIDEO_KEY_CLR;
    DWORD val_OV0_VIDEO_KEY_MSK;
    DWORD val_OV0_VIDEO_KEY_FN;
    DWORD val_OV0_SCALE_Y2R_DIS;
    DWORD val_OV0_SUBPIC_ONLY;
    DWORD val_OV0_EXTSENSE;
    DWORD val_OV0_SWAP_UV;

    DWORD val_OV0_GRAPHICS_KEY_CLR;
    DWORD val_OV0_GRAPHICS_KEY_MSK;
    DWORD val_OV0_GRAPHICS_KEY_FN;

   DWORD val_OV0_COMPARE_MIX;

} OVERLAY_REG_FIELDS;

typedef struct othfield {
    DWORD PIX1_M;
    DWORD PIX1_N;
    DWORD PIX1_P;
    DWORD CTotal_One;
    DWORD HDisp_One;
    DWORD VDisp_One;
    DWORD HTotal_One;
    DWORD VTotal_One;
    DWORD INTERLACE_EN;
} OTHFIELD;

typedef struct Surface {

    DWORD ImageWidth;
    DWORD ImageHeight;
    DWORD SurfaceFormat;
    DWORD TiledSurface;

    /* Used for all surface types */
    DWORD Base;
    DWORD Pitch;
    DWORD Height;

    /* The following are only used for the V and U planes planer surface types */
    DWORD UBase;
    DWORD UPitch;
    DWORD UHeight;
    DWORD VBase;
    DWORD VPitch;
    DWORD VHeight;

} SURFACE;

/* -------------------------------------------------------------------------- */
/*                       Function Prototypes */
/*                  New overlay/scaler driver ov1 */
/* -------------------------------------------------------------------------- */

void R128_InitOverlayDescriptor (OVERLAY_DESCRIPTOR *OverlayDescriptor);

void R128_GetBandwidthInfo (OVERLAY_DESCRIPTOR *OverlayDescriptor);

void R128_SetupOverlay (OVERLAY_DESCRIPTOR *OverlayDescriptor, OVERLAY_REG_FIELDS *OverlayRegFields);

void Calc_H_INC_STEP_BY(
    int fieldvalue_OV0_SURFACE_FORMAT,
    double H_scale_ratio,
    int DisallowFourTapVertFiltering,
    int DisallowFourTapUVVertFiltering,
    int AllowPreDownScale,
    DWORD *val_OV0_P1_H_INC,
    DWORD *val_OV0_P1_H_STEP_BY,
    DWORD *val_OV0_P23_H_INC,
    DWORD *val_OV0_P23_H_STEP_BY,
    int *P1GroupSize,
    int *P1StepSize,
    int *P23StepSize,
    DWORD *val_OV0_P1_PREDWNSC_RATIO,
    DWORD *val_OV0_P23_PREDWNSC_RATIO
);

void CalcScalerHBlank (
    DWORD SURFACE_FORMAT,
    DWORD rDestRight,
    DWORD rDestLeft,
    DWORD PrimaryPixelDepthInBytes,
    DWORD HTOTAL_InCharactersMinusOne,

    DWORD OV0_P1_X_END,
    DWORD OV0_P2_X_END,

    double XCLKPeriodInNSec,
    double VCLKPeriodInNSec,
    double ECPPeriodInNSec,
    DWORD ECP_DivideDownFactor,

    DWORD *EarliestDataTransfer,
    DWORD *LatestDataTransfer,
    DWORD *VCLK_Offset
);

void R128_FilterSetup (
    /* submitted parameters */
    OVERLAY_DESCRIPTOR *OverlayDescriptor,
    /* returned field values to program into registers */
    OVERLAY_REG_FIELDS *OverlayRegFields
);

void CreateSurface (DWORD Width, DWORD Height, DWORD ScalerSourceFormat,
            DWORD Tiled, DWORD mem_offset, SURFACE* surface);

void R128_SetOverlayDefaults (
    DWORD SrcFormat,
    DWORD SrcWidth,
    DWORD SrcHeight,
    double H_scale_ratio,
    double V_scale_ratio,
    OVERLAY_DESCRIPTOR *OverlayDescriptor
);

void R128_WriteOverlayFields (OVERLAY_REG_FIELDS *OverlayRegFields);

void R128_SetOverlay (WORD x, WORD y, WORD width, WORD height);
void R128_DisableOverlay (void);
void R128_EnableOverlay (void);

#endif // OVERLAY_H
