/*	$NetBSD$	*/

/*
 * 98625A/B HPIB driver
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD$");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/callout.h>
#include <sys/device.h>
#include <sys/kernel.h>
#include <sys/malloc.h>

#include <machine/bus.h>

#include <dev/gpib/gpibvar.h>

#include <hp300/dev/diovar.h>
#include <hp300/dev/diodevs.h>
#include <hp300/dev/dmavar.h>
#include <hp300/dev/fhpibreg.h>

#ifdef DEBUG
int fhpibdebug = 0x00;
int dopriodma = 0;	/* use high priority DMA */
int fhpibppolldelay = 50;
#define FDB_FOLLOW	0x01
#define FDB_INTR	0x02
#define FDB_FAIL	0x04
#define FDB_DMA		0x08
#define FDB_WAIT	0x10
#define FDB_PPOLL	0x20
#define FDB_REPORTTIME	0x40
#define DPRINTF(mask, str)	if (fhpibdebug & (mask)) printf str
#else
#define DPRINTF(mask, str)	/* nothing */
#endif

#define CHIP_READ(sc, reg)						\
	bus_space_read_1((sc)->sc_bst, (sc)->sc_chip_bsh, (reg))

#define CHIP_WRITE(sc, reg, val)					\
	bus_space_write_1((sc)->sc_bst, (sc)->sc_chip_bsh, (reg), (val))

struct fhpib_softc {
	struct device sc_dev;		/* generic device glue */

	bus_space_tag_t sc_bst;
	bus_space_handle_t sc_bsh;
	bus_space_handle_t sc_chip_bsh;
	struct dmaqueue *sc_dq;

	int sc_myaddr;			/* my address */
	struct gpib_softc *sc_gpib;

	volatile int sc_flags;
#define	HPIBF_IO	0x1
#define	HPIBF_DONE	0x2
#define	HPIBF_PPOLL	0x4
#define	HPIBF_READ	0x8
#define	HPIBF_TIMO	0x10
#define	HPIBF_DMA16	0x8000
	int sc_cmd;

	/* record of the current pending dma xfer */
	/* XXX probably recorded in the new dma_segment */
	struct {
		int active;
		int slave;
		int sec;
		u_int8_t *addr;
		int count;
		int dir;
		int timo;
		int curcnt;
	} sc_dmastate;

	int sc_ppoll_slave;		/* XXX stash our ppoll address */
	struct callout sc_timeout_ch;
};

int	fhpibmatch(struct device *, struct cfdata *, void *);
void	fhpibattach(struct device *, struct device *, void *);

struct cfattach fhpib_ca = {
	sizeof(struct fhpib_softc), fhpibmatch, fhpibattach
};

void	fhpibreset(void *);
int	fhpibsend(void *, int, int, void *, int);
int	fhpibrecv(void *, int, int, void *, int);
int	fhpibpptest(void *, int);
void	fhpibppwatch(void *, int);
void	fhpibppclear(void *);
void	fhpibxfer(void *, int, int, void *, int, int, int);
void	fhpibgo(void *);
void	fhpibdone(void *);
int	fhpibintr(void *);

int	fhpibsendcmds(void *, void *, int);
int	fhpibsenddata(void *, void *, int);
int	fhpibrecvdata(void *, void *, int);
int	fhpibgts(void *);
int	fhpibtc(void *, int);

static void	fhpib_init(struct fhpib_softc *);
static int	fhpib_setaddress(void *, int, int);
static void	fhpibifc(struct fhpib_softc *);
static void	fhpibtimeout(void *);
static int	fhpibwait(struct fhpib_softc *, int);

/*
 * Our chipset structure.
 */
struct	gpib_chipset_tag fhpib_ic = {
	NULL,
	fhpibreset,
	fhpibsend,
	fhpibrecv,
	fhpibpptest,
	fhpibppwatch,
	fhpibppclear,
	fhpibxfer,
};

int fhpibdmathresh = 3;		/* byte count beyond which to attempt dma */
int fhpibwtimeout = 100000;	/* # of status tests before we give up */

int
fhpibmatch(parent, match, aux)
	struct device *parent;
	struct cfdata *match;
	void *aux;
{
	struct dio_attach_args *da = aux;

	if (da->da_id == DIO_DEVICE_ID_FHPIB)
		return (1);

	return (0);
}

void
fhpibattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct fhpib_softc *sc = (struct fhpib_softc *)self;
	struct dio_attach_args *da = aux;
	struct gpibdev_attach_args ga;
	u_int16_t ids;

	sc->sc_bst = da->da_bst;
	if (bus_space_map(sc->sc_bst, da->da_addr, da->da_size,
	    BUS_SPACE_MAP_LINEAR, &sc->sc_bsh)) {
		printf("\n%s: can't map registers\n", self->dv_xname);
		return;
	}

	if (bus_space_subregion(sc->sc_bst, sc->sc_bsh, FHPIB_CHIPOFFSET, 8,
	    &sc->sc_chip_bsh)) {
		printf("\n%s: can't map registers\n", sc->sc_dev.dv_xname);
		return;
	}

	printf(": %s\n", DIO_DEVICE_DESC_FHPIB);

	/*
	 * Initialise the DMA queue entry.
	 */
	MALLOC(sc->sc_dq, struct dmaqueue *, sizeof(struct dmaqueue),
	    M_DEVBUF, M_NOWAIT);
	if (sc->sc_dq == NULL) {
		printf("%s: can't allocate DMA queue entry\n", self->dv_xname);
		return;
	}
	sc->sc_dq->dq_softc = sc;
	sc->sc_dq->dq_start = fhpibgo;
	sc->sc_dq->dq_done = fhpibdone;

	/* XXX */
	sc->sc_myaddr = 30;

	fhpibreset(sc);
	(void) fhpib_setaddress(sc, sc->sc_myaddr, -1);

	/*
	 * See if we can do word dma.
	 * If so, we should be able to write and read back the appropos bit.
	 */
	ids = bus_space_read_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR);
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR, ids | CSR_WDMA);
	ids = bus_space_read_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR);
	if (ids & CSR_WDMA) {
		bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR,
		    ids & ~CSR_WDMA);
		sc->sc_flags |= HPIBF_DMA16;
	}

	DPRINTF(FDB_FOLLOW, ("%s: %sword-DMA capable\n",
	    sc->sc_dev.dv_xname, (sc->sc_flags & HPIBF_DMA16 ? "" : "not ")));

	/* establish the interrupt handler */
	(void) dio_intr_establish(fhpibintr, sc, da->da_ipl, IPL_BIO);

	callout_init(&sc->sc_timeout_ch);

	/* attach MI GPIB bus */
	fhpib_ic.cookie = (void *)sc;
	ga.ga_ic = &fhpib_ic;
	ga.ga_address = sc->sc_myaddr;
	sc->sc_gpib =
	    (struct gpib_softc *)config_found(self, &ga, gpibdevprint);
}

void
fhpibreset(v)
	void *v;
{
	struct fhpib_softc *sc = v;
//	u_int8_t cmd;

	DPRINTF(FDB_FOLLOW, ("fhpibreset: sc=%p\n", sc));

	fhpib_init(sc);
	fhpibifc(sc);
	/* we're now the system controller */

	/* XXX should be pushed higher */

#if 0
	/* universal device clear */
//	cmd = GPIBCMD_DCL;
//	(void) fhpibsendcmds(sc, &cmd, 1);
#else
	CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_DCL);
#endif
	/* delay for devices to clear */
	DELAY(100000);
}

static int
fhpib_setaddress(v, pri, sec)
	void *v;
	int pri;
	int sec;
{
	struct fhpib_softc *sc = v;

	DPRINTF(FDB_FOLLOW, ("fhpib_setaddress: sc=%p\n", sc));

	sc->sc_myaddr = pri & GPIB_ADDRMASK;
	return (0);
}

static void
fhpib_init(struct fhpib_softc *sc)
{

	DPRINTF(FDB_FOLLOW, ("fhpib_init: sc=%p\n", sc));
	
	/* reset the chip? */
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CID, 0xff);

	DELAY(100);

	/* ??? */
	CHIP_WRITE(sc, FHPIB_CMD, CMD_8BIT);
	CHIP_WRITE(sc, FHPIB_AR, AR_ARONC);

	/* enable FHPIB interrupts */
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR, CSR_IE);
}

int
fhpibtc(v, sync)
	void *v;
	int sync;
{
	struct fhpib_softc *sc = v;

	DPRINTF(FDB_FOLLOW, ("fhpibtc: sc=%p\n", sc));

	CHIP_WRITE(sc, FHPIB_STAT, 0);
	CHIP_WRITE(sc, FHPIB_IMASK, IMASK_IDLE);
	if (fhpibwait(sc, IMASK_IDLE)) {
		CHIP_WRITE(sc, FHPIB_IMASK, 0);
		return (1);
	}
	CHIP_WRITE(sc, FHPIB_STAT, STAT_ATN);
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	return (0);
}

static void
fhpibifc(sc)
	struct fhpib_softc *sc;
{
	u_int8_t val;

	DPRINTF(FDB_FOLLOW, ("fhpibifc: sc=%p\n", sc));

	val = CHIP_READ(sc, FHPIB_CMD);
	CHIP_WRITE(sc, FHPIB_CMD, val | CMD_IFC);

	val = CHIP_READ(sc, FHPIB_CMD);
	CHIP_WRITE(sc, FHPIB_CMD, val | CMD_INITFIFO);

	DELAY(100);

	val = CHIP_READ(sc, FHPIB_CMD);
	CHIP_WRITE(sc, FHPIB_CMD, val & ~CMD_IFC);

	val = CHIP_READ(sc, FHPIB_CMD);
	CHIP_WRITE(sc, FHPIB_CMD, val | CMD_REN);

/*XXX*/	CHIP_WRITE(sc, FHPIB_STAT, STAT_ATN);
}

int
fhpibgts(v)
	void *v;
{
	struct fhpib_softc *sc = v;

	DPRINTF(FDB_FOLLOW, ("fhpibgts: sc=%p\n", sc));

	/* XXX */
	CHIP_WRITE(sc, FHPIB_STAT, STAT_READ0);
	CHIP_WRITE(sc, FHPIB_DATA, 0);

	return (0);
}

int
fhpibsendcmds(v, ptr, origcnt)
	void *v;
	void *ptr;
	int origcnt;
{
	struct fhpib_softc *sc = v;
	int cnt = origcnt;
	u_int8_t *addr = ptr;

	DPRINTF(FDB_FOLLOW, ("fhpibsendcmds: v=%p, ptr=%p, cnt=%d\n",
	    v, ptr, origcnt));

	CHIP_WRITE(sc, FHPIB_IMASK, IMASK_IDLE);
	while (--cnt >= 0)
		CHIP_WRITE(sc, FHPIB_DATA, *addr++);
	(void) fhpibwait(sc, IMASK_IDLE);
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	return (origcnt - cnt - 1);
}

int
fhpibrecvdata(v, ptr, origcnt)
	void *v;
	void *ptr;
	int origcnt;
{
	struct fhpib_softc *sc = v;
	int cnt = origcnt;
	u_int8_t *addr = ptr;

	DPRINTF(FDB_FOLLOW, ("fhpibrecvdata: v=%p ptr=%p cnt=%d\n",
	    v, ptr, origcnt));

	CHIP_WRITE(sc, FHPIB_IMASK, IMASK_ROOM | IMASK_BYTE);
	if (cnt) {
		while (--cnt >= 0) {
			if (fhpibwait(sc, IMASK_BYTE))
				goto end;
			*addr++ = CHIP_READ(sc, FHPIB_DATA);
		}
		(void) fhpibwait(sc, IMASK_ROOM);
	}
end:
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	return (origcnt);
}

int
fhpibsenddata(v, ptr, origcnt)
	void *v;
	void *ptr;
	int origcnt;
{
	struct fhpib_softc *sc = v;
	int cnt = origcnt;
	u_int8_t *addr = ptr;

	DPRINTF(FDB_FOLLOW, ("fhpibsenddata: v=%p ptr=%p cnt=%d\n",
	    v, ptr, origcnt));

	CHIP_WRITE(sc, FHPIB_IMASK, IMASK_ROOM);
	if (cnt) {
		CHIP_WRITE(sc, FHPIB_STAT, STAT_WRITE);	/* sending data */
		while (--cnt > 0) {
			CHIP_WRITE(sc, FHPIB_DATA, *addr++);
			if (fhpibwait(sc, IMASK_ROOM))
				goto end;
		}
		CHIP_WRITE(sc, FHPIB_STAT, STAT_EOI);
		CHIP_WRITE(sc, FHPIB_DATA, *addr);
		(void) fhpibwait(sc, IMASK_ROOM);
		cnt--;
	}
end:
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	return (origcnt - cnt - 1);
}

int
fhpibsend(v, slave, sec, ptr, origcnt)
	void *v;
	int slave, sec, origcnt;
	void *ptr;
{
	struct fhpib_softc *sc = v;
#if 0
	int cnt;
	u_int8_t cmds[4];
	int i = 0;

	DPRINTF(FDB_FOLLOW, ("fhpibsend: v=%p slave=%d sec=%d ptr=%p cnt=%d\n",
	    v, slave, sec, ptr, cnt));

	if (fhpibtc(sc, 0))
		goto senderr;
	cmds[i++] = GPIBCMD_UNL;
	cmds[i++] = GPIBCMD_TAG | sc->sc_myaddr;
	cmds[i++] = GPIBCMD_LAG | slave;
	if (sec >= 0 || sec == -2) {
		if (sec == -2)		/* selected device clear KLUDGE */
			cmds[i++] = GPIBCMD_SDC;
		else
			cmds[i++] = GPIBCMD_SCG | sec;
	}
	if (fhpibsendcmds(sc, cmds, i) != i)
		goto senderr;
	if (fhpibgts(sc))
		goto senderr;
	if (origcnt) {
		if ((cnt = fhpibsenddata(sc, ptr, origcnt)) != origcnt)
			goto senderr;
		if (fhpibtc(sc, 0))
			goto senderr;
	}
	return (origcnt);

senderr:
	fhpibifc(sc);
	DPRINTF(FDB_FAIL,
	    ("%s: fhpibsend failed: slave %d sec %x, sent %d of %d bytes\n",
	    sc->sc_dev.dv_xname, slave, sec, cnt, origcnt));
	return (cnt);

#else
	int cnt = origcnt;
	u_int8_t *addr = ptr;

	DPRINTF(FDB_FOLLOW, ("fhpibsend: v=%p slave=%d sec=%d ptr=%p cnt=%d\n",
	    v, slave, sec, ptr, cnt));

	CHIP_WRITE(sc, FHPIB_STAT, 0);
	CHIP_WRITE(sc, FHPIB_IMASK, IMASK_IDLE | IMASK_ROOM);
	if (fhpibwait(sc, IMASK_IDLE))
		goto senderr;
	CHIP_WRITE(sc, FHPIB_STAT, STAT_ATN);
	CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_UNL);
	CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_TAG | sc->sc_myaddr);
	CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_LAG | slave);
	if (sec < 0) {
		if (sec == -2)		/* selected device clear KLUDGE */
			CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_SDC);
	} else
		CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_SCG | sec);
	if (fhpibwait(sc, IMASK_IDLE))
		goto senderr;
	if (cnt) {
		CHIP_WRITE(sc, FHPIB_STAT, STAT_WRITE);
		while (--cnt) {
			CHIP_WRITE(sc, FHPIB_DATA, *addr++);
			if (fhpibwait(sc, IMASK_ROOM))
				goto senderr;
		}
		CHIP_WRITE(sc, FHPIB_STAT, STAT_EOI);
		CHIP_WRITE(sc, FHPIB_DATA, *addr);
		(void) fhpibwait(sc, IMASK_ROOM);
		CHIP_WRITE(sc, FHPIB_STAT, STAT_ATN);
#if 0
		/* XXX: HP-UX claims bug with CS80 transparent messages */
		if (sec == 0x12)
			DELAY(150);
		CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_UNL);
		(void) fhpibwait(sc, IMASK_IDLE);
#endif
	}
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	return (origcnt);

senderr:
	CHIP_WRITE(sc, FHPIB_IMASK, 0);

	fhpibifc(sc);
	DPRINTF(FDB_FAIL,
	    ("%s: fhpibsend failed: slave %d sec %x, sent %d of %d bytes\n",
	    sc->sc_dev.dv_xname, slave, sec, origcnt - cnt - 1, origcnt));
	return (origcnt - cnt - 1);
#endif

}

int
fhpibrecv(v, slave, sec, ptr, origcnt)
	void *v;
	int slave, sec, origcnt;
	void *ptr;
{
#if 0
	struct fhpib_softc *sc = v;
	int cnt;
	u_int8_t cmds[4];
	int i = 0;

	DPRINTF(FDB_FOLLOW, ("fhpibrecv: v=%p slave=%d sec=%d ptr=%p cnt=%d\n",
	    v, slave, sec, ptr, origcnt));

	/*
	 * Slave < 0 implies continuation of a previous receive
	 * that probably timed out.
	 */
	if (slave >= 0) {
		if (fhpibtc(sc, 0))
			goto recverror;
		cmds[i++] = GPIBCMD_UNL;
		cmds[i++] = GPIBCMD_LAG | sc->sc_myaddr;
		cmds[i++] = GPIBCMD_TAG | slave;
		if (sec != -1)
			cmds[i++] = GPIBCMD_SCG | sec;
		if (fhpibsendcmds(sc, cmds, i) != i)
			goto recverror;
		if (fhpibgts(sc))
			goto recverror;
	}
	if (origcnt) {
		if ((cnt = fhpibrecvdata(sc, ptr, origcnt)) != origcnt)
			goto recvbyteserror;
		if (fhpibtc(sc, 0))
			goto recverror;
		cmds[0] = (slave == GPIB_BROADCAST_ADDR) ?
		    GPIBCMD_UNA : GPIBCMD_UNT;
		if (fhpibsendcmds(v, cmds, 1) != 1)
			goto recverror;
	}
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	return (origcnt);

recverror:
	fhpibifc(sc);
recvbyteserror:
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	DPRINTF(FDB_FAIL,
	    ("fhpibrecv: failed, sc=%p slave %d sec %x, got %d of %d bytes\n",
	    sc, slave, sec, cnt, origcnt));
	return (cnt);

#else

	struct fhpib_softc *sc = v;
	int cnt = origcnt;
	char *addr = ptr;

	DPRINTF(FDB_FOLLOW, ("fhpibrecv: v=%p slave=%d sec=%d ptr=%p cnt=%d\n",
	    v, slave, sec, ptr, origcnt));

	if (slave >= 0) {
		CHIP_WRITE(sc, FHPIB_STAT, 0);
		CHIP_WRITE(sc, FHPIB_IMASK, IMASK_IDLE|IMASK_ROOM|IMASK_BYTE);
		if (fhpibwait(sc, IMASK_IDLE))
			goto recverror;
		CHIP_WRITE(sc, FHPIB_STAT, STAT_ATN);
		CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_UNL);
		CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_LAG + sc->sc_myaddr);
		CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_TAG + slave);
		if (sec != -1)
			CHIP_WRITE(sc, FHPIB_DATA, GPIBCMD_SCG + sec);;
		if (fhpibwait(sc, IMASK_IDLE))
			goto recverror;
		CHIP_WRITE(sc, FHPIB_STAT, STAT_READ0);
		CHIP_WRITE(sc, FHPIB_DATA, 0);
	}
	if (cnt) {
		while (--cnt >= 0) {
			if (fhpibwait(sc, IMASK_BYTE))
				goto recvbyteserror;
			*addr++ = CHIP_READ(sc, FHPIB_DATA);
		}
		(void) fhpibwait(sc, IMASK_ROOM);
		CHIP_WRITE(sc, FHPIB_STAT, STAT_ATN);
		CHIP_WRITE(sc, FHPIB_DATA, (slave == GPIB_BROADCAST_ADDR) ?
		    GPIBCMD_UNA : GPIBCMD_UNT);
		(void) fhpibwait(sc, IMASK_IDLE);
	}
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	return (origcnt);

recverror:
	fhpibifc(sc);
recvbyteserror:
	CHIP_WRITE(sc, FHPIB_IMASK, 0);
	DPRINTF(FDB_FAIL,
	    ("fhpibrecv: failed, sc=%p slave %d sec %x, got %d of %d bytes\n",
	    sc, slave, sec, origcnt - cnt - 1, origcnt));
	return (origcnt - cnt - 1);
#endif
}

void
fhpibxfer(v, slave, sec, ptr, count, dir, timo)
	void *v;
	int slave;
	int sec;
	void *ptr;
	int count;
	int dir;
	int timo;
{
	struct fhpib_softc *sc = v;

	DPRINTF(FDB_FOLLOW, ("fhpibxfer: sc=%p slave=%d sec=%d ptr=%p count=%d flags=0x%x timo=%d\n",
	    sc, slave, sec, ptr, count, dir, timo));

	/* record the DMA transfer request */
	/* should be loaded into dma_segment */
	sc->sc_dmastate.slave = slave;
	sc->sc_dmastate.sec = sec;
	sc->sc_dmastate.addr = ptr;
	sc->sc_dmastate.dir = dir;
	sc->sc_dmastate.count = count;
	sc->sc_dmastate.timo = timo;

	sc->sc_dq->dq_chan = DMA0 | DMA1;
	if (dmareq(sc->sc_dq))
		fhpibgo(sc);
}

void
fhpibgo(v)
	void *v;
{
	struct fhpib_softc *sc = v;
	int flags = 0;

	DPRINTF(FDB_FOLLOW, ("fhpibgo: sc=%p\n", sc));

	sc->sc_flags |= HPIBF_IO;
	if (sc->sc_dmastate.timo)
		sc->sc_flags |= HPIBF_TIMO;
	if (sc->sc_dmastate.dir == GPIB_READ)
		sc->sc_flags |= HPIBF_READ;
#ifdef DEBUG
	else if (sc->sc_flags & HPIBF_READ) {
		printf("fhpibgo: HPIBF_READ still set\n");
		sc->sc_flags &= ~HPIBF_READ;
	}
#endif

	/*
	 * Check if we should attempt word DMA.
	 */
	if ((sc->sc_flags & HPIBF_DMA16) &&
	    ((int)sc->sc_dmastate.addr & 1) == 0 && sc->sc_dmastate.count &&
	    (sc->sc_dmastate.count & 1) == 0) {
		flags |= DMAGO_WORD;
		bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_LATCH, 0);
	}
#ifdef DEBUG
	if (dopriodma)
		flags |= DMAGO_PRI;
#endif

	/*
	 * Do DMA read.
	 */
	if (sc->sc_flags & HPIBF_READ) {

		sc->sc_cmd = CMD_REN | CMD_8BIT;
		sc->sc_dmastate.curcnt = sc->sc_dmastate.count;

		dmago(sc->sc_dq->dq_chan, sc->sc_dmastate.addr,
		    sc->sc_dmastate.curcnt, flags | DMAGO_READ);
		if (fhpibrecv(sc, sc->sc_dmastate.slave, sc->sc_dmastate.sec,
		    0, 0) < 0) {
			DPRINTF(FDB_DMA,
			    ("fhpibgo: recv failed, retrying...\n"));
			(void) fhpibrecv(sc, sc->sc_dmastate.slave,
			    sc->sc_dmastate.sec, 0, 0);
		}
		(void) CHIP_READ(sc, FHPIB_CMD);
		CHIP_WRITE(sc, FHPIB_CMD, sc->sc_cmd);
		bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR,
		    CSR_DMA(sc->sc_dq->dq_chan) |
		    ((flags & DMAGO_WORD) ? CSR_WDMA : 0));
		return;
	}

	/*
	 * Otherwise, do DMA write.
	 */
	sc->sc_cmd = CMD_REN | CMD_8BIT | CMD_FIFOSEL;
	if (sc->sc_dmastate.count < fhpibdmathresh) {
		sc->sc_dmastate.curcnt = sc->sc_dmastate.count;
		(void) fhpibsend(sc, sc->sc_dmastate.slave,
		    sc->sc_dmastate.sec, sc->sc_dmastate.addr,
		    sc->sc_dmastate.count);
		fhpibdone(sc);
		return;
	}
	sc->sc_dmastate.curcnt = sc->sc_dmastate.count - 
	    (flags & DMAGO_WORD) ? 2 : 1;
	dmago(sc->sc_dq->dq_chan, sc->sc_dmastate.addr,
	    sc->sc_dmastate.curcnt, flags);
	if (fhpibsend(sc, sc->sc_dmastate.slave,
	    sc->sc_dmastate.sec, 0, 0) < 0) {
		DPRINTF(FDB_DMA, ("fhpibgo: send failed, retrying...\n"));
		(void) fhpibsend(sc, sc->sc_dmastate.slave,
		    sc->sc_dmastate.sec, 0, 0);
	}
	(void) CHIP_READ(sc, FHPIB_CMD);
	CHIP_WRITE(sc, FHPIB_CMD, sc->sc_cmd);
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR,
	    CSR_DMA(sc->sc_dq->dq_chan) | CSR_WRITE |
	    ((flags & DMAGO_WORD) ? CSR_WDMA : 0));
}

/*
 * A DMA read can finish but the device can still be waiting (MAG-tape
 * with more data than we're waiting for).  This timeout routine
 * takes care of that.  Somehow, the thing gets hosed.  For now, since
 * this should be a very rare occurence, we RESET it.
 */
void
fhpibtimeout(v)
	void *v;
{
	struct fhpib_softc *sc = v;
	int s;

	DPRINTF(FDB_FOLLOW, ("fhpibtimeout: sc=%p\n", sc));

	s = splbio();
	if (sc->sc_flags & HPIBF_IO) {
		CHIP_WRITE(sc, FHPIB_IMASK, 0);
		bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CID, 0xff);
		DELAY(100);
		CHIP_WRITE(sc, FHPIB_CMD, CMD_8BIT);
		CHIP_WRITE(sc, FHPIB_AR, AR_ARONC);
		fhpibifc(sc);
		bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR, CSR_IE);
		sc->sc_flags &= ~(HPIBF_DONE|HPIBF_IO|HPIBF_READ|HPIBF_TIMO);
		dmafree(sc->sc_dq);
		gpibintr(sc->sc_gpib);
	}
	splx(s);
}

void
fhpibdone(v)
	void *v;
{
	struct fhpib_softc *sc = v;
	u_int8_t *addr;
	int cnt;

	DPRINTF(FDB_FOLLOW, ("fhpibdone: addr %p cnt %d\n",
	    sc->sc_dmastate.addr, sc->sc_dmastate.count));

	cnt = sc->sc_dmastate.curcnt;
	sc->sc_dmastate.addr += cnt;
	sc->sc_dmastate.count -= cnt;
	sc->sc_flags |= HPIBF_DONE;

	if (sc->sc_flags & HPIBF_TIMO)
		callout_reset(&sc->sc_timeout_ch, hz >> 2, fhpibtimeout, sc);

	if (sc->sc_flags & HPIBF_READ) {
		CHIP_WRITE(sc, FHPIB_IMASK, IMASK_IDLE | IMASK_BYTE);
	} else {
		cnt = sc->sc_dmastate.count;
		if (cnt) {
			addr = sc->sc_dmastate.addr;
			CHIP_WRITE(sc, FHPIB_IMASK, IMASK_IDLE | IMASK_ROOM);
			(void) fhpibwait(sc, IMASK_IDLE);
			CHIP_WRITE(sc, FHPIB_STAT, STAT_WRITE);
			while (--cnt) {
				CHIP_WRITE(sc, FHPIB_DATA, *addr++);
				(void) fhpibwait(sc, IMASK_ROOM);
			}
			CHIP_WRITE(sc, FHPIB_STAT, STAT_EOI);
			CHIP_WRITE(sc, FHPIB_DATA, *addr);
		}
		CHIP_WRITE(sc, FHPIB_IMASK, IMASK_IDLE);
	}

	CHIP_WRITE(sc, FHPIB_STAT, STAT_IENAB);
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR, CSR_IE);
}

int
fhpibintr(v)
	void *v;
{
	struct fhpib_softc *sc = v;
	int stat0;

	DPRINTF(FDB_FOLLOW, ("fhpibintr: sc=%p\n", sc));

	stat0 = bus_space_read_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR);
	if ((stat0 & (CSR_IE | CSR_IR)) != (CSR_IE | CSR_IR)) {
#ifdef DEBUG
		if ((fhpibdebug & FDB_FAIL) && (stat0 & CSR_IR) &&
		    (sc->sc_flags & (HPIBF_IO | HPIBF_DONE)) != HPIBF_IO)
			printf("%s: fhpibintr: bad status %x\n",
			    sc->sc_dev.dv_xname, stat0);
#endif
		return (0);
	}

	if ((sc->sc_flags & (HPIBF_IO | HPIBF_DONE)) == HPIBF_IO)
		return (0);

	DPRINTF(FDB_DMA, ("fhpibintr: flags %x\n", sc->sc_flags));

	if (sc->sc_flags & HPIBF_IO) {

		if (sc->sc_flags & HPIBF_TIMO)
			callout_stop(&sc->sc_timeout_ch);

		(void) CHIP_READ(sc, FHPIB_CMD);
		CHIP_WRITE(sc, FHPIB_CMD, sc->sc_cmd & ~CMD_8BIT);
		CHIP_WRITE(sc, FHPIB_STAT, 0);
		CHIP_WRITE(sc, FHPIB_CMD, CMD_REN | CMD_8BIT);
		(void) CHIP_READ(sc, FHPIB_INTR);
		CHIP_WRITE(sc, FHPIB_IMASK, 0);
		sc->sc_flags &= ~(HPIBF_DONE|HPIBF_IO|HPIBF_READ|HPIBF_TIMO);
		dmafree(sc->sc_dq);
		gpibintr(sc->sc_gpib);
	} else if (sc->sc_flags & HPIBF_PPOLL) {
		(void) CHIP_READ(sc, FHPIB_INTR);
#ifdef DEBUG
		if ((fhpibdebug & FDB_FAIL) && (stat0 & IMASK_PPRESP) == 0)
			printf("%s: fhpibintr: bad intr reg %x\n",
			    sc->sc_dev.dv_xname, stat0);
#endif
		CHIP_WRITE(sc, FHPIB_STAT, 0);
		CHIP_WRITE(sc, FHPIB_IMASK, 0);
#ifdef DEBUG
		if (fhpibpptest(sc, sc->sc_ppoll_slave) == 0) {
			/*
			 * XXX give it another shot (68040)
			 */
			DELAY(fhpibppolldelay);
			fhpibpptest(sc, sc->sc_ppoll_slave);
		}
#endif
		sc->sc_flags &= ~HPIBF_PPOLL;
		gpibintr(sc->sc_gpib);
	}
	return (1);
}

int
fhpibpptest(v, slave)
	void *v;
	int slave;
{
	struct fhpib_softc *sc = v;
	int ppoll;

	DPRINTF(FDB_FOLLOW, ("fhpibpptest: sc=%p\n", sc));

	/* clear status */
	CHIP_WRITE(sc, FHPIB_STAT, 0);

	/* clear ppoll info */
	CHIP_WRITE(sc, FHPIB_PSENSE, 0);

	/* parallel poll */
	CHIP_WRITE(sc, FHPIB_PMASK, 0xff);

	/* enable the interrupts we want */
	CHIP_WRITE(sc, FHPIB_IMASK, IMASK_PPRESP | IMASK_PABORT);

	/* wait for ppoll response */	
	DELAY(25);

	/* we're not interested in ppoll response any longer */
	CHIP_WRITE(sc, FHPIB_INTR, IMASK_PABORT);

	/* read the parallel poll response */
	ppoll = CHIP_READ(sc, FHPIB_DATA);

	/* check if we actually aborted */
	if (CHIP_READ(sc, FHPIB_INTR) & IMASK_PABORT)
		ppoll = 0;

	/* disable all interrupts */
	CHIP_WRITE(sc, FHPIB_IMASK, 0);

	/* no more parallel polls */
	CHIP_WRITE(sc, FHPIB_PMASK, 0);

	/* enable interrupts XXX ??? */
	CHIP_WRITE(sc, FHPIB_STAT, STAT_IENAB);

	DPRINTF(FDB_FOLLOW, ("ppoll=0x%x\n", ppoll));

	return ((ppoll & (0x80 >> slave)) != 0);
}

int
fhpibwait(sc, x)
	struct fhpib_softc *sc;
	int x;
{
	int timo = fhpibwtimeout;

	DPRINTF(FDB_FOLLOW, ("fhpibwait: sc=%p\n", sc));

	while ((CHIP_READ(sc, FHPIB_INTR) & x) == 0 && --timo)
		DELAY(1);
	if (timo == 0) {
		DPRINTF(FDB_REPORTTIME, ("fhpibwait: %x timeout\n", x));
		return (1);
	}
	return (0);
}

/*
 * XXX: this will have to change if we ever allow more than one
 * pending operation per HP-IB.
 */
void
fhpibppwatch(v, slave)
	void *v;
	int slave;
{
	struct fhpib_softc *sc = v;
	int mask;

	DPRINTF(FDB_FOLLOW, ("fhpibppwatch: sc=%p\n", sc));

	mask = (0x80 >> slave);
	sc->sc_flags |= HPIBF_PPOLL;
	sc->sc_ppoll_slave = slave;

	/* parallel poll response we're looking for */
	CHIP_WRITE(sc, FHPIB_PSENSE, ~mask);
	CHIP_WRITE(sc, FHPIB_PMASK, mask);
	/* enable chip interrupts */
	CHIP_WRITE(sc, FHPIB_STAT, STAT_IENAB);
	/* generate ppoll interrupts */
	CHIP_WRITE(sc, FHPIB_IMASK, IMASK_PPRESP | IMASK_PABORT);

	/* propagate interrupts to host */
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, FHPIB_CSR, CSR_IE);
}

void
fhpibppclear(v)
	void *v;
{
	struct fhpib_softc *sc = v;

	DPRINTF(FDB_FOLLOW, ("fhpibppclear: sc=%p\n", sc));

	sc->sc_flags &= ~HPIBF_PPOLL;
}
