/*	$NetBSD$	*/

/*
 * Internal/98624 HP-IB driver
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD$");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/callout.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/device.h>

#include <machine/bus.h>

#include <dev/gpib/gpibvar.h>

#include <hp300/dev/intiovar.h>
#include <hp300/dev/diovar.h>
#include <hp300/dev/diodevs.h>
#include <hp300/dev/dmavar.h>

#include <hp300/dev/nhpibreg.h>
#include <dev/ic/tms9914reg.h>

#ifdef DEBUG
int nhpibdebug = 0xff;
#define DBG_FOLLOW	0x01
#define DBG_INTR	0x02
#define DBG_REPORTTIME	0x04
#define DBG_FAIL	0x08
#define DPRINTF(mask, str)	if (nhpibdebug & mask) printf str
#else
#define DPRINTF(mask, str)	/* nothing */
#endif

struct nhpib_softc {
	struct device sc_dev;		/* generic device glue */

	bus_space_tag_t sc_bst;
	bus_space_handle_t sc_bsh;
	bus_space_handle_t sc_tms_bsh;	/* tms9914 bus handle */

	int sc_dchan;
	struct dmaqueue *sc_dq;

	int sc_myaddr;

	volatile int sc_flags;		/* misc flags */
#define	HPIBF_IO	0x1
#define	HPIBF_DONE	0x2
#define	HPIBF_PPOLL	0x4
#define	HPIBF_READ	0x8
#define	HPIBF_TIMO	0x10
#define HPIBF_HOLDOFF	0x20
#define	HPIBF_DMA16	0x8000

	/* record of the current pending dma xfer */
	/* XXX probably should be recorded in the new dma_segment */
	struct {
		int active;
		int slave;
		int sec;
		u_int8_t *addr;
		int count;
		int dir;
		int timo;
		int curcnt;
	} sc_dmastate;

	struct gpib_softc *sc_gpib;

	int sc_ppoll_slave;		/* stash our ppoll address */

	struct callout sc_timeout_ch;
	struct callout sc_ppwatch_ch;
};

int	nhpib_intio_match(struct device *, struct cfdata *, void *);
void	nhpib_intio_attach(struct device *, struct device *, void *);
int	nhpib_dio_match(struct device *, struct cfdata *, void *);
void	nhpib_dio_attach(struct device *, struct device *, void *);

struct cfattach nhpib_intio_ca = {
	sizeof(struct nhpib_softc), nhpib_intio_match, nhpib_intio_attach
};

struct cfattach nhpib_dio_ca = {
	sizeof(struct nhpib_softc), nhpib_dio_match, nhpib_intio_attach
};

void nhpib_common_attach(struct nhpib_softc *, const char *);

static void	nhpibifc(struct nhpib_softc *);
static void	nhpibtimeout(void *);
static int	nhpibwait(struct nhpib_softc *, int);
static void	nhpibppwatch_callout(void *v);

#define TMS9914_READ(sc, reg)						\
	bus_space_read_1((sc)->sc_bst, (sc)->sc_tms_bsh, ((reg)<<1)+1)

#define TMS9914_WRITE(sc, reg, val)					\
	bus_space_write_1((sc)->sc_bst, (sc)->sc_tms_bsh, ((reg)<<1)+1, (val))

void	nhpibreset(void *); 
int	nhpibsend(void *, int, int, void *, int);
int	nhpibrecv(void *, int, int, void *, int);
int	nhpibpptest(void *, int);
void	nhpibppwatch(void *, int);
void	nhpibppclear(void *);
void	nhpibxfer(void *, int, int, void *, int, int, int);
void	nhpibgo(void *v);
void	nhpibdone(void *);
int	nhpibintr(void *);

static void	nhpib_init(struct nhpib_softc *);
static int	nhpib_setaddress(void *, int, int);
int	nhpibsendcmds(void *, void *, int);
int	nhpibsenddata(void *, void *, int);
int	nhpibrecvdata(void *, void *, int);
int	nhpibgts(void *);
int	nhpibtc(void *, int);

static void	dump_registers(struct nhpib_softc *);

/*
 * Our chipset structure.
 */
struct gpib_chipset_tag nhpib_ic = {
	NULL,
	nhpibreset,
	nhpibsend,
	nhpibrecv,
	nhpibpptest,
	nhpibppwatch,
	nhpibppclear,
	nhpibxfer,
};

/*
 * ODD parity table for talk/listen addresses and secondary commands.
 * The TI9914A doesn't produce the parity bit.
 */
const u_int8_t oddparity[] = {
	0x80,0x01,0x02,0x83,0x04,0x85,0x86,0x07,
	0x08,0x89,0x8a,0x0b,0x8c,0x0d,0x0e,0x8f,
	0x10,0x91,0x92,0x13,0x94,0x15,0x16,0x97,
	0x98,0x19,0x1a,0x9b,0x1c,0x9d,0x9e,0x1f,
	0x20,0xa1,0xa2,0x23,0xa4,0x25,0x26,0xa7,
	0xa8,0x29,0x2a,0xab,0x2c,0xad,0xae,0x2f,
	0xb0,0x31,0x32,0xb3,0x34,0xb5,0xb6,0x37,
	0x38,0xb9,0xba,0x3b,0xbc,0x3d,0x3e,0xbf,
	0x40,0xc1,0xc2,0x43,0xc4,0x45,0x46,0xc7,
	0xc8,0x49,0x4a,0xcb,0x4c,0xcd,0xce,0x4f,
	0xd0,0x51,0x52,0xd3,0x54,0xd5,0xd6,0x57,
	0x58,0xd9,0xda,0x5b,0xdc,0x5d,0x5e,0xdf,
	0xe0,0x61,0x62,0xe3,0x64,0xe5,0xe6,0x67,
	0x68,0xe9,0xea,0x6b,0xec,0x6d,0x6e,0xef,
	0x70,0xf1,0xf2,0x73,0xf4,0x75,0x76,0xf7,
	0xf8,0x79,0x7a,0xfb,0x7c,0xfd,0xfe,0x7f,
};
#define ADDPARITY(x)		(oddparity[(x)])

int nhpibdmathresh = 3;		/* byte count beyond which to attempt dma */
int nhpibwtimeout = 100000;	/* # of status tests before we give up */

int
nhpib_intio_match(parent, match, aux)
	struct device *parent;
	struct cfdata *match;
	void *aux;
{
	struct intio_attach_args *ia = aux;

	if (strcmp("hpib", ia->ia_modname) == 0)
		return (1);

	return (0);
}

int
nhpib_dio_match(parent, match, aux)
	struct device *parent;
	struct cfdata *match;
	void *aux;
{
	struct dio_attach_args *da = aux;

	if (da->da_id == DIO_DEVICE_ID_NHPIB)
		return (1);

	return (0);
}

void
nhpib_intio_attach(parent, self, aux)
	struct device *parent;
	struct device *self;
	void *aux;
{
	struct nhpib_softc *sc = (struct nhpib_softc *)self;
	struct intio_attach_args *ia = aux;
	const char *desc = "internal HP-IB";

	sc->sc_bst = ia->ia_bst;
	if (bus_space_map(ia->ia_bst, ia->ia_iobase, INTIO_DEVSIZE, 0,
	    &sc->sc_bsh)) {
		printf("%s: can't map registers\n", sc->sc_dev.dv_xname);
		return;
	}

	sc->sc_myaddr = -1;
	sc->sc_dchan = DMA0;

	nhpib_common_attach(sc, desc);

	/* establish the interrupt handler */
	(void) intio_intr_establish(nhpibintr, sc, ia->ia_ipl, IPL_BIO);
}

void
nhpib_dio_attach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct nhpib_softc *sc = (struct nhpib_softc *)self;
	struct dio_attach_args *da = aux;
	const char *desc = DIO_DEVICE_DESC_NHPIB;

	sc->sc_bst = da->da_bst;
	if (bus_space_map(sc->sc_bst, da->da_addr, da->da_size, 0,
	    &sc->sc_bsh)) {
		printf("\n%s: can't map registers\n", sc->sc_dev.dv_xname);
		return;
	}

	/* read address off switches */
	sc->sc_myaddr = bus_space_read_1(sc->sc_bst, sc->sc_bsh,
	    NHPIB_CSA) & GPIB_ADDRMASK;
	sc->sc_dchan = DMA0 | DMA1;

	nhpib_common_attach(sc, desc);

	/* establish the interrupt handler */
	(void) dio_intr_establish(nhpibintr, sc, da->da_ipl, IPL_BIO);
}

void
nhpib_common_attach(sc, desc)
	struct nhpib_softc *sc;
	const char *desc;
{
	struct gpibdev_attach_args ga; 

	if (bus_space_subregion(sc->sc_bst, sc->sc_bsh, NHPIB_TMSOFFSET, 16,
	    &sc->sc_tms_bsh)) {
		printf("\n%s: can't map registers\n", sc->sc_dev.dv_xname);
		return;
	}

	printf(": %s\n", desc);

	/*
	 * Initialise the DMA queue entry.
	 */
	MALLOC(sc->sc_dq, struct dmaqueue *, sizeof(struct dmaqueue),
	    M_DEVBUF, M_NOWAIT);
	if (sc->sc_dq == NULL) {
		printf("%s: can't allocate DMA queue entry\n",
		    sc->sc_dev.dv_xname);
		return;
	}
	sc->sc_dq->dq_softc = sc;
	sc->sc_dq->dq_start = nhpibgo;
	sc->sc_dq->dq_done = nhpibdone;

	sc->sc_myaddr = 21;	/* XXX */
	/* reset controller (XXX and bus?) */
	nhpibreset(sc);
	nhpib_setaddress(sc, sc->sc_myaddr, -1);

	callout_init(&sc->sc_timeout_ch);
	callout_init(&sc->sc_ppwatch_ch);

	/* attach MI GPIB bus */
	nhpib_ic.cookie = (void *)sc;
	ga.ga_ic = &nhpib_ic;
	ga.ga_address = sc->sc_myaddr;
	sc->sc_gpib =
	    (struct gpib_softc *)config_found((void *)sc, &ga, gpibdevprint);
}

void
nhpib_init(sc)
	struct nhpib_softc *sc;
{

	DPRINTF(DBG_FOLLOW, ("nhpib_init: sc=%p\n", sc));

	/* software reset - logically disconnect bus */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_SWRST);

	/* clear pending interrupts */
	TMS9914_READ(sc, TMS9914_CPTR);
	TMS9914_READ(sc, TMS9914_ISR0);
	TMS9914_READ(sc, TMS9914_ISR1);

	/* select tms9914 interrupts */
	TMS9914_WRITE(sc, TMS9914_IMR0, 0);
	TMS9914_WRITE(sc, TMS9914_IMR1, IMR1_ERR);

	/* enable tms9914 interrupts */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_CLEAR | AUXCMD_DAI);

	/* parallel poll unconfigure */
	TMS9914_WRITE(sc,TMS9914_PPR, 0);

	/* set our GPIB address */
	TMS9914_WRITE(sc, TMS9914_ADDR, 0);

	/* setup bus delay times */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_STD1);
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_VSTD1);

	/* disable shadow handshaking */
	TMS9914_WRITE(sc, TMS9914_AUXCR,  AUXCMD_CLEAR | AUXCMD_SHDW);

	/* clear any outstanding parallel-poll requests */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_CLEAR | AUXCMD_RPP);

	/* don't perform RFD holdoff */
	TMS9914_WRITE(sc, TMS9914_AUXCR,  AUXCMD_CLEAR | AUXCMD_HDFA);
	TMS9914_WRITE(sc, TMS9914_AUXCR,  AUXCMD_CLEAR | AUXCMD_HDFE);

	/* release NRFD, just in case */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_RHDF);
	sc->sc_flags |= HPIBF_HOLDOFF;

	/* clear software reset - logically reconnect bus */
	TMS9914_WRITE(sc, TMS9914_AUXCR,  AUXCMD_CLEAR | AUXCMD_SWRST);
}

/* XXX push hpib protocol stuff higher */
void
nhpibreset(v)
	void *v;
{
	struct nhpib_softc *sc = v;
	u_int8_t cmd;

	DPRINTF(DBG_FOLLOW, ("nhpibreset: sc=%p\n", sc));

	nhpib_init(sc);

	/* place devices in quiesient state for remote programming */
	nhpibifc(sc);

	/* enable NHPIB device interrupts */
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, NHPIB_CSR, CSR_IE);

	/* XXX should be pushed higher */
	/* universal device clear */
	cmd = GPIBCMD_DCL;
	(void) nhpibsendcmds(sc, &cmd, 1);
	/* delay for devices to clear */
	DELAY(100000);
}

static int
nhpib_setaddress(v, pri, sec)
	void *v;
	int pri;
	int sec;
{
	struct nhpib_softc *sc = v;

	DPRINTF(DBG_FOLLOW, ("nhpib_setaddress: sc=%p\n", sc));

	/* XXX push higher */
	sc->sc_myaddr = pri & GPIB_ADDRMASK;

	TMS9914_WRITE(sc, TMS9914_ADDR, sc->sc_myaddr);

	return (0);
}

/*
 * Place all devices on the bus into quiescient state ready for
 * remote programming.  We remain the active controller upon exit.
 */
static void
nhpibifc(sc)
	struct nhpib_softc *sc;
{

	DPRINTF(DBG_FOLLOW, ("nhpibifc: sc=%p\n", sc));

	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_TCA);
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_CLEAR | AUXCMD_SRE);
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_SIC);
	/* wait for devices to enter quiesient state */
	DELAY(100);
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_CLEAR | AUXCMD_SIC);
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_SRE);

	printf("ISR0: 0x%x, ISR1: 0x%x\n",
		TMS9914_READ(sc, TMS9914_ISR0),
		TMS9914_READ(sc, TMS9914_ISR1));
}

int
nhpibtc(v, sync)
	void *v;
	int sync;
{
	struct nhpib_softc *sc = v;
	u_int8_t adsr;
	int timo = nhpibwtimeout;

	DPRINTF(DBG_FOLLOW, ("nhpibtc: sc=%p\n", sc));

	adsr = TMS9914_READ(sc, TMS9914_ADSR);
	if ((adsr & ADSR_ATN) == ADSR_ATN) {
		printf("nhpibtc: ATN already asserted\n");
		return (0);
	}

	if (sync) {
		/* release NRFD, just in case */
		if ((sc->sc_flags & HPIBF_HOLDOFF) != 0) {
			TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_RHDF);
			sc->sc_flags &= ~HPIBF_HOLDOFF;
		}
		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_TCS);
	} else
		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_TCA);

	/* wait until ATN is asserted */
	for (;;) {
		adsr = TMS9914_READ(sc, TMS9914_ADSR);
		if (--timo == 0) {
			DPRINTF(DBG_REPORTTIME, ("nhpibtc: timeout\n"));
			return (1);
		}
		if ((adsr & ADSR_ATN) == ADSR_ATN)
			break;
		DELAY(1);
	}

	return (0);
}

int
nhpibgts(v)
	void *v;
{
	struct nhpib_softc *sc = v;
	u_int8_t adsr;
	int timo = nhpibwtimeout;

	DPRINTF(DBG_FOLLOW, ("nhpibgts: sc=%p\n", sc));

	/* release ATN */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_GTS);

	/* wait until ATN is released */
	for (;;) {
		adsr = TMS9914_READ(sc, TMS9914_ADSR);
		if (--timo == 0) {
			DPRINTF(DBG_REPORTTIME, ("nhpibgts: timeout\n"));
			return (1);
		}
		if ((adsr & ADSR_ATN) == 0)
			break;
		DELAY(1);
	}

	return (0);
}

int
nhpibsendcmds(v, ptr, origcnt)
	void *v;
	void *ptr;
	int origcnt;
{
	struct nhpib_softc *sc = v;
	int cnt = origcnt;
	u_int8_t *addr = ptr;

	DPRINTF(DBG_FOLLOW, ("nhpibsendcmds: v=%p, ptr=%p, cnt=%d\n",
	    v, ptr, origcnt));

	while (--cnt >= 0) {
		TMS9914_WRITE(sc, TMS9914_CDOR, ADDPARITY(*addr++));
		if (nhpibwait(sc, ISR0_BO))
			return (origcnt - cnt - 1);
	}
	return (origcnt);
}

int
nhpibsenddata(v, ptr, origcnt)
	void *v;
	void *ptr;
	int origcnt;
{
	struct nhpib_softc *sc = v;
	int cnt = origcnt;
	u_int8_t *addr = ptr;

	DPRINTF(DBG_FOLLOW, ("nhpibsenddata: v=%p ptr=%p cnt=%d\n",
	    v, ptr, origcnt));

	/* become active talker */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_TON);

	if (cnt) {
		while (--cnt > 0) {
			TMS9914_WRITE(sc, TMS9914_CDOR, *addr++);
			if (nhpibwait(sc, ISR0_BO))
				return (origcnt - cnt - 1);
		}
		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SEOI);
		TMS9914_WRITE(sc, TMS9914_CDOR, *addr);
		(void) nhpibwait(sc, ISR0_BO);
	}

//	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_CLEAR | AUXCMD_TON);

	return (origcnt);
}

int
nhpibrecvdata(v, ptr, origcnt)
	void *v;
	void *ptr;
	int origcnt;
{
	struct nhpib_softc *sc = v;
	int cnt = origcnt;
	u_int8_t *addr = ptr;

	DPRINTF(DBG_FOLLOW, ("nhpibrecvdata: v=%p ptr=%p cnt=%d\n",
	    v, ptr, origcnt));

	/* become active listener */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_LON);

	if ((sc->sc_flags & HPIBF_HOLDOFF) != 0) {
		DPRINTF(DBG_FOLLOW, ("nhpibrecvdata: releasing holdoff\n"));
		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_RHDF);
		sc->sc_flags &= ~HPIBF_HOLDOFF;
	}

	if (cnt) {
//		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_CLEAR | AUXCMD_HDFA);
		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_HDFE);
		while (--cnt > 0) {
			int timo = nhpibwtimeout;
			u_int8_t isr0, isr1;

			for (;;) {
				isr0 = TMS9914_READ(sc, TMS9914_ISR0);
				isr1 = TMS9914_READ(sc, TMS9914_ISR1);
				if (--timo == 0) {
					DPRINTF(DBG_FOLLOW,
					    ("nhpibrecvdata: timeout\n"));
					return (origcnt - cnt - 1);
				}
				if ((isr0 & ISR0_BI) != 0)
					break;
				DELAY(1);
			}
			printf("ISRO: 0x%x ISR1: 0x%x\n", isr0, isr1);
			*addr++ = TMS9914_READ(sc, TMS9914_DIR);
			if ((isr0 & ISR0_END) != 0) {
				DPRINTF(DBG_FOLLOW,
				    ("nhpibrecvdata: premature end\n"));
				goto end;
			}
		}
//		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_HDFA);
		if (nhpibwait(sc, ISR0_BI))
			return (origcnt - cnt - 1);
		*addr =  TMS9914_READ(sc, TMS9914_DIR);
		cnt--;
end:
		sc->sc_flags |= HPIBF_HOLDOFF;
	}

//	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_CLEAR | AUXCMD_LON);

	return (origcnt - cnt);
}


/* XXX push HPIB protocol stuff higher */
/* If we actually send something, we remain the active controller,
 * otherwise we become the standby controller.
 */
int
nhpibsend(v, slave, sec, ptr, origcnt)
	void *v;
	int slave, sec, origcnt;
	void *ptr;
{
	struct nhpib_softc *sc = v;
	int cnt;
	u_int8_t cmds[4];
	int i = 0;

	DPRINTF(DBG_FOLLOW, ("nhpibsend: sc=%p\n", sc));

	if (nhpibtc(sc, 0))
		goto senderror;
	cmds[i++] = GPIBCMD_UNL;
	cmds[i++] = GPIBCMD_TAG | sc->sc_myaddr;
	cmds[i++] = GPIBCMD_LAG | slave;
	if (sec >= 0 || sec == -2) {
		if (sec == -2)		/* selected device clear KLUDGE */
			cmds[i++] = GPIBCMD_SDC;
		else
			cmds[i++] = GPIBCMD_SCG | sec;
	}
	if (nhpibsendcmds(sc, cmds, i) != i)
		goto senderror;
	if (nhpibgts(sc))
		goto senderror;
	if (origcnt) {
		if ((cnt = nhpibsenddata(sc, ptr, origcnt)) != origcnt)
			goto senderror;
		if (nhpibtc(sc, 0))
			goto senderror;
		cmds[0] = GPIBCMD_UNL;
		if (nhpibsendcmds(v, cmds, 1) != 1)
			goto senderror;
	}
	return (origcnt);

senderror:
	nhpibifc(sc);
	DPRINTF(DBG_FAIL,
	    ("%s: nhpibsend failed: slave %d, sec %x, sent %d of %d bytes\n",
	    sc->sc_dev.dv_xname, slave, sec, cnt, origcnt));
	return (cnt);
}

/* XXX push hpib protocol stuff higher */
/* If we actually receive something, we remain the active controller,
 * otherwise we become the standby controller.
 */
int
nhpibrecv(v, slave, sec, ptr, origcnt)
	void *v;
	int slave, sec, origcnt;
	void *ptr;
{
	struct nhpib_softc *sc = v;
	int cnt;
	u_int8_t cmds[4];
	int i = 0;

	DPRINTF(DBG_FOLLOW,
	    ("nhpibrecv: sc=%p slave %d sec=%d ptr=%p cnt=%d\n",
	    sc, slave, sec, ptr, origcnt));

	dump_registers(sc);

	if (nhpibtc(sc, 0))
		goto recverror;
	cmds[i++] = GPIBCMD_UNL;
	cmds[i++] = GPIBCMD_LAG | sc->sc_myaddr;
	cmds[i++] = GPIBCMD_TAG | slave;
	if (sec >= 0)
		cmds[i++] = GPIBCMD_SCG | sec;
	if (nhpibsendcmds(sc, cmds, i) != i)
		goto recverror;

#if 0
	/* if this isn't here, the QSTAT fails */
	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_LON);

	/* I guess, if we do it while ATN is set, we don't wipe away
	 * any incoming data.
	 */

//	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_RHDF);
#endif
	printf("before gts: ");
	dump_registers(sc);
	if (nhpibgts(sc))
		goto recverror;
	printf("after gts: ");
	dump_registers(sc);

	if (origcnt) {
		if ((cnt = nhpibrecvdata(sc, ptr, origcnt)) != origcnt)
			goto recverror;
		if (nhpibtc(sc, 0))
			goto recverror;
		cmds[0] = (slave == GPIB_BROADCAST_ADDR) ? 
		    GPIBCMD_UNA : GPIBCMD_UNT;
		if (nhpibsendcmds(v, cmds, 1) != 1)
			goto recverror;
	}
	return (origcnt);

recverror:
	nhpibifc(sc);
	DPRINTF(DBG_FAIL,
	    ("nhpibrecv: failed, sc=%p slave %d, sec %x, got %d of %d bytes\n",
	    sc, slave, sec, cnt, origcnt));
	return (cnt);
}

void
nhpibxfer(v, slave, sec, ptr, count, dir, timo)
	void *v;
	int slave;
	int sec;
	void *ptr;
	int count;
	int dir;
	int timo;
{
	struct nhpib_softc *sc = v;

	DPRINTF(DBG_FOLLOW, ("nhpibxfer: sc=%p slave=%d sec=%d ptr=%p count=%d dir=0x%x timo=%d\n",
	    sc, slave, sec, ptr, count, dir, timo));

	/* record the DMA transfer request */
	/* should be loaded into dma_segment */
	sc->sc_dmastate.slave = slave;
	sc->sc_dmastate.sec = sec;
	sc->sc_dmastate.addr = ptr;
	sc->sc_dmastate.dir = dir;
	sc->sc_dmastate.count = count;
	sc->sc_dmastate.timo = timo;

	sc->sc_dq->dq_chan = sc->sc_dchan;
	if (dmareq(sc->sc_dq))
		nhpibgo(sc);
}

void
nhpibgo(v)
	void *v;
{
	struct nhpib_softc *sc = v;

	DPRINTF(DBG_FOLLOW, ("nhpibgo: sc=%p\n", sc));

	sc->sc_flags |= HPIBF_IO;
	if (sc->sc_dmastate.timo)
		sc->sc_flags |= HPIBF_TIMO;
	if (sc->sc_dmastate.dir == GPIB_READ)
		sc->sc_flags |= HPIBF_READ;
#ifdef DEBUG
	else if (sc->sc_flags & HPIBF_READ) {
		printf("nhpibgo: HPIBF_READ still set\n");
		sc->sc_flags &= ~HPIBF_READ;
	}
#endif
	if (sc->sc_flags & HPIBF_READ) {
		sc->sc_dmastate.curcnt = sc->sc_dmastate.count;
		dmago(sc->sc_dq->dq_chan, sc->sc_dmastate.addr,
		    sc->sc_dmastate.count, DMAGO_BYTE | DMAGO_READ);
		nhpibrecv(sc, sc->sc_dmastate.slave, sc->sc_dmastate.sec,
		    0, 0);
		TMS9914_WRITE(sc, TMS9914_IMR0, IMR0_END);
	} else {
		TMS9914_WRITE(sc, TMS9914_IMR0, 0);
		if (sc->sc_dmastate.count < nhpibdmathresh) {
			sc->sc_dmastate.curcnt = sc->sc_dmastate.count;
			nhpibsend(sc, sc->sc_dmastate.slave,
			    sc->sc_dmastate.sec, sc->sc_dmastate.addr,
			    sc->sc_dmastate.count);
			nhpibdone(sc);
			return;
		}
		/* we transmit the last byte manually with EOI set */
		sc->sc_dmastate.curcnt = sc->sc_dmastate.count-1;
		dmago(sc->sc_dq->dq_chan, sc->sc_dmastate.addr,
		    sc->sc_dmastate.curcnt, DMAGO_BYTE);
		nhpibsend(sc, sc->sc_dmastate.slave, sc->sc_dmastate.sec,
		    0, 0);
	}
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, NHPIB_CSR,
	    CSR_IE | CSR_DMA(sc->sc_dq->dq_chan));
}

/*
 * This timeout can only happen if a DMA read finishes DMAing with the read
 * still pending (more data in read transaction than the driver was prepared
 * to accept).  At the moment, variable-record tape drives are the only things
 * capable of doing this.  We repeat the necessary code from nhpibintr() -
 * easier and quicker than calling nhpibintr() for this special case.
 */
/* XXX should be pushed higher */
static void
nhpibtimeout(v)
	void *v;
{
	struct nhpib_softc *sc;
	int s;

	DPRINTF(DBG_FOLLOW, ("nhpibtimeout: sc=%p\n", sc));

	s = splbio();
	if (sc->sc_flags & HPIBF_IO) {
		TMS9914_WRITE(sc, TMS9914_IMR0, 0);
		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_TCA);
		sc->sc_flags &= ~(HPIBF_DONE|HPIBF_IO|HPIBF_READ|HPIBF_TIMO);
		dmafree(sc->sc_dq);
		gpibintr(sc->sc_gpib);
	}
	splx(s);
}

void
nhpibdone(v)
	void *v;
{
	struct nhpib_softc *sc = v;
	int cnt;

	DPRINTF(DBG_FOLLOW, ("nhpibdone: sc=%p flags=0x%x\n",
	    sc, sc->sc_flags));

	cnt = sc->sc_dmastate.curcnt;
	sc->sc_dmastate.addr += cnt;
	sc->sc_dmastate.count -= cnt;
	sc->sc_flags |= HPIBF_DONE;

	if ((sc->sc_flags & HPIBF_TIMO)) {
		if ((bus_space_read_1(sc->sc_bst, sc->sc_bsh, NHPIB_CSR)
		    & CSR_IR) == 0)
			callout_reset(&sc->sc_timeout_ch, hz >> 2,
			    nhpibtimeout, sc);
	}

	if (sc->sc_flags & HPIBF_READ) {
	} else {
		if (sc->sc_dmastate.count == 1) {
			(void) nhpibwait(sc, ISR0_BO);
			TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SEOI);
			TMS9914_WRITE(sc, TMS9914_CDOR,
			    *(sc->sc_dmastate.addr));
			/* generate interrupt */
			TMS9914_WRITE(sc, TMS9914_IMR0, IMR0_BO);
		}
#ifdef DEBUG
		else if (sc->sc_dmastate.count)
			panic("nhpibdone");
#endif
	}
	bus_space_write_1(sc->sc_bst, sc->sc_bsh, NHPIB_CSR, CSR_IE);

}


int
nhpibintr(v)
	void *v;
{
	struct nhpib_softc *sc = v;
	int stat0;
	int stat1;

	DPRINTF(DBG_INTR, ("nhpibintr: sc=%p, flags=0x%x\n",
	    sc, sc->sc_flags));

	if ((bus_space_read_1(sc->sc_bst, sc->sc_bsh,
	     NHPIB_CSR) & CSR_IR) == 0)
		return  (0);
	stat0 = TMS9914_READ(sc, TMS9914_ISR0);
	stat1 = TMS9914_READ(sc, TMS9914_ISR1);

	if (sc->sc_flags & HPIBF_IO) {
		TMS9914_WRITE(sc, TMS9914_IMR0, 0);
		if ((sc->sc_flags & HPIBF_DONE) == 0) {
			sc->sc_flags &= ~HPIBF_TIMO;
			dmastop(sc->sc_dq->dq_chan);
		} else if (sc->sc_flags & HPIBF_TIMO)
			callout_stop(&sc->sc_timeout_ch);
		TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_TCA);
		sc->sc_flags &= ~(HPIBF_DONE|HPIBF_IO|HPIBF_READ|HPIBF_TIMO);
		dmafree(sc->sc_dq);
		gpibintr(sc->sc_gpib);
	} else if (sc->sc_flags & HPIBF_PPOLL) {
		TMS9914_WRITE(sc, TMS9914_IMR0, 0);
		if (nhpibpptest(sc, sc->sc_ppoll_slave)) {
			sc->sc_flags &= ~HPIBF_PPOLL;
			gpibintr(sc->sc_gpib);
		}
#ifdef DEBUG
		else
			printf("%s: PPOLL intr bad status %x\n",
			    sc->sc_dev.dv_xname, stat0);
#endif
	}
	return (1);
}

static int
nhpibwait(sc, x)
	struct nhpib_softc *sc;
	int x;
{
	int timo = nhpibwtimeout;
	u_int8_t isr0;
	int count = 5;

	DPRINTF(DBG_FOLLOW, ("nhpibwait: sc=%p\n", sc));

	for (;;) {
		isr0 = TMS9914_READ(sc, TMS9914_ISR0);
		if (--timo == 0) {
			DPRINTF(DBG_REPORTTIME,
			    ("nhpibwait: timeout x=0x%x, isr0=0x%x\n",
			    x, isr0));
			return (1);
		}
		if (count > 0) {
			printf("isr0: 0x%x\n", isr0);
			count--;
		}
		if ((isr0 & x) != 0)
			break;
		DELAY(1);
	}
	return (0);
}

int
nhpibpptest(v, slave)
	void *v;
	int slave;
{
	struct nhpib_softc *sc = v;
	u_int8_t adsr;
	int ppoll;

	DPRINTF(DBG_FOLLOW, ("nhpibpptest: sc=%p\n", sc));

	adsr = TMS9914_READ(sc, TMS9914_ADSR);
	if ((adsr & ADSR_ATN) != ADSR_ATN)
		printf("nhpibtc: ATN not asserted!\n");

	TMS9914_WRITE(sc, TMS9914_AUXCR, AUXCMD_SET | AUXCMD_RPP);
	DELAY(25);
	ppoll = TMS9914_READ(sc, TMS9914_CPTR);
	TMS9914_WRITE(sc, TMS9914_AUXCR,  AUXCMD_CLEAR | AUXCMD_RPP);

	printf("nhpibpptest: ppoll=0x%x\n", ppoll);
	return ((ppoll & (0x80 >> slave)) != 0);
}

void
nhpibppwatch(v, slave)
	void *v;
	int slave;
{
	struct nhpib_softc *sc = v;

	DPRINTF(DBG_FOLLOW, ("nhpibppwatch: sc=%p slave=%d\n", sc, slave));

	sc->sc_flags |= HPIBF_PPOLL;
	sc->sc_ppoll_slave = slave;
	nhpibppwatch_callout(sc);
}

static void
nhpibppwatch_callout(v)
	void *v;
{
	struct nhpib_softc *sc = v;

	if ((sc->sc_flags & HPIBF_PPOLL) == 0)
		return;

tryagain:
	if (nhpibpptest(sc, sc->sc_ppoll_slave)) {
       		TMS9914_WRITE(sc, TMS9914_IMR0, IMR0_BO);
		return;
	}
	if (cold)
		goto tryagain;
	else
		callout_reset(&sc->sc_ppwatch_ch, 1, nhpibppwatch_callout, sc);
}

void
nhpibppclear(v)
	void *v;
{
	struct nhpib_softc *sc = v;

	DPRINTF(DBG_FOLLOW, ("nhpibppclear: sc=%p\n", sc));

	sc->sc_flags &= ~HPIBF_PPOLL;
}


static void
dump_registers(sc)
	struct nhpib_softc *sc;
{

#if 0
	printf("ISR0=0x%x, ISR1=0x%x, ADSR=0x%x\n",
		TMS9914_READ(sc, TMS9914_ISR0),
		TMS9914_READ(sc, TMS9914_ISR1),
		TMS9914_READ(sc, TMS9914_ADSR));
#else
	printf("ADSR=0x%x\n", TMS9914_READ(sc, TMS9914_ADSR));
#endif
}
