/*
 * Copyright (c) 2002 Bang Jun-Young
 * All rights reserved.
 */

#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "rtld.h"
#include "debug.h"

#define RELCACHE_DIR	"/var/db/relcache/"

int
_rtld_save_relcache(Obj_Entry *first)
{
	Obj_Entry *obj;
	int fd;
	size_t nbytes = 0;
	int offset = _rtld_pagesz;
	int nents = 1;
	RelCache_Entry *rcache;
	char *buf;
	char path[64], filename[17];

	if (_rtld_objself.cksum->crc32 == 0) {
		dbg(("rtld checksum not found"));
		goto bad;
	}
	for (obj = first; obj != NULL; obj = obj->next)
		if (!obj->cksumfound || obj->cksum->crc32 == 0) {
			dbg(("shobj checksum not found"));
			goto bad;
		}
	for (obj = first; obj != NULL; obj = obj->next)
		nents++;
	nbytes = round_up(nents * sizeof(RelCache_Entry) + 4);
	/* XXX this limitation should be gone. */
	if (nbytes > _rtld_pagesz)
		goto bad;

	buf = xcalloc(nbytes);
	memcpy(buf, &nents, 4);
	rcache = (RelCache_Entry *)(buf + 4);

	for (obj = first; obj != NULL; obj = obj->next) {
		memcpy(rcache, obj->cksum, 32);
		rcache->cksum.crc32 = obj->cksum->crc32;
		rcache->cksum.adler32 = obj->cksum->adler32;
		rcache->filesize = obj->filesize;
		rcache->base = obj->database;
		rcache->offset = offset;
		rcache->size = obj->datalim - (Elf_Addr)obj->database;
		dbg(("cache offset=0x%x, base=%p, size=0x%x",
		    offset, obj->database, rcache->size));
		offset += rcache->size;
		rcache++;
	}

	memcpy(rcache, _rtld_objself.cksum, 32);
#if 0
	rcache->filesize = 0;
	rcache->base = 0;
	rcache->offset = 0;
	rcache->size = 0;
#endif

	strcpy(path, RELCACHE_DIR);
#if 0	/* XXX strange, this doesn't work */
	xsnprintf(filename, 17, "%x%x", first->cksum->crc32,
	    first->cksum->adler32);
#else
	xsnprintf(filename, 17, "%x", first->cksum->crc32);
	strcat(path, filename);
	xsnprintf(filename, 17, "%x", first->cksum->adler32);
	strcat(path, filename);
#endif
	dbg(("saving RelCache as %s", path));
	fd = open(path, O_CREAT | O_TRUNC | O_WRONLY, 0644);
	if (fd < 0) {
		xwarn("%s", path);
		return -1;
	}

	write(fd, buf, nbytes);

	for (obj = first; obj != NULL; obj = obj->next)
		write(fd, obj->database, obj->datalim - (Elf_Addr)obj->database);

	close(fd);
	free(buf);
	return 0;
bad:
	close(fd);
	return -1;
}

bool
_rtld_load_relcache(Obj_Entry *first)
{
	caddr_t cachebase;
	int nents, nobjs = 1;
	Obj_Entry *obj;
	char path[64], filename[17];
	int fd, i;
	RelCache_Entry *rcache;

	strcpy(path, RELCACHE_DIR);
#if 0	/* XXX strange, this doesn't work */
	xsnprintf(filename, 17, "%x%x", first->cksum->crc32,
	    first->cksum->adler32);
#else
	xsnprintf(filename, 17, "%x", first->cksum->crc32);
	strcat(path, filename);
	xsnprintf(filename, 17, "%x", first->cksum->adler32);
	strcat(path, filename);
#endif
	dbg(("loading RelCache from %s", path));
	fd = open(path, O_RDONLY);
	if (fd < 0)
		return false;

	cachebase = mmap(0, _rtld_pagesz, PROT_READ, MAP_FILE | MAP_PRIVATE,
	    fd, 0);
	if (cachebase == MAP_FAILED)
		goto bad2;
	nents = (int)*cachebase;
	if (_rtld_objself.cksum->crc32 == 0) {
		dbg(("rtld checksum not found"));
		goto bad;
	}
	for (obj = first; obj != NULL; obj = obj->next) {
		if (!obj->cksumfound || obj->cksum->crc32 == 0) {
			dbg(("shobj checksum not found"));
			goto bad;
		}
		nobjs++;
	}
	if (nents != nobjs) {
		dbg(("nents %d != nobjs %d", nents, nobjs));
		goto bad;
	}

	/* XXX this limitation should be gone. */
	if (nents * sizeof(RelCache_Entry) + 4 > _rtld_pagesz) {
		dbg(("header too big: %d > %d",
		    nents * sizeof(RelCache_Entry) + 4, _rtld_pagesz));
		goto bad;
	}

	rcache = (RelCache_Entry *)(cachebase + 4);
	if (rcache[nents - 1].cksum.crc32 != _rtld_objself.cksum->crc32 ||
	    rcache[nents - 1].cksum.adler32 != _rtld_objself.cksum->adler32) {
		dbg(("rtld checksum mismatch"));
		goto bad;
	}

	nobjs -= 2;

	/* Set some sanity-checking numbers in the Obj_Entry. */
	first->magic = RTLD_MAGIC;
	first->version = RTLD_VERSION;

	/* Fill in the dynamic linker entry points. */
	first->dlopen = _rtld_dlopen;
	first->dlsym = _rtld_dlsym;
	first->dlerror = _rtld_dlerror;
	first->dlclose = _rtld_dlclose;
	first->dladdr = _rtld_dladdr;

	first->relcache = &rcache[0];

	for (obj = first->next; obj != NULL; obj = obj->next) {
		for (i = 1; i < nents - 1; i++)
			if (rcache[i].cksum.crc32 == obj->cksum->crc32 &&
			    rcache[i].cksum.adler32 == obj->cksum->adler32 &&
			    rcache[i].filesize == obj->filesize &&
			    rcache[i].base == obj->database) {
				nobjs--;
				obj->relcache = &rcache[i];
			}

		/* Set some sanity-checking numbers in the Obj_Entry. */
		obj->magic = RTLD_MAGIC;
		obj->version = RTLD_VERSION;

		/* Fill in the dynamic linker entry points. */
		obj->dlopen = _rtld_dlopen;
		obj->dlsym = _rtld_dlsym;
		obj->dlerror = _rtld_dlerror;
		obj->dlclose = _rtld_dlclose;
		obj->dladdr = _rtld_dladdr;
	}

	if (nobjs != 0) {
		dbg(("shobj checksum mismatch"));
		goto bad;
	}

	for (obj = first; obj != NULL; obj = obj->next) {
		if (mmap(obj->database, obj->relcache->size,
		    obj->dataflags, MAP_FILE | MAP_PRIVATE | MAP_FIXED, fd,
		    obj->relcache->offset) == MAP_FAILED) {
			xwarn("mmap of data failed");
			goto bad;
		}
	}

	munmap(cachebase, _rtld_pagesz);
	close(fd);
	return true;
bad:
	munmap(cachebase, _rtld_pagesz);
bad2:
	close(fd);
	return false;
}
