/* $NetBSD$ */

/*
 * Copyright (C) 2009 Mihai Chelaru <kefren@netbsd.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD$");

#include "opt_inet.h"

#include <sys/param.h>
#include <sys/mbuf.h>

#include <net/netisr.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>

#ifdef INET6
#include <netinet/ip6.h>
#include <netinet6/in6_var.h>
#include <netinet6/ip6_var.h>
#endif

#include <netmpls/mpls.h>
#include <netmpls/mpls_var.h>

static struct mbuf *mpls_prepend_shim(struct mbuf *m, union mpls_shim *ms);

#ifdef INET
/*
 * We're having here an IP packet encapsulated in MPLS
 * Called from mpls_lse() in case label is MPLS_LABEL_IPV4NULL
 * This should pop the label and serve the within IP packet
 * to IP stack
 */
int
mpls_unlabel_inet(struct mbuf *m)
{
	int s, iphlen;
	struct ip *iph;
	union mpls_shim *ms;
	struct ifqueue *inq;

	if (mpls_mapttl_inet || mpls_mapprec_inet) {

		/* get shim info */
		ms = mtod(m, union mpls_shim *);
		ms->s_addr = ntohl(ms->s_addr);

		/* and get rid of it */
		m_adj(m, sizeof(union mpls_shim));

		/* get ip header */
		if (m->m_len < sizeof (struct ip) &&
		    (m = m_pullup(m, sizeof(struct ip))) == NULL)
			return ENOBUFS;
		iph = mtod(m, struct ip *);
		iphlen = iph->ip_hl << 2;

		/* get it all */
		if (m->m_len < iphlen) {
			if ((m = m_pullup(m, iphlen)) == NULL)
				return ENOBUFS;
			iph = mtod(m, struct ip *);
		}

		/* check ipsum */
		if (in_cksum(m, iphlen) != 0) {
			m_freem(m);
			return EINVAL;
		}

		/* set IP ttl from MPLS ttl */
		if (mpls_mapttl_inet)
			iph->ip_ttl = ms->shim.ttl;

		/* set IP Precedence from MPLS Exp */
		if (mpls_mapprec_inet) {
			iph->ip_tos = (iph->ip_tos << 3) >> 3;
			iph->ip_tos |= ms->shim.exp << 5;
		}

		/* reset ipsum because we modified TTL and TOS */
		iph->ip_sum = 0;
		iph->ip_sum = in_cksum(m, iphlen);
	} else
		m_adj(m, sizeof(union mpls_shim));

	/* Put it on IP queue */
	inq = &ipintrq;
	s = splnet();
	if (IF_QFULL(inq)) {
		IF_DROP(inq);
		splx(s);
		m_freem(m);
		return ENOBUFS;
	}
	IF_ENQUEUE(inq, m);
	splx(s);
	schednetisr(NETISR_IP);

	return 0;
}

/*
 * Prepend MPLS label
 */
struct mbuf *
mpls_label_inet(struct mbuf *m, const struct sockaddr_mpls *sam)
{
	union mpls_shim ms;
	struct ip *iphdr;

	ms.s_addr = ntohl(sam->smpls_addr.s_addr);
	ms.shim.ttl = mpls_defttl;
	ms.shim.bos = 1;
	ms.shim.exp = 0;

	if (mpls_mapttl_inet || mpls_mapprec_inet) {
		if ((m->m_len < sizeof(struct ip)) &&
		    ((m = m_pullup(m, sizeof(struct ip))) == 0))
			return NULL;
		iphdr = mtod(m, struct ip *);

		/* Map TTL */
		if (mpls_mapttl_inet)
			ms.shim.ttl = iphdr->ip_ttl;

		/* Copy IP precedence to EXP */
		if (mpls_mapprec_inet)
			ms.shim.exp = ((u_int8_t)iphdr->ip_tos) >> 5;
	}

	if ((m = mpls_prepend_shim(m, &ms)) == NULL)
		return NULL;
	
	return m;
}

#endif	/* INET */

#ifdef INET6

int
mpls_unlabel_inet6(struct mbuf *m)
{
	struct ip6_hdr *ip6hdr;
	union mpls_shim ms;
	struct ifqueue *inq;
	int s;

	/* TODO: mapclass */
	if (mpls_mapttl_inet6) {
		ms.s_addr = ntohl(mtod(m, union mpls_shim *)->s_addr);
		m_adj(m, sizeof(union mpls_shim));

		if (m->m_len < sizeof (struct ip6_hdr) &&
		    (m = m_pullup(m, sizeof(struct ip6_hdr))) == 0)
			return ENOBUFS;
		ip6hdr = mtod(m, struct ip6_hdr *);

		/* Because we just decremented this in mpls_lse */
		ip6hdr->ip6_hlim = ms.shim.ttl + 1;
	} else
		m_adj(m, sizeof(union mpls_shim));

	/* Put it back on IPv6 stack */
	schednetisr(NETISR_IPV6);
	inq = &ip6intrq;
	s = splnet();
	if (IF_QFULL(inq)) {
		IF_DROP(inq);
		splx(s);
		m_freem(m);
		return ENOBUFS;
	}

	IF_ENQUEUE(inq, m);
	splx(s);

	return 0;
}

struct mbuf *
mpls_label_inet6(struct mbuf *m, const struct sockaddr_mpls *sam)
{
	struct ip6_hdr *ip6h;
	union mpls_shim ms;

	ms.s_addr = ntohl(sam->smpls_addr.s_addr);
	ms.shim.bos = 1;
	ms.shim.exp = 0;
	ms.shim.ttl = mpls_defttl;

	if (mpls_mapttl_inet6 || mpls_mapclass_inet6) {
		if (m->m_len < sizeof(struct ip6_hdr))
			if ((m = m_pullup(m, sizeof(struct ip6_hdr))) == 0)
				return NULL;
		ip6h = mtod(m, struct ip6_hdr *);

		if (mpls_mapttl_inet6)
			ms.shim.ttl = ip6h->ip6_hlim;

		if (mpls_mapclass_inet6)
			ms.shim.exp = ip6h->ip6_vfc << 1 >> 5;
	}

	if ((m = mpls_prepend_shim(m, &ms)) == NULL)
		return NULL;

	return m;
}

#endif	/* INET6 */

static struct mbuf *
mpls_prepend_shim(struct mbuf *m, union mpls_shim *ms) 
{
	union mpls_shim *shim; 
 
	M_PREPEND(m, sizeof(union mpls_shim), M_DONTWAIT);
	if (m == NULL)
		return NULL;

	if (m->m_len < sizeof(union mpls_shim) &&
	    (m = m_pullup(m, sizeof(union mpls_shim))) == 0)
		return NULL;

	shim = mtod(m, union mpls_shim *);

	memcpy(shim, ms, sizeof(*shim));
	shim->s_addr = htonl(shim->s_addr);

	return m;
}
