/* $NetBSD$ */

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Mihai Chelaru <kefren@netbsd.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD$");

#include "opt_mpls.h"
#include "opt_inet.h"

#include <sys/param.h>
#include <sys/mbuf.h>
#include <sys/errno.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/route.h>

#include <netmpls/mpls.h>
#include <netmpls/mpls_var.h>

extern int mpls_forwarding;	/* should we forward MPLS frames ? */
extern int mpls_accept;		/* should we accept  MPLS frames ? */

static int mpls_send_frame(struct mbuf *, struct rtentry *);

/*
 * Label Switch Engine
 */
void
mpls_lse(struct mbuf *m)
{

	struct sockaddr_mpls dst;
	struct sockaddr_mpls *mplsgate;
	union mpls_shim tshim, *htag;
	struct rtentry *rt = NULL;
	int error = EINVAL;

	dst.smpls_len = sizeof(struct sockaddr_mpls);
	dst.smpls_family = AF_MPLS;
	dst.smpls_addr.s_addr = ntohl(mtod(m, union mpls_shim *)->s_addr);

	/* Check if we're accepting MPLS Frames */
	if (!mpls_accept)
		goto done;

	/* TTL decrement */
	if ((m = mpls_ttl_dec(m)) == NULL)
		goto done;

	if (dst.smpls_addr.shim.label <= MPLS_LABEL_RESMAX) {
		/* Reserved labels get special treatment */
		switch (dst.smpls_addr.shim.label) {
		/* Check for IPv[46] NULL labels */
		    case MPLS_LABEL_IPV4NULL:
			/* Pop shim and push mbuf to IP stack */
			if (dst.smpls_addr.shim.bos)
				error = mpls_unlabel_inet(m);
			break;
#ifdef INET6
		    case MPLS_LABEL_IPV6NULL:
			/* Pop shim and push mbuf to IPv6 stack */
			if (dst.smpls_addr.shim.bos)
				error = mpls_unlabel_inet6(m);
			break;
#endif
		    case MPLS_LABEL_RTALERT:	/* Yeah, I'm all alerted */
		    case MPLS_LABEL_IMPLNULL:	/* This is logical only */
		    default:			/* Rest are not allowed */
			break;
		}
		goto done;
	}

	/* Check if we should do MPLS forwarding */
	error = EHOSTUNREACH;
	if (!mpls_forwarding)
		goto done;

	/*
	 * We need to clear everything except label
	 * in order to get a route
	 */
	dst.smpls_addr.shim.ttl =
	    dst.smpls_addr.shim.bos =
	    dst.smpls_addr.shim.exp = 0;

	/* Get a route to dst */
	dst.smpls_addr.s_addr = htonl(dst.smpls_addr.s_addr);
	if ((rt = rtalloc1((const struct sockaddr*)&dst, 1)) == NULL)
		goto done;

	/* MPLS packet with no MPLS route ? */
	if (((rt->rt_flags & RTF_GATEWAY) == 0) ||
	    (rt->rt_gateway->sa_family != AF_MPLS))
		goto done;

	mplsgate = (struct sockaddr_mpls *)rt->rt_gateway;
	tshim.s_addr = ntohl(mplsgate->smpls_addr.s_addr);

	/* Swap labels */
	if ((m->m_len < sizeof(union mpls_shim)) &&
	    ((m = m_pullup(m, sizeof(union mpls_shim))) == 0)) {
		error = ENOBUFS;
		goto done;
	}

	/* Replace the shim */
	htag = mtod(m, union mpls_shim *);
	htag->s_addr = ntohl(htag->s_addr);
	htag->shim.label = tshim.shim.label;
	htag->s_addr = htonl(htag->s_addr);

	error = mpls_send_frame(m, rt);

done:
	if (error && m)
		m_freem(m);
	if (rt)
		RTFREE(rt);
}

static int
mpls_send_frame(struct mbuf *m, struct rtentry *rt)
{

	struct ifnet *ifp;

	/* If gateway is not MPLS we have nothing to do */
	if (((rt->rt_flags & RTF_GATEWAY) == 0) ||
	    (rt->rt_gateway->sa_family != AF_MPLS))
		return EHOSTUNREACH;

	rt->rt_use++;

	ifp = rt->rt_ifp;

	switch(ifp->if_type) {
		/* only this two are supported for now */
		case IFT_ETHER:
		case IFT_TUNNEL:
			return (*ifp->if_output)(ifp, m, rt->rt_gateway, rt);
		case IFT_LOOP:
			break;
		default:
			return ENETUNREACH;
	}
	return 0;
}
