/* $NetBSD$ */

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Mihai Chelaru <kefren@NetBSD.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the NetBSD
 *        Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netmpls/mpls.h>

#include <sys/param.h>

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "ldp.h"
#include "ldp_peer.h"
#include "ldp_errors.h"
#include "tlv_stack.h"
#include "label.h"
#include "mpls_interface.h"
#include "mpls_routes.h"

#define ROUNDUP(a) \
	((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))

int 
mpls_add_label(struct ldp_peer * p, struct rt_msg * inh_rg,
    struct in_addr * addr, int len, int label, int rlookup)
{
	char            padd[20];

	int             kount = 0, rv;
	union sockunion *so_dest, *so_pref = NULL, *so_gate;
	struct rt_msg   rg;
	struct rt_msg	*rgp = &rg;
	struct label	*lab;

	strlcpy(padd, inet_ntoa(p->address), 20);
	debugp("Trying to add %s/%d as label %d to peer %s\n", inet_ntoa(*addr),
		len, label, padd);

	/* Don't accept default route */
	if (!len)
		return LDP_E_BAD_AF;

	/* Is there a label mapping for this ? */
	if (ldp_peer_get_lm(p, addr, len) == NULL)
		return LDP_E_NOENT;

	if (!inh_rg) {
		/*
		 * XXX: Check if we have a route for that.
		 * Why the hell isn't kernel inserting the route immediatly ?
		 * let's loop until we have it..
		 */

		so_dest = make_inet_union(inet_ntoa(*addr));
		if (len != 32)
			so_pref = from_cidr_to_union(len);

		do {
			if (kount == rlookup) {
				debugp("No route for this prefix\n");
				return LDP_E_NO_SUCH_ROUTE;
			}
			kount++;
			/* Last time give it a higher chance */
			if (kount == rlookup)
				usleep(5000);

			rv = get_route(rgp, so_dest, so_pref, 1);
			if (rv != LDP_E_OK && len == 32)
				/* Host maybe ? */
				rv = get_route(rgp, so_dest, NULL, 1);
		} while (rv != LDP_E_OK);

		free(so_dest);
		if (so_pref)
			free(so_pref);

	} else
		rgp = inh_rg;

	/* Check if it's an IPv4 route */

	so_gate = (union sockunion *) rgp->m_space;

	so_gate = (union sockunion *)((char*)so_gate +
	    ROUNDUP(so_gate->sa.sa_len));

	if (so_gate->sa.sa_family != AF_INET) {
		debugp("Failed at family check\n");
		return LDP_E_BAD_AF;
	}

	/* Check if the address is bounded to the peer */
	if (check_ifaddr(p, &so_gate->sin.sin_addr) == NULL) {
		debugp("Failed at next-hop check\n");
		return LDP_E_ROUTE_ERROR;
	}

	/* CHECK IF WE HAVE A BINDING FOR THAT */
	lab = label_get_by_prefix(addr, len);

	/* We should have a label because we have a route */
	assert (lab);

	if (lab->binding == MPLS_LABEL_IMPLNULL) {
		change_local_label(lab, get_free_local_label());
		if (!lab->binding) {
			fatalp("No more free labels !!!\n");
			return LDP_E_TOO_MANY_LABELS;
		}
	}

	warnp("[mpls_add_label] Adding %s/%d as local binding %d, label %d"
	    " to peer %s\n",
		inet_ntoa(*addr), len, lab->binding, label, padd);

	/* Modify existing label */
	lab->label = label;
	lab->p = p;

	/* Add switching route */
	so_dest = make_mpls_union(lab->binding);
	so_gate = make_mplsinet_union(0, label, &p->address);
	if (add_route(so_dest, NULL, so_gate, NULL, FREESO, RTM_ADD) != LDP_E_OK)
		return LDP_E_ROUTE_ERROR;

	if (label != MPLS_LABEL_IMPLNULL) {
		/* Change IPv4 route to MPLS */
		so_dest = make_inet_union(inet_ntoa(*addr));

		/* if prefixlen == 32 check if it's inserted as host
	 	* and treat it as host. It may also be set as /32 prefix
	 	* (thanks mlelstv !)
	 	*/
		if ((len == 32) && (rgp->m_rtm.rtm_flags & RTF_HOST))
			so_pref = NULL;
		else
			so_pref = from_cidr_to_union(len);

		/* Add tag to route */
		so_gate = make_mpls_union(label);
		if (add_route(so_dest, so_pref, NULL, so_gate, FREESO, RTM_CHANGE) !=
		    LDP_E_OK)
			return LDP_E_ROUTE_ERROR;
	} /* label not POP */

	debugp("Added %s/%d as label %d to peer %s\n", inet_ntoa(*addr), len,
	    label, padd);

	return LDP_E_OK;
}

int 
mpls_add_ldp_peer(struct ldp_peer * p)
{
	return LDP_E_OK;
}

int 
mpls_delete_ldp_peer(struct ldp_peer * p)
{

	/* Reput all the routes also to IPv4 */
	label_reattach_all_peer_labels(p, LDP_READD_CHANGE);

	return LDP_E_OK;
}

int 
mpls_start_ldp()
{
	ldp_peer_init();
	label_init();

	return LDP_E_OK;
}
