/* $Id: launder_netbsd.c,v 1.20 1999/02/24 01:44:36 proff Exp $
 * $Copyright$
 * $Log: launder_netbsd.c,v $
 * Revision 1.20  1999/02/24 01:44:36  proff
 * set LAUNDER_THIEVED when packet has been stolen from the network stack
 *
 * Revision 1.19  1999/02/17 18:33:58  proff
 * remove old pfil/ip_fw hooks
 *
 * Revision 1.18  1999/02/17 18:25:17  proff
 * bugfix thieving support
 *
 * Revision 1.17  1999/02/15 10:43:12  proff
 * inline a few functions
 *
 * disable open()/ioctl() (etc) locking. We only permit one reader for now
 * anyway. Ideally we would only allow one packet stealer, and infinitely
 * many readers.
 *
 * Revision 1.16  1999/02/02 19:15:46  proff
 * spelling police
 *
 * Revision 1.15  1999/02/02 19:07:25  proff
 * support for THEIVE/ALOOF ioctls
 *
 * Revision 1.14  1999/02/01 16:18:20  proff
 * bugfix (< instead of >)
 *
 * Revision 1.13  1999/01/31 18:52:42  proff
 * fix bozotic typo in launderread (while ();{})
 *
 * spl locking consistency
 *
 * Revision 1.12  1999/01/31 12:47:11  proff
 * simplfy / rationalise length checks etc.
 *
 * Revision 1.11  1999/01/30 20:51:43  proff
 * Handle PPP and ipflow_fastfoward interception
 *
 * Revision 1.10  1999/01/30 19:01:51  proff
 * Support localhost/other output. This will work with all ethernet
 * cards, via a hook in sys/net/ethersubr.c:ether_output(). the status of
 * other devices is as yet unknown.
 *
 * Revision 1.9  1999/01/30 16:31:25  proff
 * merge struct launder_hdr into struct washing_line, as many of the fields were
 * incommon, and the former was being built out of the latter anyway.
 *
 * int/u_int consistency changes.
 *
 * add LSC_TIMEOUT flag and associated code to handle wait-for-min-queue-packets
 * time out during poll/select. there are some potential issues here: if the
 * poll is interrupted (say, by an alarm) and restarted the packet-wait timer will
 * be reset to zero. timer-since-start-of-poll isn't terribly userful.
 * timer-since-last-read-completion is probably want is called for.
 *
 * bugfix: free washingline mbufs on device close
 *
 * bugfix: ifp->if_launder was set incorrectly
 *
 * Revision 1.8  1999/01/26 16:00:26  proff
 * remove checksumming code. implementation showed launder checksuming to
 * be a bad design decision, primarily stemming from conflict between the
 * protocol-independent nature of the launder device, and inherent
 * protocol dependency of checksumming operations. cksum flags have been
 * maintained for the case where data originated from the local host, and
 * for userland clients to set when the data is being forwarded to or
 * through the localhost protocol stack.
 *
 * Revision 1.7  1999/01/26 15:46:49  proff
 * IP / TCP checksumming
 *
 * Revision 1.6  1999/01/26 06:59:21  proff
 * New ioctl:
 *
 * 	LAUNDERIOCSTATS
 *
 * Faster calculation of launder iface via ifnet->if_launder
 *
 * Revision 1.5  1999/01/24 19:59:49  proff
 * debugged. suburbia.net -> fxp0 -> launder -> launder_test (userland)
 * 	  -> launder -> fxp0 -> suburbia.net now working
 *
 * Revision 1.4  1999/01/23 08:19:07  proff
 * Expand interface attach/detach framework.
 *
 * Compiles both as a module and in-kernel. Otherwise untested.
 *
 * Revision 1.3  1999/01/23 05:13:08  proff
 * Coarse device locking implemented via LSC_LOCKED/LSC_LOCKREQ.
 *
 * New ioctls and backing code for:
 *
 * 	LAUNDERIOCSNAPLEN,
 * 	LAUNDERIOCPROMISC,
 * 	LAUNDERIOCMONOGAMOUS
 *
 * flags:
 * 	LAUNDER_IGNORE
 *
 * Interface name cache redesigned.
 *
 * Transition to low level driver [i.e protocols other than ip]
 * intercepts 70% complete. Packets originating from the localhost
 * still have to be delt with.
 *
 * Other small code cleanups.
 *
 * Compiles, but otherwise untested.
 *
 * Revision 1.2  1999/01/22 15:05:12  proff
 * few substantial additions for promisc mode. not complete.
 *
 * Revision 1.1.1.1  1999/01/22 03:32:43  proff
 * initial import
 *
 */

#include "../h/kernel_netbsd.h"
#include "../h/pqueue.h"

#include "launder-api.h"

cdev_decl(launder);

#define  DEBUG_LAUNDER

void	 launderattach __P((int));

#ifdef _LKM
void	 launderdetach __P((void));
#endif

#ifdef DEBUG_LAUNDER
#  define DB if (launder_gflags&LAUNDER_DEBUG) printf
#else
#  define DB 0 && printf
#endif

#define NLAUNDER 1
#define NUM_LAUNDER_IFACES	32	/* max number of interface attaches */
#define LQUEUE_MAX (1024*1024)	/* in bytes, excluding mbuf overhead */
#define LQUEUE_MIN (LQUEUE_MAX/2)	/* when queue becomes readable */

#ifndef M_PFIL
#  define M_PFIL M_DEVBUF	/* malloc type */
#endif
#define LHLEN LAUNDER_WORDALIGN(sizeof(struct launder_hdr))

#define launderunit(dev) (minor(dev)>>1&0xf)

static struct timeval launder_def_wait = {0, 1*1000 /* 10 miliseconds */};

typedef enum
{
    LSC_OPEN		= 1<<0,	/* device has been opened */
    LSC_ACTIVE		= 1<<1,	/* device has been configured */
    LSC_LOCKED		= 1<<2, /* device locked */
    LSC_LOCKREQ		= 1<<3, /* lock requested */
    LSC_TIMEOUT		= 1<<4	/* timeout has occurred */
} launder_sc_flags;

struct washing_line
{
    PQ_STAILQ_ENTRY(struct washing_line) washing_thread;
    struct mbuf *m;		/* packet */
    struct launder_hdr hdr;	/* launder launder -- see launder-api.h */
};

PQ_STAILQ_HEAD(washing_line_head, struct washing_line);

struct launder_softc
{
    int sc_ref;			/* reference count */
    int sc_readers;		/* tsleep()ing readers */
    u_32 sc_flags;		/* flags */
    u_32 sc_queue_max;		/* max queue size */
    u_32 sc_queue_min;		/* min queue size */
    u_32 sc_queue_size;		/* bytes used by queue */
    struct washing_line_head sc_whead;/* chain of queued packets */
    u_int sc_timeout;		/* timeout for device read/poll, in ticks */
    struct selinfo sc_selq;	/* select/poll queue */
    u_int sc_snaplen;		/* bytes of packet to capture, 0 = all */
    struct launder_stats sc_stats; /* launder statistics */
} *launder_softc;

typedef enum
{
    LI_PROMISC,			/* interface has been set to promisc mode */
    LI_THIEVE			/* steal non-promisc packets */
} lauder_iface_flag;

struct launder_iface
{
    struct ifnet *ifp;
    u_int flags;
    int ifnum;
    int ref;
} launder_iface[NUM_LAUNDER_IFACES];

static int num_launder = 0;
static u_32 launder_gflags = LAUNDER_FAILOPEN;

#define ENTERSC(dev)\
    struct launder_softc *sc;\
    do\
    {\
    int eerr;\
    if (launderunit(dev) >= num_launder)\
	return (ENXIO);\
    sc = &launder_softc[launderunit(dev)];\
    eerr = launder_lock(sc);\
    if (eerr!=0)\
	return (eerr);\
    } while(0)

#define EXITSC(err)\
    do\
    {\
        launder_unlock(sc);\
        return(err);\
    } while(0)


/*
 * Wait interruptibly for an exclusive lock.
 *
 * XXX
 * Several drivers do this; it should be abstracted and made MP-safe.
 */
static int
launder_lock(struct launder_softc *sc)
{
#if 0
    int error;
    DB("launder_lock(%p)\n", sc);
    while ((sc->sc_flags & LSC_LOCKED) != 0)
	{
	    sc->sc_flags |= LSC_LOCKREQ;
	    if ((error = tsleep(sc, PUSER | PCATCH, "launlck", 0)) != 0)
		return error;
	}
    if (num_launder<1)
	return ENXIO;
    sc->sc_flags |= LSC_LOCKED;
#endif
    return (0);
}

/*
 * Unlock and wake up any waiters. `garcon!'
 */
static void
launder_unlock(struct launder_softc *sc)
{
#if 0
    DB("launder_unlock(%p)\n", sc);
    sc->sc_flags &= ~LSC_LOCKED;
    if ((sc->sc_flags & LSC_LOCKREQ) != 0)
	{
	    sc->sc_flags &= ~LSC_LOCKREQ;
	    wakeup(sc);
	}
#endif
}

#ifdef DEBUG_LAUNDER_X
static char *
launder_inet_ntoa(struct in_addr addr)
{
    u_char *p=(char*)&addr;
#define P(x) ((int)(p[x]))
    static u_char res[3*1+4*3+1];
    sprintf(res, "%d.%d.%d.%d", P(0), P(1), P(2), P(3));
    return res;
}
#endif 

inline static struct launder_iface *
launder_num2iface(int n)
{
    DB("launder_num2iface(%d)\n", n);
    return &launder_iface[n];
}

inline static struct ifnet *
launder_num2ifnet(int n)
{
    struct launder_iface *lif;
    DB("launder_num2ifnet(%d)\n", n);
    lif = launder_num2iface(n);
    if (!lif)
	return NULL;
    return lif->ifp;
}

/* attach to an interface. returns zero on success,
 * error on failure. 
 * *ifnum will contain the assigned interface number,
 * or -1 respectively.
 */

static int
launder_iface_attach(int *ifnum, char *ifname)
{
    struct ifnet *ifp;
    int n;
    int free_n = -1;
    int s;
    struct launder_iface *lif;
    DB("launder_iface_attach(%p, %p)", ifnum, ifname);
    
    s = splnet();
    ifp = ifunit(ifname);
    if (!ifp)
	{
	    splx(s);
	    return ENXIO;
	}
    for (n=0; n<NUM_LAUNDER_IFACES; n++)
	{
	    lif = &launder_iface[n];
	    if (lif->ifp == ifp)
		{
		    splx(s);
		    return EBUSY;
		}
	    else
		if (free_n<0 && !lif->ifp)
		    free_n = n;
	}
    if (free_n<0)
	{
	    splx(s);
	    return ENOBUFS;
	}
    lif = &launder_iface[free_n];
    lif->ifp = ifp;
    lif->ifnum = free_n;
    lif->ref++;
    ifp->if_launder = (caddr_t)lif;
    splx(s);
    *ifnum = free_n;
    return 0;
}

/*
 * detach an interface
 */

static void
launder_iface_detach(struct launder_iface *lif)
{
    int s;
    DB("launder_iface_detach(%p)", lif);
    
    s = splnet();
    if (lif->ref>0 && lif->ref-- == 1)
	{
	    if (lif->flags&LI_PROMISC)
		ifpromisc(lif->ifp, 0);
	    lif->ifp->if_launder = NULL;
	    bzero(lif, sizeof lif);
	}
    splx(s);
    return;
}

static u_int
launder_tv2ticks(struct timeval *tv)
{
    u_int t;
    /* Compute number of ticks. */
    t = tv->tv_sec * hz + tv->tv_usec / tick;	/* XXX overflow */
    if (t == 0 && (tv->tv_usec != 0))
	t = 1;
    return t;
}

inline static bool
launder_readable(struct launder_softc *sc)
{
    bool t;
    int s = splnet();
    t = (sc->sc_queue_size >= sc->sc_queue_min ||
	 ((sc->sc_flags&LSC_TIMEOUT) &&
	  PQ_STAILQ_FIRST(&sc->sc_whead)));
    splx(s);
    return t;
}

static void
launder_clear (struct launder_softc *launder, struct proc *p)
{
    struct washing_line *wl;
    int n;
    int s;
    DB("launderclear(%p, %p)\n", launder, p);
    s = splnet();
    launder->sc_flags &=~LSC_ACTIVE;
    if (launder->sc_ref > 1) /* 1 ref for the fd maybe doing the ioctl() */
	printf ("launderclear() warning: reference count = %d\n", launder->sc_ref);
    while ((wl = PQ_STAILQ_FIRST(&launder->sc_whead)))
	{
	    PQ_STAILQ_REMOVE_HEAD(&launder->sc_whead, washing_thread);
	    m_freem(wl->m);
	    free(wl, M_PFIL);
	}
    for (n=0; n<NUM_LAUNDER_IFACES; n++)
	{
	    struct launder_iface *lif = launder_num2iface(n);
	    if (lif)
		launder_iface_detach(lif);
	}
    bzero(launder, sizeof *launder);
    splx(s);
}

EXPORT int launderopen (dev_t dev, int flags, int fmt, struct proc *p)
{
    ENTERSC(dev);
    DB("launderopen(%d, %d, %d, %p)\n", dev, flags, fmt, p);
    if (sc->sc_flags&LSC_OPEN)
	{
	    sc->sc_ref++;
	    return 0;
	}
    bzero(sc, sizeof *sc);
    sc->sc_ref = 1;
    sc->sc_queue_max = LQUEUE_MAX;
    sc->sc_queue_min = LQUEUE_MIN;
    sc->sc_timeout = launder_tv2ticks(&launder_def_wait);
    PQ_STAILQ_INIT(&sc->sc_whead, NULL);
    sc->sc_flags|=LSC_OPEN;
    EXITSC(0);
}

EXPORT int launderclose (dev_t dev, int flags, int mode, struct proc *p)
{
    int s;
    ENTERSC(dev);
    DB("launderclose(%d, %d, %d, %p)\n", dev, flags, mode, p);
    s = splnet();
    if (sc->sc_ref > 0)
	{
	    if (--sc->sc_ref == 0)
		launder_clear(sc, p);
	}
    splx(s);
    EXITSC (0);
}

/*
 * advance uio n bytes (but don't actually read/write
 * anything)
 */

inline static void
launder_uioskip(struct uio *uio, int n)
{
    DB("launder_uioskip(%p, %d)\n", uio, n);
    while (n > 0 && uio->uio_resid > 0)
	{
	    struct iovec *iov = uio->uio_iov;
	    u_int cnt = iov->iov_len;
	    if (cnt == 0)
		{
		    uio->uio_iov++;
		    uio->uio_iovcnt--;
		    continue;
		}
	    if (cnt > n)
		cnt = n;
	    (caddr_t)iov->iov_base += cnt;
	    iov->iov_len -= cnt;
	    uio->uio_resid -= cnt;
	    uio->uio_offset += cnt;
	    n -= cnt;
	}
}

/*
 * move out len bytes of mbuf chain m. we presume uio has been
 * checked to be size-appropriate
 */

inline static void
launder_mbuf2uio(struct mbuf *m, struct uio *uio, int len)
{
    DB("launder_mbuf2uio(%p, %p, %d)\n", m, uio, len);
    for (; m; m = m->m_next)
	{
	    int mlen = min(m->m_len, len);
	    uiomove(mtod(m, caddr_t), mlen, uio);
	    len-=mlen;
	    if (len<=0)
		break;
	}
    return;
}

    
EXPORT int launderread(dev_t dev, struct uio *uio, int flag)
{
    int s;
    int error;
    struct washing_line *wl;
    ENTERSC(dev);
    DB("launderread(%d, %p, %d)\n", dev, uio, flag);
    if (!launder_readable(sc))
	{
	    sc->sc_readers++;
	    error = tsleep((caddr_t) sc, PCATCH, "laread", sc->sc_timeout);
	    sc->sc_readers--;
	    if (error && error != EWOULDBLOCK)
		EXITSC(error);
	}
    sc->sc_flags&=~LSC_TIMEOUT;
    s = splnet();
    wl = PQ_STAILQ_FIRST(&sc->sc_whead);
    splx(s);
    while (wl)
	{
	    int ulen;
	    int len;
	    /* only whole packets */
	    if (uio->uio_resid < LHLEN + wl->hdr.lh_caplen)
		EXITSC(EINVAL);

	    /* move launder_header to userland */
	    wl->hdr.lh_hdrlen = LHLEN;
	    uiomove(&wl->hdr, LHLEN, uio);

	    /* adjust size if we are using snaplen */
	    ulen = wl->hdr.lh_caplen;
	    if (sc->sc_snaplen>0)
		ulen = MIN(ulen, sc->sc_snaplen);

	    /* packet to userland */
	    launder_mbuf2uio(wl->m, uio, ulen);

	    /* add virtual padding */
	    len = LAUNDER_WORDALIGN(wl->hdr.lh_caplen);
	    if (len > ulen)
		launder_uioskip(uio, len - ulen);

	    /* free washing and pegs */
	    s = splnet();
	    PQ_STAILQ_REMOVE_HEAD(&sc->sc_whead, washing_thread);
	    m_freem(wl->m);
	    sc->sc_queue_size -= sizeof(*wl) + wl->hdr.lh_caplen;
	    free(wl, M_PFIL);
	    wl = PQ_STAILQ_FIRST(&sc->sc_whead);
	    splx(s);
	}
    EXITSC(0);
}


inline static struct mbuf *
launder_uio2mbuf(struct uio *uio, int ilen)
{
    struct mbuf *m0 = NULL;
    struct mbuf *m = NULL;
    int len = ilen;

    DB("launder_uio2mbuf(%p, %d)\n", uio, ilen);
    for (;;)
	{
	    int mlen;
	    if (!m0)
		{
		    m0 = m_gethdr(M_NOWAIT, M_DEVBUF);
		    if (!m0)
			return NULL;
		    m = m0;
		    m->m_len = MHLEN;
		}
 	    else
		{
		    struct mbuf *n;
		    n = m_get(M_WAIT, M_DEVBUF);
		    if (!n)
			{
			    m_freem(m0);
			    return NULL;
			}
		    m->m_next = n;
		    m = n;
		    m->m_len = MLEN;
		}
	    if (len >= MINCLSIZE)
		{
		    MCLGET(m, M_WAIT);
		    if (m->m_flags & M_EXT)
			m->m_len = MCLBYTES;
		}
	    mlen = MIN(uio->uio_resid, MIN(m->m_len, len));
	    uiomove(mtod(m, caddr_t), mlen, uio);
	    m->m_len = mlen;
	    len-=mlen;
	    if (len>0)
		{
		    if (uio->uio_resid<=0) /* short write */
			{
			    m_freem(m0);
			    printf("launder_io2mbuf() short write\n");
			    return NULL;
			}
		}
	    else
		break;
	}
    m0->m_pkthdr.len = ilen-len;
    return m0;
}

EXPORT int launderwrite(dev_t dev, struct uio *uio, int flag)
{
    struct ifnet *ifp;
    int error = 0;
    ENTERSC(dev);
    DB("launderwrite(%d, %p, %d)\n", dev, uio, flag);

    while (uio->uio_resid > 0) /* more data to go? */
	{
	    char buf[LHLEN];
	    struct launder_hdr *h = (struct launder_hdr *) buf;
	    struct mbuf *m;
	    int ulen;
	    int len;

	    /* userland has at least a launder_header for us? */
	    if (uio->uio_resid < LHLEN)
		{
		einv:
		    error = EINVAL;
		    goto err;
		}

	    /* okay, bring in the header */
	    uiomove((caddr_t)h, LHLEN, uio);

	    /* header size okay? */
	    if (h->lh_hdrlen != LHLEN)
		goto einv;

	    ulen = h->lh_caplen;

	    /* ignore zero byte packets */
	    if (ulen == 0)
		continue;

	    /* rest of the packet the right size? */
	    if (ulen<0 || ulen>uio->uio_resid)
		goto einv;

	    /* align upto the boundary of the next packet */
	    len = MIN(LAUNDER_WORDALIGN(ulen), uio->uio_resid);

	    /* ignore this packet altogether? */
	    if (h->lh_flags&LAUNDER_IGNORE)
		{
		    launder_uioskip(uio, len);
		    continue;
		}

	    /* iface okay? */
	    if (!(ifp = launder_num2ifnet(h->lh_ifnum)))
		goto einv;

	    /* pull entire packet into an mbuf chain */
	    m = launder_uio2mbuf(uio, ulen);
	    if (!m)
		{
		    error = ENOBUFS;
		    goto err;
		}

	    /* move past padding, if any */
	    if (len != ulen)
		launder_uioskip(uio, len-ulen);
#ifdef notneededanymore
	    m->m_flags&=~M_CANFASTFWD;	/* prevent fast routing */
#endif
	    m->m_flags|=M_LAUNDER;	/* prevent loopback */
	    if (h->lh_flags&LAUNDER_MULTICAST)
		m->m_flags|=M_MCAST;
	    if (h->lh_flags&LAUNDER_BROARDCAST)
		m->m_flags|=M_BCAST;

	    /* intercepted this packet on the way in? */
	    if (h->lh_flags&LAUNDER_INPUT)
		{
		    /* okay, restore receiving iface information */
		    m->m_pkthdr.rcvif = ifp;
		    /*
		     * inject packet into appropriate input queue
		     */
		    switch (h->lh_type)
			{
			    case LP_ETHER:	/* ethernet */
				{
				    struct ether_header *eh = mtod(m, struct ether_header *);
				    m->m_data += sizeof(*eh);
				    m->m_pkthdr.len -= sizeof(*eh);
				    m->m_len -= sizeof(*eh);
				    ether_input(ifp, eh, m);
				}
				break;
			    case LP_IP:		/* raw ip, no link-layer */
				{
				    int s;
				    struct ifqueue *ifq = &ipintrq;
#ifdef GATEWAY
				    if ((h->lh_flags&LAUNDER_IPFLOW) &&
					ipflow_fastforward(m)) /* XXX splnet()/ */
					break;
#endif
				    s = splnet();
				    if (IF_QFULL(ifq))
					{
					    IF_DROP(ifq); /* XXX blocking */
					    m_freem(m);
					    splx(s);
					}
				    else
					{
					    IF_ENQUEUE(ifq, m);
					    /* schedule software interrupt to process ip_input queue */
					    splx(s);
					    schednetisr(NETISR_IP);
					}
				}
				break;
			default:
			    DB("launderwrite(): unknown packet type %d", h->lh_type);
			    break;
				
			}
		    continue;
		}
	    else 
		{    /*
		      * i.e LAUNDER_OUTPUT 
		      */
		    m->m_pkthdr.rcvif = NULL;
		    if (h->lh_flags&LAUNDER_PPP)
			{
			    struct sockaddr dst;
			    /* ppp supports AF_UNSPEC (used for
			     * control information) and AF_INET only */
			    dst.sa_family = AF_INET;
			    (void)pppoutput(ifp, m, &dst, NULL);
			}
		    else
			{
			    int s = splnet();
			    /*
			     * queue message on interface, and start it up if
			     * not yet active.
			     */
			    if (IF_QFULL(&ifp->if_snd))
				{
				    /* XXX count dropped packets */
				    IF_DROP(&ifp->if_snd);
				}
			    else
				{
				    ifp->if_obytes += m->m_pkthdr.len;
				    IF_ENQUEUE(&ifp->if_snd, m);
				    if ((ifp->if_flags & IFF_OACTIVE) == 0)
					(*ifp->if_start)(ifp);
				    if (m->m_flags & M_MCAST)
					ifp->if_omcasts++;
				}
			    splx(s);
			}
		    continue;
		}

	}
 err:
    EXITSC(error);
}

EXPORT int launderioctl (dev_t dev, u_long cmd, caddr_t data, int flag, struct proc *p)
{
    int error;
    ENTERSC(dev);
    DB("launderioctl(%d, %lu, %p, %d, %p)\n", dev, cmd, data, flag, p);
    switch (cmd)
	{
	case LAUNDERIOCGSET:
	    launder_gflags = *(u_int *)data;
	    break;
	case LAUNDERIOCGGET:
	    *(u_int *)data = launder_gflags;
	    break;
	case LAUNDERIOCTIMEOUT:
	    sc->sc_timeout = launder_tv2ticks((struct timeval *)data);
	    break;
	case LAUNDERIOCMINQUEUE:
	    sc->sc_queue_min = *(u_int *)data;
	    break;
	case LAUNDERIOCMAXQUEUE:
	    sc->sc_queue_max = *(u_int *)data;
	    break;
	case LAUNDERIOCACTIVATE:
	    if (*(int*)data)
		sc->sc_flags |= LSC_ACTIVE;
	    else
		sc->sc_flags &= ~LSC_ACTIVE;
	    break;
	case LAUNDERIOCATTACH:
	    ;{
		struct launder_attach *p = (struct launder_attach *)data;
		error = launder_iface_attach(&p->la_ifnum, p->la_ifname);
		if (error)
		    EXITSC (error);
	    }
	    break;
	case LAUNDERIOCDETACH:
	    ;{
		struct launder_iface *lif = launder_num2iface(*(int *)data);
		if (!lif)
		    EXITSC (ENXIO);
		launder_iface_detach(lif);
	    }
	    break;
	case LAUNDERIOCSNAPLEN:
	    sc->sc_snaplen = *(u_int *)data;
	    break;
	case LAUNDERIOCPROMISC:
	    {
		struct launder_iface *lif = launder_num2iface(*(int *)data);
		if (!lif)
		    EXITSC (ENXIO);
		if (!(lif->flags&LI_PROMISC))
		    {
			error = ifpromisc(lif->ifp, 1);
			if (error)
			    EXITSC (error);
			lif->flags|=LI_PROMISC;
		    }
	    }
	    break;
	case LAUNDERIOCMONOGAM:
	    {
		struct launder_iface *lif = launder_num2iface(*(int *)data);
		if (!lif)
		    EXITSC (ENXIO);
		if (lif->flags&LI_PROMISC)
		    {
			error = ifpromisc(lif->ifp, 0);
			if (error)
			    EXITSC (error);
			lif->flags&=~LI_PROMISC;
		    }
	    }
	    break;
	case LAUNDERIOCSTATS:
	    memcpy(data, &sc->sc_stats, sizeof sc->sc_stats);
	    break;
	case LAUNDERIOCTHIEVE:
	    {
		struct launder_iface *lif = launder_num2iface(*(int *)data);
		if (!lif)
		    EXITSC (ENXIO);
		lif->flags|=LI_THIEVE;
	    }
	    break;
	case LAUNDERIOCALOOF:
	    {
		struct launder_iface *lif = launder_num2iface(*(int *)data);
		if (!lif)
		    EXITSC (ENXIO);
		lif->flags&=~LI_THIEVE;
	    }
	    break;
	default:
	    EXITSC (ENOTTY);
	}
    EXITSC (0);
}

static void
launder_poll_timeout(void *arg)
{
    struct launder_softc *sc = arg;

    /* device shutdown under us? */
    if (!(sc->sc_flags&LSC_ACTIVE))
	return;

    sc->sc_flags|=LSC_TIMEOUT;
    if (PQ_STAILQ_FIRST(&sc->sc_whead))
	selwakeup(&sc->sc_selq);
}

EXPORT int launderpoll(dev_t dev, int events, struct proc *p)
{
    int revents;
    ENTERSC(dev);
    DB("launderpoll(%d, %d, %p)\n", dev, events, p);
    /* we are always writable */
    revents = events & (POLLOUT | POLLWRNORM);
    /* save some work if not checking for reads */
    if ((events & (POLLIN | POLLRDNORM)) == 0)
	EXITSC (revents);

    /* anything on the washingline? */

    if (launder_readable(sc) ||
	(sc->sc_timeout == 0 && PQ_STAILQ_FIRST(&sc->sc_whead)))
	revents|=events & (POLLIN | POLLRDNORM);
    else
	{
	    /* leave the mark of our tapping foot */
	    selrecord(p, &sc->sc_selq);
	    if (sc->sc_timeout>0)
		timeout(launder_poll_timeout, sc, sc->sc_timeout);
	}
   EXITSC(revents);
}

/*
 * inject packet into launder queue. returns 0 if the caller
 * can keep a copy of the packet, 1 if the caller must nolonger touch it.
 * in this latter case *mp is set to NULL.
 */

EXPORT int
launder_input(struct ifnet *ifp, struct mbuf **mp, launder_pkt_t type, int flags)
{
    struct washing_line *wl;
    int s;
    int mlen;
    struct mbuf *m = *mp;
    struct mbuf *new_m;
    struct launder_softc *sc;
    struct launder_iface *lif;
    /* seen this packet before? */
    if (m->m_flags & M_LAUNDER)
	return 0;

    sc = &launder_softc[0];

    /* launder activated? correct interface? */
    lif = (struct launder_iface *)(ifp->if_launder);
    if (!lif || !(sc->sc_flags&LSC_ACTIVE))
	{
	failtest:
	    if (launder_gflags & LAUNDER_FAILOPEN)
		{
		    return 0;
		}
	    else
		{
		dropqueue:
		    sc->sc_stats.ls_drop_queue++;
		    sc->sc_stats.ls_dropped++;
		    *mp = NULL;
		    m_freem(m);
		    return 1;
		}
	}

    DB("launder_input(%p, %p, %d, %d)\n", ifp, mp, type, flags);

    mlen = sizeof(struct washing_line) + m->m_pkthdr.len;

    /* room on the washing line? */
    if (mlen + sc->sc_queue_size > sc->sc_queue_max)
	goto dropqueue;
    
    /* malloc a peg */
    wl = malloc(sizeof *wl, M_PFIL, M_NOWAIT);
    if (!wl)
	goto dropqueue;

    if (flags&LAUNDER_PROMISC)
	new_m = NULL;
    else
	{
	    /* are we stealing this packet from the network stack? */
	    if (lif->flags&LI_THIEVE)
		{
		    flags|=LAUNDER_THIEVED;
		    new_m = NULL;
		}
	    else
		{
		    /* leave a copy for the network stack */
		    new_m = m_copym(m, 0, M_COPYALL, M_DONTWAIT);
		    if (!new_m)
			{
			    free(wl, M_PFIL);
			    goto failtest;
			}
		    m = new_m;
		}
	}
    wl->m = m;
    wl->hdr.lh_caplen = m->m_pkthdr.len;
    wl->hdr.lh_tstamp = time; /* XXX microtime() is slow on some archs, but has better resolution */
    if (m->m_flags&M_MCAST)
	flags|=LAUNDER_MULTICAST;
    if (m->m_flags&M_BCAST)
	flags|=LAUNDER_BROARDCAST;
    wl->hdr.lh_flags = flags;
    wl->hdr.lh_type = type;
    wl->hdr.lh_ifnum = ((struct launder_iface *)(ifp->if_launder))->ifnum;
    s = splnet();
    /* hang mr mbuf out to dry */
    PQ_STAILQ_INSERT_TAIL(&sc->sc_whead, wl, washing_thread);
    sc->sc_queue_size += mlen;
    /* `time to bring the washing in man' */
    if (launder_readable(sc))
	{
	    splx(s);
	    if (sc->sc_readers>0)
		{
		    wakeup((caddr_t) sc);
		}
	    selwakeup(&sc->sc_selq);
	}
    else
	{
	    splx(s);
	}
    return (new_m==NULL);
}

#ifdef _LKM
/*
 * detach all launder devices
 */

EXPORT void launderdetach (void)
{
    int i;
    int s;
    DB("launderdetach()\n");
    s = splnet();
#if 0
    pfil_remove_hook(launder_fw_hook, PFIL_IN|PFIL_OUT);
#endif
    for (i = 0; i < num_launder; i++)
	{
	    struct launder_softc *sc = &launder_softc[i];
	    if (sc->sc_flags&LSC_OPEN)
		{
		    launder_clear (sc, initproc);
		}
	}
    splx(s);
}
#endif

/*
 * allocate and initialise num launder devices
 */

EXPORT void launderattach(int num)
{
    struct launder_softc *mem;
    u_long size;
    DB("launderattach(%d)\n", num);
    if (num < 1)
	return;
    size = num * sizeof(struct launder_softc);
    mem = malloc(size, M_PFIL, M_NOWAIT);
    if (mem == NULL)
	{
	    printf("WARNING: couldn't allocate %d bytes for %d launder devices\n", (int)size, num);
	    return;
	}
    bzero(mem, size);
    launder_softc = mem;
    num_launder = num;
#if 0
    pfil_add_hook(launder_fw_hook, PFIL_IN|PFIL_OUT);
#endif
}
