#include "standard.h"
#include "network.h"

#include "launder-api.h"

static void
print_flags(u_int flags)
{
    int count=0;
    u_int mask;
    for (mask = 1; mask; mask<<=1)
	{
	    char *s;
	    switch(flags&mask)
		{
		case 0:
		    s = NULL;
		    break;
		case LAUNDER_INPUT:
		    s = "INPUT";
		    break;
		case LAUNDER_PROMISC:
		    s = "PROMISC";
		    break;
		case LAUNDER_IGNORE:
		    s = "IGNORE";
		    break;
		case LAUNDER_IPSUM:
		    s = "IPSUM";
		    break;
		case LAUNDER_IPSUMOK:
		    s = "IPSUMOK";
		    break;
		case LAUNDER_TCPSUM:
		    s = "TCPSUM";
		    break;
		case LAUNDER_TCPSUMOK:
		    s = "TCPSUMOK";
		    break;
		case LAUNDER_MULTICAST:
		    s = "MULTICAST";
		    break;
		case LAUNDER_BROARDCAST:
		    s = "BROARDCAST";
		    break;
		case LAUNDER_PPP:
		    s = "PPP";
		    break;
		case LAUNDER_IPFLOW:
		    s = "IPFLOW";
		    break;
		default:
		    s = "UNKNOWN";
		    break;
		}
	    if (s)
		printf("%s%s", (count++>0)? ",": "", s);
	}
}

static void
usage(char *argv0)
{
    fprintf(stderr, "usage: %s [-cCdvpPTv] [-i interface] [-m min_queue] [-M max_queue] [-t timeout_secs[,timeout_usecs]] [-s snaplen] -w dump_file\n", argv0);
    fprintf(stderr, "\
-c		set launder FAILOPEN flag\n\
-C		clear launder FAILOPEN flag\n\
-d		set launder DEBUG flag\n\
-i interface	attach interface interface to laudner device (multiple)\n\
-m min_queue	set MINQUEUE (bytes)\n\
-M max_queue	set MAXQUEUE (bytes)\n\
-p		place the last -i iface into promiscuous mode (multiple)\n\
-P		place the last -i iface into monogamuous mode (multiple)\n\
-s snaplen	set snaplen (bytes) (0=inifinite)\n\
-t secs[,usecs] time to block waiting for MINQUEUE packets\n\
-T		inject thieved packets back onto stack\n\
-v		increase verbosity (multiple)\n\
-w dump_file	dump launder packets to dump_file\n");
}

static u_64 count, last_count;
static struct timeval last_time;
static int launder_fd;

static void
sig_alrm()
{
    if (count!=last_count)
	{
	    float secs;
	    struct launder_stats ls;
	    struct timeval tv2;
	    gettimeofday(&tv2, NULL);
	    secs = (tv2.tv_sec - last_time.tv_sec) +
		(tv2.tv_usec - last_time.tv_usec)/1000000.0;
	    if (ioctl(launder_fd, LAUNDERIOCSTATS, &ls) != 0)
		err(1, "LAUNDERIOCSTATS");
	    printf("%.2f Kb, %.2f Kb/s dropped: %d\n", (float)count/1024.0, (float)(count-last_count)/1024.0 / secs, (int)ls.ls_drop_queue);
	    last_count = count;
	    last_time = tv2;
	}
    alarm(1);
}

int
main(int argc, char **argv)
{
    int c;
    char *dev = "/dev/launder";
    int fd;
    u_long flags = 0;
    int opt_verbose = 0;
    bool opt_write = FALSE;
    char *opt_write_file = NULL;
    int wfd;
    int on = 1;
    int current_ifnum = -1;

    launder_fd = fd = open(dev, O_RDWR);
    if (fd<0)
	err(1, dev);

    while ((c = getopt(argc, argv, "cCdi:m:M:pPs:t:Tvw:")) != -1)
	switch(c)
	    {
	    case 'c':
		flags |= LAUNDER_FAILOPEN;
		break;
	    case 'C':
		flags &= ~LAUNDER_FAILOPEN;
		break;
	    case 'd':
		flags |= LAUNDER_DEBUG;
		break;
	    case 'i':
		{
		    struct launder_attach la;
		    strncpy(la.la_ifname, optarg, sizeof la.la_ifname);
		    if (ioctl(fd, LAUNDERIOCATTACH, &la)!=0)
			err(1, "LAUNDERIOCATTACH: %s", dev);
		    current_ifnum = la.la_ifnum;
		}
		break;
	    case 'm':
		{
		    u_32 val = atoi(optarg);
		    if (ioctl(fd, LAUNDERIOCMINQUEUE, &val)!=0)
			err(1, "LAUNDERIOCMINQUEUE: %s", dev);
		}
		break;
	    case 'M':
		{
		    u_32 val = atoi(optarg);
		    if (ioctl(fd, LAUNDERIOCMAXQUEUE, &val)!=0)
			err(1, "LAUNDERIOCMAXQUEUE: %s", dev);
		}
		break;
	    case 'p':
		if (ioctl(fd, LAUNDERIOCPROMISC, &current_ifnum)!=0)
		    err(1, "LAUNDERIOCPROMISC: %s", dev);
		break;
	    case 'P':
		if (ioctl(fd, LAUNDERIOCMONOGAM, &current_ifnum)!=0)
		    err(1, "LAUNDERIOCMONOGAM: %s", dev);
		break;
	    case 's':
		{
		    u_int val = atoi(optarg);
		    if (ioctl(fd, LAUNDERIOCSNAPLEN, &val)!=0)
			err(1, "LAUNDERIOCSNAPLEN: %s", dev);
		}
		break;
	    case 't':
		{
		    char *p;
		    struct timeval tv;
		    tv.tv_sec = atoi(optarg);
		    if ((p = strchr(optarg, ',')) != NULL)
			tv.tv_usec = atoi(++p);
		    else
			tv.tv_usec = 0;
		    if (ioctl(fd, LAUNDERIOCTIMEOUT, &tv)!=0)
			err(1, "LAUNDERIOCTIMEOUT: %s", dev);
		}
		break;
	    case 'T':
		if (ioctl(fd, LAUNDERIOCTHIEVE, &current_ifnum)!=0)
		    err(1, "LAUNDERIOCTHIEVE: %s", dev);
		opt_write = TRUE;
		break;
	    case 'v':
		opt_verbose++;
		break;
	    case 'w':
		opt_write_file = optarg;
		break;
	    default:
		usage(argv[0]);
		exit(1);
	    }
    if (opt_write_file)
	{
	    wfd = open(opt_write_file, O_WRONLY|O_CREAT|O_TRUNC, 0700);
	    if (wfd<0)
		err(1, "%s", opt_write_file);
	}
    if (ioctl(fd, LAUNDERIOCGSET, &flags)!=0)
	err(1, "LAUNDERIOCGET: %s", dev);
    if (ioctl(fd, LAUNDERIOCACTIVATE, &on)!=0)
	err(1, "LAUNDERIOCACTIVATE: %s", dev);
    signal(SIGALRM, sig_alrm);
    sig_alrm();
    do
	{
	    int n;
	    int cc;
	    char buf[1024*1024];
	    struct launder_hdr *h = (struct launder_hdr*)buf;
	    struct pollfd pfd;
	    pfd.fd=fd;
	    pfd.events = POLLIN;
	    pfd.revents = 0;
	    
	    if (poll(&pfd, 1, -1)<1)
		{
		    if (errno == EINTR)
			continue;
		    err(1, "poll");
		}
	    if (!(pfd.revents&pfd.events))
		continue;
	    cc = read(fd, buf, sizeof buf);
	    if (!cc)
		continue;
	    if (cc<0)
		err(1, dev);
	    if (opt_write_file && write(wfd, buf, cc) != cc)
		err(1, "%s", opt_write_file);
	    if (last_count == count)
		{
		    gettimeofday(&last_time, NULL);
		}
	    for (n=0;;n++)
		{
		    if (opt_verbose>1)
			{
			    printf("iface:%d %d bytes flags=",
				   h->lh_ifnum,
				   h->lh_caplen);
			    print_flags(h->lh_flags);
			    printf("\n");
			}
		    count+=h->lh_caplen;
		    if (!(h->lh_flags&LAUNDER_THIEVED))
			h->lh_flags|=LAUNDER_IGNORE; /* don't route non-thieved packets */
		    h = (struct launder_hdr*)((char*)h + h->lh_hdrlen + LAUNDER_WORDALIGN(h->lh_caplen));
		    if ((char*)h>=(buf+cc))
			break;
		}
	    if (opt_verbose>0)
		{
		    printf("[%d] ", n);
		    fflush(stdout);
		}
	    if (opt_write && write (fd, buf, cc)!=cc)
		err(1, "write failed: %s", dev);
	} while (1);
    if (opt_write_file)
	close(wfd);
    sig_alrm();
}

