$NetBSD$
--- zaptel/zaptel.c.orig	2007-09-13 18:24:38.000000000 +0100
+++ zaptel/zaptel.c	2008-10-30 09:13:43.000000000 +0000
@@ -60,6 +60,8 @@
 #include <sys/filio.h>
 #include <sys/kthread.h>
 
+/* workarounds and #defines for portability cruft */
+#include "zaptel-compat.h"
 #include "zconfig.h"
 
 #include "zaptel.h"
@@ -73,6 +75,7 @@
 /* arith.h *must* be after  zaptel.h */
 #include "arith.h"
 
+static void zt_init(void);
 
 #ifdef MALLOC_DEFINE
 MALLOC_DEFINE(M_ZAP, "zaptel", "ZapTel interface data structures");
@@ -80,6 +83,16 @@
 #define M_ZAP M_DEVBUF
 #endif
 
+static int zapdebug = 2;
+#ifdef DEBUG
+#define DPRINTF(m, x)   if (zapdebug >= (m)) printf x
+#define ZAPATA_DEBUG 1
+#else
+#define DPRINTF(m, x)   /* nop */
+#define ZAPATA_DEBUG 0
+#endif
+
+#ifdef __FreeBSD__
 MODULE_VERSION(zaptel, 1);
 
 static devclass_t zt_devclass;
@@ -115,23 +128,95 @@
 		.d_poll			= zt_poll,
 		.d_name			= "zaptel"
 };
-
 static eventhandler_tag zt_ehtag;
 
+#elif defined(__NetBSD__)
+dev_type_open(zt_open);
+const struct cdevsw zaptel_cdevsw = {
+        zt_open, noclose, noread, nowrite, noioctl, nostop, notty,
+        nopoll, nommap, nokqfilter, 0
+};
+
+int zaptelmatch(struct device *, struct cfdata *, void *);
+void zaptelattach(struct device *, struct device *, void *);
+struct device *zaptel_attach_mi(struct device *);
+int zaptelprint(void *, const char *);
+
+CFATTACH_DECL(zaptel, sizeof(struct device), zaptelmatch, zaptelattach,
+        NULL, NULL);
+/* this should only ever be called on a match */
+void
+zaptelattach(struct device *dev, struct device *parent, void *aux) {
+        zt_init();
+}
+
+int
+zaptelmatch(struct device *dev, struct cfdata *cf, void *aux) {
+	return 1;
+}
+
+struct device *
+zaptel_attach_mi(struct device *dev)
+{
+        return (config_found(dev, NULL, zaptelprint));
+}
+
+int
+zaptelprint(void *aux, const char *pnp)
+{
+        if (pnp != NULL)
+                aprint_normal("zaptel at %s", pnp);
+        return (UNCONF);
+}
+
+typedef int     fo_rdwr_t(struct file *, off_t *, struct uio *,
+                                kauth_cred_t, int);
+typedef int     fo_ioctl_t(struct file *, u_long, void *, zt_proc_t *p);
+typedef int     fo_fcntl_t(struct file *, u_int, void *, zt_proc_t *p);
+typedef int     fo_poll_t(struct file *, int, zt_proc_t *);
+typedef int     fo_kqfilter_t(struct file *, struct knote *);
+typedef int     fo_stat_t(struct file *, struct stat *, zt_proc_t *);
+typedef int     fo_close_t(struct file *, zt_proc_t *);
+
+fo_fcntl_t                      zt_fcntl;
+fo_close_t                      zt_close;
+fo_ioctl_t                      zt_ioctl;
+fo_rdwr_t                       zt_read;
+fo_rdwr_t                       zt_write;
+fo_poll_t                       zt_poll;
+fo_stat_t                       zt_stat;
+fo_kqfilter_t           zt_kqfilter;
+
+static struct fileops zt_fileops = {
+        .fo_close = zt_close,
+        .fo_ioctl = zt_ioctl,
+        .fo_fcntl = zt_fcntl,
+        .fo_read = zt_read,
+        .fo_write = zt_write,
+        .fo_poll =  zt_poll,
+        .fo_stat = zt_stat
+};
+
+#endif /* __NetBSD__ */
+
 extern int pulse_pause;
 extern int pulse_make;
 extern int pulse_break;
+static int zt_open_timers = 0;
 
+/* save device types for corespondent units */
+static int unit_2_minor[32*32];
+#define MASK_BLOCKS 32
+static u_int32_t zap_units_mask[MASK_BLOCKS];
 extern struct zt_span *spans[ZT_MAX_SPANS];
+  
+#ifdef __FreeBSD__
 
 /* 
  * These are masks for units bit 0 - means it's free and vice versa. 
 */
 
-#define MASK_BLOCKS 32
-
-static int zt_open_timers = 0;
-static u_int32_t zap_units_mask[MASK_BLOCKS];
+static lock_t toselwakeup_lock;
 static void * toselwakeup[256];
 static struct proc    *kthread;
 static int            kthread_must_exit;
@@ -140,13 +225,20 @@
 void zt_sysctl_init(void);
 void zt_sysctl_destroy(void);
 
-/* save device types for corespondent units */
-static int unit_2_minor[32*32];
-
-#ifdef __FreeBSD__
 /* sysctl stuff */
 static struct sysctl_oid * zt_root, * pulse_root;
 static struct sysctl_ctx_list zt_clist;
+#elif defined(__NetBSD__)
+static int zt_nodenum;
+static int zt_debug_nodenum;
+static int zt_deftaps_nodenum;
+static int zt_pulse_nodenum;
+static int zt_pulse_make_nodenum;
+static int zt_pulse_break_nodenum;
+static int zt_pulse_pause_nodenum;
+static struct sysctllog *zaptel_log;
+static int zt_sysctl_init(void);
+static int zt_sysctl_verify(SYSCTLFN_PROTO);
 #endif
 
 
@@ -156,8 +248,6 @@
 int debug = 0;
 
 
-static lock_t toselwakeup_lock;
-
 /* There are three sets of conference sum accumulators. One for the current
 sample chunk (conf_sums), one for the next sample chunk (conf_sums_next), and
 one for the previous sample chunk (conf_sums_prev). The following routine 
@@ -200,9 +290,59 @@
 #define PPP_FCS(fcs, c)	  (((fcs) >> 8) ^ fcstab[((fcs) ^ (c)) & 0xff])
 
 
+static void
+zt_init(void) {
+	zt_sysctl_init();
+	/* 
+	 * A kind of hack. Without any pre-created device 
+	 * zt_clone will get "zap" string as name instead of 
+	 * "zap/timer" or something
+	 */
+#ifdef __FreeBSD__
+	dev_ctl = make_dev(&zt_devsw, 0, UID_ROOT, GID_WHEEL, 0664, 
+			   "zap/ctl");
+#endif
+
+	printf("Zapata Telephony Interface Registered on major %d\n", 
+		   ZT_MAJOR);
+	printf("Echo Canceller: %s\n", ZAPTEL_ECHO_CANCELLER ZAPTEL_ECHO_AGGRESSIVE);
+
+	bzero(zap_units_mask, sizeof(zap_units_mask));
+	/* 
+	 * Note, that this is indirect initialization 
+	 * of unit_2_minor[0] to devtype 0 (ctl)
+	 */
+	bzero(unit_2_minor, sizeof(unit_2_minor));
+
+#ifdef __FreeBSD__
+
+	zt_devclass = devclass_find("zaptel");
+#endif
+	zt_conv_init();
+	tone_zone_init();
+	rotate_sums();
+
+	zt_timers_start();
+	zt_channels_start();
+	zt_tonezone_start();
+#ifdef __FreeBSD__
+	bzero(toselwakeup, sizeof(toselwakeup));
+	ZAP_LOCK_INIT(toselwakeup_lock, "toselwakeup_lock");
+
+	kthread_must_exit = 0;
+	if (kthread_create(selwakeup_thread_handler, (void *)toselwakeup, &kthread, 0, 0, "zt_selwakeup_thread"))
+	{
+		printf("Failed to create kthread\n");
+	}
+#endif
 
+#ifdef CONFIG_ZAPTEL_WATCHDOG
+	watchdog_init();
+#endif	
+}
 
 
+#ifdef __FreeBSD__
 /*
  * This hack is attempt of INTR_FAST/selwakeup incompatibility 
  * fix. INTR_FAST nature denies any work with sleep mutexes
@@ -266,6 +406,7 @@
 		tsleep(kthread, PZERO, "waitdata", 10);
 	}
 }
+#endif
 
 /*
  * Find first free block and give it away. -1 if no free blocks
@@ -289,6 +430,7 @@
 	return -1;
 }
 
+#ifdef __FreeBSD__
 static __inline int zt_get_unit(struct cdev * dev)
 {
 	struct dev_info * info;
@@ -298,8 +440,16 @@
 	if(!info)
 		return 0;
 	return info->unit;
-	
 }
+#elif defined(__NetBSD__)
+static __inline int zt_get_unit(struct file * dev) {
+	struct zt_fileinfo *fi = dev->f_data;
+	if (fi)
+		return fi->unit;
+	else
+		return 0;
+}
+#endif
 
 /* 
  * mark the unit busy/free.
@@ -323,16 +473,27 @@
 	unit_2_minor[unit] = real_minor;
 }
 
+#ifdef __FreeBSD__
 static __inline int zt_get_minor(struct cdev * dev)
 {
 	int unit = zt_get_unit(dev);
 	return unit_2_minor[unit];
 }
+#elif defined(__NetBSD__)
+static __inline int zt_get_minor(struct file * dev)
+{
+	struct zt_fileinfo *fi = dev->f_data;
+	if (fi)
+		return fi->minor;
+	else
+		return 0;
+}
+#endif
 
+#ifdef __FreeBSD__
 void zt_sysctl_init()
 {
 
-#ifdef __FreeBSD__
 	sysctl_ctx_init(&zt_clist);
 
 	zt_root = SYSCTL_ADD_NODE(&zt_clist,
@@ -365,16 +526,136 @@
 	SYSCTL_ADD_INT(&zt_clist, SYSCTL_CHILDREN(pulse_root),
 		OID_AUTO, "pause", CTLFLAG_RW, &pulse_pause, 0, 
 		"pause between series(milliseconds)");
-#endif
 }
+#endif
 
 void zt_sysctl_destroy()
 {
 #ifdef __FreeBSD__
 	sysctl_ctx_free(&zt_clist);
+#elif defined(__NetBSD__)
+	sysctl_teardown(&zaptel_log);
 #endif
 }
 
+#ifdef __NetBSD__
+static int
+zt_sysctl_init(void)
+{
+#if defined (__NetBSD__) && (__NetBSD_Version__ < 399000600)
+	struct sysctlnode *node;
+#else
+	const struct sysctlnode *node;
+#endif
+	int error = 0;
+
+	if ((error = sysctl_createv(&zaptel_log, 0, NULL, NULL, 0,
+	    CTLTYPE_NODE, "hw", NULL, NULL, 0, NULL, 0,
+	    CTL_HW, CTL_EOL)) != 0)
+		return error;
+
+	if ((error = sysctl_createv(&zaptel_log, 0, NULL, &node, 0,
+	    CTLTYPE_NODE, "zaptel", SYSCTL_DESCR("zaptel controls"),
+	    NULL, 0, NULL, 0, CTL_HW, CTL_CREATE, CTL_EOL)) != 0)
+		return error;
+	zt_nodenum = node->sysctl_num;
+	DPRINTF(2,("%s: added sysctl node %d\n", __func__, zt_nodenum));
+
+	if ((error = sysctl_createv(&zaptel_log, 0, NULL, &node,
+	    CTLFLAG_READWRITE, CTLTYPE_INT, "debug",
+	    SYSCTL_DESCR("zaptel debug level"), zt_sysctl_verify,
+	    0, &zapdebug, 0, CTL_HW, zt_nodenum, CTL_CREATE,
+	    CTL_EOL)) != 0)
+		return error;
+	zt_debug_nodenum = node->sysctl_num;
+	DPRINTF(2,("%s: added sysctl node %d\n", __func__,
+	    zt_debug_nodenum));
+
+	if ((error = sysctl_createv(&zaptel_log, 0, NULL, &node,
+	    CTLFLAG_READWRITE, CTLTYPE_INT, "deftaps",
+	    SYSCTL_DESCR("deftaps"), zt_sysctl_verify,
+	    0, &deftaps, 0, CTL_HW, zt_nodenum, CTL_CREATE,
+	    CTL_EOL)) != 0)
+		return error;
+	zt_deftaps_nodenum = node->sysctl_num;
+	DPRINTF(2,("%s: added sysctl node %d\n", __func__,
+	    zt_deftaps_nodenum));
+
+	if ((error = sysctl_createv(&zaptel_log, 0, NULL, &node,
+	    0, CTLTYPE_NODE, "pulse",
+	    SYSCTL_DESCR("zaptel pulse dialing settings"), NULL,
+	    0, NULL, 0, CTL_HW, zt_nodenum, CTL_CREATE,
+	    CTL_EOL)) != 0)
+		return error;
+	zt_pulse_nodenum = node->sysctl_num;
+	DPRINTF(2,("%s: added sysctl node %d\n", __func__,
+	    zt_pulse_nodenum));
+
+	if ((error = sysctl_createv(&zaptel_log, 0, NULL, &node,
+	    CTLFLAG_READWRITE, CTLTYPE_INT, "make",
+	    SYSCTL_DESCR("make time (milliseconds)"), zt_sysctl_verify,
+	    0, &pulse_make, 0, CTL_HW, zt_nodenum, zt_pulse_nodenum,
+	    CTL_CREATE, CTL_EOL)) != 0)
+		return error;
+	zt_pulse_make_nodenum = node->sysctl_num;
+	DPRINTF(2,("%s: added sysctl node %d\n", __func__,
+	    zt_pulse_make_nodenum));
+
+	if ((error = sysctl_createv(&zaptel_log, 0, NULL, &node,
+	    CTLFLAG_READWRITE, CTLTYPE_INT, "break",
+	    SYSCTL_DESCR("break time (milliseconds)"), zt_sysctl_verify,
+	    0, &pulse_break, 0, CTL_HW, zt_nodenum, zt_pulse_nodenum,
+	    CTL_CREATE, CTL_EOL)) != 0)
+		return error;
+	zt_pulse_break_nodenum = node->sysctl_num;
+	DPRINTF(2,("%s: added sysctl node %d\n", __func__,
+	    zt_pulse_break_nodenum));
+
+	if ((error = sysctl_createv(&zaptel_log, 0, NULL, &node,
+	    CTLFLAG_READWRITE, CTLTYPE_INT, "pause",
+	    SYSCTL_DESCR("pause between series (milliseconds)"),
+	    zt_sysctl_verify, 0, &pulse_pause, 0, CTL_HW,
+	    zt_nodenum, zt_pulse_nodenum, CTL_CREATE, CTL_EOL)) != 0)
+		return error;
+	zt_pulse_pause_nodenum = node->sysctl_num;
+	DPRINTF(2,("%s: added sysctl node %d\n", __func__,
+	    zt_pulse_pause_nodenum));
+
+	return 0;
+}
+
+static int
+zt_sysctl_verify(SYSCTLFN_ARGS)
+{
+	int error, t;
+	struct sysctlnode node;
+
+	node = *rnode;
+	t = *(int *)rnode->sysctl_data;
+	node.sysctl_data = &t;
+	error = sysctl_lookup(SYSCTLFN_CALL(&node));
+	if (error || newp == NULL)
+		return error;
+
+	if (node.sysctl_num == zt_debug_nodenum ||
+		node.sysctl_num == zt_deftaps_nodenum ||
+		node.sysctl_num == zt_pulse_make_nodenum ||
+		node.sysctl_num == zt_pulse_break_nodenum ||
+		node.sysctl_num == zt_pulse_pause_nodenum) {
+		if (t < 0)
+			return EINVAL;
+	} else {
+		DPRINTF(1,("%s: got invalid sysctl_num %d\n", __func__,
+			node.sysctl_num));
+		return EINVAL;
+	}
+
+	*(int *)rnode->sysctl_data = t;
+
+	return 0;
+}
+#endif /* __NetBSD__ */
+
 
 
 int schluffen(void *q)
@@ -389,6 +670,7 @@
 	return 0;
 }
 
+#ifdef __FreeBSD__
 /* Clone device */
 static void
 #if __FreeBSD_version < 600000
@@ -457,21 +739,107 @@
 	return res;
 }
 	
+	
+#elif defined(__NetBSD__)
+
+int
+zt_clone(struct file **filep, dev_t dev, int flags, void *data, zt_proc_t *p)
+{
+	/* Clone device
+	 * We need one instance for every open() call since the linux
+	 * version operates with ZT_SPECIFY and sends several channels on
+	 * /dev/zap/channel.
+	 */
+
+	int fd, error, ret, u;
+	struct file *fp;
+
+	struct zt_fileinfo *info;
+
+	if (filep)
+		*filep = 0;  /* XXX riz is this leaking memory?? */
+
+	info = malloc(sizeof(struct zt_fileinfo), M_ZAP, M_NOWAIT);
+	if (!info)
+		return ENOMEM;
+
+	u = next_free_unit(zap_units_mask, MASK_BLOCKS);
+
+	if(u < 0) /* no free units */
+		return;
+
+	mark_unit(zap_units_mask, MASK_BLOCKS, u, 1);
+
+	info->minor = minor(dev);
+	info->unit = next_free_unit(zap_units_mask, MASK_BLOCKS);
+	zt_set_minor(u, info->minor);
+	info->data = data;
+
+	if (curlwp->l_dupfd >= 0) {
+		return ENODEV;
+	}
+
+	/* Create a new file and allocate a descriptor for proc p */
+#if defined(__NetBSD__)
+	if ((error = falloc(p, &fp, &fd)) != 0)
+#else
+	if ((error = falloc(p->l_proc, &fp, &fd)) != 0)
+#endif
+		return error;
+	if (filep)
+		*filep = fp;
+
+	ret = fdclone(p, fp, fd, flags & FMASK, &zt_fileops, NULL);
+	fp->f_data = info;
+	return ret;
+}
+
+static int zt_ctl_open(dev_t dev, int flags, zt_proc_t * threadp)
+{
+	int res;
+	res = zt_clone(0, dev, flags, 0, threadp);
+	return res;
+}
+
+#endif
+
+
+
+#ifdef __FreeBSD__
 static int zt_ctl_close(struct cdev * dev)
 {
 	return 0;
 }
 
+#elif defined(__NetBSD__)
+static int zt_ctl_close(struct file *fp)
+{
+	/* because we allocated some data in the clone call */
+	free(fp->f_data, M_ZAP);
+	return 0;
+}
+#endif
 
 	
-
-int zt_open(struct cdev * dev, int flag, int mode, struct thread *threadp)
+#ifdef __FreeBSD__
+int zt_open(struct cdev * dev, int flag, int mode, zt_proc_t *threadp)
+#elif defined (__NetBSD__)
+int zt_open(dev_t dev, int flag, int mode, zt_proc_t *threadp)
+#endif
 {
+#ifdef __FreeBSD__
 	int unit = zt_get_minor(dev);
+#elif defined (__NetBSD__)
+	int unit = minor(dev);
+	struct file *fp;
+#endif
 	struct zt_chan *chan;
 	int res;
 
+#ifdef __FreeBSD__
+	/* In NetBSD, this is handled by the open function directly */
 	zt_set_flags(dev, flag);
+#endif
 	/* Minor 0: Special "control" descriptor */
 	if (!unit) 
 		return zt_ctl_open(dev, flag, threadp);
@@ -495,9 +863,20 @@
 		if (maxspans) {
 			chan = zt_alloc_pseudo();
 			if (chan) {
+#ifdef __FreeBSD__
 
 				dev->si_drv1 = chan;
 				res = zt_specchan_open(dev, chan->channo, 1);
+#elif defined(__NetBSD__)
+                                zt_clone(&fp, dev, flag, chan, threadp);
+                                res = zt_specchan_open(fp, chan->channo, 1);
+                                /* XXX - riz */
+                                DPRINTF(2,("ztopen: zt_specchan_open returned %d\n",res));
+                                if(!res)
+                                        return ENXIO;
+                                else
+                                        return res;
+#endif
 				return res;
 			} else {
 				return ENXIO;
@@ -506,14 +885,17 @@
 			return ENXIO;
 	}
 
-	return zt_specchan_open(dev, unit, 1);
+	zt_clone(&fp, dev, flag, chan, threadp);
+	return zt_specchan_open(fp, unit, 1);
 }
 
-
-
-
+#ifdef __FreeBSD__
 int 
 zt_read(struct cdev *dev, struct uio *uio, int ioflag)
+#elif defined(__NetBSD__)
+int zt_read(struct file *dev, off_t * off, struct uio * uio,
+        kauth_cred_t active_cred, int flags)
+#endif
 {
 	int unit = zt_get_minor(dev);
 	struct zt_chan *chan;
@@ -527,14 +909,14 @@
 		return EINVAL;
 	
 	if (unit == ZAP_CHANNEL_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+		chan = ZT_CHAN(dev);
 		if (!chan)
 			return EINVAL;
 		return zt_chan_read(uio, chan->channo);
 	}
 	
 	if (unit == ZAP_PSEUDO_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+		chan = ZT_CHAN(dev);
 		if (!chan) {
 			printf("No pseudo channel structure to read?\n");
 			return EINVAL;
@@ -545,8 +927,13 @@
 	return zt_chan_read(uio, unit);
 }
 
+#ifdef __FreeBSD__
 int 
 zt_write(struct cdev *dev, struct uio *uio, int ioflag)
+#elif defined(__NetBSD__)
+int zt_write(struct file *dev, off_t * off, struct uio * uio,
+        kauth_cred_t active_cred, int flags)
+#endif
 {
 	int unit = zt_get_minor(dev);
 	struct zt_chan *chan;
@@ -565,13 +952,13 @@
 		return EINVAL;
 
 	if (unit == ZAP_CHANNEL_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+		chan = ZT_CHAN(dev);
 		if (!chan)
 			return EINVAL;
 		return zt_chan_write(uio, chan->channo);
 	}
 	if (unit == ZAP_PSEUDO_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+		chan = ZT_CHAN(dev);
 		if (!chan) {
 			printf("No pseudo channel structure to read?\n");
 			return EINVAL;
@@ -583,15 +970,12 @@
 }
 
 
-
-
-
-
-
-
-
+#ifdef __FreeBSD__
 int 
-zt_close(struct cdev *dev, int flag, int otyp, struct thread *td)
+zt_close(struct cdev *dev, int flag, int otyp, zt_proc_t *td)
+#elif defined(__NetBSD__)
+int zt_close(struct file *dev, zt_proc_t *p)
+#endif
 {
 	int unit = zt_get_minor(dev);
 	int res;
@@ -611,8 +995,8 @@
 			zt_open_timers--;
 	}
 	else if (unit == ZAP_CHANNEL_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
-		dev->si_drv1 = NULL;
+		chan = ZT_CHAN(dev);
+		ZT_CHAN(dev) = NULL;
 		if (!chan)
 			res = zt_chan_close(dev);
 		else
@@ -620,7 +1004,7 @@
 		mark_unit(zap_units_mask, MASK_BLOCKS, zt_get_unit(dev), 0); 
 	}
 	else if (unit == ZAP_PSEUDO_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+		chan = ZT_CHAN(dev);
 		if (chan) {
 			res = zt_specchan_close(dev, chan->channo);
 			zt_free_pseudo(chan);
@@ -635,10 +1019,14 @@
 		res = zt_specchan_close(dev, unit);
 	}
 
-	/* Free dev_info if exist */
+	/* Free dev_info if exist */
+#ifdef __FreeBSD__
 	if(dev->si_drv2) free(dev->si_drv2, M_ZAP);
 	dev->si_drv2 = NULL;
-	destroy_dev(dev);	
+  	destroy_dev(dev);	
+#elif defined(__NetBSD__)
+	free(dev->f_data, M_ZAP);
+#endif
 	return res;
 }
 
@@ -687,7 +1075,11 @@
 #endif
 }
 
+#ifdef __FreeBSD__
 static int zt_ctl_ioctl(struct cdev *dev, u_long cmd, void * arg)
+#elif defined(__NetBSD__)
+static int zt_ctl_ioctl(struct file *dev, u_long cmd, void * arg)
+#endif
 {
 	/* I/O CTL's for control interface */
 	int i,j;
@@ -1018,38 +1410,42 @@
 }
 
 
+#ifdef __FreeBSD__
 int zt_ioctl(struct cdev *dev, u_long cmd, caddr_t arg, int mode,
-    struct thread *td)
+    zt_proc_t *td)
+#elif defined(__NetBSD__)
+int zt_ioctl(struct file *dev, u_long cmd, void *arg, zt_proc_t *td)
+#endif
 {
-	int unit = zt_get_minor(dev);
+	int minor = zt_get_minor(dev);
 	struct zt_chan *chan;
 	struct zt_timer *timer;
 	int res;
 
-	if (!unit)
+	if (!minor)
 		return zt_ctl_ioctl(dev, cmd, arg);
 
-	if (unit == ZAP_TIMER_MINOR) {
-		timer = (struct zt_timer *)dev->si_drv1;
+	if (minor == ZAP_TIMER_MINOR) {
+		timer = ZT_TIMER(dev);
 		if (timer)
 			return zt_timer_ioctl(dev, cmd, arg, timer);
 		else
 			return EINVAL;
 	}
-	if (unit == ZAP_CHANNEL_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+	if (minor == ZAP_CHANNEL_MINOR) {
+		chan = ZT_CHAN(dev);
 		if (chan)
 		{
 			res = zt_chan_ioctl(dev, cmd, arg, chan->channo);
 		}
 		else
 		{
-			res = zt_prechan_ioctl(dev, cmd, arg, unit);
+			res = zt_prechan_ioctl(dev, cmd, arg, minor);
 		}
 		return res;
 	}
-	if (unit == ZAP_PSEUDO_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+	if (minor == ZAP_PSEUDO_MINOR) {
+		chan = ZT_CHAN(dev);
 		if (!chan) {
 			printf("No pseudo channel structure to read?\n");
 			return EINVAL;
@@ -1064,7 +1460,7 @@
 		}
 		return zt_chanandpseudo_ioctl(dev, cmd, arg, chan->channo);
 	}
-	return zt_chan_ioctl(dev, cmd, arg, unit);
+	return zt_chan_ioctl(dev, cmd, arg, minor);
 }
 
 int zt_register(struct zt_span *span, int prefmaster)
@@ -1129,15 +1525,12 @@
 }
 
 
-
-
-
-
-
-
-
+#ifdef __FreeBSD__
 int 
-zt_poll(struct cdev *dev, int events, struct thread *t)
+zt_poll(struct cdev *dev, int events, zt_proc_t *t)
+#elif defined(__NetBSD__)
+int     zt_poll(struct file *dev, int events, zt_proc_t *t)
+#endif
 {
 	int unit = zt_get_minor(dev);
 	struct zt_chan * chan;
@@ -1150,13 +1543,13 @@
 		return zt_timer_poll(dev, events, t);
 		
 	if (unit == ZAP_CHANNEL_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+		chan = ZT_CHAN(dev);
 		if (!chan)
 			return 0;
 		return zt_chan_poll(dev, events, t, chan->channo);
 	}
 	if (unit == ZAP_PSEUDO_MINOR) {
-		chan = (struct zt_chan *)dev->si_drv1;
+		chan = ZT_CHAN(dev);
 		if (!chan) {
 			printf("No pseudo channel structure to read?\n");
 			return EINVAL;
@@ -1168,67 +1561,57 @@
 	return zt_chan_poll(dev, events, t, unit);
 }
 
+#ifdef __NetBSD__
+int zt_fcntl(struct file *fp, u_int cmd, void *data, zt_proc_t *p)
+{
+        int flags;
+
+        /* Take appropriate actions */
+        switch(cmd) {
+        case F_GETFL:
+                return(fp->f_flag);
+                break;
+        case F_SETFL:
+                flags = (u_int)data;
+                fp->f_flag = flags | FREAD | FWRITE;
+                return(0);
+                break;
+        default:
+                return(EINVAL);
+                break;
+        }
+
+        /* NOTREACHED */
+        return(0);
+}
+
+int zt_stat(struct file * fp, struct stat *sb, zt_proc_t *p)
+{
+        bzero(sb, sizeof(*sb));
+        sb->st_mode = S_IFCHR;
+        /*
+        * Left as 0: st_dev, st_ino, st_nlink, st_rdev, st_flags, st_gen.
+        * XXX (st_dev, st_ino) should be unique.
+        */
+        return (0);
+}
 
+#endif
 
 
 #ifdef __FreeBSD__
 static int
 ztdev_load(module_t mod, int cmd, void *arg)
-#else
-static int
-ztdev_load( lkmtp, cmd)
-struct lkm_table        *lkmtp;
-int                     cmd;
+#elif defined(__NetBSD__)
+static int ztdev_load(struct lkm_table *lkmtp, int cmd)
 #endif
 {
 	int  err = 0;
 
 	switch (cmd) {
 	case MOD_LOAD:
-		
-		zt_sysctl_init();
-		/* 
-		 * A kind of hack. Without any pre-created device 
-		 * zt_clone will get "zap" string as name instead of 
-		 * "zap/timer" or something
-		 */
-		dev_ctl = make_dev(&zt_devsw, 0, UID_ROOT, GID_WHEEL, 0664, 
-				   "zap/ctl");
-
-		printf("Zapata Telephony Interface Registered on major %d\n", 
-			   ZT_MAJOR);
-		printf("Echo Canceller: %s\n", ZAPTEL_ECHO_CANCELLER ZAPTEL_ECHO_AGGRESSIVE);
-
-		/* 
-		 * Note, that this is indirect initialization 
-		 * of unit_2_minor[0] to devtype 0 (ctl)
-		 */
-		bzero(zap_units_mask, sizeof(zap_units_mask));
-		bzero(unit_2_minor, sizeof(unit_2_minor));
-
-#ifdef __FreeBSD__
-		zt_devclass = devclass_find("zaptel");
-#endif
-		zt_conv_init();
-		tone_zone_init();
-		rotate_sums();
-
-		bzero(toselwakeup, sizeof(toselwakeup));
-
-		zt_timers_start();
-		zt_channels_start();
-		zt_tonezone_start();
-		ZAP_LOCK_INIT(toselwakeup_lock, "toselwakeup_lock");
-
-	        kthread_must_exit = 0;
-		if (kthread_create(selwakeup_thread_handler, (void *)toselwakeup, &kthread, 0, 0, "zt_selwakeup_thread"))
-		{
-			printf("Failed to create kthread\n");
-		}
-
-#ifdef CONFIG_ZAPTEL_WATCHDOG
-		watchdog_init();
-#endif	
+		/* broken out so it can be called at attach */
+		zt_init();
 		break;		  /* Success*/
 
 	case MOD_UNLOAD:
@@ -1238,12 +1621,14 @@
 			return EBUSY;
 		}
 
+#ifdef __FreeBSD__
                 if (kthread) {
                         kthread_must_exit = 1;
                         wakeup(kthread);
 			/* exit1(9) will wake us up */
                         tsleep((void *)kthread, PZERO, "waitthrd", 0);
                 }
+#endif
 
 		zt_sysctl_destroy();
 		printf("Unloaded kld character device driver\n");
@@ -1255,10 +1640,12 @@
 		zt_channels_stop();
 		zt_tonezone_stop();
 
+#ifdef __FreeBSD__
 		ZAP_LOCK_DESTROY(toselwakeup_lock);
 
 		if(dev_ctl)
 			destroy_dev(dev_ctl);
+#endif
 
 		break;		  /* Success*/
 
@@ -1270,6 +1657,7 @@
 	return(err);
 }
 
+#ifdef __FreeBSD__
 static void
 zt_drvinit(void * unused)
 {
@@ -1284,13 +1672,41 @@
 }
 
 
-
 /* Now declare the module to the system */
 
 SYSINIT(zt_drvinit, SI_SUB_DRIVERS, SI_ORDER_MIDDLE, zt_drvinit,NULL);
 SYSUNINIT(zt_drvuninit, SI_SUB_DRIVERS, SI_ORDER_MIDDLE, zt_drvuninit, NULL);
 
 DEV_MODULE(ztdev, ztdev_load, NULL);
+#elif defined(__NetBSD__)
+#ifdef _LKM
+#ifdef __OpenBSD__
+MOD_DEV("zaptel", LM_DT_CHAR, -1, &zt_devsw)
+
+/*
+ * Our external entry point. 
+ */
+
+int
+mod_zaptel(lkmtp, cmd, ver)
+        struct lkm_table *lkmtp;
+        int cmd;
+        int ver;
+#endif
+#ifdef __NetBSD__
+MOD_DEV("zaptel", "zaptel", NULL, -1, &zaptel_cdevsw, ZT_MAJOR)
+
+int mod_zaptel_lkmentry(struct lkm_table *lkmtp, int cmd, int ver);
+
+int
+mod_zaptel_lkmentry(struct lkm_table *lkmtp, int cmd, int ver) 
+#endif
+{
+	DISPATCH(lkmtp, cmd, ver, ztdev_load, ztdev_load, lkm_nofunc)
+}
+
+#endif /* _LKM */
+#endif
 
 #ifdef CONFIG_ZAPATA_NET
 MODULE_DEPEND (ztp, sppp, 1, 1, 1);
