/* $Id: fsu_ecp.c,v 1.9 2009/01/07 19:36:10 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/syslimits.h>
#include <sys/stat.h>

#include <dirent.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <rump/ukfs.h>

#include <fsu_mount.h>

#include "fsu_flist.h"

#define FSU_ECP_COPY_LINK (0x01)
#define FSU_ECP_RECURSIVE (FSU_ECP_COPY_LINK<<1)
#define FSU_ECP_VERBOSE (FSU_ECP_RECURSIVE<<1)
#define FSU_ECP_GET (FSU_ECP_VERBOSE<<1)
#define FSU_ECP_PUT (FSU_ECP_GET<<1)

DECLARE_UKFS(ukfs)

#ifdef FSU_CONSOLE
#define main(a, b) fsu_ecp_main(a, b)

int  fsu_ecp_main(int, char *[]);

static char *progname;
#endif /* !FSU_CONSOLE */

#define BUFSIZE (8192)

static int copy_dir(struct ukfs *, const char *, const char *, int);
static int copy_dir_rec(struct ukfs *, const char *, char *, int);
static int copy_fifo(struct ukfs *, const char *, const char *, int);
static int copy_file(struct ukfs *, const char *, const char *, int);
static int copy_link(struct ukfs *, const char *, const char *, int);
static int copy_special(struct ukfs *, const char *, const char *, int);
static int copy_to_dir(struct ukfs *, const char *, struct stat *,
		       const char *, int);
static int copy_to_file(struct ukfs *, const char *, struct stat *,
			const char *, int);
static int fsu_ecp(struct ukfs *, const char *, const char *, int);
static int fsu_ecp_parse_arg(int *, char ***);
static void usage(void);

int
main(int argc, char *argv[])
{
	size_t len;
	int cur_arg, flags, rv;

	FSU_MOUNT(argc, argv, ukfs);

	flags = fsu_ecp_parse_arg(&argc, &argv);
	if (flags == -1 || argc < 2) {
		usage();
		return -1;
	}

	for (rv = 0, cur_arg = 0; cur_arg < argc-1; ++cur_arg) {
		len = strlen(argv[cur_arg]);
		while (argv[cur_arg][--len] == '/')
			argv[cur_arg][len] = '\0';
		rv |= fsu_ecp(ukfs, argv[cur_arg], argv[argc-1], flags);
	}

	return rv;
}

static int
fsu_ecp_parse_arg(int *argc, char ***argv)
{
	int flags, rv;

	flags = 0;

#ifdef FSU_CONSOLE
	if (strcmp((*argv)[0], "get") == 0)
		flags |= FSU_ECP_GET;
	else if (strcmp((*argv)[0], "put") == 0)
		flags |= FSU_ECP_PUT;
#else
	if (strcmp((*argv)[0], "fsu_get") == 0)
		flags |= FSU_ECP_GET;
	else if (strcmp((*argv)[0], "fsu_put") == 0)
		flags |= FSU_ECP_PUT;
#endif

	while ((rv = getopt(*argc, *argv, "gPpRv")) != -1) {
		switch (rv) {
		case 'g':
			flags |= FSU_ECP_GET;
			flags &= ~FSU_ECP_PUT;
			break;
		case 'P':
			flags |= FSU_ECP_COPY_LINK;
			break;
		case 'p':
			flags |= FSU_ECP_PUT;
			flags &= ~FSU_ECP_GET;
			break;
		case 'R':
			flags |= FSU_ECP_RECURSIVE;
			break;
		case 'v':
			flags |= FSU_ECP_VERBOSE;
			break;
		case '?':
		default:
			return -1;
		}
	}
	*argc -= optind;
	*argv += optind;

	if ((flags & (FSU_ECP_GET | FSU_ECP_PUT)) == 0) {
		warnx("-g or -p should be specified");
		return -1;
	}

	return flags;
}

static int
fsu_ecp(struct ukfs *fs, const char *from, const char *to, int flags)
{
	struct stat from_stat, to_stat;
	int rv;
	
	if (flags & FSU_ECP_PUT)
		rv = lstat(from, &from_stat);
	else
		rv = ukfs_lstat(fs, from, &from_stat);
	if (rv == -1) {
		warn("%s", from);
		return -1;
	}

	if (S_ISDIR(from_stat.st_mode)) {
		if (!(flags & FSU_ECP_RECURSIVE)) {
			fprintf(stderr, "%s: is a directory\n", from);
			return -1;
		}
		return copy_dir(fs, from, to, flags);
	}

	if (flags & FSU_ECP_GET)
		rv = stat(to, &to_stat);
	else
		rv = ukfs_stat(fs, to, &to_stat);
	if (rv == 0 && S_ISDIR(to_stat.st_mode))
		return copy_to_dir(fs, from, &from_stat, to, flags);

	if (from_stat.st_ino == to_stat.st_ino) {
		fprintf(stderr, "%s and %s are identical (not copied).\n",
			from, to);
		return -1;
	}
	return copy_to_file(fs, from, &from_stat, to, flags);
}

static int
copy_dir(struct ukfs *fs, const char *from, const char *to, int flags)
{
	const char *filename;
	int res, rv;
	struct stat file_stat;
	char to_p[PATH_MAX + 1];
	size_t tlen, flen;

	tlen = strlen(to);
	rv = strlcpy(to_p, to, PATH_MAX + 1);
	if (rv != tlen) {
		warn("%s", to);
		return -1;
	}
	
	if (flags & FSU_ECP_GET)
		rv = lstat(to, &file_stat);
	else
		rv = ukfs_lstat(fs, to, &file_stat);
	if (rv == 0) {
		if (S_ISDIR(file_stat.st_mode)) {
			filename = strrchr(from, '/');
			if (filename == NULL)
				filename = from;
			else {
				printf("filename = \"%s\"\n", filename);
				if (filename[1] == '\0') {
					printf("brdl\n");
					for (rv = strlen(from) - 2;
					     rv >= 0 && from[rv] != '/';
					     --rv)
						continue;
				}
				++filename;
			}
			
			if (to_p[tlen - 1] != '/') {
				if (filename[0] != '/') {
					to_p[tlen] = '/';
					to_p[tlen + 1] = '\0';
				}
			} else
				--tlen;

			flen = strlen(filename);
			
			rv = strlcat(to_p, filename, PATH_MAX + 1);
			if (rv != flen + tlen + 1) {
				warn("%s/%s", to_p, filename);
				return -1;
			}
		} else {
			warnx("%s: not a directory", to);
			return -1;
		}
	}

	res = copy_dir_rec(fs, from, to_p, flags);

	return res;
}

static int
copy_dir_rec(struct ukfs *fs, const char *from_p, char *to_p, int flags)
{
	FSU_FENT *root, *cur;
	size_t len;
	int flist_options, res, rv, off;

	res = 0;

	if (flags & FSU_ECP_COPY_LINK)
		flist_options = FSU_FLIST_STATLINK;
	else
		flist_options = 0;

	if (flags & FSU_ECP_RECURSIVE)
		flist_options |= FSU_FLIST_RECURSIVE;

	if (flags & FSU_ECP_PUT)
		flist_options |= FSU_FLIST_REALFS;

	len = strlen(to_p) - 1;
	
	root = fsu_flist_build(fs, from_p, flist_options);

	if (root == NULL)
		return -1;
	
	if (flags & FSU_ECP_GET)
		rv = mkdir(to_p, root->sb.st_mode);
	else
		rv = ukfs_mkdir(fs, to_p, root->sb.st_mode);
	if (rv == -1) {
		warn("%s", to_p);
		goto out;
	}

	if (!(flags & FSU_ECP_GET)) {
		rv = ukfs_chown(fs, to_p, root->sb.st_uid, root->sb.st_gid);
		if (rv == -1) {
			warn("%s", from_p);
			goto out;
		}
	}

	off = root->pathlen;

	for (cur = root->next; cur != NULL; cur = cur->next) {
		
		rv = strlcat(to_p, cur->path + off, PATH_MAX+1);
		if (rv != len + cur->pathlen - off + 1) {
			warn("%s%s", to_p, cur->path + off);
			res = -1;
			break;
		}

		if (S_ISDIR(cur->sb.st_mode)) {
			if (flags & FSU_ECP_GET) {
				rv = mkdir(to_p, cur->sb.st_mode);
				if (rv == -1) {
					warn("%s", to_p);
					res = -1;
					break;
				}
			} else {
				rv = ukfs_mkdir(fs, to_p, cur->sb.st_mode);
				if (rv == -1) {
					warn("%s", to_p);
					res = -1;
					break;
				}
				
				rv = ukfs_chown(fs, to_p, cur->sb.st_uid,
						cur->sb.st_gid);
				if (rv == -1) {
					warn("%s", to_p);
					res = -1;
					break;
				}
			}
		} else {
			res |= copy_to_file(fs, cur->path, &(cur->sb), to_p,
					  flags);
			if (errno == ENOSPC) {
				warn(NULL);
				goto out;
			}
		}
		to_p[len + 1] = '\0';
	}
	
out:
	fsu_flist_free(root);

	return res;
}

static int
copy_to_dir(struct ukfs *fs, const char *from, struct stat *frstat,
	    const char *to, int flags)
{
	char path[PATH_MAX+1];
	const char *filename;
	int len;

	filename = strrchr(from, '/');
	if (filename == NULL)
		filename = from;
	else
		++filename;

	len = snprintf(path, PATH_MAX, "%s/%s", to, filename);
	path[len] = '\0';

	return copy_to_file(fs, from, frstat, path, flags);
}

static int
copy_to_file(struct ukfs *fs, const char *from, struct stat *frstat,
	     const char *to, int flags)
{
	int rv;

	switch ((frstat->st_mode & S_IFMT)) {
	case S_IFIFO:
		rv = copy_fifo(fs, from, to, flags);
		break;
	case S_IFLNK:
		if (flags & FSU_ECP_COPY_LINK)
			rv = copy_link(fs, from, to, flags);
		else
			rv = copy_file(fs, from, to, flags);
		break;
	case S_IFCHR: /* FALLTHROUGH */
	case S_IFBLK:
		rv = copy_special(fs, from, to, flags);
		break;
	case S_IFREG:
		rv = copy_file(fs, from, to, flags);
		break;
	default:
		return -1;
		/* NOTREACHED */
	}

	if (rv != 0)
		return -1;

	if (flags & FSU_ECP_GET)
		return rv;

	if (flags & FSU_ECP_COPY_LINK)
		rv = ukfs_lchown(fs, to, frstat->st_uid, frstat->st_gid);
	else
		rv = ukfs_chown(fs, to, frstat->st_uid, frstat->st_gid);
	if (rv == -1) {
		warn("%s", to);
		return -1;
	}

	return 0;
}

static int
copy_file(struct ukfs *fs, const char *from, const char *to, int flags)
{
	uint8_t buf[BUFSIZE];
	ssize_t rd, wr;
	off_t off;
	int fd, rv;
	struct stat from_stat;

	fd = -1;

	if (flags & FSU_ECP_VERBOSE)
		printf("%s -> %s\n", from, to);

	if (flags & FSU_ECP_PUT)
		rv = stat(from, &from_stat);
	else
		rv = ukfs_stat(fs, from, &from_stat);
	if (rv == -1) {
		warn("%s", from);
		return -1;
	}

	if (flags & FSU_ECP_GET) {
		rv = open(to, O_WRONLY|O_CREAT|O_EXCL, 0777);
		fd = rv;
	} else if (flags & FSU_ECP_PUT) {
		rv = open(from, O_RDONLY, from_stat.st_mode);
		fd = rv;
		rv = ukfs_create(fs, to, from_stat.st_mode);
	} else {
		rv = ukfs_create(fs, to, from_stat.st_mode);
	}
	if (rv == -1) {
		warn("%s", to);
		return -1;
	}
	
	off = 0;
	do {
		if (flags & FSU_ECP_PUT)
			rd = read(fd, buf, BUFSIZE);
		else
			rd = ukfs_read(fs, from, off, buf, BUFSIZE);
		if (rd == -1) {
			warn("%s", from);
			rv = -1;
			goto out;
		}
		if (flags & FSU_ECP_GET)
			wr = write(fd, buf, rd);
		else
			wr = ukfs_write(fs, to, off, buf, rd);
		if (wr == -1 || wr != rd) {
			warn("%s", to);
			rv = -1;
			goto out;
		}
		off += rd;
	} while (rd == BUFSIZE);

	rv = 0;
out:
	if (flags & (FSU_ECP_GET | FSU_ECP_PUT))
		close(fd);
	return rv;

}

static int
copy_fifo(struct ukfs *fs, const char *from, const char *to, int flags)
{
	int rv;
	struct stat file_stat;

	if (flags & FSU_ECP_VERBOSE)
		printf("Copying fifo %s -> %s\n", from, to);

	if (flags & FSU_ECP_GET)
		rv = lstat(to, &file_stat);
	else
		rv = ukfs_lstat(fs, to, &file_stat);
	if (rv != -1) {
		if (flags & FSU_ECP_GET)
			rv = remove(to);
		else
			rv = ukfs_remove(fs, to);
		if (rv == -1) {
			warn("%s", to);
			return -1;
		}
	}

	if (flags & FSU_ECP_PUT)
		rv = lstat(from, &file_stat);
	else
		rv = ukfs_lstat(fs, from, &file_stat);
	if (rv == -1) {
		warn("%s", from);
		return -1;
	}

	if (flags & FSU_ECP_GET)
		rv = mkfifo(to, file_stat.st_mode);
	else
		rv = ukfs_mkfifo(fs, to, file_stat.st_mode);
	if (rv == -1) {
		warn("%s", to);
		return -1;
	}
	
	return 0;
}

static int
copy_special(struct ukfs *fs, const char *from, const char *to, int flags)
{
	int rv;
	struct stat file_stat;

	if (flags & FSU_ECP_VERBOSE)
		printf("%s -> %s\n", from, to);

	if (flags & FSU_ECP_GET)
		rv = lstat(to, &file_stat);
	else
		rv = ukfs_lstat(fs, to, &file_stat);
	if (rv != -1) {
		rv = ukfs_remove(fs, to);
		if (rv == -1) {
			warn("%s", to);
			return -1;
		}
	}

	if (flags & FSU_ECP_PUT)
		rv = lstat(from, &file_stat);
	else
		rv = ukfs_lstat(fs, from, &file_stat);
	if (rv == -1) {
		warn("%s", from);
		return -1;
	}

	if (flags & FSU_ECP_GET)
		rv = mknod(to, file_stat.st_mode, file_stat.st_rdev);
	else
		rv = ukfs_mknod(fs, to, file_stat.st_mode, file_stat.st_rdev);
	if (rv == -1) {
		warn("%s", to);
		return -1;
	}
	
	return 0;
}

static int
copy_link(struct ukfs *fs, const char *from, const char *to, int flags)
{
	char target[PATH_MAX+1];
	int rv;
	struct stat file_stat;

	if (flags & FSU_ECP_PUT)
		rv = readlink(from, target, PATH_MAX);
	else
		rv = ukfs_readlink(fs, from, target, PATH_MAX);
	if (rv == -1) {
		warn("%s", from);
		return -1;
	}
	target[rv] = '\0';

	if (flags & FSU_ECP_VERBOSE)
		printf("%s -> %s : %s", from, to, target);

	if (flags & FSU_ECP_GET)
		rv = lstat(to, &file_stat);
	else
		rv = ukfs_lstat(fs, to, &file_stat);
	if (rv != -1) {
		if (flags & FSU_ECP_GET)
			rv = remove(to);
		else
			rv = ukfs_remove(fs, to);
		if (rv == -1) {
			warn("%s", to);
			return -1;
		}
	}

	if (flags & FSU_ECP_GET)
		rv = symlink(target, to);
	else
		rv = ukfs_symlink(fs, target, to);
	if (rv == -1) {
		warn("%s", target);
		return -1;
	}
	return 0;
}

static void
usage(void)
{

#ifdef FSU_CONSOLE
	fprintf(stderr, "usage: %s [-gPpRv] src target\n"
		"usage: %s [-gPpRv] src... directory\n", progname, progname);
#else
	fprintf(stderr,	"usage: %s %s [-gPpRv] src target\n"
		"usage: %s %s [-gPpRv] src... directory\n",
		getprogname(), fsu_mount_usage(), 
		getprogname(), fsu_mount_usage());

#endif
	exit(EXIT_FAILURE);
}
