/*	$Id: fsu_write.c,v 1.10 2009/01/07 19:36:10 stacktic Exp $	*/

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <fsu_mount.h>
#include <fsu_utils.h>

DECLARE_UKFS(ukfs)

#ifdef FSU_CONSOLE
#define main(a, b) fsu_write_main(a, b)

int fsu_write_main(int, char *[]);

static char *progname;
#endif /* !FSU_CONSOLE */

static void	usage(void);
int		fsu_write(struct ukfs *, int, const char *, int);

int
main(int argc, char *argv[])
{
	int append, rv;

#ifdef FSU_CONSOLE
	progname = argv[0];
#endif
	setprogname(argv[0]);

#ifdef USE_UKFS
	FSU_MOUNT(argc, argv, ukfs);
#endif
	
	append = 0;
	while ((rv = getopt(argc, argv, "a")) != -1) {
		switch(rv) {
		case 'a':
			append = 1;
			break;

		case '?':
		default:
			usage();
			/* NOTREACHED */
		}
	}

	if (optind >= argc)
		usage();

	rv = fsu_write(ukfs, STDIN_FILENO, argv[optind], append);

	return rv != 0;
}

int
fsu_write(struct ukfs *fs, int fd, const char *fname, int append)
{
	int rd, wr;
	size_t total;
	uint8_t buf[8192];
	FSU_FILE *fdout;

	if (fname == NULL)
		return -1;

	fdout = fsu_fopen(fs, fname, append ? "a" : "w");
	if (fdout == NULL) {
		warn("%s", fname);
		return -1;
	}

	total = 0;
	do {
		rd = read(fd, buf, sizeof(buf));
		if (rd == -1) {
			warn("read");
			goto out;
		}
		wr = fsu_fwrite(buf, 1, rd, fdout);
		if (wr == -1 || wr != rd) {
			warn("%s", fname);
			goto out;
		}
		total += wr;
	} while (rd > 0 && errno != ENOSPC);

out:
	fsu_fclose(fdout);
	return total;
}

static void
usage(void)
{

#ifdef FSU_CONSOLE
	fprintf(stderr, "usage: %s [-a] file\n", progname);
#else
	fprintf(stderr, "usage: %s %s [-a] file\n",
		getprogname(), fsu_mount_usage());
#endif
	exit(EXIT_FAILURE);
}
